/* snprintf.c */

/*
 * int snprintf (char *str, size_t n, const char *fmt, ...)
 *	like sprintf, but copy at most `n' characters into `str', including the
 *	trailing NULL
 *
 * Return the number of bytes that would be required to hold the entire
 * formatted result, or -1 on error.
 *
 * This implementation of snprintf is hacked from the 4.4 BSD C Stdio Library,
 * written by Chris Torek. The original copyright notice is reproduced below.
 *
 * It differs from the 4.4 BSD version in the following ways
 *    (1) it does NOT support floating point types (!)
 *    (2) it does NOT support "quads" (long long)
 *    (3) if n==0 || str==NULL it returns the number of bytes required to
 *        hold the formatted result
 *
 * If you would like a version of snprintf with floating point support, see the
 * the GNU C Library, the GNU Libio Library, the 4.4 BSD C Library, or the
 * Korn/Vo SFIO Library. All these libraries are publicly available.
 */

/*-
 * Copyright (c) 1990, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Chris Torek.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <limits.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <sys/types.h>

#ifndef NULL
#define	NULL	0
#endif

#ifndef MIN
#define	MIN(a, b)	((a) < (b) ? (a) : (b))
#endif

#undef u_int
#undef u_long
#undef u_short
#define u_int		unsigned int
#define u_long		unsigned long
#define u_short		unsigned short

typedef	struct {
	unsigned char	*_p;	/* current position in (some) buffer */
	int		_w;	/* write space left */
} xFILE;

/*
 * I/O descriptors for __sfvwrite().
 */

struct __siov {
	void	*iov_base;
	size_t	iov_len;
};

struct __suio {
	struct	__siov *uio_iov;
	int	uio_iovcnt;
	int	uio_resid;
};

/*
 * Write some memory regions.
 */
static void
__sfvwrite (fp, uio)
	register xFILE *fp;
	register struct __suio *uio;
{
	register size_t len;
	register char *p;
	register struct __siov *iov;
	register int w;

	if ((len = uio->uio_resid) == 0) return;

	iov = uio->uio_iov;
	p = iov->iov_base;
	len = iov->iov_len;
	iov++;

	/*
	 * Write as many bytes as fit, but pretend we wrote everything.
	 * This makes snprintf() return the number of bytes needed, rather
	 * than the number used.
	 */
	do {
		while (len == 0) {
			p = iov->iov_base;
			len = iov->iov_len;
			iov++;
		}
		w = MIN (fp->_w, len); /* copy MIN (fp->_w, len) */
		(void) memcpy ((void *) fp->_p, (void *) p, (size_t) w);
		fp->_w -= w;
		fp->_p += w;
		w = len;	/* but pretend copied all */
		p += w;
		len -= w;
	} while ((uio->uio_resid -= w) != 0);
}

/*
 * Flush out all the vectors defined by the given uio,
 * then reset it so that it can be reused.
 */
static void
__sprint(fp, uio)
	xFILE *fp;
	register struct __suio *uio;
{
	if (uio->uio_resid == 0) {
		uio->uio_iovcnt = 0;
		return;
	}
	__sfvwrite(fp, uio);
	uio->uio_resid = 0;
	uio->uio_iovcnt = 0;
}

/*
 * Macros for converting digits to letters and vice versa
 */
#define	to_digit(c)	((c) - '0')
#define is_digit(c)	((unsigned) to_digit (c) <= 9)
#define	to_char(n)	((n) + '0')

/*
 * Convert an unsigned long to ASCII for printf purposes, returning
 * a pointer to the first character of the string representation.
 * Octal numbers can be forced to have a leading zero; hex numbers
 * use the given digits.
 */
static char *
__ultoa(val, endp, base, octzero, xdigs)
	register u_long val;
	char *endp;
	int base, octzero;
	char *xdigs;
{
	register char *cp = endp;
	register long sval;

	/*
	 * Handle the three cases separately, in the hope of getting
	 * better/faster code.
	 */
	switch (base) {
	case 10:
		if (val < 10) {	/* many numbers are 1 digit */
			*--cp = to_char(val);
			return (cp);
		}
		/*
		 * On many machines, unsigned arithmetic is harder than
		 * signed arithmetic, so we do at most one unsigned mod and
		 * divide; this is sufficient to reduce the range of
		 * the incoming value to where signed arithmetic works.
		 */
		if (val > LONG_MAX) {
			*--cp = to_char (val % 10);
			sval = val / 10;
		} else
			sval = val;
		do {
			*--cp = to_char (sval % 10);
			sval /= 10;
		} while (sval != 0);
		break;

	case 8:
		do {
			*--cp = to_char (val & 7);
			val >>= 3;
		} while (val);
		if (octzero && *cp != '0')
			*--cp = '0';
		break;

	case 16:
		do {
			*--cp = xdigs[val & 15];
			val >>= 4;
		} while (val);
		break;

	default:			/* oops */
		abort ();
	}
	return (cp);
}

#define	BUF		68

/*
 * Flags used during conversion.
 */
#define	ALT		0x001		/* alternate form */
#define	HEXPREFIX	0x002		/* add 0x or 0X prefix */
#define	LADJUST		0x004		/* left adjustment */
#define	LONGDBL		0x008		/* long double; unimplemented */
#define	LONGINT		0x010		/* long integer */
#define	SHORTINT	0x020		/* short integer */
#define	ZEROPAD		0x040		/* zero (as opposed to blank) pad */

/*
 * MUST be declared static so that it doesn't interfere with the native C
 * Library vfprintf, if there is one.
 */

static int
vfprintf(fp, fmt0, ap)
	xFILE *fp;
	const char *fmt0;
	va_list ap;
{
	register char *fmt;	/* format string */
	register int ch;	/* character from fmt */
	register int n;		/* handy integer (short term usage) */
	register char *cp;	/* handy char pointer (short term usage) */
	register struct __siov *iovp;/* for PRINT macro */
	register int flags;	/* flags as above */
	int ret;		/* return value accumulator */
	int width;		/* width from format (%8d), or 0 */
	int prec;		/* precision from format (%.3d), or -1 */
	char sign;		/* sign prefix (' ', '+', '-', or \0) */
	u_long	ulval;		/* integer arguments %[diouxX] */
	int base;		/* base for [diouxX] conversion */
	int dprec;		/* a copy of prec if [diouxX], 0 otherwise */
	int fieldsz;		/* field size expanded by sign, etc */
	int realsz;		/* field size expanded by dprec */
	int size;		/* size of converted field or string */
	char *xdigs = 0;	/* digits for [xX] conversion */
#define NIOV 8
	struct __suio uio;	/* output information: summary */
	struct __siov iov[NIOV];/* ... and individual io vectors */
	char buf[BUF];		/* space for %c, %[diouxX], %[eEfgG] */
	char ox[2];		/* space for 0x hex-prefix */

	/*
	 * Choose PADSIZE to trade efficiency vs. size.  If larger printf
	 * fields occur frequently, increase PADSIZE and make the initialisers
	 * below longer.
	 */

#define	PADSIZE	16		/* pad chunk size */

	static char blanks[PADSIZE] =
	 {' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',' '};

	static char zeroes[PADSIZE] =
	 {'0','0','0','0','0','0','0','0','0','0','0','0','0','0','0','0'};

#define	PRINT(ptr, len) {				\
	iovp->iov_base = (ptr);				\
	iovp->iov_len = (len);				\
	uio.uio_resid += (len);				\
	iovp++;						\
	if (++uio.uio_iovcnt >= NIOV) {			\
		__sprint (fp, &uio);			\
		iovp = iov;				\
	}						\
}

#define	PAD(howmany, with) {				\
	if ((n = (howmany)) > 0) {			\
		while (n > PADSIZE) {			\
			PRINT(with, PADSIZE);		\
			n -= PADSIZE;			\
		}					\
		PRINT(with, n);				\
	}						\
}

#define	FLUSH() {					\
	if (uio.uio_resid)				\
		__sprint(fp, &uio);			\
	uio.uio_iovcnt = 0;				\
	iovp = iov;					\
}

	/*
	 * To extend shorts properly, we need both signed and unsigned
	 * argument extraction methods.
	 */

#define	SARG()								\
	(flags & LONGINT ? va_arg (ap, long) :				\
	    flags & SHORTINT ? (long) (short) va_arg (ap, int) :	\
	    (long) va_arg (ap, int))

#define	UARG()								\
	(flags & LONGINT ? va_arg (ap, u_long) :			\
	    flags & SHORTINT ? (u_long) (u_short) va_arg (ap, int) :	\
	    (u_long) va_arg (ap, u_int))

	fmt = (char *) fmt0;
	uio.uio_iov = iovp = iov;
	uio.uio_resid = 0;
	uio.uio_iovcnt = 0;
	ret = 0;

	/*
	 * Scan the format for conversions (`%' character).
	 */
	for (;;) {
		for (cp = fmt; (ch = *fmt) != '\0' && ch != '%'; fmt++)
			/* void */;
		if ((n = fmt - cp) != 0) {
			PRINT(cp, n);
			ret += n;
		}
		if (ch == '\0')
			goto done;
		fmt++;		/* skip over '%' */

		flags = 0;
		dprec = 0;
		width = 0;
		prec = -1;
		sign = '\0';

rflag:		ch = *fmt++;
reswitch:	switch (ch) {
		case ' ':
			/*
			 * ``If the space and + flags both appear, the space
			 * flag will be ignored.''
			 *	-- ANSI X3J11
			 */
			if (!sign)
				sign = ' ';
			goto rflag;
		case '#':
			flags |= ALT;
			goto rflag;
		case '*':
			/*
			 * ``A negative field width argument is taken as a
			 * - flag followed by a positive field width.''
			 *	-- ANSI X3J11
			 * They don't exclude field widths read from args.
			 */
			if ((width = va_arg (ap, int)) >= 0)
				goto rflag;
			width = -width;
			/* FALLTHROUGH */
		case '-':
			flags |= LADJUST;
			goto rflag;
		case '+':
			sign = '+';
			goto rflag;
		case '.':
			if ((ch = *fmt++) == '*') {
				n = va_arg (ap, int);
				prec = n < 0 ? -1 : n;
				goto rflag;
			}
			n = 0;
			while (is_digit (ch)) {
				n = 10 * n + to_digit (ch);
				ch = *fmt++;
			}
			prec = n < 0 ? -1 : n;
			goto reswitch;
		case '0':
			/*
			 * ``Note that 0 is taken as a flag, not as the
			 * beginning of a field width.''
			 *	-- ANSI X3J11
			 */
			flags |= ZEROPAD;
			goto rflag;
		case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9':
			n = 0;
			do {
				n = 10 * n + to_digit (ch);
				ch = *fmt++;
			} while (is_digit (ch));
			width = n;
			goto reswitch;
		case 'h':
			flags |= SHORTINT;
			goto rflag;
		case 'l':
			flags |= LONGINT;
			goto rflag;
		case 'c':
			*(cp = buf) = va_arg (ap, int);
			size = 1;
			sign = '\0';
			break;
		case 'D':
			flags |= LONGINT;
			/*FALLTHROUGH*/
		case 'd':
		case 'i':
			ulval = SARG();
			if ((long) ulval < 0) {
				ulval = -ulval;
				sign = '-';
			}
			base = 10;
			goto number;
		case 'n':
			if (flags & LONGINT)
				*va_arg (ap, long *) = ret;
			else if (flags & SHORTINT)
				*va_arg (ap, short *) = ret;
			else
				*va_arg (ap, int *) = ret;
			continue;	/* no output */
		case 'O':
			flags |= LONGINT;
			/*FALLTHROUGH*/
		case 'o':
			ulval = UARG ();
			base = 8;
			goto nosign;
		case 'p':
			/*
			 * ``The argument shall be a pointer to void.  The
			 * value of the pointer is converted to a sequence
			 * of printable characters, in an implementation-
			 * defined manner.''
			 *	-- ANSI X3J11
			 */
			ulval = (u_long) va_arg (ap, void *);
			base = 16;
			xdigs = "0123456789abcdef";
			flags |= HEXPREFIX;
			ch = 'x';
			goto nosign;
		case 's':
			if ((cp = va_arg (ap, char *)) == NULL)
				cp = "(null)";
			if (prec >= 0) {
				/*
				 * can't use strlen; can only look for the
				 * NUL in the first `prec' characters, and
				 * strlen() will go further.
				 */
				char *p = memchr (cp, 0, prec);

				if (p != NULL) {
					size = p - cp;
					if (size > prec)
						size = prec;
				} else
					size = prec;
			} else
				size = strlen (cp);
			sign = '\0';
			break;
		case 'U':
			flags |= LONGINT;
			/*FALLTHROUGH*/
		case 'u':
			ulval = UARG ();
			base = 10;
			goto nosign;
		case 'X':
			xdigs = "0123456789ABCDEF";
			goto hex;
		case 'x':
			xdigs = "0123456789abcdef";
hex:			ulval = UARG ();
			base = 16;
			/* leading 0x/X only if non-zero */
			if ((flags & ALT) && ulval != 0)
				flags |= HEXPREFIX;

			/* unsigned conversions */
nosign:			sign = '\0';
			/*
			 * ``... diouXx conversions ... if a precision is
			 * specified, the 0 flag will be ignored.''
			 *	-- ANSI X3J11
			 */
number:			if ((dprec = prec) >= 0)
				flags &= ~ZEROPAD;

			/*
			 * ``The result of converting a zero value with an
			 * explicit precision of zero is no characters.''
			 *	-- ANSI X3J11
			 */
			cp = buf + BUF;
			if (ulval != 0 || prec != 0)
				cp = __ultoa(ulval, cp, base,
				   		 flags & ALT, xdigs);
			size = buf + BUF - cp;
			break;
		default:	/* "%?" prints ?, unless ? is NUL */
			if (ch == '\0')
				goto done;
			/* pretend it was %c with argument ch */
			cp = buf;
			*cp = ch;
			size = 1;
			sign = '\0';
			break;
		}

		/*
		 * All reasonable formats wind up here.  At this point, `cp'
		 * points to a string which (if not flags&LADJUST) should be
		 * padded out to `width' places.  If flags&ZEROPAD, it should
		 * first be prefixed by any sign or other prefix; otherwise,
		 * it should be blank padded before the prefix is emitted.
		 * After any left-hand padding and prefixing, emit zeroes
		 * required by a decimal [diouxX] precision, then print the
		 * string proper, then emit zeroes required by any leftover
		 * floating precision; finally, if LADJUST, pad with blanks.
		 *
		 * Compute actual size, so we know how much to pad.
		 * fieldsz excludes decimal prec; realsz includes it.
		 */
		fieldsz = size;
		if (sign)
			fieldsz++;
		else if (flags & HEXPREFIX)
			fieldsz += 2;
		realsz = dprec > fieldsz ? dprec : fieldsz;

		/* right-adjusting blank padding */
		if ((flags & (LADJUST|ZEROPAD)) == 0)
			PAD (width - realsz, blanks);

		/* prefix */
		if (sign) {
			PRINT (&sign, 1);
		} else if (flags & HEXPREFIX) {
			ox[0] = '0';
			ox[1] = ch;
			PRINT(ox, 2);
		}

		/* right-adjusting zero padding */
		if ((flags & (LADJUST|ZEROPAD)) == ZEROPAD)
			PAD (width - realsz, zeroes);

		/* leading zeroes from decimal precision */
		PAD (dprec - fieldsz, zeroes);

		/* the string or number proper */
		PRINT (cp, size);

		/* left-adjusting padding (always blank) */
		if (flags & LADJUST)
			PAD (width - realsz, blanks);

		/* finally, adjust ret */
		ret += width > realsz ? width : realsz;

		FLUSH ();	/* copy out the I/O vectors */
	}
done:
	FLUSH ();
	return ret;
}

int
vsnprintf (char *str, size_t n, const char *fmt, va_list ap)
{
	int ret;
	xFILE f;

	if (n==0 || !str) { /* just compute required number of bytes */
		str = 0;
		n = 1;
	}
	else if ((int) n < 1) return -1;

	f._p = (unsigned char *) str;
	f._w = n-1;
	ret = vfprintf (&f, fmt, ap);
	if (str) *f._p = 0;
	return ret;
}

int
snprintf (char *str, size_t n, char const *fmt, ...)
{
	int ret;
	va_list ap;

	va_start (ap, fmt);
	ret = vsnprintf (str, n, fmt, ap);
	va_end (ap);
	return ret;
}
