/*
    brs_collect : Browsing an Analysis Tool for C-source code;
	          Collects browser-files and create control-file
                  

    Copyright (C) 1994  Eckehard Stolz

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation (version 2 of the License).

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

%{



   #include <stdio.h>
   #include <stdlib.h>
   #include <string.h>
   #include <unistd.h>
   #include <errno.h>
   #include <time.h>
   
   #include "c-bat.h"
   #include "hash.h"

   #define TYPE_BRS       1
   #define TYPE_BRP       2
   #define TYPE_BRC       3
   #define TYPE_BRI       4
   #define TYPE_FILE     10
   #define TYPE_MACRO    11
   #define TYPE_REMOVE   12
   
   typedef struct brs_info  t_brs_info;
   
   struct  brs_info {
      char  *file;
      char  *dir;
      short type;
      t_brs_info *next;
     };
   
   
   static char   browser_file[MAX_FILENAME];
   static char   browser_dir[MAX_FILENAME];
   static char   working_dir[MAX_FILENAME];
   static char   output_file[MAX_FILENAME] = "all.brc";
   static char   input_file[MAX_FILENAME];
   static char   act_macro[MAX_FILENAME];
   static char   act_file[MAX_FILENAME];
   
   static t_brs_info   *brs_list = NULL;
   static t_brs_info   *act_brs_list = NULL;
   static short         brs_type;
   static short         sys_include_flag;


   static long   lineno = 1;
   static long   macro_line;
   
   /* All symbols, which have to be printed, are stored in the hashtable
      and aditionally stored in this array (by their hash-indizes).
      So we don't have to check the whole hash-table for nonnull-entries
    */
   static long *hash_array = NULL;
   static long hash_array_cnt = 0;
   static long hash_array_size = 0;
   
   typedef struct hash_add_data  t_hash_add;

   struct hash_add_data
     { long          file;  /* File, where Function is defined (hash-index) */
       long          line;  /* Line, where Function is defined              */
       t_hash_add   *next;  /* next entry (for files)                       */
      };
   
   void Add_to_browser_list( char * brs_file, char *brs_dir, short type);
   long Add_macro_to_hash(char *ident, char *file, long line, int sys_flag );
   void dump_macros( FILE *fp, t_hash_data *file);
   void Add_to_Hash_Index_Array(long index);
%}

DIGIT   [0-9]
ID      [a-zA-Z_][a-zA-Z_0-9]*

INT     {DIGIT}*
FILEN   [^\"<>]*
WHITE   [\t ]*

%s control control2 includes definition definition2 definition3 bri_file bri_def

%%

          

<bri_file>"[I0]\""{FILEN}"\""	{ /* read include-filename */
        strcpy( browser_file, yytext + 5 );
        browser_file[yyleng-6] = 0;
        sys_include_flag = 0;
        }

<bri_file>"[I1]\""{FILEN}"\""	{ /* read include-filename */
        strcpy( browser_file, yytext + 5 );
        browser_file[yyleng-6] = 0;
        sys_include_flag = 1;
        }

<bri_file>"[I1]\"<predefined>\""	{ /* read include-filename */
        strcpy( browser_file, yytext + 5 );
        browser_file[yyleng-6] = 0;
        sys_include_flag = 1;
        }

<bri_file>"[EOF0]"	{ /* end of .bri-file */
        return;
        }

<bri_file>"[P]["{INT}     { /* definition */
	macro_line = atoi( yytext + 4 );

	BEGIN( bri_def);
	}
   
<bri_def>"]\""{ID}"\""    { /* macro definition */
	strcpy( act_macro, yytext + 2 );
	act_macro[yyleng-3] = 0;

        Add_macro_to_hash( act_macro,  browser_file, macro_line, 
             sys_include_flag);
     
	BEGIN( bri_file );
	}
   
<control>"\""{FILEN}"\""	 {  /* name of the browser-file */
        BEGIN( control2 );
        strcpy( browser_file, yytext + 1 );
        browser_file[yyleng-2] = 0;
        if( strcmp( browser_file + yyleng - 5, "brs" ) )
          brs_type = TYPE_BRS;
        else if( strcmp( browser_file + yyleng - 5, "brp" ) )
          brs_type = TYPE_BRP;
        else if( strcmp( browser_file + yyleng - 5, "brc" ) )
          brs_type = TYPE_BRC;
        }

<control2>{WHITE}	 {     /* eat up whitespaces */
        ;
        }

<control2>"\""{FILEN}"\""	{ /* read working dir */
        BEGIN( control );
        strcpy( browser_dir, yytext + 1 );
        browser_dir[yyleng-2] = 0;
        Add_to_browser_list( browser_file, browser_dir, brs_type);
        }

<control>\n	{
        lineno++;
        }

<includes>"[P]["{INT}     { /* definition */
	macro_line = atoi( yytext + 4 );

	BEGIN(definition);
	}
   

<definition>"]\""{ID}"\""    { /* macro definition */
	strcpy( act_macro, yytext + 2 );
	act_macro[yyleng-3] = 0;
     
	BEGIN( definition2 );
	}

<definition2>"\""{FILEN}"\""    { /* file, where macro defined */
	strcpy( act_file, yytext + 1 );
	act_file[yyleng-2] = 0;
     
	BEGIN( definition3 );
	}

<definition2>"\"<predefined>\""    { /* predefined macro: maybe later
                                        a special treatment  */
	strcpy( act_file, yytext + 1 );
	act_file[yyleng-2] = 0;
     
	BEGIN( definition3 );
	}

<definition3>"n"|"s"    { /* normal-/system-include */
     
        Add_macro_to_hash( act_macro,  act_file, macro_line, 
             (yytext[0] == 's' ) ? 1 : 0 );

	BEGIN( includes );
	}

<includes,control,INITIAL,bri_file>\n	{
   	lineno++;
 	}

"[browser_control]"\n      {
	BEGIN( control );
        lineno++;
	}

"[I1]\"\""	{ /* begin of an .bri-file */
   	BEGIN( bri_file );
 	}

"[P]["{INT}     { /* definition */
	macro_line = atoi( yytext + 4 );

	BEGIN(definition);
	}

.	{ /* syntax error in .brp-file */
   	fprintf(stderr, "Syntax error in .brp-file: '%s' \n", yytext );
        }

%%


/* This function add's a macro (and the correspondng include file)
   to the symbol-table. 
 */
long Add_macro_to_hash(char *ident, char *file, long line, int sys_flag )
{ long index, f_index;
  t_hash_data  h_entry, f_entry;
  t_hash_add   *next_entry;


  index = search_token( ident, NULL );
  if (index != TOKEN_NOT_FOUND )
    return( TOKEN_NOT_FOUND ); /* macro allready in hash */
   
  f_index = search_token( file, &f_entry );
  if( f_index == TOKEN_NOT_FOUND )
   { /* create new entry for the include-file
      */
     strncpy( f_entry.ident, file, SZE_HASH_IDENT - 1 );
     f_entry.ident[SZE_HASH_IDENT - 1] = 0;
     f_entry.type  = TYPE_FILE;
     f_entry.flags = sys_flag;
     f_entry.data  = NULL;
     next_entry = (t_hash_add *)malloc( sizeof(t_hash_add) );
     if( !next_entry )
       { fprintf( stderr, "Not enough memory !\n" );
         exit(1);
        }
     f_index = insert_token( &f_entry );
     Add_to_Hash_Index_Array( f_index);
     next_entry->next = NULL;
    }
  else
   { /* add the define to the list
      */
     next_entry = (t_hash_add *)malloc( sizeof(t_hash_add) );
     if( !next_entry )
       { fprintf( stderr, "Not enough memory !\n" );
         exit(1);
        }
      
     next_entry->next = f_entry.data;
    }

  f_entry.data = next_entry;
  update_token( &f_entry, f_index);
  strncpy( h_entry.ident, ident, SZE_HASH_IDENT - 1 );
  h_entry.ident[SZE_HASH_IDENT - 1] = 0;
  h_entry.type  = TYPE_MACRO;
  h_entry.flags = 0;
  h_entry.data  = (t_hash_add *)malloc( sizeof(t_hash_add) );
  if( !h_entry.data )
    { fprintf( stderr, "Not enough memory !\n" );
      exit(1);
     }
  ((t_hash_add *)(h_entry.data))->line = line;
  ((t_hash_add *)(h_entry.data))->file = f_index;
  index = insert_token( &h_entry );
   
  next_entry->file = index;
  Add_to_Hash_Index_Array( index);
  return( index );
}

/* This function writes a complete include-file and the 
   macro-definitions to the file *fp
 */
void dump_macros( FILE *fp, t_hash_data *file)
{ t_hash_add  *ptr;
  t_hash_data h_entry;
   
  ptr = file->data;
   
  fprintf( fp, "[I%d]\"%s\"\n", file->flags, file->ident);
   
  while( ptr )
   { get_token(&h_entry, ptr->file );
     fprintf( fp, "[P][%ld]\"%s\"\n", 
          ((t_hash_add *)(h_entry.data))->line, h_entry.ident );
     ptr = ptr->next;
    }
 }
     
   
/* Add the last saved hash-ident to the hash-array-list
 */
void Add_to_Hash_Index_Array(long index)
{

  hash_array[hash_array_cnt] = index;
  hash_array_cnt++;

  if( hash_array_cnt == hash_array_size)
    { hash_array_size += HASH_ARRAY_INIT_SIZE;
      hash_array = realloc( hash_array,
		       sizeof( long ) * hash_array_size );

      if( !hash_array )
	{ fprintf(stderr, "Error ! Cannot allocate %ld Bytes of Memory\n",
		   sizeof( long ) * hash_array_size );
	  exit(1);
	 }
     }
 }


/* add actual browser-file and wd to list of
   browser-files
 */
void Add_to_browser_list( char * brs_file, char *brs_dir, short type)
{ char        *ptr;
  t_brs_info  *info_ptr;
   
  if( !brs_list )
   { brs_list = malloc( sizeof( t_brs_info ) );
     if( !brs_list )
       { fprintf(stderr, "Not enough memory !\n" );
         exit(1);
        }
     act_brs_list = brs_list;
    }
   
  act_brs_list->file = strdup( brs_file );
  act_brs_list->dir  = strdup( brs_dir  );
  act_brs_list->type = type;
  act_brs_list->next = malloc( sizeof( t_brs_info ) );
  if( !act_brs_list->next )
    { fprintf(stderr, "Not enough memory !\n" );
      exit(1);
     }
  act_brs_list = act_brs_list->next;
  /* reset file-entry of new created entry
   */
  act_brs_list->file = NULL;
}
   
/* This function stores the file in the hash-table with the
   type "TYPE_REMOVE", so it will be removed after creation
   of the *.bri file
 */
void Store_file_for_remove( char *input_file )
{ t_hash_data   f_entry;
  long          f_index;
   
  strncpy( f_entry.ident, input_file, SZE_HASH_IDENT - 1 );
  f_entry.ident[SZE_HASH_IDENT - 1] = 0;
  f_entry.type  = TYPE_REMOVE;
  f_entry.flags = 0;
  f_entry.data  = NULL;
   
  f_index = insert_token( &f_entry );
  Add_to_Hash_Index_Array( f_index);
 }
  

main( int argc, char *argv[] )
{ short        i, flag,
               first = 1;
  char         *ptr,
               *p_slash,
               *extension,
               *p_dot;
  long         cnt;
  FILE         *fp;
  t_hash_data  h_data;

   if( !getcwd( working_dir, MAX_FILENAME ) )
     { fprintf( stderr, "Working-directory too long !\n" );
       exit(1);
      }
   
   if( argc <= 1 || argv[1][1] == '?' )
     { fprintf(stderr, "Usage: brs_collect [Options] <file1> <file2> ...\n" );
       fprintf(stderr, "Collects all given browser-files in that directory and creates a \n");
       fprintf(stderr, "control-file (default-name: 'all.brc' ) \n");
       fprintf(stderr, "\nOptions:\n");
       fprintf(stderr, "Overall Options:\n");
       fprintf(stderr, "-o <file>    specify name for output control-file\n");
       fprintf(stderr, "\n");
       fprintf(stderr, "<file?> can be one of the following:\n");
       fprintf(stderr, "*.brs     ...  explicitly given browser-file\n");
       fprintf(stderr, "*.brp     ...  explicitly given browser preprocessor-file\n");
       fprintf(stderr, "*.brc     ...  control-file (might be in a different directory)\n");
       fprintf(stderr, "*.o, *.c  ...  takes basename and check's for *.brs and *.brc\n");
       exit(1);
      }

   hash_array      = malloc(sizeof(long) * HASH_ARRAY_INIT_SIZE);
   if( !hash_array )
     { fprintf(stderr, "Error ! Cannot allocate %ld Bytes of Memory\n",
		       sizeof( long ) * HASH_ARRAY_INIT_SIZE );
       exit( 1 );
      }
   hash_array_size = HASH_ARRAY_INIT_SIZE;
   hash_array_cnt  = 0;
   
   if( first )
      { init_hashtable ();
        first = 0;
       }
   
   yyin = stdin;
   
   for( i = 1; i < argc; i++ )
     { /* ordinary option given: */
       if( argv[i][0] == '-' )
        { switch( argv[i][1] )
             { case 'o': if( argv[i][2] )
                           strcpy( output_file, argv[i] + 2); 
                         else
                           { /* next argument */
                             i++;
                             strcpy( output_file, argv[i] ); 
                            }
               
                         if( strcmp( output_file + strlen(output_file) - 4,
                                ".brc" ) )
                           strcat( output_file, ".brc" );
                         break;
               default:
	          fprintf(stderr, "Unknown Option \"%s\" !\n", argv[i] );
	          fprintf(stderr, "Type \"brs_collect -?\" for help !\n" );
	          exit(1);
              }
	  }
       else
         { /* normal file */ 
           strcpy( input_file, argv[i] );
           p_slash = strchr( input_file, '/' );
           p_dot   = strrchr( input_file, '.' );
           if( p_dot )
            { *p_dot = 0;
              extension = p_dot + 1;
             }
           else
              extension = "";
         
           if( !strcmp( extension, "brc" ) )
             { /* control-file - scan this file via lex
                  and continue processing
                */
               yyin = fopen( argv[i], "r");
               if( !yyin )
                 { fprintf(stderr, "Unable to open '%s' !\n", argv[i] );
                   exit(1);
                  }
               lineno = 1;
               yylex();
               yyrestart(yyin);
               fclose( yyin );
               continue;
              }
         
           if( p_slash )
             { fprintf(stderr, "No path's allowed ('%s') !\n", argv[i] );
               exit(1);
              }
         
           if( !strcmp( extension, "o" ) ||
                    !strcmp( extension, "c" )    )
             { /* .o or .c file - create .brs and .bri files
                */
               strcat( input_file, ".brs" );
               fp = fopen( input_file, "r" );
               if( !fp )
                 { fprintf(stderr, "Unable to open '%s' !\n", input_file );
                   exit(1);
                  }
               fclose( fp );
            
               Add_to_browser_list( input_file, working_dir, TYPE_BRS);

               /* Now try to check for .brp-file
                */
               *p_dot = 0;  
               strcat( input_file, ".brp" );
               yyin = fopen( input_file, "r" );
               if( yyin )
                { lineno = 1;
                  yylex();
                  yyrestart(yyin);
                  fclose( yyin );
                  Store_file_for_remove( input_file );
                 }

              }
           else if( !strcmp( extension, "brs" ) )
             { /* .brs-file ! do look for *.brp-file
                */
               fp = fopen( argv[i], "r" );
               if( !fp )
                 { fprintf(stderr, "Unable to open '%s' !\n", argv[i] );
                   exit(1);
                  }
               fclose( fp );
            
               Add_to_browser_list( argv[i], working_dir, TYPE_BRS);
              }
           else if( !strcmp( extension, "brp" ) )
             { /* .brp-file ! do look for .brs-file
                */
               strcat( argv[i], ".brp" );
               yyin = fopen( input_file, "r" );
               if( !yyin )
                 { fprintf(stderr, "Unable to open '%s' !\n", argv[i] );
                   exit(1);
                  }
            
               lineno = 1;
               yylex();
               yyrestart(yyin);
               fclose( yyin );
            
               Store_file_for_remove( input_file );
            
               /* Now store this file in the hashtable so we can remove it
                  after we dumped the new *.bri-file
                */
              }
           else
             { /* unknown extension
                */
               fprintf(stderr, "Unknown extension of file: '%s' !\n", argv[i] );
               exit(1);
              }
          }
      }

   if( hash_array_cnt )
     { /* if we have found #define-macros, we have to create a collected
          file with all definitions
        */
       ptr = strdup( output_file );
       strcpy( ptr + strlen( ptr ) - 4, ".bri" );
       Add_to_browser_list( ptr, working_dir, TYPE_BRI);

       yyin = fopen( ptr, "r" );
       if( yyin )
         { /* there is an existing .bri-file ! We scan this file
              to have the definition of macro's defined in files
              which have not been recompiled. Because the 
              definitions of the new compiled files have been
              scanned first, only not processed macro definitions
              will be added
            */
           lineno = 1;
           yylex();
           yyrestart(yyin);
           fclose( yyin );
          }
      }

   /* Now we have to create the new browser-control-file
    */
   
   fp = fopen( output_file, "w" );
   
   if( !fp )
     { fprintf( stderr, "Cannot open output-file '%s' !\n", output_file);
       exit(1);
      }
   
   fprintf( fp, "[browser_control]\n" );

   act_brs_list = brs_list;

   while( act_brs_list->file )
     { fprintf( fp, "\"%s\"     \"%s\"\n", act_brs_list->file,
                    act_brs_list->dir );
       act_brs_list = act_brs_list->next;
      }
   
   fclose( fp );

   if( hash_array_cnt )
     { /* Now we have to create to collected file with defines
        */
       strcpy( output_file + strlen(output_file) - 4, ".bri" );

       fp = fopen( output_file, "w" );
       if( !fp )
         exit(1);

       fprintf(fp, "[I1]\"\"\n" ); /* illegal file: ignore */
       for( i = 0; i < hash_array_cnt; i++ )
         { get_token( &h_data , hash_array[i] );

           if( h_data.type == TYPE_FILE )
             dump_macros( fp, &h_data );
           else   if( h_data.type == TYPE_REMOVE )
             { if( remove( h_data.ident ) )
                 fprintf( stderr, "Cannot remove file '%s' !\n", h_data.ident );
              }
          }
       fprintf( fp, "[EOF0]\n" );
       fclose( fp );
      
      }

  return( 0 );
}
