/* Signals stubs.
   This file is part of Checker.
   Copyright 1994 Tristan Gingold
		  Written January 1994 by Tristan Gingold

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License 
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 The author may be reached (Email) at the address gingold@amoco.saclay.cea.fr,
 or (US/French mail) as Tristan Gingold 
   			  8 rue Parmentier
   			  F-91120 PALAISEAU
   			  FRANCE
*/
#include "checker.h"
#include "errlist.h"
#include <signal.h>

#ifndef MDCHECKER

/* This function deliver a signal and does not return. */
void _sig_jump_to(void *, void *);

/* This function saves the signal arguments and resumes.  It will be delivered
 *  later. */
void _sig_save(int nsignal, STRUCT_SIG_CONTEXT *context, void *func, 
          unsigned int mask);

/* Contains the action of signals. */
static struct sigaction sig_tab[NSIGNALS+1];

/* True if we are in Checker: the program is checking a memory access.
 * Set and clear by codecheck.S */
int chkr_in_checker = 0;

/* The really used handler.
 * It calls your handler after changing the rights of the stack.
 * When this functions is called, all signals are blocked. */
static void 
chkr_sig_handler(int nsignal, STRUCT_SIG_CONTEXT context)
{
 if (chkr_in_checker)
   {
     /* The processus has received a signal, but it must not be delivered
      * now, since we are in Checker and Checker is not recursive. So, we
      * have to store all informations and deliver the signal as soon as
      * possible.
      */
     _sig_save(nsignal, &context, sig_tab[nsignal].sa_handler, 
               sig_tab[nsignal].sa_mask);
     /* Must never be reached. */
     chkr_abort();
   }
   
 /* Set the right of the stack for the informations given by the OS. */
 known_stack_limit = &nsignal;
 chkr_set_right(&nsignal, sizeof(int), CHKR_RW);
 chkr_set_right(&context, sizeof(STRUCT_SIG_CONTEXT), CHKR_RO);
 
 /* Restore sa_mask. */
 sigprocmask(SIG_SETMASK, &(sig_tab[nsignal].sa_mask), 0);
 /* Deliver the signal. This function must call sigreturn.*/
 _sig_jump_to(sig_tab[nsignal].sa_handler, &nsignal);
}

/*
 * Save signal before calling main().
 */
void
save_signals(void)
{
 int i;
 struct sigaction t;
 
 for (i=1; i < NSIGNALS; i++)
   {
     chkr_sigaction(i, (struct sigaction*)0, &sig_tab[i]);
     if (i != SIGSTOP && i != SIGKILL 
         && sig_tab[i].sa_handler != SIG_DFL 
         && sig_tab[i].sa_handler != SIG_IGN)
       {
         /* Set the Checker handler. */
         t = sig_tab[i];
         t.sa_handler = (SIG_HANDLER_T)chkr_sig_handler;
         /* All signals are blocked. */
         t.sa_mask = ~0L;
         chkr_sigaction(i, &t, (struct sigaction*)0);
       }
   }
}

/*
 * The new sigaction syscall.
 */
int
__sigaction(int sig, struct sigaction *act, struct sigaction *oldact)
{
  int ret;
  struct sigaction action;

  /* Check the arguments are readable. */
  chkr_check_addr(&sig, sizeof(int), CHKR_RO);
  chkr_check_addr(&act, sizeof(struct sigaction*), CHKR_RO);
  chkr_check_addr(&oldact, sizeof(struct sigaction*), CHKR_RO);
    
  /* ACT is optional and is a pointer. */
  if (act)
#ifdef linux
    /* There is an usused and shadow argument in linux. */
    chkr_check_addr(act, sizeof(struct sigaction) - sizeof(int), CHKR_RO);
#else
    chkr_check_addr(act, sizeof(struct sigaction), CHKR_RO);
#endif
  /* OLDACT is optional... */
  if (oldact)
    chkr_check_addr(oldact, sizeof(struct sigaction), CHKR_WO);
    
  /* These signals can't be catched or nothing is changed. */
  if (sig == SIGSTOP || sig == SIGKILL || act == (struct sigaction*)0)
    return chkr_sigaction(sig, act, oldact);
    
  if (act->sa_handler == SIG_DFL || act->sa_handler == SIG_IGN)
    {
      /* These signals are not delivered anymore. */
      ret = chkr_sigaction(sig, act, oldact);
      if (oldact)
        {
          /* Must be filled with the user value saved and not with what
           *  Checker has used. */
          oldact->sa_handler = sig_tab[sig].sa_handler;
          oldact->sa_mask = sig_tab[sig].sa_mask;
        }
      return ret;
    }
   
  action = *act;
  /* Use the handler of Checker. */
  action.sa_handler = (SIG_HANDLER_T)chkr_sig_handler;
  /* Mask all signals. */
  action.sa_mask = ~0L;	
  ret = chkr_sigaction(sig, &action, oldact);
  if (oldact)
    {
      /* Must be filled with the saved value. */
      oldact->sa_handler = sig_tab[sig].sa_handler;
      oldact->sa_mask = sig_tab[sig].sa_mask;
    }
  sig_tab[sig] = *act;
  return ret;
}

#endif /* !MDCHECKER */