/* free(3) implementation
   Copyright 1993, 1994 Tristan Gingold
		  Written December 1993 by Tristan Gingold
		
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 The author may be reached (Email) at the address gingold@amoco.saclay.cea.fr,
 or (US/French mail) as Tristan Gingold
   			  8 rue Parmentier
   			  F91120 PALAISEAU
   			  FRANCE
*/
#define _MALLOC_INTERNAL
#include "malloc.h"

#include "errlist.h"
#include "message.h"

/* Number of aged block kept in memory */
unsigned int _block_age = DEFAULT_BLOCK_AGE;

/* Subroutine of free
 * _internal_free is also called by malloc
 */
void
_internal_free (struct mdesc *mdp, struct malloc_header *block)
{
  struct malloc_header *tmp;
  size_t size;
  int l_size;

  size = block->size;

  /* Can we free memory ? */
  if (mdp->_lastblock == block && size >= FINAL_FREE_SIZE)
    {
#ifndef CHKR_IS_SAFE
      if (block->next)
	chkr_printf (M_BAD_NLINK_4BLOCK, block);
#endif /* CHKR_IS_SAFE */
      /* Updates mdp->_lastblock */
      mdp->_lastblock = block->prev;
      if (mdp->_lastblock)
	mdp->_lastblock->next = NULL_HEADER;
      /* Is there only one block ? */
      if (mdp->_firstblock == block)
	mdp->_firstblock = NULL_HEADER;
      /* Free memory */
      mdp->morecore (mdp, -(size + HEADER_SIZE));
      return;			/* That's all */
    }

  /* compute the log size */
  l_size = log_size (size);

  block->state = MDFREE;
#ifndef NO_HEAPINDEX
  /* Update _heapindex, the biggest size of free blocks available */
  if (l_size > mdp->_heapindex)
    mdp->_heapindex = l_size;
#endif /* NO_HEAPINDEX */

  tmp = mdp->_heapinfo[l_size];
  /* The list of free blocks must be sorted by size
   * It is the smallest free block of the list ?
   * This can be the case if there was no block in the list */
  if (!tmp || tmp->size < size)
    {
      /* put it at the begin of the list */
      block->info.free.next = tmp;
      block->info.free.prev = NULL_HEADER;
      /* Set the 'prev' link of the next block, if necessary */
      if (tmp)
	tmp->info.free.prev = block;
      mdp->_heapinfo[l_size] = block;
    }
  else
    {
      /* Find the good slot */
      while (tmp->info.free.next && tmp->info.free.next->size > size)
	tmp = tmp->info.free.next;
      /* put it after tmp */
      block->info.free.prev = tmp;
      block->info.free.next = tmp->info.free.next;
      /* Set the 'prev' link of the next block */
      if (tmp->info.free.next)
	tmp->info.free.next->info.free.prev = block;
      tmp->info.free.next = block;
    }
}

/* The well-known free function */
void
_free (struct mdesc *mdp, PTR ptr)
{
  struct malloc_header *tmp;
  struct malloc_header *tmp1;
  int l_size;

  /* ANSI-C test */
  if (ptr == (PTR) 0)
    return;

  mdp = MD_TO_MDP (mdp);
  
  /* Compute the address of the header from ptr */
  tmp = (struct malloc_header *) (ptr - HEADER_SIZE - be_red_zone);
  
  /* Check if the address is good */
  if (tmp != find_header (mdp, ptr, 1))
    {
      chkr_perror (M_M_FBA_FR_ET);
      return;
    }
  /* A little check: the block must be busy, to be freed */
  if (tmp->state != MDBUSY)
    {
      chkr_perror (M_M_FFB_FR_ET);
      return;
    }

#ifdef CHKR_HEAPBITMAP
  /* Disable reading and writing right in the block */
  chkr_set_right ((PTR) tmp + HEADER_SIZE + be_red_zone,
		  tmp->info.busy.real_size, CHKR_UN);
#endif /* CHKR_HEAPBITMAP */

#ifndef NO_AGE
  /* save the stack */
#ifdef CHKR_SAVESTACK
  chkr_save_stack (		/* save the stack */
		    (PTR *) ((u_int) tmp + HEADER_SIZE),
		    0,		/* number of frame to forget. with 0, show malloc */
		    (tmp->size - af_red_zone) / sizeof (void *));	/* number of frames to save */
#endif /* CHKR_SAVESTACK */
  /* Put it in the aged list */
  tmp->info.aged.next = mdp->_youngerblock;
  tmp->info.aged.prev = NULL_HEADER;
  tmp->state = MDAGED;
  if (mdp->_youngerblock)
    mdp->_youngerblock->info.aged.prev = tmp;
  mdp->_youngerblock = tmp;
  if (mdp->_olderblock == NULL_HEADER)
    mdp->_olderblock = tmp;
  if (mdp->_agedblock >= _block_age)
    {
      /* 'tmp' is now the older block, which must be freed */
      tmp = mdp->_olderblock;
      if (tmp->state != MDAGED)
	chkr_printf (M_FOUND_AGED_BLOCK, tmp);
      mdp->_olderblock = tmp->info.aged.prev;
      mdp->_olderblock->info.aged.next = NULL_HEADER;
    }
  else
    {
      /* No too old block */
      mdp->_agedblock++;
      return;
    }
#endif /* NO_AGE */

  /* try to coalise this block with its next one */
  if ((tmp1 = tmp->next) && tmp->next->state == MDFREE)
    {
      l_size = log_size (tmp1->size);
      if (mdp->_heapinfo[l_size] == tmp1)
	mdp->_heapinfo[l_size] = tmp1->info.free.next;	/* BUG: _heapindex */
      if (tmp1->info.free.next)
	tmp1->info.free.next->info.free.prev = tmp1->info.free.prev;
      if (tmp1->info.free.prev)
	tmp1->info.free.prev->info.free.next = tmp1->info.free.next;
      tmp->size += tmp1->size + HEADER_SIZE;
      tmp->next = tmp1->next;
      if (tmp->next)
	tmp->next->prev = tmp;
      if (mdp->_lastblock == tmp1)
	mdp->_lastblock = tmp;
    }

  /* try to coalise this block with its predecessor */
  if ((tmp1 = tmp->prev) && tmp->prev->state == MDFREE)
    {
      l_size = log_size (tmp1->size);
      if (mdp->_heapinfo[l_size] == tmp1)
	mdp->_heapinfo[l_size] = tmp1->info.free.next;	/* BUG: _heapindex */
      if (tmp1->info.free.next)
	tmp1->info.free.next->info.free.prev = tmp1->info.free.prev;
      if (tmp1->info.free.prev)
	tmp1->info.free.prev->info.free.next = tmp1->info.free.next;
      tmp1->size += tmp->size + HEADER_SIZE;
      tmp1->next = tmp->next;
      if (tmp->next)
	tmp->next->prev = tmp1;
      if (mdp->_lastblock == tmp)
	mdp->_lastblock = tmp1;
      tmp = tmp1;
    }

  _internal_free (mdp, tmp);
}

/* The well-known free function */
void
free (PTR ptr)
{
#ifdef CHKR_STACKBITMAP
  chkr_check_addr (&ptr, sizeof (PTR), CHKR_RO);
#endif

  if (!__malloc_initialized)	/* Imagine you call free before malloc */
    {
      chkr_perror (M_M_FBM_FR_ET);
      return;
    }

  _free (NULL_MDESC, ptr);
}

void
mfree (struct mdesc *mdp, PTR ptr)
{
#ifdef CHKR_STACKBITMAP
  chkr_check_addr (&mdp, sizeof (mdp), CHKR_RO);
  chkr_check_addr (&ptr, sizeof (PTR), CHKR_RO);
#endif

  if (!__malloc_initialized)	/* Imagine you call free before malloc */
    {
      chkr_perror (M_M_FBM_FR_ET);
      return;
    }

  _free (mdp, ptr);
}
