/* gnu.a.out.c Functions to handle the symbol table of a a.out exec file.
   from nlist.c and objdump.c Modifications by T.Gingold */

/* Copyright (C) 1991 Free Software Foundation, Inc.
This file is part of the GNU C Library.

The GNU C Library is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

The GNU C Library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with the GNU C Library; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#define NEED_MM    
#include <nlist.h>
#include <stab.h>
#include <fcntl.h>
#include "checker.h"
#include "machine.h"
#include "message.h"

#ifdef CHKR_SAVESTACK
/* The prototype of the file, used by mktemp(). */
static char symtabfileproto[] = TMP_FILE;

/* This variable contains the temp file where the symbol table is stored. */
static char *symtabfile;

/* Size of the temp file. NULL if symtab has been never loaded.
   1 if not saved, 2 if error when loading */
static unsigned int symtabfilesize;

/* Number of symbols in the temp file. */
static size_t nsymbols;

/* A pointer to the symbol table when loaded in memory. */
static struct nlist *symbols;

/* A pointer to the string table when loaded in memory. */
static char *string_table;

/* The header structure of the executable file. */
static struct exec my_header;

/* 0: not loaded; >= 1: loaded */
int symtab_available = 0;

/* True if no symtab available.  Set by parse-args.c */
extern int nosymtab_flag;

/* Return true if the symbol is useful
   useful symbols are:
        N_SLINE		line number in text segment
        N_SO		name of main source file
        N_SOL		name of source file
        N_FUNC		function name
 */
/*#define GOODSYM(x) ( ((x)& ~N_EXT) == N_TEXT || ((x)& ~0x20) == 0x44 )*/
#define GOODSYM(x) ((x) == N_SO || (x) == N_SOL || (x) == N_SLINE || (x) == N_FUN)

/* Function called at exit, to remove the temp file. */
void
chkr_remove_symtabfile(void)
{
  if (symtabfilesize > 2)
  {
    unlink(symtabfile);
    symtabfilesize = 0;		/* don't try to load it */
  }
}

/* Test if eheader is right.
 */
static int
is_good_image(struct exec eheader)
{
  /* Check that this is really an executable:
     The magic number must be valid and
     there must be no relocation infos (on linux) */
  if( N_BADMAG(eheader) || eheader.a_trsize || eheader.a_drsize )
    return 0;
     
#define PAGE_ROUND(a) (((unsigned)(a) + PAGESIZE -1) & ~(PAGE_SIZE - 1))     

  /* Check that this is not a bad file */
  switch(N_MAGIC(eheader))
    {
      case ZMAGIC:
      case NMAGIC:
        if (eheader.a_text != (unsigned)(objects->text_end)
     	    || eheader.a_text + eheader.a_data != PAGE_ROUND(objects->data_end)
     	    || eheader.a_text + eheader.a_data + eheader.a_bss != PAGE_ROUND(objects->bss_end))
     	  return 0;
     	break;
      case OMAGIC:
        if (eheader.a_text != (unsigned)(objects->text_end)
            || eheader.a_text + eheader.a_data != (unsigned)(objects->data_end)
            || eheader.a_text + eheader.a_data + eheader.a_bss != (unsigned)(objects->bss_end))
          return 0;
        break;
      case QMAGIC:
        /* Really easy! (I like it !!) */
        if (memcmp(&eheader, (struct exec*)PAGESIZE, sizeof(struct exec)) != 0)
          return 0;
        break;
      default:
        return 0;
    }
  /* seems OK */
  return 1;
}

/* Prepare the symtab file: create it.
 * All unuseful symbols are forgotten. */
static void
prepare_symtab_file(void)
{
 int fd;
 struct nlist *sym, *sym1;
 char *string_table1;	/* second string table */
 unsigned long int string_table_size;	/* size of the string table */
 int  str_index=0;
 int i;

 if (symtabfilesize < 2)
 {
   /* Symtab never loaded or symtab temp file not saved */
   symtabfile = mktemp(symtabfileproto);	/* make the temp file name */
   fd = open(chkr_prog_path, O_RDONLY);		/* open the binary file */
   if (fd == -1)
     goto error;

   /* Go to the symbol table. */
   if (lseek(fd, N_SYMOFF(my_header), 0) < 0)
     goto error;
     
   /* Allocate (stack) memory for loading the symbols. */
   symbols = (struct nlist *) __alloca(my_header.a_syms);
   nsymbols = my_header.a_syms / sizeof(symbols[0]);
   
   /* Load the symbols table. */
   if (read(fd, (__ptr_t) symbols, sizeof(symbols[0]) * nsymbols) != 
 		sizeof(symbols[0])*nsymbols)
     goto error;
  
   /* Load the size of the strings table. */
   if (read(fd, (__ptr_t) &string_table_size, sizeof(string_table_size)) 
   				!= sizeof(string_table_size))
     goto error;
     
   /* Allocate space for the strings. */
   string_table_size -= sizeof(string_table_size);
   string_table = (char *) __alloca(string_table_size);
   
   /* This second string table will contain a relocated and shorter copy 
        of the first string table */
   string_table1 = (char*) __alloca(string_table_size + 30); /* security */
   
   /* Load the strings */ 
   if (read(fd, (__ptr_t) string_table, string_table_size)
        != string_table_size)
      goto error;
     
   /* Prepare the second string table */
#define BAD_STR M_BAD_STR
   string_table1[str_index++] = '\0';	
   strcpy(&string_table1[str_index], BAD_STR);
   str_index += strlen(BAD_STR) + 1;	/* +1 is the '\0' */
   
   /* Shift the symtab: forget unuseful symbols. The second string table
        is initialised */
   for (sym = symbols, sym1 = symbols; sym < &symbols[nsymbols]; sym++ )
     if (GOODSYM(sym->n_type))
       {
         *sym1 = *sym;
         /* Taken from objdump.c (binutils 1.9) */
         if (sym1->n_un.n_strx < 0 || sym1->n_un.n_strx > string_table_size) 
       	   sym1->n_un.n_strx = 1;		/* 1 is BAD_STR */
         else
           {
             char *str = &string_table1[str_index];
             /* copy the string. FIXME: try to see if it already exits. */
             strcpy(str, &string_table[ sym1->n_un.n_strx 
           		- sizeof(string_table_size) ]);
             sym1->n_un.n_strx = str_index;
             str_index += strlen(&string_table1[str_index]) + 1;
           } 
         sym1++;
       }
   nsymbols = sym1 - symbols;
     
   close(fd);		/* all is right */
 
   /* relocation */
   for(i=0; i < nsymbols; i++)
     symbols[i].n_un.n_name = (char*)(MM_SYM + nsymbols * sizeof(symbols[0]) 
       			        + symbols[i].n_un.n_strx);
       			        
   /* Write the symtab to the temp file */
   fd = open(symtabfile, O_WRONLY|O_CREAT, 0600);
   if (fd != -1)
     {
       /* write the symbol table */
       if (write(fd, symbols, nsymbols * sizeof(symbols[0])) !=
           nsymbols * sizeof(symbols[0]))
         goto error_bis;
       /* write the string table */
       if (write(fd, string_table1, str_index) != str_index)
         goto error_bis;
       /* write symtabfilesize */
       if (write(fd, &symtabfilesize, sizeof(int)) != sizeof(int))
         goto error_bis;
       /* write nsymbols */
       if (write(fd, &nsymbols, sizeof(int)) != sizeof(int))
         goto error_bis;
       if (close(fd) == -1)
         goto error_bis;
       symtabfilesize = nsymbols * sizeof(symbols[0]) + str_index;
#if 0         
       /* This file will be removed at the end */
       atexit(chkr_remove_symtabfile);
#endif /* 0 */         
     }
   else
     goto error;
 }
 symbols = (struct nlist*)0;
 return;
   error:	/* in case of error */
 symtabfilesize = 2;	/* don't try again to load symbol table */
 symbols = (struct nlist*)0;
 return;
   error_bis:
 unlink(symtabfile);	/* error after creating: remove the file */
 symtabfilesize = 2;	/* try again next time */
 symbols = (struct nlist*)0;
 return;
}

/* Dup the symtab file.
 * Used at fork time
 */
void
fork_symtab_file(void)
{
 char tmp_file[] = TMP_FILE;
 char *tmpf;
 
 if (symtabfilesize < 2)
   return;	/* nothing to do */
 tmpf = mktemp(tmp_file);
 
 link(symtabfile, tmpf);
 strcpy(symtabfileproto, tmpf);
}

/* prepare exec(2).
 * Must be OK to recreate the file.
 */
void
exec_symtab_file(void)
{
  symtabfilesize = 0;
  symtab_available = 0;
}

/* Load the symbols in memory */
void
chkr_load_symtab(void)
{
 int fd;
 
 if (nosymtab_flag || (++symtab_available) > 1) /* no symtab or already loaded */
   return;
   
 if (symtabfilesize < 2)
   prepare_symtab_file();
 if ( symtabfilesize > 2)
   {
     /* load into memory the symbol table */
     fd = open(symtabfile, O_RDONLY);
     if (fd == -1)
       goto error;
     symbols = (struct nlist*)(mmap((char*)MM_SYM, symtabfilesize, PROT_READ,
               MAP_FIXED | MAP_FILE | MAP_PRIVATE, fd, 0));
     string_table = (char*)&symbols[nsymbols];
     if (symbols == (struct nlist*)MM_SYM)
       symtab_available = 1;
     else
       symtab_available = 0;
     close(fd);
     error:
     if (symtab_available == 0)
       symtabfilesize = 2;	/* don't try again */
   }
 else
   symtab_available = 0;
 return;
}

/* Unload the symbols */
void
chkr_unload_symtab(void)
{
 if (symtab_available)
   {
     symbols = (struct nlist*)0;
     munmap((char*)MM_SYM, symtabfilesize);
     symtab_available--;
   }
}

/* NOTE: This comes from bfd/aoutx.h */
void
chkr_show_addr(__ptr_t *ptr)
{
  /* Run down the file looking for the filename, function and linenumber */
  char buffer[100];
  char filename_buffer[200];
  uint line = 0;
  char *functionname = NULL;
  char *directory_name = NULL;
  char *main_file_name = NULL;
  char *current_file_name = NULL;
  char *line_file_name = NULL; /* Value of current_file_name at line number. */
  ulong low_line_vma = 0;
  ulong low_func_vma = 0;
  struct nlist *func = 0;
  struct nlist *p;
  
  if (symtab_available == 0)
    {
      chkr_printf(M_PC__NOSYMTAB, *ptr);
      return;
    }
  filename_buffer[0] = '\0';
  for (p = symbols; p < &symbols[nsymbols]; p++)
    {
    next:
      switch (p->n_type)
      {
      case N_SO:
	main_file_name = current_file_name = p->n_un.n_name;
	/* Look ahead to next symbol to check if that too is an N_SO. */
	p++;
	if (p == &symbols[nsymbols])
	  break;
	if (p->n_type != (int)N_SO)
	  goto next;

	/* Found a second N_SO  First is directory; second is filename. */
	directory_name = current_file_name;
	main_file_name = current_file_name = p->n_un.n_name;
	break;
      case N_SOL:
	current_file_name = p->n_un.n_name;
	break;
      case N_SLINE:
	/* We'll keep this if it resolves nearer than the one we have already */
	if (p->n_value >= low_line_vma && p->n_value <= (ulong)*ptr)
	  {
	    line = p->n_desc;
	    low_line_vma = p->n_value;
	    line_file_name = current_file_name;
	  }
	break;
      case N_FUN:
	{
	  /* We'll keep this if it is nearer than the one we have already */
	  if (p->n_value >= low_func_vma && p->n_value <= (ulong)*ptr)
	    {
	      low_func_vma = p->n_value;
	      func = p;
	    }
	}
	break;
      }
  }

  if (line)
    main_file_name = line_file_name;
  if (func)
  {
      char *function = func->n_un.n_name;
      char *p;
      strncpy(buffer, function, sizeof(buffer)-1);
      buffer[sizeof(buffer)-1] = 0;
      /* Have to remove : stuff */
      p = strchr(buffer, ':');
      if (p != NULL) 
	  *p = '\0';
      functionname = buffer;
  }
#if 0
  if (main_file_name)
    {
      if (main_file_name[0] == '/' || directory_name == NULL)
        strncpy(filename_buffer, main_file_name, 150);
      else
        {
          strncpy(filename_buffer, directory_name, 140);
          strncat(filename_buffer, main_file_name, 50);
        }
    }
#else /* !0 */
  strncpy(filename_buffer, main_file_name, 150);
#endif /* !0 */

  chkr_printf(M_PC___IN___AT_,
  	*ptr, /* pc or ip */
  	functionname ? functionname : M_UNKNOWN, /* function name */ 
    	filename_buffer[0] ? filename_buffer : M_UNKNOWN, /* file name */
    	line);
}

/* Show all the symbol table.  */
void
chkr_dump_symtab(void)
{
  int i;
  struct nlist *sp;

  if (symtab_available == 0)
    return;
  /* From objdump.c (binutils 1.9) */
  chkr_printf("%3s: %4s %5s %4s %8s",
	  "#", "type", "other", "desc", "val");
  for (i = 0, sp = symbols; i < nsymbols; i++, sp++)
    {
      chkr_printf("%3d: %4x %5x %4x %8x %s\n",
	      i,
	      sp->n_type & 0xff,
	      sp->n_other & 0xff,
	      sp->n_desc & 0xffff,
	      sp->n_value,
	      sp->n_un.n_name);
    }
}

/* Initialize the objects structure: seek etext, edata, end.
 * If LINKED is true, ETEXT are good for the program
 * eg: if checker is a module, ETEXT is for checker, not for your program.
 */
void
init_main_object(int linked, char *argv0, int nlibs, char **libs)
{
  struct exec hdr;
  char *path;
  int fd, i;
  
  objects = (struct object*)sys_malloc((nlibs + 1) * sizeof(struct object));
  
  /* The main program */
  objects[0].name = chkr_prog_path ? chkr_prog_path : "Main program";
  path = chkr_prog_path ? chkr_prog_path : "/proc/self/exe";
  fd = open(path, O_RDONLY);
  if (fd == -1)
    {
      chkr_printf(M_CANT_OPEN_BINARY, path);
      chkr_abort();
    }
  i = read(fd, &my_header, sizeof(struct exec));
  if (i != sizeof(struct exec) || (linked && !is_good_image(my_header)) )
    {
       chkr_header(M_CANT_F_GOOD_INFO);
       chkr_abort();
    }
  close(fd);
  objects[0].text_org = my_header.a_entry;
  objects[0].text_beg = my_header.a_entry & ~(PAGESIZE - 1);
  objects[0].text_end = objects[0].text_beg + my_header.a_text;
  objects[0].data_beg = objects[0].text_end;
  objects[0].data_end = objects[0].data_beg + my_header.a_data;
  objects[0].bss_end =  objects[0].data_end + my_header.a_bss;
  objects[0].next = (struct object*)0;
  
#ifndef MDCHECKER
   /* Perhaps the text segment is writable. */
   if (N_MAGIC(my_header) == OMAGIC)
     is_text_writable = 1;	/* can write in the text segment */
#endif /* MDCHECKER */  
  
  /* The libraries */
  for (i = 0; i < nlibs; i++)
    {
     /* Link it */
     objects[i].next = &objects[i+1];
     objects[i+1].next = (struct object*)0;
     
     /* The name */
     objects[i+1].name = (char*)sys_malloc(strlen(libs[i]) + 1);
     strcpy(objects[i+1].name, libs[i]);
     
     /* Get info from the lib. It can't fail. */
     fd = open(libs[i], O_RDONLY);
     read(fd, &hdr, sizeof(struct exec));
     objects[i+1].text_org = hdr.a_entry;
     objects[i+1].text_beg = hdr.a_entry & ~(PAGESIZE - 1);
     objects[i+1].text_end = objects[i+1].text_beg + hdr.a_text;
     objects[i+1].data_beg = objects[i+1].text_end;
     objects[i+1].data_end = objects[i+1].data_beg + hdr.a_data;
     objects[i+1].bss_end = objects[i+1].data_end + hdr.a_bss;
     close(fd);
   }
}
  
#endif /* CHKR_SAVESTACK */
