//
// Qt Sample Application: desktop.cpp
//
// Demonstrates that the desktop can be painted, just like any other
// widget.
//

#include <qimage.h>
#include <qbitmap.h>
#include <qpainter.h>
#include <qapp.h>
#include <stdio.h>

static double seed = 0.353535353535;
static const int KINDA_RAND_MAX = 32767;

static int kindaRand()
{
    seed = seed*147;
    seed = seed - (double) ((int) seed);
    return (int) ( seed*(KINDA_RAND_MAX + 1) );
}

static int velocity( int i )			// change velocity
{
    const velmax = 15;
    const velmin = 4;
    if ( i == 1 || i == 2 )
	i = (kindaRand()&0x7fff % velmax)/3 + velmin;
    else
	i = (kindaRand()&0x7fff % velmax) + velmin;
    return i;
}

//
// Draw polygon on desktop.
//

void poly()
{
    QWidget *d = QApplication::desktop();
    d->setBackgroundColor( white );		// white desktop

    const int maxpoints = 5;
    const int maxcurves = 8;
    static int xvel[maxpoints];
    static int yvel[maxpoints];
    int head = 0;
    int tail = -maxcurves + 2;
    QPointArray *a = new QPointArray[ maxcurves ];
    register QPointArray *p;
    QRect r = d->rect();			// desktop rectangle

    int i;
    for ( i=0; i<maxcurves; i++ )
	a[i].resize( maxpoints );
    p = &a[0];
    for ( i=0; i<maxpoints; i++ ) {		// setup first polygon points
	p->setPoint( i, (kindaRand()&0x7fff) % r.width(),
		        (kindaRand()&0x7fff) % r.height() );
	xvel[i] = velocity(i);
	yvel[i] = velocity(i);
    }

    QPainter paint;
    paint.begin( d );				// start painting desktop

    for ( int ntimes=0; ntimes<2000; ntimes++ ) {
	QColor c;
	c.setHSV( kindaRand()%360, 180, 255 );	// random color
	paint.setBrush( c );
	paint.drawPolygon( a[head] );
	if ( ++tail >= maxcurves )
	    tail = 0;

	int minx=r.left(), maxx=r.right();
	int miny=r.top(),  maxy=r.bottom();
	int x, y;
	p = &a[head];
	if ( ++head >= maxcurves )
	    head = 0;
	for ( int i=0; i<maxpoints; i++ ) {	// calc new curve
	    p->point( i, &x, &y );
	    x += xvel[i];
	    y += yvel[i];
	    if ( x >= maxx ) {
		x = maxx - (x - maxx + 1);
		xvel[i] = -velocity(i);
	    }
	    if ( x <= minx ) {
		x = minx + (minx - x + 1);
		xvel[i] = velocity(i);
	    }
	    if ( y >= maxy ) {
		y = maxy - (y - maxy + 1);
		yvel[i] = -velocity(i);
	    }
	    if ( y <= miny ) {
		y = miny + (miny - y + 1);
		yvel[i] = velocity(i);
	    }
	    a[head].setPoint( i, x, y );
	}
    }
    paint.end();				// painting done
    delete[] a;
}


//
// Rotate pattern on desktop.
//

void rotate()
{
    int i;
    const int w = 64;
    const int h = 64;
    QImage image( w, h, 8, 128 );		// create image
    for ( i=0; i<128; i++ )			// build color table
	image.setColor( i, QRGB(i,0,0) );
    for ( int y=0; y<h; y++ ) {			// set image pixels
	uchar *p = image.scanline(y);
	for ( int x=0; x<w; x++ )
	    *p++ = (x+y)%128;
    }

    QPixmap pm;
    pm = image;					// convert image to pixmap
    pm.enableImageCache( TRUE );		// rotation will be faster

    QWidget *d = QApplication::desktop();	// w = desktop widget

    for ( i=0; i<=360; i += 2 ) {
	Q2DMatrix m;
	m.rotate( i );				// rotate coordinate system
	QPixmap rpm = pm.xForm( m );		// rpm = rotated pixmap
	d->setBackgroundPixmap( rpm );		// set desktop pixmap
	d->update();				// repaint desktop
    }
}

void generateStone( const QPixmap &pm, 
                    const QColor &c1, const QColor &c2,  const QColor &c3 )
{
    QPainter p;
    p.begin( &pm );
    for( int i = 0 ; i < pm.width() ; i++ )
        for( int j = 0 ; j < pm.height() ; j++ ) {
            int r = kindaRand();
            if ( r < KINDA_RAND_MAX / 3 )
                p.pen().setColor( c1 );
            else if ( r < KINDA_RAND_MAX / 3 * 2 )
                p.pen().setColor( c2 );
            else
                p.pen().setColor( c3 );
            p.drawPoint( i,j );
        }
    p.end();
}

void drawShadeText( QPainter *p, int x, int y, const char *text, 
                    const QColor &topColor, const QColor &bottomColor,
                    int sw = 2,   // shadeWidth
                    const QColor &fillColor = black, bool fill = FALSE )
{
    if ( !p->isActive() ) {
        warning("drawShadeText: QPainter not active"
                " (you probably forgot to call begin).");
        return;
    }
    QFontMetrics fm( p->font() );
    QRect r = fm.boundingRect( text );

    QBitmap mask( r.size() + QSize(sw*2,sw*2), TRUE );
    QBitmap b1 = mask;
    QBitmap b2 = mask;

    QPainter tmp;
    tmp.begin( &mask );
    tmp.drawText( -r.x() + sw, -r.y() + sw, text );
    tmp.end();
    bitBlt( &b1,    0,    0, &mask, sw, sw, r.width() ,r.height() );
    bitBlt( &b1,   sw,   sw, &mask, sw, sw, r.width(), r.height(), EraseROP );
    bitBlt( &b2, sw*2, sw*2, &mask, sw, sw, r.width(), r.height() );
    bitBlt( &b2,   sw,   sw, &mask, sw, sw, r.width(), r.height() , EraseROP );
    int xpos =  x + r.x() - sw;
    int ypos =  y + r.y() - sw;
    p->pen().setColor( topColor );
    p->drawPixmap( xpos, ypos, b1 );
    p->pen().setColor( bottomColor );
    p->drawPixmap( xpos, ypos, b2 );
    if ( fill ) {
        p->pen().setColor( fillColor );
        p->drawPixmap( xpos, ypos, mask );
    }

}

class DesktopWidget : public QWidget
{
public:
    DesktopWidget( const char *s, QWidget *parent=0, const char *name=0 );
    ~DesktopWidget();
    void paintEvent( QPaintEvent * );
private:
    QPixmap *pm;
    QString text;
};

DesktopWidget::DesktopWidget( const char *s, QWidget *parent, const char *name )
        : QWidget( parent, name, WType_Desktop | WPaintDesktop)
{
    text = s;   // makes a copy of s
    pm   = 0;
}

DesktopWidget::~DesktopWidget()
{
    delete pm;
}

void DesktopWidget::paintEvent( QPaintEvent * )
{
    QColor c1 = backgroundColor();
    QColor c2 = c1.light(104);
    QColor c3 = c1.dark(106);
    if ( !pm ) {
        pm = new QPixmap( 64, 64 );
        generateStone( *pm, c1, c2, c3 );
        setBackgroundPixmap( *pm );
        update();
    }
    QPainter p;
    p.begin( this );
    QFontMetrics fm = p.fontMetrics();
    QRect r = fm.boundingRect( text );
    int x = ( width() - r.width() )/2 - r.x();
    int y = ( height() - r.height() )/2 - r.y();
    drawShadeText( &p, x, y, text, c2, c3, 3, c1,TRUE );
    p.end();
}

void desktopWidget( const char *s = "Troll Tech" )
{
    DesktopWidget *t = new DesktopWidget(s);
    t->update();
    qApp->exec( t );
    delete t;
}

void desktopText( const char *s = "Troll Tech" )
{
    const int border = 20;

    QColor c1 =  qApp->palette()->normal().background();
    QColor c2 = c1.light(104);
    QColor c3 = c1.dark(106);
    QFontMetrics fm( *(qApp->font()) );

    QRect r   = fm.boundingRect( s );
    int appWidth  =  qApp->desktop()->width();
    int appHeight =  qApp->desktop()->height();
    if ( r.width() > appWidth - border*2 )
        r.setWidth( appWidth - border*2 );
    if ( r.height() > appHeight - border*2 )
        r.setHeight( appHeight - border*2 );

    QPixmap pm( r.size() + QSize( border*2, border*2 ) );
    generateStone( pm, c1, c2, c3 );

    QPainter p;
    p.begin( &pm );
    drawShadeText( &p, -r.x() + border, -r.y() + border, s, c2, c3 );
    p.end();

    qApp->desktop()->setBackgroundPixmap( pm );
}

//
// The program starts here.
//

int main( int argc, char **argv )
{
    QFont f("charter", 96, QFont::Black );
    f.setStyleHint( QFont::Times );
    QFont::setDefaultFont( f );
    QApplication app( argc, argv );
    bool validOptions = FALSE;

    if ( argc == 2 ) {
        validOptions = TRUE;
	if ( strcmp(argv[1],"-poly") == 0 )
	    poly();
	else if ( strcmp(argv[1],"-rotate") == 0 )
	    rotate();
	else if ( strcmp(argv[1],"-troll") == 0 )
            desktopText();
	else if ( strcmp(argv[1],"-trollwidget") == 0 )
	    desktopWidget();
	else
	    validOptions = FALSE;
    }
    if ( argc == 3 ) {
        validOptions = TRUE;
        if ( strcmp(argv[1],"-shadetext") == 0 )
            desktopText( argv[2] );
        else if ( strcmp(argv[1],"-shadewidget") == 0 )
            desktopWidget( argv[2] );
	else
	    validOptions = FALSE;
    }
    if ( !validOptions ) {
	fprintf( stderr, "Usage:\n\tdesktop -poly"
                               "\n\tdesktop -rotate"
                               "\n\tdesktop -troll"
                               "\n\tdesktop -trollwidget"
                               "\n\tdesktop -shadetext <text>"
                               "\n\tdesktop -shadewidget <text>\n" );
	return 1;
    }
    return 0;
}

