//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_                C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_                          N E T W O R K S
 * |_        |_  |_  |_  |_        |_                                C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_                      L I B R A R Y
 *
 * $Id: LRE.h,v 0.27 1995/01/20 15:13:27 cncl-adm Exp cncl-adm $
 *
 * Class: CNLRE --- Abstract LRE base class
 *
 *****************************************************************************
 * Copyright (C) 1992-1995   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@dfv.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 **EHEADER********************************************************************/

#ifndef __LRE_h
#define __LRE_h


#include <CNCL/CNCL.h>
#include <CNCL/Class.h>
#include <CNCL/Param.h>
#include <CNCL/Statistics.h>            // Base class

#include <values.h>
#include <math.h>
//#include <stdlib.h>
//#include <stream.h>
#include <fstream.h>
                        
extern CNClassDesc CN_LRE;      // CNClass CNLRE description

/*
 * The class CNLRE
 */


class CNLRE : public CNStatistics
{
  public:
    enum Scale { LIN,LOG };

  protected:
    // Private types
    struct sm                           /* sortierte Tabelle            */
    {
        double  rvakt ;                 /* x-Wert                       */
        double rvnach ;                 /* Nachfolgerwert               */
    }; 

    struct result                       /* Struktur f. Ergebnisse der   */
    {                                   /* Ausfuehrung d. LRE-Algorithm.*/
        double  xlevlim ;               /* Grenzwert x-Bereich          */
        int nxvalue ;                   /* Haeufigkeit des Vorkommens   */
                                        /* des Grenzwertes selbst       */
        int  sortmsize ;                /* Zahl der sortierten Werte    */
        int transcount ;                /* Zahl der Uebergaenge         */
                                        /* Spalte in der Ergebnistab.   */
    };


  protected:
    char *name;                 // name of LRE-evaluation
    char *text;                 // explanation to evaluation (fuss 25.08.93)
    Phase phase;                // state of calculation 
    bool end_reached;           // end of evaluation reached ? (fuss 03.12.93)
    Scale fl_log;               // logarithmic scale ?
    int fl_init;                // initialize ?       
    int nsec;                   // no. of sections of distribution function(df)
    int maxsortfeldgroesse;     // max. size of sort array
    int sortfeldgroesse;        // size of sort array
    unsigned long nrv;          // no. of evaluated values
    long n0;                    // reqired no. of values for initialize
    double min_value;           // minimal value of trials
    double max_value;           // maximal value of trials
    double rv_sum;               // necess. for mean
    double rv_square_sum;        // and variance
    double levfac;              // factor for calculation of next F-level
    double minlev;              // minimal level of df resp. cdf
    double maxlev;              // maximal level of df resp. cdf
    double relerror;            // desired relative error
    double maxreler;            // sqare of desired relative error
    double rvakt;               // current random number
    double rvnach;              // succeeding random number
    struct result *pr;          // result table od df
    struct result *pfirst;      // first column of result table
    struct sm *sortquick;       // sort array
    struct sm *sortquickend;    // end of sort array
    struct sm *sortmerge;       // table of sorted values
    struct sm *sortmergeend;    // end of table
    double base;                // for cond. df.
    
    double compreler(long,struct result *);
    void schaufelum();
    void next_level();
    void make_level();

    virtual double end_lev() = 0;
    virtual void neusort(struct result*) = 0;
    virtual void quicksort(struct sm*, struct sm*) = 0;
    virtual struct sm *mergesort(struct sm*) = 0;
    virtual void verbesser(struct result*, struct result*, double, double) = 0;

    // Private members
  private:
    inline double MAX(double x,double y) {return(x > y)? (x) : (y);}
    inline double MIN(double x, double y) {return (x < y)? (x) : (y);}

    inline double FX(struct result * x) 
    { return (double)x->sortmsize/(double)nrv;}
    inline double AR(struct result * x) 
    { return (double)x->transcount/(double)x->sortmsize;}
    inline double KORREKTUR(struct result * x) 
    { return (2.0 - 2*FX(x) - AR(x))/AR(x);}
    inline double NEWN0RECH(struct result * x) 
    { return ((1.0-FX(x))/(FX(x) * maxreler))*KORREKTUR(x);}
    inline double NEWA0RECH(struct result * x) 
    { return ((1.0-FX(x))/(maxreler))*KORREKTUR(x);}
    inline long NEWN0(struct result * x) 
    { return (long) (( x->transcount == 0)? (0) : (NEWN0RECH(x)) ); }
    inline long NEWA0(struct result * x)
    { return (long) (( x->transcount == 0)? (0) : (NEWA0RECH(x)) ); }
    inline double FIRSTLEVEL() 
    { return (int)(pr->sortmsize * maxlev);}
    inline double LOGLEVEL() 
    { return (int)(levfac*(double)pr->sortmsize);}
    inline double LINLEVEL() 
    { return (int)(((double)pr->sortmsize/(double)nrv - levfac)*(double)nrv);}
    
public:
    virtual void          put ( double ) = 0;
    virtual double        mean()     const;
    virtual double        variance() const;
    virtual unsigned long trials()   const { return nrv; }
    virtual double        min()      const { return min_value; }
    virtual double        max()      const { return max_value; }
    virtual void          reset();
    virtual bool          end()      const { return end_reached; } // 03.12.93 
    virtual Phase         status()   const { return phase; }       // (fuss)
    virtual void          print( CNStatistics::Type type, 
                                 ostream &strm ) const;
                                                                 


    /***** Constructors ******************************************************/
public:
    CNLRE (double min=0.01, double max=0.99, double reler=0.05, 
           int levn=100, Scale scale=LIN, int maxsort=0,
           char *n="no name", char *t="no text" );
    CNLRE ( CNParam *param) {}  // CNParam constructor
    ~CNLRE();

    /***** Member functions required by CNCL *********************************/
public:
    virtual CNClassDesc class_desc() const      // CNClass description
    {
        return CN_LRE;
    };
            
    virtual bool is_a(CNClassDesc desc) const   // Type checking
    {
        return desc == CN_LRE ? TRUE : CNStatistics::is_a(desc);
    };
        
    static CNLRE *cast_from_object(CNObject *obj) // Safe type cast
    {
#   ifdef NO_TYPE_CHECK
        return (CNLRE *)obj;
#   else
        return (CNLRE *)( obj->is_a(CN_LRE)
               ? obj : fatal_type(obj->class_desc(), CN_LRE) );
#   endif
    }
    
    // Print/debug output
    virtual void print(ostream &strm = cout) const
        { print( CNStatistics::DF, strm ); }
    virtual void dump (ostream &strm = cout) const;

};

#endif /**__LRE_h**/
