//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: SDLManager.c,v 0.27 1995/01/20 15:15:32 cncl-adm Exp cncl-adm $
 *
 * Class: CNSDLManager --- ...
 *
 *****************************************************************************
 * Copyright (C) 1992-1995   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@dfv.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 **EHEADER********************************************************************/

#include "SDLManager.h"


CNSDLManager::CNSDLManager(int sys, CNEventBaseSched* sim)
{
    system = sys;
    actual_process = 0;
    sched = sim;
}



CNSDLManager::~CNSDLManager()
{
    for (int i=2; i<pr_size; i++)
        delete processes[i];
    delete[] processes;
    
    for (i=0; i<ch_size; i++)
        delete channels[i];
    delete[] channels;
    
    scheduler()->delete_events(this);   // delete all signals adressed to
                                        // the actual system
}




// the eventhandling routine called by the scheduler for every signal
/*
   SDLManager receives Signals from the EventScheduler. It transfers them
   to the appropriate process and handles the saved signals. It also reacts
   to procedure calls in sending an INIT signal.
*/
virtual void CNSDLManager::event_handler(const CNEvent *ev)
{
        // save incoming signal
    signal = (CNSDLSignal*) ev;
    
        // get neccessary informations about current process
    actual_process = signal->receiver();        
    ProcessType pt = 1;                         
    while (!( processes[pt]->is_of_this_type(actual_process) ))
        if (++pt == pr_size) 
            fatal("CNSDLManager::event_handler", " strange error");
    thatdata = processes[ pt ];
    that = thatdata->get_process_by_PId( actual_process );
    
        // first try new signal
    in_save_list = false;
    int old_state = that->state();
    that->sender( signal->sender() );
    
        // deliver signal to process
    int changed_handler = that->signalhandler(signal->type());
       
        // initialize procedure when called or continue behind procedure call
        // when returning from procedure
    while( changed_handler ) changed_handler = 
    	that->signalhandler( (changed_handler == 1 ? 0 : -changed_handler) );
        
        // if process died or remains in same state nothing more is to be done    
    if (!that || (that->state() == old_state)) return;
    
        // if state has changed try saved signals
    in_save_list = true;
    CNDLObject *saved_sig;
    do
    {
        old_state = that->state();
        saved_sig = that->save_list->first(); // at begin of save list
        while( saved_sig )
        {
            signal = (CNSDLSignal*) saved_sig;

            that->sender( signal->sender() );
            has_saved = false;
            
            changed_handler = that->signalhandler(signal->type());
            
                // initialize procedure when called or continue behind 
                // procedure call when returning from procedure
            while( changed_handler ) changed_handler = that->signalhandler( 
                (changed_handler == 1 ? 0 : -changed_handler) );
            
            if (!that) return; // process died, so stop processing
            
            if (!has_saved)
            {
                    // remove signal from save list and delete it
                    // automagically take next signal from list
                saved_sig = that->save_list->delete_object( saved_sig );
                    // restart at begin of save list if state has changed
                if (that->state() != old_state) break;	
            }
            else saved_sig = that->save_list->next( saved_sig );
        }
    } while( that->state() != old_state );
}


    
             
// save signal
void CNSDLManager::save(int name)
{
    has_saved = true;
    if (!in_save_list) 
    {
            // copy signal (the original one will be deleted by the scheduler)
        CNSDLSignal *sig = new CNSDLSignal(signal->type(), signal->scheduled(),
                                          signal->from(), signal->to(),
                                          signal->sender(), signal->receiver(),
                                          signal->system(), signal->object());
        that->save( sig );
    }
}



/*
    SDLManager creates a SDLSignal and delivers it to the EventScheduler.
    The function send is inherited from EventHandler as is scheduler and
    now and some others.
*/
// SDL output functions (first the local ones)
virtual void CNSDLManager::output_x(SignalType name, PId receiver, 
                                    CNObject *data)
{
    // here explicit addressing is used
    ProcessType pt = 2;

    while ( !(processes[ pt ]->is_of_this_type(receiver)) )
    {
        pt++;
        if (pt == pr_size)
            fatal("CNSDLManager::output_x", " strange error");
    }

    if ( !(processes[ pt ]->get_process_by_PId(receiver)) )
        fatal("SDLManager::output_x", " no receiver availlable");
     
    send(new CNSDLSignal(name, now(), actual_process, receiver, system, data));
}



virtual void CNSDLManager::output(SignalType name, CNObject *data)
{
    ProcessType rec = signal_receiver[name];    // implicit addressing
  
    CNSDLProcess *proc_ad = processes[ rec ]->get_process_by_type();
    if (!proc_ad)
        fatal("SDLManager::output", " no receiver availlable");

    PId receiver = proc_ad->self();

    send(new CNSDLSignal(name, now(), actual_process, receiver, system, data));
}



virtual void CNSDLManager::output(SignalType name, ProcessType rec, 
                                  CNObject *data)
{
    CNSDLProcess *proc_ad = processes[ rec ]->get_process_by_type();
    if (!proc_ad)
        fatal("SDLManager::output", " no receiver availlable");
        
    PId receiver = proc_ad->self(); 
    
    send(new CNSDLSignal(name, now(), actual_process, receiver, system, data));
}


// SDL output functions (now the global one)
virtual void CNSDLManager::output_x(SignalType name, PId receiver, 
                              CNSDLChannel *ch, CNObject *data, int do_not_use)
{
    // signals experience a nondeterministic delay when using a channel
    CNSimTime arrival = ch->delay(now());
    			ch->disturb(name, data);
    
    if (receiver == 1) 
    {
        if (processes[1] == NIL)
            fatal("SDLManager::output", " no environment defined");
            
        send(new CNSDLSignal(name, arrival, this, 
                             processes[1]->get_process_by_type()->System, 
                             actual_process, receiver, system, data));
        // addressed to Environment
    }
    else
    {
        ProcessType pt = 2;
        
        while ( !(processes[ pt ]->is_of_this_type(receiver)) )
        {
            pt++;
            if (pt == pr_size)
                fatal("CNSDLManager::output", " strange error");
        }

        if ( !(processes[ pt ]->get_process_by_PId(receiver)) )
            fatal("SDLManager::output", " no receiver availlable");
    
        send(new CNSDLSignal(name, arrival, actual_process, receiver, 
                             system, data));
    }
}

virtual void CNSDLManager::output(SignalType name, ProcessType rec, 
                              CNSDLChannel *ch, CNObject *data, int do_not_use)
{
    // signals experience a nondeterministic delay when using a channel
    CNSimTime arrival = ch->delay(now());
    			ch->disturb(name, data);
    			
    PId receiver;
    if (rec == 1) {
        receiver = 1;
        
        if (processes[1] == NIL)
            fatal("SDLMananger::output", " no environment defined");
        
        send(new CNSDLSignal(name, arrival, this, 
                             processes[1]->get_process_by_type()->System, 
                             actual_process, receiver, system, data));
        // addressed to Enevironment
    }
    else 
    {
        CNSDLProcess *proc_ad = processes[ rec ]->get_process_by_type();
        if (!proc_ad)
            fatal("SDLManager::output", " no receiver availlable");

        receiver = proc_ad->self(); 

        send(new CNSDLSignal(name, arrival, actual_process, receiver, 
                             system, data));
    }                          
}


    

// timer functions
/*
   SDLManager translates SDL compatible Timer calls into the syntax used
   by the SDLTimer class. These are much more complex due to the actions
   taken by this class. The management of timers is done in that class.
*/
// set timer to an absolute time value
void CNSDLManager::set(CNSimTime time, TimerType timer)
{
    CNSDLSignal *sig = that->p_timer->set(time, timer, that->save_list,
                       scheduler(), actual_process, system); 
    CNEventID id = send( sig );
    that->p_timer->tell_ID( id );
}

// reset timer (and free it)
void CNSDLManager::reset(TimerType timer)
{
    that->p_timer->reset(timer, that->save_list, scheduler());
}

// ask wether timer is used
bool CNSDLManager::active(TimerType timer)
{
    return that->p_timer->active( timer );
}

// free timer (make it inactive). This function is necessary because a timer
// in SDL is active until the timer signal is consumed (not delivered!).
void CNSDLManager::remove(TimerType timer)
{
    that->p_timer->remove( timer );
}



/*
   Dynamic SDL. Most of this moved into the class SDLProcessData which manages
   a hash table of process instances of a type. Here we have to find the right
   type and insert the process into the hash table (if possible). Delete frees
   the entry from there. 
*/
// static and dynamic creation of SDL processes
void CNSDLManager::create(CNSDLProcess *proc_ad, PId proc)
{
    ProcessType pt = proc_ad->type();
    PId pid;
    
    if (proc) 
    {
        // static creation (PId is given)
        pid = processes[ pt ]->add_process( proc_ad, proc );
        proc_ad->parent(0);
        proc_ad->self(pid);
        sched->send_event(new CNSDLSignal(0, 0, this, this, proc, proc,
                                          system, NIL));
             // initialization
    }
    else
    {
        // dynamic creation (no PId is given)
        pid = processes[ pt ]->add_process( proc_ad );
        that->offspring( pid );
        proc_ad->parent(actual_process);
        proc_ad->self(pid);
        if( pid ) 
            output_x( 0, pid );   // initialization
        else
            delete proc_ad;       // shouldn't have been created anyway
    }
}


void CNSDLManager::stop()
{
    thatdata->delete_process( actual_process );  // clear from table
    delete that;                                 // clear from memory
    that = NIL;

    CNEvent *ev;
    CNSDLSignal *sig;
    CNEventIterator *it = scheduler()->create_iterator();
    while (ev = it->next_event())	// look through the list of events
    {					// and delete every signal address
        if (ev->is_a(CN_SDLSIGNAL))     // to the stopped process
        {
            sig = (CNSDLSignal *)ev;
            if (sig->receiver()==actual_process && sig->to()==this)
                it->delete_current_event();
        }
    }
    delete it;  // iterator was created on the heap, so delete it
}



virtual CNSDLManager *CNSDLManager::S(int num)
{
    fatal("SDLManager::S ", 
          "this function can be used by the environment only");
    return NIL;
}



/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNSDLManager::print(ostream &strm) const
{
    strm << "CNSDLManager " << " system=" << system << endl;
}

// Debug output
void CNSDLManager::dump(ostream &strm) const
{
    strm << "CNSDLManager { $Revision: 0.27 $ ..." << endl;
    strm << "system=" << system << endl;
    strm << "actual process:" << endl;
        if (that) that->dump(strm);
        if(thatdata) thatdata->dump(strm);
    strm << "last signal:" << endl; 
        if (signal) signal->dump(strm);
    strm << " }" << endl;
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNSDLManager
static CNClass CNSDLManager_desc("CNSDLManager", "$Revision: 0.27 $", NIL);

// "Type" for type checking functions
CNClassDesc CN_SDLMANAGER = &CNSDLManager_desc;

