//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: FSetLR.c,v 0.27 1995/01/20 15:15:21 cncl-adm Exp cncl-adm $
 *
 * Class: CNFSetLR --- Fuzzy set with L and R functions
 *
 *****************************************************************************
 * Copyright (C) 1992-1995   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@dfv.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 **EHEADER********************************************************************/

#include "FSetLR.h"

#include <math.h>
#include <CNCL/minmax.h>


/*
 * Contructors
 */
CNFSetLR::CNFSetLR()
    : CNFSet(),
      m1(0), m2(0), alpha(0), beta(0), L(CNFuncLin), R(CNFuncLin)
{}


CNFSetLR::CNFSetLR(CNParam *param)
    : CNFSet(),
      m1(0), m2(0), alpha(0), beta(0), L(CNFuncLin), R(CNFuncLin)
{}


CNFSetLR::CNFSetLR(double min, double max,
		   double xm1, double xm2, double xalpha, double xbeta,
		   CNFuncType fL, CNFuncType fR                            )
    : CNFSet(min, max),
      m1(xm1), m2(xm2), alpha(xalpha), beta(xbeta), L(fL), R(fR)
{}

      
CNFSetLR::CNFSetLR(double xm1, double xm2, double xalpha, double xbeta,
		   CNFuncType fL, CNFuncType fR                            )
    : CNFSet(),
      m1(xm1), m2(xm2), alpha(xalpha), beta(xbeta), L(fL), R(fR)
{}

      
CNFSetLR::CNFSetLR(CNStringR xname, double min, double max,
		   double xm1, double xm2, double xalpha, double xbeta,
		   CNFuncType fL, CNFuncType fR                            )
    : CNFSet(xname, min, max),
      m1(xm1), m2(xm2), alpha(xalpha), beta(xbeta), L(fL), R(fR)
{}

      
CNFSetLR::CNFSetLR(CNStringR xname, 
		   double xm1, double xm2, double xalpha, double xbeta,
		   CNFuncType fL, CNFuncType fR                            )
    : CNFSet(xname),
      m1(xm1), m2(xm2), alpha(xalpha), beta(xbeta), L(fL), R(fR)
{}



/*
 * Initialize
 */
void CNFSetLR::init(CNStringR xname,
		    double xm1, double xm2, double xalpha, double xbeta,
		    CNFuncType fL, CNFuncType fR,
		    double min, double max                              )
{
    CNFSet::init(xname, min, max);
    m1    = xm1;
    m2    = xm2;
    alpha = xalpha;
    beta  = xbeta;
    L     = fL;
    R     = fR;
}


      

/*
 * Compute membership value
 */
double CNFSetLR::get_membership(double x) const
{
    double dy = ymax - ymin;

    if(x>=m1 && x<=m2)
	return ymax;
    if(x < m1)
	return ymin + dy * L((m1-x)/alpha);
    return ymin + dy * R((x-m2)/beta);
}



/*
 * Functions
 */
double CNFuncExp(double x)
{
    if (x < 0)
	x *= -1.0;
    return exp(-x);
}

double CNFuncExp2(double x)
{
    return exp(-x*x);
}

double CNFuncLin(double x)
{
    if (x < 0)
	x *= -1.0;
    return 1-x > 0.0 ? 1-x : 0;
}

double CNFuncSqr(double x)
{
	 return max(0.0, 1.0-x*x);
}

double CNFuncHyp(double x)
{
    if (x < 0)
	x *= -1.0;
    return 1 / (1+x);
}

double CNFuncHyp2(double x)
{
    return 1 / (1+x*x);
}



/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNFSetLR::print(ostream &strm) const
{
    strm << "\"" << name() << "\": "  
	 << "ymin=" << ymin << " ymax=" << ymax << " m1=" << m1
	 << " m2=" << m2 << " alpha=" << alpha << " beta=" << beta << endl;
}

// Debug output
void CNFSetLR::dump(ostream &strm) const
{
    strm << "CNFSetLR { $Revision: 0.27 $" << endl;
    print(strm);
    strm << "L="  << hex << (unsigned long)(L)
	 << " R=" << hex << (unsigned long)(R) << dec << endl
	 << "}" << endl;
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNFSetLR
static CNClass CNFSetLR_desc("CNFSetLR", "$Revision: 0.27 $", CNFSetLR::new_object);

// "Type" for type checking functions
CNClassDesc CN_FSETLR = &CNFSetLR_desc;






