//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: Manager.c,v 0.27 1995/01/20 15:13:35 cncl-adm Exp cncl-adm $
 *
 * CNClass: CNManager --- CNObject management frontend
 *
 *****************************************************************************
 * Copyright (C) 1992-1995   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@dfv.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 * 
 * As an exception to this rule you may use this template to generate
 * your own classes. This does not cause these classes to be covered by
 * the GNU Library General Public License. This exception does not
 * however invalidate any other reasons why the resulting program must be
 * covered by the GNU Library General Public License.
 **EHEADER********************************************************************/

#include "Manager.h"

CNManager::CNManager(char *object_name)
{
    name = (object_name == NIL) ? "root" : object_name;
    if (*name == EOS)
	fatal("You MUST pass a valid object name to CNManager()");
    
    table = new CNHashDynamic(DEFAULT_CAPACITY);
    if (table == NIL)
	fatal("Out of memory!");
}

CNManager::CNManager(CNParam *param)
{
    CNManager();
}

CNManager::~CNManager()
{
    if (table != NIL) {
	if (table->get_num_entries())
	    table->reset_absolutely();
	delete table;
    }
}

CNObject *CNManager::new_object(char *object_name, CNClassDesc desc, 
				CNParam *param)
{
    CNObject *obj;
    char *slash;
    
    if (object_name == NIL)
	fatal("You MUST pass a valid object_name to new_object()\n");
    if (desc == NIL)
	fatal("You MUST pass a valid class descriptor to new_object()\n");

    if (*object_name == EOS)
	return NIL;
    
    // remove leading slash
    if (*object_name == SLASH)
	object_name++;
 
    slash = strchr(object_name, SLASH);

    // name contains directories
    if (slash) {
	char *dup, *dslash;
	CNManager *manager;
	
	dup = new char[strlen(object_name) + 1];
	strcpy(dup, object_name);
	dslash = strchr(dup, SLASH);
	*dslash = EOS;
	
	// does current dir contain the specified subdir?
	obj = table->get_object(CNKeyString(dup));
	delete dup;
	
	if (obj) {
	    // this is a save cast
	    manager = cast_from_object(obj);
	    // pass creation of new object to subentry
	    return manager->new_object(slash, desc, param);
	} else {
	    warning("You passed an improper object_name to new_object\n",
		    "NIL returned\n");
	    return NIL;
	}
    }
    
    // find out whether the hash table already contains an object with the 
    // same name 
    if (table->get_key(CNKeyString(object_name)) != NIL)
	fatal("Unable to create duplicate entries in one directory\n");
    
    obj = (is_a(desc)) ? new CNManager(object_name) : desc->new_object(param);
    if (obj == NIL)
	fatal("Out of memory!\n");
    
    table->store_key(new CNKeyString(object_name, obj));
    
    return obj;
}

CNObject *CNManager::get_object(char *object_name) const
{
    CNObject *obj;
    CNManager *manager;
    char *slash, *dup, *dslash;

    if (object_name == NIL)
	fatal("You MUST pass a valid object_name to get_object()\n");

    if (*object_name == EOS)
	return NIL;
    
    // remove leading slash
    if (*object_name == SLASH)
	object_name++;
    
    slash = strchr(object_name, SLASH);
    // object_name is base name
    if (!slash) 
	return table->get_object(CNKeyString(object_name));

    dup = new char[strlen(object_name) + 1];
    strcpy(dup, object_name);
    dslash = strchr(dup, SLASH);
    *dslash = EOS;
	
    // does current dir contain the specified subdir?
    obj = table->get_object(CNKeyString(dup));
    delete dup;
	
    if (obj == NIL || !obj->is_a(CN_MANAGER))
    	return NIL;

    // this is a save cast
    manager = cast_from_object(obj);
    // climb down the tree
    return manager->get_object(slash);
}

bool CNManager::delete_object(char *object_name)
{
    CNObject *obj;
    CNManager *manager;
    char *slash, *dup, *dslash;

    if (object_name == NIL)
	fatal("You MUST pass a valid object_name to delete_object()\n");

    if (*object_name == EOS)
	return FALSE;
    
    // remove leading slash
    if (*object_name == SLASH)
	object_name++;
    
    slash = strchr(object_name, SLASH);
    // object_name is base name
    if (!slash) {
	CNObject *obj;
	
	obj = table->get_object(CNKeyString(object_name));
        if (obj == NIL) 
	    return FALSE;
	
	// does user want to delete a non-empty directory?
	if (obj->is_a(CN_MANAGER)) {
	    CNManager *manager;
	    
	    // this is a safe cast
	    manager = (CNManager *)obj;

	    // non-empty?
	    if (!manager->is_empty())
		error("You wanted to delete the CNManager object '",
		      manager->get_name(),
		      "',\nwhich still has valid subentries!");
	}
		    
	table->delete_key_absolutely(CNKeyString(object_name));
	delete obj;
	
	return TRUE;
    }

    dup = new char[strlen(object_name) + 1];
    strcpy(dup, object_name);
    dslash = strchr(dup, SLASH);
    *dslash = EOS;
	
    // does current dir contain the specified subdir?
    obj = table->get_object(CNKeyString(dup));
    delete dup;
	
    if (obj == NIL || !obj->is_a(CN_MANAGER))
    	return FALSE;

    // this is a save cast
    manager = cast_from_object(obj);
    // climb down the tree
    return manager->delete_object(slash);
}

bool CNManager::is_empty() const
{
    return (table->get_num_entries() == 0) ? TRUE : FALSE;
}

    

/***** Default I/O member function for CNCL classes **************************/

// CNNormal output
void CNManager::print(ostream &strm) const
{
    strm << "name " << name << endl;
}

// Debug output
void CNManager::dump(ostream &strm) const
{
    strm << "CNManager { $Revision: 0.27 $"
	 << " name " << name
	 << " }" << endl;
}



/***** CNCL stuff for type information and exemplar objects ******************/

// Describing object for class CNManager
static CNClass CNManager_desc("CNManager", "$Revision: 0.27 $",
			  CNManager::new_object);

// "Type" for type checking functions
CNClassDesc CN_MANAGER = &CNManager_desc;






