static char rcsid[] = "$Id: command.c,v 1.2 1992/01/17 23:02:02 jtsillas Exp $";

/*****************************************************************************
 *
 *  Copyright 1989 The University of Texas at Austin
 *  Copyright 1990 Microelectronics and Computer Technology Corporation
 *  Copyright 1990 Thomson Consumer Electronics, Inc.
 *  Copyright 1991 Bull HN Worldwide Info Systems, Inc.
 *
 *****************************************************************************/

/*  command.c
 *
 *    Create the command window, the command buttons and their callbacks.
 *
 *    CreateButtons() :		Create command buttons in panel
 *    AddButton() :		Add a command button into the command window
 *    ButtonSet() :		Action proc for command button translation
 *
 *    Command callbacks for the command buttons:
 *
 *    forwardSearch() :		forward string search (static).
 *    reverseSearch() :		reverse string search (static).
 *    Search() :		call either forwardSearch() or reverseSearch()
 *    PopupSearch() :		command callback for search button
 *    DoneSearch() :		command callback for DONE button in search 
 *    CreateSearchPopup() :	create search panel
 *    XttyCallback() :          create the xterm on a open pty and tell gdb
 *    DoIt():                   execute the gdb command.
 *    Break():                  set a breakpoint.
 *    Clear():                  clear a breakpoint.
 *    Print():                  print the value of a symbol.
 *    Quit():                   quit mxgdb and gdb.
 *    Help():                   access gdb's help.
 *    Display_():               display a value.
 *    Undisplay():              undisplay a value.
 *    CancelSearch():           cancel the search and returns to prev (static).
 *    StopSearch():             stop the search at the current pos (static).
 *
 *    Command queue manipulation routines:
 *	send_command():		send a command to gdb and record in the queue
 *	get_command():		read command off head of queue
 *	insert_command():	insert command at the head of queue
 *	delete_command():	delete command from head of queue
 */
 
#include <signal.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "global.h"
#include "bitmaps.h"
#include <Xm/Xm.h>
#include <Xm/SelectioB.h>

#define	 REVERSE	0
#define	 FORWARD	1

static int forwardSearch(char*, XmTextPosition, char*, XmTextPosition*,
			 XmTextPosition*);
static int reverseSearch(char*, XmTextPosition, char*, XmTextPosition*,
			 XmTextPosition*);
static void StopSearch(Widget, XtPointer, XtPointer);
static void CancelSearch(Widget, XtPointer, XtPointer);

static Widget searchPopupShell;
static Widget AddButton();
static char SearchString[LINESIZ] = "";	/* search string buffer */
static char command[LINESIZ];
static CommandRec *commandQueue = NULL;
char *xttyname = NULL;


/*  Create an xterm for the debug process to use
 */
void XttyCallback (w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
  
  if(!xttyname)
    {
      if((xttyname = callxtty()))
	 {
	   sprintf(command, "tty %s\n", xttyname);
	   send_command(command);
	   AppendDialogText(command);
	 }
	 else			/* Problem in creating Xtty */
	 {
	   UpdateMessageWindow("Could not open Xtty.", NULL);
	 }
    }
  else
      kill(xttypid, SIGKILL);
}

/*  Execute the gdb command specifed in client_data
 */
void DoIt (w, command, call_data)
    Widget w;
    XtPointer command;
    XtPointer call_data;
{
    /* run, cont, next, step, where, up, down, status */
    send_command(command);
    AppendDialogText(command);
}

/*  here client_data is "break" or "tbreak"
*/
void Break(w, client_data, call_data)
     Widget w;
     XtPointer client_data;
     XtPointer call_data;
{
    XmTextPosition pos;
    int line;
    char *string1;
    char *string2;
    char *s;

    string1 = (char *) XmTextGetSelection(sourceWindow);
    string2 = (char *) XmTextGetSelection(dialogWindow);

    if(string2 && !string1)
      {
        string1 = string2;
        string2 = NULL;
      }

    if(string2) XtFree(string2);

    if (string1 && *string1) 
      {
	s = string1;
	while (*s == ' ') s++;	/* skip leading spaces (if any) */
	if ((*s >= '0') && (*s <= '9'))
	  sprintf(command, "%s *%s\n",client_data,string1);
	else
	  sprintf(command, "%s %s\n",client_data,string1);
      }
    else
      {
	if (displayedFile != NULL)
	  {
	    pos = XmTextGetInsertionPosition(sourceWindow);
	    line = TextPositionToLine(pos);
	    sprintf(command, "%s %d\n",client_data,line);
	  }
	else
	  {
	    UpdateMessageWindow(BREAK_HELP, NULL);
	    bell(0);
	    if(string1) XtFree(string1);
	    return;
	  }
      }
    
    if(string1) XtFree(string1);
    send_command(command);
    AppendDialogText(command);
  }

/*  Clear removes the stop_no associated with a given line number or
 *  a given function name.
 *  RemoveStop() is called to undisplay the stop sign only when there
 *  are no more stop_no's associated with that line number.
 */
				/* A selection-box of all available break 
				   points should be added somehwere.
				 - Jim (5-4-91)*/
void Clear(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    XmTextPosition pos;
    int		    line;
    char           *string1, *string2;

    string1 = (char *) XmTextGetSelection(sourceWindow);
    string2 = (char *) XmTextGetSelection(dialogWindow);

    if(string2 && !string1)
      {
	string1 = string2;
	string2 = NULL;
      }

    if(string2) XtFree(string2);

    if(string1 && *string1 != '\0')
      {
	sprintf(command, "clear %s\n", string1);
	send_command(command);
	AppendDialogText(command);
	XtFree(string1);
	return;
      }
    if(string1) XtFree(string1);

    if (displayedFile) {
      pos = XmTextGetInsertionPosition(sourceWindow);
      line = TextPositionToLine(pos);
      if (LineToStop_no(line)) {
	sprintf(command, "clear %d\n", line);
	send_command(command);
	AppendDialogText(command);
	return;
      }
    }
    UpdateMessageWindow(CLEAR_HELP, NULL);
    bell(0);
}

void Print(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    char *string1, *string2;
    int nbytes;

    string1 = (char *) XmTextGetSelection(sourceWindow);
    string2 = (char *) XmTextGetSelection(dialogWindow);

    if(string2 && !string1)
      {
	string1 = string2;
	string2 = NULL;
      }

    if(string2) XtFree(string2);

    if(!string1 || *string1 == '\0')
      {
	UpdateMessageWindow(PRINT_HELP, NULL);
	bell(0);
	if(string1) XtFree(string1);
	return;
      }

    if (client_data == (XtPointer)0)
	sprintf(command, "print %s\n", string1);
    else if (client_data == (XtPointer)1)
	sprintf(command, "print *%s\n", string1);

    send_command(command);
    AppendDialogText(command);
    if(string1) XtFree(string1);
}

void Quit(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    write_gdb("quit\n");
    XtDestroyApplicationContext(app_context);
    if(xttypid)
      kill(xttypid, SIGKILL);
    kill(gdbpid, SIGKILL);
    exit(0);
}

extern int helpstackidx;	/* Stack Index for the help levels */

void Help(w, client_data, call_data)
     Widget w;
     XtPointer client_data;
     XtPointer call_data;
{
  helpstackidx = 0;
  query_gdb("help\n");
}

void Display_(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    char *string1, *string2;
    int nbytes;

    if(!XtIsManaged(displayFD)) XtManageChild(displayFD);

    string1 = (char *)XmTextGetSelection(sourceWindow);
    string2 = (char *)XmTextGetSelection(dialogWindow);

    if(string2 && !string1)
      {
	string1 = string2;
	string2 = NULL;
      }

    if(string2) XtFree(string2);

    if(string1 && *string1 != '\0')
      {
	sprintf(command, "display %s\n", string1);
	send_command(command);
	AppendDialogText(command);
	XtFree(string1);
	return;
      }
    if(string1) XtFree(string1);

}

void Undisplay(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
  char *string1, s[5], *st1, *st2;
  int	 stop_no, nbytes;
  
  string1 = (char *) XmTextGetSelection(displayWindow);
  
  if (string1 && *string1)
    {
      st1 = s;
      st2 = string1;
      while((*st1++ = *st2++)!=':' && *st1);
      *(st1-1) = '\0';
      XtFree(string1);
      if ((stop_no = atoi(s)) > 0)
	{
	  sprintf(command, "undisplay %d\n", stop_no);
	  send_command(command);
	  AppendDialogText(command);
	}
      else
	{
	  UpdateMessageWindow(UNDISPLAY_HELP, NULL);
	  bell(0);
	  return;
	}
  }
  else
    {
      if(XtIsManaged(displayFD))
	XtUnmanageChild(displayFD);
      if(string1) XtFree(string1);
      return;
    }
}

/*  Beginning from startpos, this routine searches text forward for 
 *  searchstring, and returns 1 if searchstring is found, also returning 
 *  the left and right positions of the matched string in left and right; 
 *  else 0 is returned. It also does wrap-around search. 
 */
	
static int forwardSearch(text, startpos, searchstring, left, right)
     char *text;
     XmTextPosition startpos;
     char *searchstring;
     XmTextPosition *left, *right;
{
    int  searchlength, searchsize, i, n=0;
    char *s1, *s2;

    searchlength = strlen(searchstring);
    searchsize = strlen(text) - searchlength;
    for (i=startpos+1; i < searchsize; i++) {
	n = searchlength;
	s1 = &text[i];
	s2 = searchstring;
	while (--n >= 0 && *s1++ == *s2++);
	if (n < 0) break;
    }
    if (n < 0) {
    	*left = i;
    	*right = i+searchlength;
    	return 1;
    }
    else {
	for (i=0; i <= startpos; i++) {
	    n = searchlength;
	    s1 = &text[i];
	    s2 = searchstring;
	    while (--n >= 0 && *s1++ == *s2++);
	    if (n < 0) break;
	}
	if (n < 0) {
	    *left = i;
	    *right = i+searchlength;
	    return 1;
	}
	return 0;
    }
}
	

/*  Similar to forwardSearch(), except that it does a reverse search
 */
static int reverseSearch(text, startpos, searchstring, left, right)
    char 	    *text;
    XmTextPosition  startpos;
    char 	    *searchstring;
    XmTextPosition  *left, *right;
{
    int  searchlength, i, n=0;
    char *s1, *s2;

    searchlength = strlen(searchstring);
    for (i=startpos; i >= searchlength; i--) {
	n = searchlength;
	s1 = &text[i];
	s2 = &searchstring[searchlength-1];
	while (--n >= 0 && *--s1 == *s2--);
	if (n < 0) break;
    }
    if (n < 0) {
    	*right = i;
    	*left = *right-searchlength;
    	return 1;
    }
    else {
	for (i=strlen(text); i > startpos; i--) {
	    n = searchlength;
	    s1 = &text[i];
	    s2 = &searchstring[searchlength-1];
	    while (--n >= 0 && *--s1 == *s2--);
	    if (n < 0) break;
	}
	if (n < 0) {
            *right = i;
            *left = *right-searchlength;
	    return 1;
	}
	return 0;
    }
}

static XmTextPosition last_pos_before_search; /* This saves the cursor
					         position when we start
						 searching so we can return
						 to it if we wish.
					      */

void PopupSearch(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    if (!displayedFile) {
	UpdateMessageWindow(SEARCH_HELP, NULL);
	bell(0);
    }
    else {
        last_pos_before_search = XmTextGetInsertionPosition(sourceWindow);
	XtManageChild(searchPopupShell);
    }
}

/* This callback is used by the Stop button to stop the search at the
 * current position.
 */
static void StopSearch(w, client_data, call_data)
     Widget w;
     XtPointer client_data;
     XtPointer call_data;
{
  XtUnmanageChild(searchPopupShell);
}

/* This callback is used by the Cancel button to return to the saved text
 * position after a search.
 */
static void CancelSearch(w, client_data, call_data)
     Widget w;
     XtPointer client_data;
     XtPointer call_data;
{
        AdjustText(TextPositionToLine(last_pos_before_search));
	XmTextSetInsertionPosition(sourceWindow, last_pos_before_search);
}


/*  This routine handles both forward and reverse text search.
 *  If no text has been entered, the contents of the cut buffer are used
 *  for searching.
 */ 
static void Search(w, direction, call_data)
     Widget w;
     XtPointer direction;
     XtPointer call_data;
{
  XmTextPosition	pos, left, right;
  char		*searchString;
  
  searchString = 
    (char *) XmTextGetString(XmSelectionBoxGetChild(searchPopupShell,
							XmDIALOG_TEXT));
  if(searchString && *searchString)
    {
      pos = XmTextGetInsertionPosition(sourceWindow);
      if ((direction == (XtPointer)FORWARD && 
	   forwardSearch(displayedFile->buf, pos, 
			 searchString, &left, &right)) ||
	  (direction == (XtPointer)REVERSE && 
	   reverseSearch(displayedFile->buf, pos, 
			 searchString, &left, &right))) {
	AdjustText(TextPositionToLine(left));
	XmTextSetSelection(sourceWindow, left, right, 0);
	XmTextSetInsertionPosition(sourceWindow, left);
      }
      else {
	if (direction == (XtPointer)FORWARD)
	  UpdateMessageWindow("String not found", NULL);
	else if (direction == (XtPointer)REVERSE)
	  UpdateMessageWindow("String not found", NULL);
#ifdef DEBUG
	else
	  abort();
#endif
	bell(0);
      }
    }
  if(searchString) XtFree(searchString);
}

void CreateSearchPopup(parent)
     Widget parent;
{
  Arg args[MAXARGS];
  Display *topdisplay;
  Pixel bg;
  Widget applybutton, okbutton, cancelbutton, helpbutton;

  searchPopupShell = XmCreatePromptDialog(parent, "searchWindow", 
					  NULL, 0);
  applybutton = XmSelectionBoxGetChild(searchPopupShell, 
				       XmDIALOG_APPLY_BUTTON);
  okbutton = XmSelectionBoxGetChild(searchPopupShell, 
				    XmDIALOG_OK_BUTTON);
  cancelbutton = XmSelectionBoxGetChild(searchPopupShell, 
					XmDIALOG_CANCEL_BUTTON);
  helpbutton = XmSelectionBoxGetChild(searchPopupShell, 
				      XmDIALOG_HELP_BUTTON);
  XtManageChild(applybutton);

  topdisplay = XtDisplay(toplevel);
  XtSetArg(args[0], XmNbackground, &bg);
  XtGetValues(searchPopupShell, args, 1);
  
  XtSetArg(args[0], XmNlabelType, XmPIXMAP);
  XtSetArg(args[1], XmNlabelPixmap, 
	   XCreatePixmapFromBitmapData (topdisplay, 
			DefaultRootWindow(topdisplay), 
			arrowup_bits, arrowup_width, 
			arrowup_height, 
			app_resources.stop_color, bg, 
			DefaultDepth(topdisplay, 
			DefaultScreen(topdisplay))));

  XtSetValues(applybutton, args, 2);
  XtSetArg(args[1], XmNlabelPixmap, 
	   XCreatePixmapFromBitmapData (topdisplay, 
			DefaultRootWindow(topdisplay), 
			arrowdown_bits, arrowdown_width, 
			arrowdown_height, app_resources.stop_color, bg, 
			DefaultDepth(topdisplay, 
			DefaultScreen(topdisplay))));

  XtSetValues(okbutton, args, 2);
  XtSetArg(args[0],  XmNlabelString, XmStrMk("Stop"));
  XtSetValues(helpbutton, args, 1);
  XtRemoveAllCallbacks(applybutton, XmNactivateCallback);
  XtRemoveAllCallbacks(okbutton, XmNactivateCallback);
  XtAddCallback(applybutton, XmNactivateCallback, 
		Search, (XtPointer)REVERSE);
  XtAddCallback(okbutton, XmNactivateCallback, 
		Search, (XtPointer)FORWARD);
  XtAddCallback(cancelbutton, XmNactivateCallback, CancelSearch, NULL);
  XtAddCallback(helpbutton, XmNactivateCallback, StopSearch, NULL);

}

/**************************************************************************
 *
 *  Command queue functions
 *
 **************************************************************************/

/*  
 *  Append command to end of the command queue and send the command to gdb 
 */
void send_command(command)
     char *command;
{
  CommandRec *p, *q, *r;

  p = (CommandRec *)XtNew(CommandRec);
  p->command = XtNewString(command);
  p->next = NULL;
  if (!commandQueue)
    commandQueue = p;
  else {
    q = commandQueue;
    while (r = q->next)
      q = r;
    q->next = p;
  }
  write_gdb(command);
}

/*  
 * Read command at the head of the command queue 
 */
char *get_command()
{
  if (commandQueue) {
    return (commandQueue->command);
  }
  else
    return NULL;
}

/*  
 *Delete command from the head of the command queue 
 */
void delete_command()
{
  CommandRec *p;
  
  if (p = commandQueue) {
    commandQueue = p->next;
    XtFree(p->command);
    XtFree((char *)p);
  }
}

/*  
 * Insert command into head of queue 
 */
void insert_command(command)
     char *command;
{
  CommandRec *p;
  
  p = (CommandRec *)XtNew(CommandRec);
  p->command = XtNewString(command);
  p->next = NULL;
  if (!commandQueue)
    commandQueue = p;
  else {
    p->next = commandQueue;
    commandQueue = p;
  }
}
