/* bdmcpu32.c - routines to talk to CPU32 target system
 * Copyright (C) 1992 by Scott Howard, all rights reserved
 * Permission is hereby granted to freely copy and use this code or derivations thereof
 * as long as no charge is made to anyone for its use
 */

#include	<stdio.h>
#include	<stdarg.h>
#include	"bdmcalls.h"
#include	"bdm.h"
#include	"trgtstat.h"
#include	"regs-32.h"
#include	"bdmerror.h"

#define         BDM_RDREG       0x2180
#define         BDM_WRREG       0x2080
#define         BDM_RSREG       0x2580
#define         BDM_WSREG       0x2480
#define         BDM_READ        0x1900
#define         BDM_WRITE       0x1800
#define         BDM_DUMP        0x1D00
#define         BDM_FILL        0x1C00
#define         BDM_GO          0x0C00
#define         BDM_CALL        0x0800
#define         BDM_RST         0x0400
#define         BDM_NOP         0x0000

#define         BDM_BYTESIZE    0x00
#define         BDM_WORDSIZE    0x40
#define         BDM_LONGSIZE    0x80
#define         BDM_RDBIT       0x100

#define         BDM_RPC         0x0
#define         BDM_PCC         0x1
#define         BDM_SR          0xb
#define         BDM_USP         0xc
#define         BDM_SSP         0xd
#define         BDM_SFC         0xe
#define         BDM_DFC         0xf
#define         BDM_ATEMP       0x8
#define         BDM_FAR         0x9
#define         BDM_VBR         0xa

#define         BDM_NOTREADY    0x10000
#define         BDM_BERR        0x10001
#define         BDM_ILLEGAL     0x1FFFF
#define         BDM_CMDCMPLTE   0x0FFFF

static void bdm_clrerror (void);
static BYTE fc_set = 0, fc = 5;
static LONG old_sfc, old_dfc;
static char last_rw;
static LONG last_addr;
unsigned go_cmd = BDM_GO, CommandBitCount = 17;
static Initted = 0;
char RegsValid;

int set_fc (void)
{
	if (fc_set) return 0;
	old_sfc = (int) GetReg (REG_SFC);
	old_dfc = (int) GetReg (REG_DFC);
	PutReg (REG_SFC, (LONG) fc);
	PutReg (REG_DFC, (LONG) fc);
	return fc_set = 1;
}

int restore_fc (void)
{
	if (!fc_set) return 0;
	PutReg (REG_SFC, (LONG) old_sfc);
	PutReg (REG_DFC, (LONG) old_dfc);
	fc_set = 0;
	return 1;
}

/* ValidPorts returns unsigned integer where each bit position
 * indicates an installed LPT port (Bit 0 = LPT1, bit 1 = LPT2, etc)
 * we will need eventually to look in /proc/ioports.  for now, return 1
 */

unsigned ValidPorts (void)
{
	return 1;
}

void SetFC (unsigned NewFC)
{
	if (NewFC == UserData || NewFC == UserCode
	|| NewFC == SupervisorData || NewFC == SupervisorCode)
		fc = NewFC;
}

/* DeInit must be called before program quits or shells to DOS
 * un-do any bad things which have been done to talk to target
 */

void DeInit (void)
{
	if (Initted) bdm_deinit ();
	Initted = 0;
}

/* Init initializes parallel port to talk to target */

int Init (unsigned port, unsigned baud)
{
	DeInit ();
	if (!(ValidPorts () & (1 << (port - 1)))) return -1;

	bdm_init (0x378, baud);
	GetStatus ();
	Initted = 1;
	return 0;
}

/* bdm_error handles call to DriverError if error detected by bdm routines */

void bdm_error (int type)
{
	if ((type > BDM_FAULT_BERR) || (type < 0)) type = 0;
	if (type == BDM_FAULT_BERR) restore_fc ();
	DriverError (type, last_rw, last_addr);
}
static void bdm_clrerror (void)
{
	int count;
	long response;

	bdm_clk (BDM_NOP, CommandBitCount);
	bdm_clk (BDM_NOP, CommandBitCount);
	bdm_clk (BDM_NOP, CommandBitCount);
	bdm_clk (BDM_NOP, CommandBitCount);
	while (bdm_clk (0, 1)) ;
	while (!bdm_clk (0, 1)) ;
	bdm_clk (0, 15);
}

static LONG bdm_read (LONG addr, int pcount, WORD cmd, ...)
{
	_G_va_list arg;
	int err,count;
	LONG response,result;

	last_addr = addr;
	last_rw = 1;
	result = 0L;
        if (!Initted) bdm_error (BDM_FAULT_PORT);
	for (err = 3; err; err--)
	{
		count = pcount;
		va_start (arg,cmd);
		response = bdm_clk (cmd, CommandBitCount);
		while (count--)
			if ((response = bdm_clk (va_arg(arg,WORD), CommandBitCount)) > BDM_NOTREADY)
			{
				if (response == BDM_BERR) bdm_error (BDM_FAULT_BERR);
				continue;
			}
		for (count = (cmd & BDM_LONGSIZE) ? 2 : 1; count; count--)
		{
			while ((response = bdm_clk (BDM_NOP, CommandBitCount)) == BDM_NOTREADY) ;
			if (response < BDM_NOTREADY)
			{
				result <<= 16;
				result |= response;
			}
			else
			{
				if (response == BDM_BERR) bdm_error (BDM_FAULT_BERR);
				break;
			}
		}
		if (response > BDM_NOTREADY)
		{
			if (response == BDM_BERR) bdm_error (BDM_FAULT_BERR);
			bdm_clrerror ();
			continue;
		}
		va_end (arg);
		break;
	}
        if (!err) bdm_error (BDM_FAULT_UNKNOWN);
	response = bdm_clk (BDM_NOP, CommandBitCount);
	if (response == BDM_BERR) bdm_error (BDM_FAULT_BERR);
	return result;
}

static void bdm_write (LONG addr, int pcount, WORD cmd, ...)
{
	va_list arg;
	int err,count;
	LONG response,result;

	last_addr = addr;
	last_rw = 0;
	if (!Initted) bdm_error (BDM_FAULT_PORT);
	for (err = 3; err; err--)
	{
		count = pcount;
		va_start (arg,cmd);
		response = bdm_clk (cmd, CommandBitCount);
		while (count--)
			if ((response = bdm_clk (va_arg(arg,WORD), CommandBitCount)) > BDM_NOTREADY)
			{
				if (response == BDM_BERR) bdm_error (BDM_FAULT_BERR);
				bdm_clrerror ();
				continue;
			}
		while ((response = bdm_clk (BDM_NOP, CommandBitCount)) == BDM_NOTREADY) ;
		if (response == BDM_CMDCMPLTE) break;
		else if (response == BDM_BERR) bdm_error (BDM_FAULT_BERR);

	}
	va_end (arg);
	if (!err) bdm_error (BDM_FAULT_UNKNOWN);
	response = bdm_clk (BDM_NOP, CommandBitCount);
	if (response == BDM_BERR) bdm_error (BDM_FAULT_BERR);
}

void RunChip (LONG where)
{
	StopChip ();
	if (where)
		bdm_write (where, 2, BDM_WSREG + BDM_RPC, (WORD) (where >> 16), (WORD) where);
	bdm_clk (BDM_GO, CommandBitCount);
}

LONG GetByte (LONG x)
{
	int frozen,save,c;
	LONG result;

	frozen = StopChip ();
	save = set_fc ();
	result = bdm_read (x, 2, BDM_READ, (WORD) (x >> 16), (WORD) x);
	if (save) restore_fc ();
	if (frozen) RunChip (0L);
	return result;
}

void PutByte (LONG x, BYTE y)
{
	int frozen,save;

	frozen = StopChip ();
	save = set_fc ();
	bdm_write (x, 3, BDM_WRITE, (WORD) (x >> 16), (WORD) x, (WORD) y);
	if (save) restore_fc ();
	if (frozen) RunChip (0L);
}

void FillByte (LONG x, BYTE y)
{
	int frozen,save;

	frozen = StopChip ();
	save = set_fc ();
	bdm_write (x, 1, BDM_FILL, (WORD) y);
	if (save) restore_fc ();
	if (frozen) RunChip (0L);
}

LONG DumpByte (LONG x)
{
	return GetByte (x);
}

LONG GetWord (LONG x)
{
	int frozen,save;
	LONG result;

	frozen = StopChip ();
	save = set_fc ();
	result = bdm_read (x, 2, BDM_READ + BDM_WORDSIZE, (WORD) (x >> 16), (WORD) x);
	if (save) restore_fc ();
	if (frozen) RunChip (0L);
	return result;
}

void PutWord (LONG x, WORD y)
{
	int frozen,save;

	frozen = StopChip ();
	save = set_fc ();
	bdm_write (x, 3, BDM_WRITE + BDM_WORDSIZE, (WORD) (x >> 16), (WORD) x, y);
	if (save) restore_fc ();
	if (frozen) RunChip (0L);
}

void FillWord (LONG x, WORD y)
{
	int frozen,save;

	frozen = StopChip ();
	save = set_fc ();
	bdm_write (x, 1, BDM_FILL + BDM_WORDSIZE, y);
	if (save) restore_fc ();
	if (frozen) RunChip (0L);
}

LONG DumpWord (LONG x)
{
	return GetWord (x);
}

LONG GetLong (LONG x)
{
	int frozen,save;
	LONG result;

	frozen = StopChip ();
	save = set_fc ();
	result = bdm_read (x, 2, BDM_READ + BDM_LONGSIZE, (WORD) (x >> 16), (WORD) x);
	if (save) restore_fc ();
	if (frozen) RunChip (0L);
	return result;
}

void PutLong (LONG x, LONG y)
{
	int frozen,save;

	frozen = StopChip ();
	save = set_fc ();
	bdm_write (x, 4, BDM_WRITE + BDM_LONGSIZE, (WORD) (x >> 16), (WORD) x, (WORD) (y >> 16), (WORD) y);
	if (save) restore_fc ();
	if (frozen) RunChip (0L);
}

void FillLong (LONG x, LONG y)
{
	int frozen,save;

	frozen = StopChip ();
	save = set_fc ();
	bdm_write (x, 2, BDM_FILL + BDM_LONGSIZE, (WORD) (y >> 16), (WORD) y);
	if (save) restore_fc ();
	if (frozen) RunChip (0L);
}

LONG DumpLong (LONG x)
{
	return GetLong (x);
}

static WORD RegCodes [] = {
BDM_WSREG + BDM_RPC, BDM_WSREG + BDM_USP,
BDM_WSREG + BDM_SSP, BDM_WSREG + BDM_VBR,
BDM_WSREG + BDM_SR, BDM_WSREG + BDM_SFC,
BDM_WSREG + BDM_DFC, BDM_WSREG + BDM_ATEMP,

BDM_WRREG + 0, BDM_WRREG + 1, BDM_WRREG + 2, BDM_WRREG + 3,
BDM_WRREG + 4, BDM_WRREG + 5, BDM_WRREG + 6, BDM_WRREG + 7,

BDM_WRREG + 8, BDM_WRREG + 9, BDM_WRREG + 10, BDM_WRREG + 11,
BDM_WRREG + 12, BDM_WRREG + 13, BDM_WRREG + 14, BDM_WRREG + 15
};

LONG GetReg (unsigned which)
{
	int frozen;
	LONG result;

	frozen = StopChip ();
	result = bdm_read (0L,0,BDM_RDBIT + RegCodes [which]);
	if (frozen) RunChip (0L);
	return result;
}

void PutReg (unsigned which, LONG data)
{
	int frozen;

	frozen = StopChip ();
	bdm_write (0L, 2,RegCodes [which], (WORD) (data >> 16), (WORD) data);
	if (frozen) RunChip (0L);
}

/* end of bdmcpu32.c */
