/*
 *	This file contains an error handling function.
 */


#include <errno.h>
#include <stdio.h>
#include "sm_client.h"

#include <stdlib.h>
#include <strings.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/resource.h>
#include "widget.h"
#include "error_handler.h"

#ifndef __cplusplus
	extern char * getenv(char*);
#endif

static int ConnectRefuseCount = 0;


/*
 *	This function handles server initiated transaction abort
 *	(esmTRANSABORTED) and server failure (esmNOTCONNECTED).
 *	It attempts to reestablish contact with the server if the 
 *	server fails.
 *	
 *	If an internal error is detected, the client will exit.
 *	For all other errors a message is printed and handleError returns.
 */	
void handleError(int code, int line, char* file)
{
	
	switch (code) {
	case esmTRANSABORTED:
		if (sm_AbortTransaction(global_tid) != esmNOERROR) {
			printf("Error aborting transaction (line: %d, file: %s, code: %s) : %s\n", line, file, sm_ErrorId(sm_errno), sm_Error(sm_errno));
			exit(1);
		}
		INVALIDATE_TID(global_tid);
		break;
	case ECONNREFUSED:
		if (ConnectRefuseCount == 0) {
			printf("Could not connect to the server (line: %d, file: %s, code: %s) : %s \n", line, file, sm_ErrorId(sm_errno), sm_Error(sm_errno));
			ConnectRefuseCount++;
			break;
		}
		ConnectRefuseCount++;
	case esmNOTCONNECTED:
		if (sm_AbortTransaction(global_tid) != esmNOERROR) {
			if (sm_errno == esmBADTRANSID) {
				printf("Transaction already aborted\n");	
			} else {
				printf("Error aborting transaction (line: %d, file: %s, code: %s) : %s\n", line, file, sm_ErrorId(sm_errno), sm_Error(sm_errno));
				exit(1);
			}
		}
		printf("Server Died, attempting new transaction in 5 seconds\n");
		INVALIDATE_TID(global_tid);
		sleep(5);
		break;
	case esmLOCKCAUSEDDEADLOCK:
		/* Theoretically, if we retry we may get the lock */
		printf("deadlock, retrying opration on (line: %d, file: %s\n", line, file);
		break;
	case esmINTERNAL:
		printf("Internal error in the storage manager\n");
		exit(1);
		break;
	default:
		printf("unsupported error from storage manager (line: %d, file: %s, code: %s) : %s \n", line, file, sm_ErrorId(sm_errno), sm_Error(sm_errno));
		break;
	}
	
	return;
}
