/*
 * $RCSfile: et.c,v $
 * $Revision: 1.20 $
 * $Date: 1993/04/29 14:16:27 $
 */

#include "et.h"
#include "etError.h"
#include "func.h"
#include "command_funcs.h"

BEGIN_EXTERNC
extern void DumpVolRecTable();
END_EXTERNC


int		reportableProblems = 0;
int 	Numretries = NUMOFRECONNECT;
char	*argv0;

/*
 *	global variables
 */
int     InvalidCount = 0 ;
TID		tid = NULL_TID;		/* transaction id */

int		    bufGroup = NO_BUFGROUP;

char		volName[STRINGSIZE];
FID			Fids[100];
VOLIDINFO		Volids[40];
int			Nvolumes=0;
ObjInfo         UA[ARRAYSIZE];
ObjInfo         CA[ARRAYSIZE];
int		objUACount = -1;
int		objCACount = -10;

int 	objSizeCap = 10000;

char	rootEntryName[STRINGSIZE];

BOOL		interactive = TRUE;
char	varString[VARIABLESIZE];

FILE		*OVFPtr = NULL;	


/*  void handler(int sig, int code) */
void handler()
{
	errno = EPIPE;
	HANDLE_ERROR(CAUSE_UNIX, TREAT_INFO, NOINFO);
}

int
find_volid_index(VOLID v)
{
	register int j; 
	for(j=0; j<Nvolumes; j++) 
		if(Volids[j].volid == v) return j;
	return -1;
}

void
mark_used(VOLID volid) 
{
	register int i;
	i = find_volid_index((int)volid);

	DEBUGINFO
		"mark_used volume %d, index %d, tid %d\n", volid, i, tid
	ENDINFO
	if(i >= 0) {
		Volids[i].lastused = tid;
		Volids[i].empty = FALSE;
	}
}

void
reinit_volid(int n)
{
	Volids[n].connected = FALSE;
	Volids[n].temp = FALSE;
	Volids[n].empty = TRUE; /* as far as we know */
	Volids[n].lastused = NULL_TID;
	Volids[n].crashloc = 0;
	Volids[n].vote = YESVOTE /*(default)*/;
}

VOLID
add_volid(char *str)
{
	int i;

	if( getnum(str, &i)!= esmNOERROR) {
		exit(EXIT_INTERNAL);
	}
	/* don't add it again if it's already there */
	if(find_volid_index(i) >= 0) return i;

	reinit_volid(Nvolumes);
	Volids[Nvolumes++].volid = i;
	return i;
}

static BOOL
anytemp(TID tid)
{
	register int j; 
	for(j=0; j<Nvolumes; j++)  {
		if( (Volids[j].lastused == tid) &&
			istemp(Volids[j].volid)  )
		return TRUE;
	}
	return FALSE;
}

int
doRootEntry(VOLID _volid, FID *fid, BOOL startover, int tries)
{
    int		rootEntryLen;
	BOOL	createData, istempvol;
	jmp_buf	rootentry;

	(void) gettemp(_volid, -1, Numretries);
	istempvol = istemp(_volid);

	if(istempvol) {
		rootEntryLen = 0;
	} else {
		/*
		 * See if the data file is already created.
		 */
		rootEntryLen = sizeof(FID); 

		set_retry();
		sm_errno=esmNOERROR;
		if (sm_GetRootEntry(_volid, rootEntryName, fid, (int *)&rootEntryLen)) {
			if(sm_errno == esmBADROOTNAME) {
				rootEntryLen = 0;
			} else {
				HANDLE_ERRWRECONNECT(CAUSE_SM, TREAT_INFO, NOINFO, _volid, tries);
				return esmFAILURE;
			} /* else ignore */
		} else if (rootEntryLen != sizeof(FID)) {
			HANDLE_ERROR(CAUSE_VALIDATION, TREAT_FATAL, VAL_ROOTENTRY_CHANGED);
		}
		clr_retry();
	}

	if(rootEntryLen == 0) {
		/* Data file does not exist */
		createData = TRUE;
	} else if(startover || istempvol) {
		if(startover) {
			INFO
				"Destroying old data file %d.%d.%d \n", 
					fid->pid.page, fid->pid.volid, fid->unique
			ENDINFO
		}
		set_retry();
		sm_errno = esmNOERROR;
		if (sm_DestroyFile(bufGroup, fid) == esmFAILURE) {
			/* if it's a temp file, we expect to get esmBADFID */
			/* Anything else is an error */
			if((sm_errno != esmBADFID) || (istempvol==FALSE)) {
				HANDLE_ERRWRECONNECT(CAUSE_SM, TREAT_FATAL, SM_CD_FILE_ERR, _volid, tries);
				return esmFAILURE;
			}
		} else {
			clr_retry();
			/* temp file should be gone; should have return esmFAILURE */
			if(istempvol) {
				INFO
					"File %d.%d.%d  on temp volume gave no error\n\t(sm_errno %d) when destroyed!\n",
					fid->pid.page, fid->pid.volid, fid->unique, sm_errno
				ENDINFO
				HANDLE_ERROR(CAUSE_VALIDATION, TREAT_FATAL, SM_CD_FILE_ERR);
			}
		}
		createData = TRUE;
	} else {
		OID oid;
		OBJHDR objHdr;
		BOOL empty;

		/* file should be there!  do a sanity check anyway.*/
		set_retry();
		if(sm_GetFirstOid(bufGroup, fid,  &oid, &objHdr, &empty)) {
			if(sm_errno == esmEMPTYFILE) {
				if(!empty) {
					HANDLE_ERROR(CAUSE_VALIDATION, TREAT_FATAL, SM_CD_FILE_ERR);
				}
			} else {
				HANDLE_ERRWRECONNECT(CAUSE_SM, TREAT_NONFATAL, SM_CD_FILE_ERR, _volid, tries);
				return esmFAILURE;
			}
		}
		clr_retry();
		createData = FALSE;
	}

    if (createData) {
		INFO
		"Creating data file "
		ENDINFO
		set_retry();
		if (sm_CreateFile(bufGroup, _volid, fid)) {
			DEBUGINFO
				"create file failed %d\n", _volid
			ENDINFO
			HANDLE_ERRWRECONNECT(CAUSE_SM, TREAT_FATAL, SM_CD_FILE_ERR, _volid, tries);
			return esmFAILURE;
		}
		clr_retry();
		INFO
		"%d.%d.%d \n", 
				fid->pid.page, fid->pid.volid, fid->unique
		ENDINFO

		if(!istempvol) {
			/*
			 *  set root entry for the data file just created, if this fails, 
			 *  exit with SM_INIT_ERR, and the transaction is NOT committed.
			 */
			set_retry();
			DEBUGINFO
				"set root entry vol %d\n", _volid
			ENDINFO
			if (sm_SetRootEntry(_volid, rootEntryName, fid, sizeof(FID))) {
				HANDLE_ERRWRECONNECT(CAUSE_SM, TREAT_FATAL, SM_SETRTENTRY_ERR, _volid, tries);
			DEBUGINFO
				"failed set root entry vol %d\n", _volid
			ENDINFO
				return esmFAILURE;
			}
			clr_retry();
		}

		/*
		 *  If fail to commit this transaction which created a new data
		 *  file, no further operations can be performed, exit with SM_INIT_ERR.
		 *
		 *  We would like to just call commit_tx(), but this requires
		 * slightly different handling.
		 *  We cannot commit if this is a temp volume.
		 */
		mark_used(fid->pid.volid);
		if(anytemp(tid) == FALSE){
			if (sm_CommitTransaction(tid)) {
				HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, SM_COMMIT_TRANS_ERR);
				/* no point in retrying a commit */
				return esmFAILURE;
			}
			tid = NULL_TID;
			(void) begin_tx(TRUE, TRUE);
		}
	} else {
		INFO
		"Data file %d.%d.%d exists. \n", 
				fid->pid.page, fid->pid.volid, fid->unique
		ENDINFO
		mark_used(fid->pid.volid);
	}

	return esmNOERROR;
}

void
use_evolids()
{
	Nvolumes = 0;
	add_evolids();
}

void
add_evolids()
{
	char *envp;
	char *v;
	char vid[512];

	envp = getenv("EVOLID");
	if((envp == NULL) || (atoi(envp)==0)) {
		if( interactive) {
			INFO
				"No volumes specified. You must use -v on your session command(s).\n"
			ENDINFO
			return;
		} else { 
			INFO
			"No volumes specified. \n"
			ENDINFO
			exit(EXIT_USER);
		}
	}
	strcpy(vid, envp);

	for(v = strtok(vid, " "); v != NULL; v = strtok(NULL, " ") ) {
		(void) add_volid(v);
	}
	if(interactive) {
		int i;

		INFO
			"Using %d volumes: ", Nvolumes
		ENDINFO
		for(i=0; i<Nvolumes; i++ ) {
			INFO
				"%d ",Volids[i].volid
			ENDINFO
		}
	}
}


BOOL debugon = FALSE;

/********************************************************
 *
 *	main
 *
 *******************************************************/
main (int argc, char **argv)
{
	int		cmd;
    OID		oid[MAXOBJS];	/* object id */
	char	*errorMsg;

	argv0	= argv[0];
	sm_errno=esmNOERROR;

	setbuf(stdout, NULL); /* make it unbuffered */

	(void) malloc(1); /* grot */
	checkMalloc("begin", TRUE);

#ifdef DEBUG
	{
		extern void SetFakeFailure(int);

		char *fake;

		FakingFailure = FALSE;
		if(fake = getenv("EFAKEFAIL"))  {
			if(!strcmp(fake, "esmMALLOCFAILED")) {
				SetFakeFailure(esmMALLOCFAILED);
				FakingFailure = TRUE;
			} else if(!strcmp(fake, "esmCANTSWAP")) {
				SetFakeFailure(esmCANTSWAP);
				FakingFailure = TRUE;
			} else if(!strcmp(fake, "esmDISKFULL")) {
				SetFakeFailure(esmDISKFULL);
				FakingFailure = TRUE;
			} else if(!strcmp(fake, "esmNOFREEPAGEHASH")) {
				SetFakeFailure(esmNOFREEPAGEHASH);
				FakingFailure = TRUE;
			} else if(!strcmp(fake, "esmBADVOLID")) {
				SetFakeFailure(esmBADVOLID);
				FakingFailure = TRUE;
			} else {
				fprintf(stderr, "EFAKEFAIL type not supported: %s\n",
					fake);
				exit(1);
			}
		}
	}
#endif DEBUG
	if(getenv("EDEBUG")) debugon = TRUE;
	{ 	char *c;
		if(c = getenv("ERETRIES"))  {
			Numretries = atoi( c);
			if(Numretries == 0)
				Numretries = NUMOFRECONNECT;
		}
	}

    /*
     *	If a file name is specified with 'et' on the command
     *	line, commands are read from the script file.
     */
	/*
    if (argc > 1) {
		interactive = FALSE;
        close(0);

		open_script(argv[1]);
    } else {
	*/
        interactive = isatty(0);
		/*
		 *  If a pipe is used, then isatty returns FALSE.
		 */
	/*
	}
	*/
#if defined(hpux) || defined(linux)
    signal(SIGPIPE, (void (*)(int)) handler);
#else
    signal(SIGPIPE, (SIG_TYP)handler);
#endif hpux

	init_scan();

	/* read the default config file(s) */
	if(sm_ReadConfigFile(NULL, argv0, &errorMsg) != esmNOERROR) {
		INFO
			"Missing or bad config file: %s\n", errorMsg
		ENDINFO
		HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, SM_INIT_ERR);
	}
	if(--argc > 0) {
		argv++;
		if(sm_ParseCommandLine(&argc, argv, &errorMsg) != esmNOERROR) {
			INFO
				"Error processing command line arguments: %s\n", errorMsg
			ENDINFO
			HANDLE_ERROR(CAUSE_SM, TREAT_FATAL, SM_INIT_ERR);
		}
	}

    currentVar =  createVar("current");

	/* 
	 * Get the volids that we are interested in using - should be a subset
	 * of those in the config file.
	 */

	initializeSM(TRUE); /* just in case the first command is a set... */

	if( probe_handle() == FALSE )
		use_evolids();

	for(;;) {
		cmd = get_command(GET_COMMAND_SESSION, TRUE /* prompt */);
		if(cmd != WORD_SESSION) {
			/* shouldn't happen */
			HANDLE_ERROR(CAUSE_ET, TREAT_FATAL, esmINTERNAL);
		}
		if(do_command(cmd) == esmNOERROR) 
			break;
		/* o.w. try again */
	}

	for (;;) {
		sm_errno=esmNOERROR;

		cmd = get_command(GET_COMMAND_ANY, TRUE /* prompt */);
		if(cmd == WORD_NONE) {
			INFO
				"Ignored.\n"
			ENDINFO
		} else {
			do_command(cmd);
		}

	} /* end for */

}

#define RANDOM_VOLID 0x1
#define SPECIFIC_VOLID 0x2
#define CYCLE_VOLID 0x3

static int how_volid = CYCLE_VOLID;
static VOLID specific_volid;
static int cycle_on = 0;

int
set_volid(char * val)
{
	char *arg = next_word(val, Whitespace);

	/* a hack */
	if(strcmp(arg, "random")==0) {
		how_volid = RANDOM_VOLID;
	} else if(strcmp(arg, "cycle")==0) {
		how_volid = CYCLE_VOLID;
	} else {
		how_volid = SPECIFIC_VOLID;
		specific_volid = add_volid(arg); 
	}
	print_volid();
	return esmNOERROR;
}

VOLID 
get_volid()
{
	VOLID retval;

	if(Nvolumes == 0) {
		HANDLE_ERROR(CAUSE_USER, TREAT_FATAL, NOVOLUMES);
	}
	switch(how_volid) {
	case RANDOM_VOLID:
		retval = Volids[ (int) random() % Nvolumes ].volid;
		break;

	case CYCLE_VOLID:
		retval =  Volids[(cycle_on++ % Nvolumes)].volid;
		break;

	case SPECIFIC_VOLID:
		retval = specific_volid;
		break;

	default:
		HANDLE_ERROR(CAUSE_USER, TREAT_FATAL, NOINFO);
	}
	DEBUGINFO
		"get_volid returns %d", retval
	ENDINFO
	return retval;
}

FID *
get_fid_of(VOLID v)
{
	/* find the index of specific_volid */
	/* If file is not there, it does root entry
	 * stuff and creates one 
	 */
	register int j; 

	for(j=0; j<Nvolumes; j++) if(Volids[j].volid == v ) {
		if( Volids[j].empty ) {
			DEBUGINFO
				"No (known) file on volume %d.\n", Volids[j].volid
			ENDINFO
			/* recreate the files */
			if(doRootEntry(Volids[j].volid, &Fids[j], FALSE, Numretries) != esmNOERROR) {
				DEBUGINFO
					"Can't re-create the root entries!\n"
				ENDINFO
				HANDLE_ERROR(CAUSE_USER, TREAT_FATAL, NOINFO);
			}
		}
		DEBUGINFO
			"get_fid_of(%d) returns fid #%d (%d.%d.%d)\n",
			v, j, 
			Fids[j].pid.page,
			Fids[j].pid.volid,
			Fids[j].unique
		ENDINFO
		return &Fids[j];
	}
	HANDLE_ERROR(CAUSE_ET, TREAT_FATAL, NOINFO);
	return NULL; /* no reached */
}

FID *
get_fid()
{
	/* gets a fid for a file on which to operate,
	 * based on the reference pattern: random, cycle, etc
	 */
	VOLID 	volid;
	FID 	*retval = NULL;

	volid = get_volid();
	retval = get_fid_of(volid);
	DEBUGINFO
		"get_fid returns %d.%d.%d\n", 
			retval->pid.page, retval->pid.volid, retval->unique
	ENDINFO
	return retval;
}

void
print_volid()
{
	/* based on volid */
	switch(how_volid) {
	case RANDOM_VOLID:
		INFO
			"volid is random."
		ENDINFO
		break;

	case CYCLE_VOLID:
		INFO
			"volid is cycle."
		ENDINFO
		break;

	case SPECIFIC_VOLID:
		INFO
			"volid is %d.", specific_volid
		ENDINFO
		break;
	}
}

int
set_objsizelimit(char *val)
{
	char *arg = next_word(val, Whitespace);
	int i;

	if( getnum(val, &i)!= esmNOERROR) {
		return esmNOERROR;
	}
	objSizeCap = i;
	return esmNOERROR;
}

void
print_objsizelimit()
{
	INFO
		"object size limit is %d", objSizeCap
	ENDINFO
}

ObjInfo *
oid2objinfo(OID *oid, ObjInfo *array, int count)
{
	ObjInfo		*objinfo;
	int i;

	for(i=0, objinfo = array; i<count; i++, objinfo++) {
		if(objinfo->objID.diskAddr.page != oid->diskAddr.page) continue;
		if(objinfo->objID.diskAddr.volid != oid->diskAddr.volid) continue;
		if(objinfo->objID.diskAddr.slot != oid->diskAddr.slot) continue;
		if(objinfo->objID.diskAddr.unique != oid->diskAddr.unique) continue;
		/* FOUND! */
		return objinfo;
	}
	DEBUGINFO
		"no objinfo for oid %d.%d.%d.%d\n", 
		oid->diskAddr.page,oid->diskAddr.volid,oid->diskAddr.slot,oid->diskAddr.unique
	ENDINFO
	return NULL;
}

void
clear_visited(ObjInfo *array, int count)
{
	ObjInfo		*objinfo;
	int			i;

	for(i=0, objinfo = array; i<count; i++, objinfo++) {
		objinfo->visited = FALSE;
	}
}
void
set_visited(ObjInfo *objinfo)
{
	if(objinfo->visited) {
		OID *oid = &objinfo->objID;
		INFO
			"Object %d, oid=%d.%d.%d.%d WAS FOUND MORE THAN ONCE in the scan!\n",
				oid->diskAddr.page,oid->diskAddr.volid,oid->diskAddr.slot,oid->diskAddr.unique
		ENDINFO
		HANDLE_ERROR(CAUSE_VALIDATION, TREAT_FATAL, NOINFO);
	}
	objinfo->visited = TRUE;
}

void
check_visited(ObjInfo *array, int count, VOLID volid)
{
	OID *oid;
	ObjInfo		*objinfo;
	int			i;

	for(i=0, objinfo = array; i<count; i++, objinfo++) {
		if( objinfo->destroyed )  
			continue;

		oid = &objinfo->objID;

		if(oid->diskAddr.volid != volid)
			continue;

		if( !objinfo->visited ) {
			INFO
				"Object %d, oid=%d.%d.%d.%d was not found in scan!\n",
					objinfo->index,
					oid->diskAddr.page,oid->diskAddr.volid,
					oid->diskAddr.slot,oid->diskAddr.unique
			ENDINFO
			HANDLE_ERROR(CAUSE_VALIDATION, TREAT_FATAL, NOINFO);
		}
	}
}

void
dismount()
{
	register int j; 
	for(j=0; j<Nvolumes; j++)  {
		if(Volids[j].lastused > 0) {
			if(sm_DismountVolume(Volids[j].volid) !=  esmNOERROR ) {
				HANDLE_ERROR(CAUSE_SM, TREAT_INFO, SM_DISMOUNT_VOL_ERR);
			}
			Volids[j].lastused  = 0;
		}
	}
}

void
mark_temp_volumes_empty()
{
	register int j; 
	for(j=0; j<Nvolumes; j++)  {
		if(Volids[j].temp) {
			Volids[j].empty  = TRUE;
		}
	}
}

int
set_loglevel(char *str)
{
	register int i, level;
	char *arg = next_word(str, Whitespace);

	switch( i = word2cmd(arg) ) {
	case WORD_ALL:
		level = LOG_ALL;
		break;
	case WORD_SPACE:
		level = LOG_SPACE;
		break;
	case WORD_NADA:
		level = LOG_NONE;
		break;
	case WORD_NONE:
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, SYNTAXERROR);
		return esmFAILURE;
	}

	if(sm_SetLogLevel(level, Nvolumes, Fids) != esmNOERROR) {
		HANDLE_ERROR(CAUSE_USER, TREAT_NONFATAL, NOINFO);
		return esmFAILURE;
	}

	return esmNOERROR;
}
