#ifndef __LHDIRPAGE_H__
#define __LHDIRPAGE_H__

/*
 *	$RCSfile: LHDIRPAGE.h,v $
 *	$Revision: 1.9 $
 *	$Date: 1993/05/10 15:29:17 $
 */
/**********************************************************************
* EXODUS Database Toolkit Software
* Copyright (c) 1991 Computer Sciences Department, University of
*                    Wisconsin -- Madison
* All Rights Reserved.
*
* Permission to use, copy, modify and distribute this software and its
* documentation is hereby granted, provided that both the copyright
* notice and this permission notice appear in all copies of the
* software, derivative works or modified versions, and any portions
* thereof, and that both notices appear in supporting documentation.
*
* THE COMPUTER SCIENCES DEPARTMENT OF THE UNIVERSITY OF WISCONSIN --
* MADISON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION.  
* THE DEPARTMENT DISCLAIMS ANY LIABILITY OF ANY KIND FOR ANY DAMAGES
* WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
*
* The EXODUS Project Group requests users of this software to return 
* any improvements or extensions that they make to:
*
*   EXODUS Project Group 
*     c/o David J. DeWitt and Michael J. Carey
*   Computer Sciences Department
*   University of Wisconsin -- Madison
*   Madison, WI 53706
*
*	 or exodus@cs.wisc.edu
*
* In addition, the EXODUS Project Group requests that users grant the 
* Computer Sciences Department rights to redistribute these changes.
**********************************************************************/
#include "lh_extfuncs.h"  // change
#include "LH_Hash.h"

/*
 *	Size of a LHDIRPAGE (same as LHINDPAGE) defined in object.h
 */

/*
 *	load factor threshold is defaulted as 75%
 */
#define 	LH_THRESHOLD		0.75

#ifdef __cplusplus			/* LHDIRPAGE defs for C++ only */

// 
//	LHDIR page control structure 
//
struct LHDIRCONTROL {
	LRC         lrc;            // for recovery
	PID         selfID;         // id of this page
	FOUR		curPos;			// position of next entry 
	FOUR        M;              // number of primary pages
	FOUR        P;              // pointer to next page to be splitted
	ONE         I;              // current level
	ONE			type;           // key type
	ONE			height;			// the height of the directory tree			
	ONE 		rootFlag;		// flag to indicate if this is a root page
	float		loadThreshold;	// the threshold of load factor
	float		load;			// current load of primary pages
	};



const LHDIR_MAGIC = 14021967;   // change

//
// 	LHDIR page definition
//
//const LH_TBLSIZE = 3;
const LH_TBLSIZE = (BTREE_PAGESIZE - sizeof(LHDIRCONTROL) - 100) / sizeof(SHORTPID);

class LHDIRPAGE {
	
	// 
	//	Physical undos and redos must access page directly
	//
	friend void undoLHash_fr(LOGRECORDHDR*);
	friend void redoLHash_fr(LOGRECORDHDR*);

		LHDIRCONTROL	lhDirCtrl;				// control information
		SHORTPID		indPid[LH_TBLSIZE];		// index page table

	public:
		//
		//	General queries
		//
		PID&		SelfID()		{ return lhDirCtrl.selfID; }
		int			Volume()		{ return lhDirCtrl.selfID.volid; }
		SMDATATYPE	KeyType()		{ return (SMDATATYPE) lhDirCtrl.type; }
		FOUR		Primary()		{ return lhDirCtrl.M; }
		FOUR		SplitPtr()		{ return lhDirCtrl.P; }
		int			Level()			{ return (int) lhDirCtrl.I; }
		float		Load()			{ return lhDirCtrl.load; }
		float		Threshold()		{ return lhDirCtrl.loadThreshold; }
		int			Height()		{ return (int) lhDirCtrl.height; }
		int			CurPos()		{ return (int) lhDirCtrl.curPos; } 
		int			IsRoot()		{ return lhDirCtrl.rootFlag; }

		//
		//	address of start of table
		//
		char*		FirstEntry()	{ return (char*) &indPid[0]; }
		
		//
		//	LRC 
		//
		LRC&		lrc()			{ return lhDirCtrl.lrc; }

		//
		//	Return TRUE if this page is safe
		//
//		int			IsSafe(LH_OPER oper, int maxKeyLen);

		//
		//	Get the index page id at slot position within index table
		//
		SHORTPID	GetEntry(FOUR slot)	{ 
								ASSERT1( slot>=0 && slot < lhDirCtrl.curPos );
								return indPid[slot]; 
								}
		//
		//	Set the next entry to contain the new index page id.
		//
		void		SetNextEntry(SHORTPID indPage)	{ 
								ASSERT1( lhDirCtrl.curPos < LH_TBLSIZE );
								indPid[lhDirCtrl.curPos] = indPage;
								lhDirCtrl.curPos++;
								}
		//
		//	UnSet the next entry to contain the new index page id.
		//
		void		UnSetNextEntry(SHORTPID unused /*indPage*/)	{ 
								ASSERT1( lhDirCtrl.curPos > 0 );
								indPid[--lhDirCtrl.curPos] = NULLPID;
								}

		//
		//	Set the specified entry to contain the new index page id.
		//	For original primary pages only
		//
		void		SetEntry(SHORTPID indPage, FOUR slot)	{ 
								ASSERT1( slot < INIT_PRIMARY );
								// asserts may cause failure in recovery 
								//  under debug mode.
								// ASSERT1( lhDirCtrl.M == INIT_PRIMARY );
								// ASSERT1( lhDirCtrl.curPos == INIT_PRIMARY );
								indPid[slot] = indPage;
								}

		//
		//  Copy the whole directory page
		//
        //
		void        CopyData(LHDIRPAGE* target) {
						bcopy((char*)indPid, (char*)target->indPid, 
								LH_TBLSIZE*sizeof(SHORTPID));
					 	target->lhDirCtrl.height = lhDirCtrl.height;
						target->lhDirCtrl.curPos = lhDirCtrl.curPos;
						}

		//
		//	Initialize a new directory page
		//
		void 		Init(PID& const pid, int rootFlag, SMDATATYPE keyType) {
								lhDirCtrl.selfID = pid;
								lhDirCtrl.type = keyType;
								lhDirCtrl.M = 0;
								lhDirCtrl.P = 0;
								lhDirCtrl.I = 0;
								lhDirCtrl.load = 0.0;
								lhDirCtrl.height = 1;
								lhDirCtrl.curPos = 0;
								lhDirCtrl.rootFlag = rootFlag;
								lhDirCtrl.loadThreshold = LH_THRESHOLD; 
								}
		//
		//  update the global index page count
		//	only root directory page can perform the function
		//
		void 		IncrCount() { ASSERT1(lhDirCtrl.rootFlag);
								  lhDirCtrl.M++;
								  }
	
		//
		//	calculate the index value
		//
		int			IndexCalc(int keyLen, void* keyVal);

		//
		//	calculate the directory capacity with given height
		//
		int			Capacity(int height);

        //
		//  update function used in index page splitting
		//
		void        IndSplitUpdate(float oldFromLoad, float fromLoad,
									float toLoad);
        
		//
		//  update function used in index page shrinking
		//
		void        IndShrinkUpdate(float fromLoad, float toLoad, 
									float newFromLoad);
		
		//
		//	update function used in directory page splitting
		//		reset curPos and increment height
		//
		void        DirSplitUpdate() {
								lhDirCtrl.curPos = 0;
								lhDirCtrl.height++;
								}
		//
		//  update load change
		//
		void 		LoadUpdate(float oldLoad, float newLoad);

		//
		//  set load 
		//
		void		SetLoad(float load) {
						ASSERT1(lhDirCtrl.rootFlag);
						lhDirCtrl.load = load;
						}
		//
		//  set up load factor threshold
		//
		void		SetThreshold(float threshold) {
						ASSERT1(lhDirCtrl.rootFlag);
						lhDirCtrl.loadThreshold = threshold;
						}

		//
		// set up height: used in bulk load only
		//
		void		SetHeight(int level) {
						lhDirCtrl.height = level;
						}
								
		//
		//	Debugging utilities
		//
		int  		CheckContent();
		void		Print();
	};

#endif		/* #ifdef C++ */

#endif __LHDIRPAGE_H__
