#ifndef BT_TUPLE_H
#define BT_TUPLE_H
/*
 *	$RCSfile: BT_Tuple.h,v $
 *	$Revision: 1.13 $
 *	$Date: 1993/05/07 15:10:26 $
 */
/**********************************************************************
 * EXODUS Database Toolkit Software
 * Copyright (c) 1991 Computer Sciences Department, University of
 *                    Wisconsin -- Madison
 * All Rights Reserved.
 *
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 *
 * THE COMPUTER SCIENCES DEPARTMENT OF THE UNIVERSITY OF WISCONSIN --
 * MADISON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION.  
 * THE DEPARTMENT DISCLAIMS ANY LIABILITY OF ANY KIND FOR ANY DAMAGES
 * WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * The EXODUS Project Group requests users of this software to return 
 * any improvements or extensions that they make to:
 *
 *   EXODUS Project Group 
 *     c/o David J. DeWitt and Michael J. Carey
 *   Computer Sciences Department
 *   University of Wisconsin -- Madison
 *   Madison, WI 53706
 *
 *	 or exodus@cs.wisc.edu
 *
 * In addition, the EXODUS Project Group requests that users grant the 
 * Computer Sciences Department rights to redistribute these changes.
 **********************************************************************/

/*
 *	A tuple could be of type LEAF or NODE. LEAF tuple contains elCnt 
 *	and ovFlag. NODE tuple contains childPid.
 */
enum BT_TupleType { LEAFTUPLE = 0x37, NODETUPLE };

struct LeafAttr {
    ONE	ovFlag;		// overflow flag
    ONE	elSize;		// element size
    TWO	elCnt;		// oid counter
};

struct NodeAttr {
    SHORTPID	childPid;
};

union Attr_U {
    LeafAttr	leafAttr;
    NodeAttr	nodeAttr;
};

/*
 *	Size declaration of a tuple. 
 */
const MAXTUPLESIZE = BT_DATASIZE / 2 - sizeof(SLOT);
const FIXEDMEM	   = sizeof(TWO) + 2 * sizeof(ONE) + sizeof(Attr_U);
const MAXKEYLEN	   = MAXTUPLESIZE - FIXEDMEM - SM_MAXELEMLEN;



/*
 * 	BT_Tuple class. 
 */
class BT_Tuple {
    
    TWO		klen;		// length of key 
    ONE		align;		// alignment added to key
    ONE		type;		// BT_TupleTupe
    Attr_U	attr;		// attribute of the tuple 
    union {	
	char	key[1];		// key and (oidArray or overflowRoot)
	ONE	total_space[MAXTUPLESIZE - FIXEDMEM];
    };
    
public:
    
    /*
     *	Functions to manipulate OidArray
     */
    void*	ElArray(int idx);
    int		AddElem(const void* newElem);
    int		RemoveElem(const void* doomedElem);
    
    //
    //	Functions to manipulate oid counter
    //
    int		ElCnt();
    int		ElSize();
    void	IncrElCnt();
    void 	DecrElCnt();
    
    //
    //	Functions to get key attributes
    //
    int 	KeyLen()		{ return klen; }
    void* 	KeyValue()		{ return key; }
    
    //
    //	Get child pointer
    //
    SHORTPID Vector()		{ 
	ASSERT3(IsNodeTuple());
	return attr.nodeAttr.childPid; 
    }
    
    //
    //	Functions dealing with tuple overflow
    //
    int		WillOverflow()	{ 
	ASSERT3(IsLeafTuple());
	return Size() + ElSize() > MAXTUPLESIZE; 
    }
    int		IsOverflow()	{ 
	ASSERT3(IsLeafTuple());
	return attr.leafAttr.ovFlag; 
    }
    void	SetOverflow(const PID& ovRoot);
    void	ResetOverflow(int cnt, const void* elList);
    PID&	OverflowRoot();
    
    //
    //	Checks
    //
    int 	IsNodeTuple()	{ return type == NODETUPLE; }
    int 	IsLeafTuple()	{ return type == LEAFTUPLE; }
    
    int		PredictSize(int keyLen)  {
	ASSERT3(this == 0);
	return ALIGNSIZE(keyLen) + (key-(ONE*)this) + SM_MAXELEMLEN;
    }
    int		Size();

    /* create leaf */
    void 	Create(int keyLen, const void* keyPtr,
		       const void* elem, int elSize);

    /* create node */
    void 	Create(int keyLen, void* keyPtr, SHORTPID child);

};


#if (defined DEBUG && defined btree_MODULE) || ! defined DEBUG

INLINE int BT_Tuple::ElSize()
{ 
    ASSERT3(IsLeafTuple());
    ASSERT3(attr.leafAttr.elSize >=0 &&
	    attr.leafAttr.elSize <= SM_MAXELEMLEN);
    return attr.leafAttr.elSize;
}

INLINE void* BT_Tuple::ElArray(int idx)
{
    ASSERT3(!IsOverflow());
    ASSERT3(idx < attr.leafAttr.elCnt);
    return key + (klen + align + idx * ElSize());
}

INLINE int BT_Tuple::ElCnt()
{ 
    ASSERT3(IsLeafTuple());
    return attr.leafAttr.elCnt; 
}


INLINE void BT_Tuple::IncrElCnt()
{ 
    ASSERT3(IsOverflow());
    attr.leafAttr.elCnt++;
}


INLINE void BT_Tuple::DecrElCnt()
{ 
    ASSERT3(IsOverflow());
    ASSERT3(ElCnt() > 0);
    attr.leafAttr.elCnt--;
}

INLINE BT_Tuple::Size()
{
    int len = align + klen + (key - (ONE*)this);
    if (type == LEAFTUPLE)
	len += ALIGNSIZE(IsOverflow() ? sizeof(PID) : ElCnt() *
			 ElSize());

    return len;
}


INLINE void BT_Tuple::SetOverflow(const PID& ovRoot)
{ 
    ASSERT3(IsLeafTuple());
    ASSERT3(attr.leafAttr.ovFlag == 0);
    attr.leafAttr.ovFlag = 1;
    OverflowRoot() = ovRoot;
}

INLINE void BT_Tuple::ResetOverflow(int cnt, const void* elList) 
{
    ASSERT3(IsLeafTuple());
    ASSERT3(IsOverflow());
    ASSERT3(cnt == ElCnt());
    attr.leafAttr.ovFlag = 0;
    BCOPY(elList, ElArray(0), cnt * ElSize());
}


INLINE PID& BT_Tuple::OverflowRoot()
{ 
    ASSERT3(IsOverflow());
    return *((PID*)(key+klen+align));
}

#endif /* (DEBUG && btree_MODULE) || !DEBUG */
#endif	// BT_TUPLE_H
