/* xtmix  - an X11 interface to all mixers supported by
 *          Hannu Savolainen's VoxWare (which is included 
 *          in most Linux distributions).
 *
 * (c) 1995 by Martin Denn <mdenn@unix-ag.uni-kl.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
/* X stuff */
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Form.h>
#include <X11/xpm.h>

/* Linux stuff */
#include <sys/types.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <stdio.h>
#include <math.h>
#include <linux/soundcard.h>

/* selfmade Widgets */
#include "Dial.h"
#include "Slide.h"

/* xpm-bitmaps */
#include "bitmaps/cd.xpm"
#include "bitmaps/synth.xpm"
#include "bitmaps/mic.xpm"
#include "bitmaps/pcm.xpm"
#include "bitmaps/line.xpm"
#include "bitmaps/speaker.xpm"
#include "bitmaps/empty.xpm"
#include "bitmaps/mix.xpm"
#include "bitmaps/pcm2.xpm"
#include "bitmaps/rec.xpm"

/* X bitmap to build the icon */
#include "bitmaps/xtmix.xbm"

/* misc defines */
#define VERSION		"0.2"
#define MIXERDEVICE     "/dev/mixer"
#define BALANCE		100
#define MAX(x,y)	(x>y?x:y)
#define SQR(x)		(x*x)
#define CIRCLE(x)	(sqrt(SQR((50.0))-SQR((x-50.0))))
#define CORRECTION	5

/* Action Prototypes */
void Messages(), Quit(), read_mixer_values();

/* Callback Prototypes */
void slideCallback(), slideRBCallback(), dialCallback();

static XtActionsRec actions[] =
{
    {"Messages", Messages},
    {"Quit", Quit},
    {"ReadValues", read_mixer_values},
};

static char topLevelTranslations[] =
    "<Message>: Messages() \n\
     <Enter>:   ReadValues() \n\
     <Expose>:  ReadValues()";

static char formTranslations[] =
    "<Key>q:    Quit()";

static String fallback_resources[] =
{
    "*Dial.width:40",
    "*Dial.height:50",
    "*Dial.borderWidth:0",
    "*Slide.width:16",
    "*Slide.height:109",
    "*Slide.borderWidth:0",
    "*vol.label:Vol",
    "*balance.label:Bal",
    "*bass.label:Bas",
    "*treble.label:Tre",
    NULL, /* Must be NULL terminated */
};



/* SOUND_DEVICE_NAMES comes from <linux/soundcard.h> */
char *soundDeviceNames[] = SOUND_DEVICE_NAMES;
char *progname;
int supported, stereodevs, recmask;
int mixer_fd;
int nr_of_devices;
int error_field;
Widget wInfo[SOUND_MIXER_NRDEVICES];
Widget topLevel, form;
static Atom delete_w;

/* ARGSUSED */
void Messages(Widget w, XClientMessageEvent *event,
                    String *params, Cardinal *count)
/*
 * This routine implements the DELETE_WINDOW protocoll, which means that
 * you can use the double-click in the title bar's left button to close
 * the window.
 */ 
{
    if ((Atom) event->data.l[0] == delete_w) exit(0);
}

/* ARGSUSED */
void Quit(Widget w, XClientMessageEvent *event,
                    String *params, Cardinal *count)
{
    exit(0);
}

void fatal()
{
    fprintf(stderr,"FATAL: your sound board probably does not have a mixer device\n");
    exit(-1);
}

void mixerInit()
/* init mixer and get some useful constants */
{
    mixer_fd = open (MIXERDEVICE, O_RDWR, 0);
    if (mixer_fd < 0 )
    {
        fprintf (stderr, "%s: Error opening mixer device\n", progname);
        fprintf (stderr, "FATAL: %s probably does not exist or has no r/w permissions\n", MIXERDEVICE);
        exit (-1);
    }
    if (ioctl(mixer_fd, SOUND_MIXER_READ_DEVMASK, &supported) == -1)
        supported = ((1 << (SOUND_MIXER_NRDEVICES))-1);

    if (ioctl(mixer_fd, SOUND_MIXER_READ_STEREODEVS, &stereodevs) == -1)
        stereodevs = ((1 << (SOUND_MIXER_NRDEVICES))-1);

    if (ioctl(mixer_fd, SOUND_MIXER_READ_RECMASK, &recmask) == -1)
        stereodevs = ((1 << (SOUND_MIXER_NRDEVICES))-1);
    error_field = supported;
        

}

void createPixmap(Display *dpy, int i, Pixmap *pix)
/* build pixmaps using the Xpm library */
{
    Pixmap dummy= XtUnspecifiedPixmap;
    char **data;
    XpmAttributes attributes;

    attributes.valuemask = XpmSize;
    switch (i)
    {
        case SOUND_MIXER_MIC:
            data = mic_xpm;
            break;
        case SOUND_MIXER_CD:
            data = cd_xpm;
            break;
        case SOUND_MIXER_SYNTH:
            data = synth_xpm;
            break;
        case SOUND_MIXER_LINE:
            data = line_xpm;
            break;
        case SOUND_MIXER_PCM:
            data = pcm_xpm;
            break;
        case SOUND_MIXER_SPEAKER:
            data = speaker_xpm;
            break;
/* Well, the next pictures are rather ugly */
         case SOUND_MIXER_IMIX:
             data = mix_xpm;
             break;
         case SOUND_MIXER_ALTPCM:
             data = pcm2_xpm;
             break;
         case SOUND_MIXER_RECLEV:
             data = rec_xpm;
             break;
/* (there was no need for it, because I only own a SB Pro). So, if someone 
 * wants to create better pictures, feel free to send them to me: 
 * mdenn@unix-ag.uni-kl.de
 * Thanks :-)
 */
        default:
            data = empty_xpm;
    }
    if (data != NULL)
        XpmCreatePixmapFromData(dpy, DefaultRootWindow(dpy),
                                data, pix, &dummy, &attributes);
    if ((dummy != XtUnspecifiedPixmap) && (dummy != 0))
    {
        /* erase mask-pixmap */
        XFreePixmap(dpy, dummy);
    }
}


void createWidgets(int *argc, char **argv, XtAppContext *app_context)
/*
 * This part is rather messy, but here we arrange all widgets:
 * the slide bars to the left, the dial buttons to the right.
 */
{
    int i, n = 0;
    char *left = NULL;
    Pixmap pix;
    Boolean RB, common;

    topLevel = XtVaAppInitialize(
        app_context,        /* Application context */
	"XtMix",           /* Application class */
        NULL, 0,            /* command line option list */
        argc, argv,        /* command line args */
        fallback_resources, /* for missing app-defaults file */
        NULL);              /* terminate varargs list */
    form = XtVaCreateManagedWidget("form", formWidgetClass, topLevel, NULL);

    for (i=0; i<SOUND_MIXER_NRDEVICES; i++)
    {
        if ((i != SOUND_MIXER_VOLUME) && (((1 << i) & supported) != 0) &&
            (i != SOUND_MIXER_BASS) && (i != SOUND_MIXER_TREBLE))
        {
            /* we create a slide bar: */
            createPixmap(XtDisplay(topLevel),i, &pix);
            RB = (((1 << i) & recmask) != 0);
            common = (((1 << i) & stereodevs) == 0);
            if (left == NULL)
            {
                wInfo[n] = XtVaCreateManagedWidget(soundDeviceNames[i],
                                               slideWidgetClass, form,
                                               XtNpixmap, pix, 
                                               XtNrBExists, RB,
                                               XtNcommon, common,
                                               NULL);
            }
            else
            {
                wInfo[n] = XtVaCreateManagedWidget(soundDeviceNames[i],
                                               slideWidgetClass, form,
                                               XtVaTypedArg, XtNfromHoriz, 
                                               XtRString, left,
                                               strlen(left)+1,
                                               XtNpixmap, pix,
                                               XtNrBExists, RB,
                                               XtNcommon, common,
                                               NULL);
            }
            left = soundDeviceNames[i];
            XtAddCallback(wInfo[n], XtNcallback, slideCallback, (XtPointer) i);
            if (RB)
                XtAddCallback(wInfo[n], XtNrBCallback, slideRBCallback,
                              (XtPointer) i);
            n++; 
        }
    } 
    /* the rest will get dial buttons */
    i = SOUND_MIXER_VOLUME;
    if (((1 << i) & supported) != 0)
    {
        wInfo[n] = XtVaCreateManagedWidget(soundDeviceNames[i],
                                       dialWidgetClass, form,
                                       XtVaTypedArg, XtNfromHoriz, 
                                       XtRString, left,
                                       strlen(left)+1,
                                       NULL);
        XtAddCallback(wInfo[n], XtNcallback, dialCallback, (XtPointer) i);
        n++;
        wInfo[n] = XtVaCreateManagedWidget("balance",
                                       dialWidgetClass, form,
                                       XtVaTypedArg, XtNfromHoriz, 
                                       XtRString, left,
                                       strlen(left)+1,
                                       XtVaTypedArg, XtNfromVert, 
                                       XtRString, soundDeviceNames[i],
                                       strlen(soundDeviceNames[i])+1,
                                       NULL);
        XtAddCallback(wInfo[n], XtNcallback, dialCallback, (XtPointer) 100);
        left = soundDeviceNames[i];
        n++;

    }
    i = SOUND_MIXER_BASS;
    if (((1 << i) & supported) != 0)
    {
        wInfo[n] = XtVaCreateManagedWidget(soundDeviceNames[i],
                                       dialWidgetClass, form,
                                       XtVaTypedArg, XtNfromHoriz, 
                                       XtRString, left,
                                       strlen(left)+1,
                                       NULL);
        XtAddCallback(wInfo[n], XtNcallback, dialCallback, (XtPointer) i);
        n++;
    }
    i = SOUND_MIXER_TREBLE;
    if (((1 << i) & supported) != 0)
    {
        wInfo[n] = XtVaCreateManagedWidget(soundDeviceNames[i],
                                       dialWidgetClass, form,
                                       XtVaTypedArg, XtNfromHoriz, 
                                       XtRString, left,
                                       strlen(left)+1,
                                       XtVaTypedArg, XtNfromVert, 
                                       XtRString, soundDeviceNames[SOUND_MIXER_BASS],
                                       strlen(soundDeviceNames[SOUND_MIXER_BASS])+1,
                                       NULL);
        XtAddCallback(wInfo[n], XtNcallback, dialCallback, (XtPointer) i);
        n++;
    }
    nr_of_devices = n;    
}

/* ARGSUSED */
void slideCallback(Widget w, XtPointer client_data, XtPointer call_data)
/* 
 * when you move the slide, this routine gets invoced and writes the mixer
 * settings.
 */
{
    int devNr = (int) client_data;
    SlideInfo *info = (SlideInfo *) call_data;
    int value;
    
    value = ( info->lvalue | ((info->rvalue) << 8));
    if (ioctl(mixer_fd,MIXER_WRITE(devNr),&value) == -1)
    {
        fprintf(stderr,"%s: Error writing %s in %s\n",
                       progname, soundDeviceNames[devNr], MIXERDEVICE);
        error_field = (error_field & ~(1 << devNr));
        if (error_field == 0) fatal();
    }
}

/* ARGSUSED */
void slideRBCallback(Widget w, XtPointer client_data, XtPointer call_data)
/*
 * This is invoced, when you change the recording source
 */
{
    int devNr = (int) client_data;
    int value, i;
    
    value = 1 << devNr;
    if (ioctl(mixer_fd,SOUND_MIXER_WRITE_RECSRC,&value) == -1)
    {
        fprintf(stderr,"%s: Error setting recording source to %s\n",
                      progname, soundDeviceNames[devNr]);
    }
    /* reset all other RadioButtons */
    for (i=0; i<SOUND_MIXER_NRDEVICES; i++)
    {
        if ((((1 << i) & recmask) != 0) && (i != devNr))
            XtVaSetValues(XtNameToWidget(form, soundDeviceNames[i]),
                          XtNrBOn, False, NULL);
    }
}

/* ARGSUSED */
void dialCallback(Widget w, XtPointer client_data, XtPointer call_data)
/*
 * when you move the dial button, this procedure is called. The
 * balance/volume stuff seems rather complicated; I used this function:
 *
 * (Percentage of volume)
 *   100%            xxxxxx|xxxxxxxxxxxxxxxx 
 *               xxxx      |
 *            xxx          |
 *           x             |
 *    50%   x              |                  (right channel, 
 *         x               |                   left is vice versa)
 *         x               |
 *        x                |
 *     0% x                |
 *        +----------------+----------------+
 *      -100               0               100 (Balance Value)
 */
{
    int devNr = (int) client_data;
    DialInfo *info = (DialInfo *) call_data;
    int value,n=0,vol,bal,left_vol, right_vol;
    
    switch (devNr)
    {
        case SOUND_MIXER_BASS:
        case SOUND_MIXER_TREBLE:
            value = ( info->value | ((info->value) << 8));
            n=devNr;
            break;
        case SOUND_MIXER_VOLUME:
        case BALANCE:
            if (devNr==SOUND_MIXER_VOLUME) 
            {
                vol=info->value;
                XtVaGetValues(XtNameToWidget(form,"balance"),
                              XtNcurValue, &bal, NULL);
            }
            if (devNr==BALANCE)
            {
                bal=info->value;
                XtVaGetValues(XtNameToWidget(form,
                              soundDeviceNames[SOUND_MIXER_VOLUME]),
                              XtNcurValue, &vol, NULL);
            }
            left_vol = (int) ((bal<50)?50:CIRCLE((bal)))*vol/50.0;
            right_vol= (int) ((bal>50)?50:CIRCLE((bal)))*vol/50.0;
            value = ( left_vol | ((right_vol) << 8));
            n=SOUND_MIXER_VOLUME;
            break; 
        default: 
            break;
    }
    if (ioctl(mixer_fd,MIXER_WRITE(n),&value) == -1)
    {
        fprintf(stderr,"%s: Error writing %s in %s\n",
                       progname, soundDeviceNames[devNr], MIXERDEVICE);
        error_field = (error_field & ~(1 << devNr));
        if (error_field == 0) fatal();
    }

}

void read_mixer_values()
/* this procedure reads the values from the mixer device */
{
    int i, value, lvalue, rvalue, vol, bal, left_vol, right_vol, rec_src;


    if (ioctl(mixer_fd,SOUND_MIXER_READ_RECSRC,&rec_src) == -1)
    {
        fprintf(stderr,"%s: Error reading recording source\n", progname);
    }
        
    for (i=0; i<SOUND_MIXER_NRDEVICES; i++)
    {
        if (((1 << i) & supported) != 0)
        {
            if (ioctl(mixer_fd,MIXER_READ(i),&value) == -1)
            {
                fprintf(stderr,"%s: Error reading %s in %s\n",
                        progname, soundDeviceNames[i], MIXERDEVICE);
                error_field = (error_field & ~(1 << i));
                if (error_field == 0) fatal();
            }
            else
            {
                lvalue=(value & 0x00FF);
                rvalue=((value & 0xFF00) >> 8);
                switch (i)
                {
                    case SOUND_MIXER_VOLUME:
                        vol = MAX((lvalue),(rvalue));
                        XtVaSetValues(XtNameToWidget(form, soundDeviceNames[i]),
                                      XtNcurValue, vol, NULL);

                        /* test old values */
                        XtVaGetValues(XtNameToWidget(form,"balance"),
                                      XtNcurValue, &bal, NULL);
                        left_vol = (int) ((bal<50)?50:CIRCLE((bal)))*vol/50.0;
                        right_vol= (int) ((bal>50)?50:CIRCLE((bal)))*vol/50.0;
                        /* if old values are incorrect -> calculate new ones */
                        if ((abs(left_vol-lvalue)>CORRECTION)
                           ||(abs(right_vol-rvalue)>CORRECTION))
                        {
                            if ((vol==0)||
                                ((vol>0)&&(abs(lvalue-rvalue)<CORRECTION))) 
                                bal=50;
                            else if ((vol>0)&&(lvalue<rvalue))
                            {
                                bal=50+CIRCLE((50-lvalue*50/rvalue));
                            }
                            else
                            {
                                bal=50-CIRCLE((50-rvalue*50/lvalue));
                            }
                            XtVaSetValues(XtNameToWidget(form,"balance"),
                                          XtNcurValue, bal, NULL);
                        }
                        break;
                    case SOUND_MIXER_BASS:
                    case SOUND_MIXER_TREBLE:
                        XtVaSetValues(XtNameToWidget(form, soundDeviceNames[i]),
                                      XtNcurValue, (lvalue+rvalue)/2, NULL);
                        break;
                    default:
                        XtVaSetValues(XtNameToWidget(form, soundDeviceNames[i]),
                                      XtNleftValue, lvalue,
                                      XtNrightValue, rvalue,
                                      XtNrBOn, (((1 << i) & rec_src) != 0),
                                      NULL);
                        break;
                }
            }
        }
    }           
}

void fix_size(Widget w)
{
    int ww, wh, dummyint;
    Window dummywin;
    
    XGetGeometry(XtDisplay(w), XtWindow(w), &dummywin, &dummyint,
                 &dummyint, &ww, &wh, &dummyint, &dummyint);
    XtVaSetValues(w, XtNminHeight, wh, XtNminWidth, ww, NULL);
}

void info()
{
    
    fprintf(stderr, "USAGE: %s [ X-Toolkit-Options ...]\n",progname);
    fprintf(stderr, "xtmix  %s - an X Windows interface to the Linux Sound Driver Mixer\n",VERSION);
    fprintf(stderr, "       (c) 1995 by Martin Denn <mdenn@unix-ag.uni-kl.de>\n");
#if defined(CTIME) && defined(CNAME)
    fprintf(stderr, "compiled %s by %s\n",CTIME,CNAME);
#endif
    exit(-1);
}

void main(int argc, char **argv)
/* main routine */
{
    XtAppContext app_context;
    XtTranslations trans;
    Pixmap icon_pixmap;
    
    progname = argv[0];

    /* init mixer and widgets */
    mixerInit();
    createWidgets(&argc, argv, &app_context);
    
    /* any options left? */
    if (argc > 1) info();

    /* add actions and translations */
    XtAppAddActions(app_context, actions, XtNumber(actions));
    trans = XtParseTranslationTable(topLevelTranslations);
    XtOverrideTranslations(topLevel, trans);
    trans = XtParseTranslationTable(formTranslations);
    XtOverrideTranslations(form, trans);

    /* add icon pixmap */
    icon_pixmap = XCreateBitmapFromData(XtDisplay (topLevel), 
			 RootWindowOfScreen (XtScreen (topLevel)),
			 xtmix_bits, xtmix_width, xtmix_height);
    XtVaSetValues (topLevel, XtNiconPixmap, icon_pixmap, NULL);

    /* read initial mixer values */
    read_mixer_values();
    
    /* Map Widgets */
    XtRealizeWidget(topLevel);
    
    /* set minimal size = actual size */
    fix_size(topLevel);
    
    /* add the "delete" - feature to xtmix */
    delete_w = XInternAtom(XtDisplay(topLevel), "WM_DELETE_WINDOW", True);
    XSetWMProtocols(XtDisplay(topLevel), XtWindow(topLevel),
                    &delete_w, 1);

    /* Xt will do the rest */
    XtAppMainLoop(app_context);
}

        
