/*
 * Dial.c - dial widget.
 *
 * (c) 1995 by Martin Denn <mdenn@unix-ag.uni-kl.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>


#include <stdio.h>
#include <math.h>

#include "bitmaps/gray1"

#include "DialP.h"

#define MIN(x,y) (x < y ? x : y)
#define MAX(x,y) (x > y ? x : y)
#define MAXVALUE 100

#define offset(field) XtOffsetOf(DialRec, field)

static XtResource resources[] = {
     {
	XtNdialBorderColor, 
	XtCDialBorderColor, 
	XtRPixel, 
	sizeof(Pixel),
	offset(dial.dial_border_color), 
	XtRString, 
	XtDefaultForeground
     },
     {
	XtNdialColor, 
	XtCDialColor, 
	XtRPixel, 
	sizeof(Pixel),
	offset(dial.dial_color), 
	XtRString, 
	XtDefaultForeground
     },
     {
	XtNdialBackColor, 
	XtCDialBackColor, 
	XtRPixel, 
	sizeof(Pixel),
	offset(dial.dial_back_color), 
	XtRString, 
	XtDefaultBackground
     },
     {
	XtNdialPointColor, 
	XtCDialPointColor, 
	XtRPixel, 
	sizeof(Pixel),
	offset(dial.dial_point_color), 
	XtRString, 
	XtDefaultBackground
     },
     {
	XtNdialPointBorderColor, 
	XtCDialPointBorderColor, 
	XtRPixel, 
	sizeof(Pixel),
	offset(dial.dial_point_border_color), 
	XtRString, 
	XtDefaultForeground
     },
     {
	XtNlabelColor, 
	XtCLabelColor, 
	XtRPixel, 
	sizeof(Pixel),
	offset(dial.label_color), 
	XtRString, 
	XtDefaultForeground
     },
     {
	XtNcallback, 
	XtCCallback, 
	XtRCallback, 
	sizeof(XtPointer),
	offset(dial.callback), 
	XtRCallback, 
	NULL
     },
     {
        XtNcurValue,
        XtCCurValue,
        XtRInt,
        sizeof(int),
        offset(dial.cur_value),
        XtRImmediate,
        (XtPointer) 0
     },
     {
        XtNfont,
        XtCFont,
        XtRFontStruct,
        sizeof(XFontStruct*),
        offset(dial.font),
        XtRString,
        XtDefaultFont
     },
     {
        XtNlabel,
        XtCLabel,
        XtRString,
        sizeof(String),
        offset(dial.label),
        XtRString,
        NULL
     },
};

/* Declaration of methods */

static void Initialize();
static void Redisplay();
static void Destroy();
static void Resize();
static Boolean SetValues();

/* the following are private functions unique to Dial */
static void UpdateDial();

/* the following are actions of Dial */
static void MoveDial(); 

static char defaultTranslations[] =
	"<Btn1Down>:    MoveDial()              \n\
	<Btn1Motion>:  MoveDial()";

static XtActionsRec actions[] = {
        {"MoveDial", MoveDial},
};

/* definition in Dial.h */
static DialInfo info;

DialClassRec dialClassRec = {
    {
    /* core_class fields */
    /* superclass	  	 */ (WidgetClass) &coreClassRec,
    /* class_name	  	 */ "Dial",
    /* widget_size	  	 */ sizeof(DialRec),
    /* class_initialize   	 */ NULL,
    /* class_part_initialize	 */ NULL,
    /* class_inited       	 */ FALSE,
    /* initialize	  	 */ Initialize,
    /* initialize_hook		 */ NULL,
    /* realize		  	 */ XtInheritRealize,
    /* actions		  	 */ actions,
    /* num_actions	  	 */ XtNumber(actions),
    /* resources	  	 */ resources,
    /* num_resources	  	 */ XtNumber(resources),
    /* xrm_class	  	 */ NULLQUARK,
    /* compress_motion	  	 */ TRUE,
    /* compress_exposure  	 */ XtExposeCompressMultiple,
    /* compress_enterleave	 */ TRUE,
    /* visible_interest	  	 */ FALSE,
    /* destroy		  	 */ Destroy,
    /* resize		  	 */ Resize,
    /* expose		  	 */ Redisplay,
    /* set_values	  	 */ SetValues,
    /* set_values_hook		 */ NULL,
    /* set_values_almost	 */ XtInheritSetValuesAlmost,
    /* get_values_hook		 */ NULL,
    /* accept_focus	 	 */ NULL,
    /* version			 */ XtVersion,
    /* callback_private   	 */ NULL,
    /* tm_table		   	 */ defaultTranslations,
    /* query_geometry		 */ NULL, 
    /* display_accelerator       */ XtInheritDisplayAccelerator,
    /* extension                 */ NULL
    },
    {
    /* dummy_field               */ 0,
    },
};

WidgetClass dialWidgetClass = (WidgetClass) & dialClassRec;


/* ARGSUSED */
static void
Initialize(Widget treq, Widget tnew, ArgList args, Cardinal *num_args)
{
    DialWidget w = (DialWidget) tnew;
    XGCValues gcValues;
    XtGCMask value, dynamic, unused;
    static Pixmap gray1;
    int screen_num = DefaultScreen(XtDisplay(w));

    
    /* 
     *  Check instance values set by resources that may be invalid. 
     */

    if (w->dial.cur_value > MAXVALUE)
    {
        w->dial.cur_value = MAXVALUE;
        XtWarning("Dial: curValue must not exceed 100.\n");
    }
    if (w->dial.cur_value < 0)
    {
        w->dial.cur_value = 0;
        XtWarning("Dial: curValue must not fall below zero.\n");
    }

    /*
     *  Get GC (and set overall values)
     */
    
    gray1 = XCreateBitmapFromData(XtDisplay(w),
                                  RootWindow(XtDisplay(w), screen_num),
                                  gray1_bits, gray1_width, gray1_height);    
    value = ( GCStipple | GCFont );
    gcValues.stipple = gray1;
    gcValues.font = w->dial.font->fid;
    dynamic = ( GCForeground | GCFillStyle | GCFont);
    unused = (XtGCMask) 0;
    w->dial.gc = XtAllocateGC((Widget) w, 0, value, &gcValues, dynamic, unused);
    w->dial.pix = XtUnspecifiedPixmap;
}

/* ARGSUSED */
static void Redisplay(Widget cw, XExposeEvent *event)
/* complete redraw */
{
    DialWidget w = (DialWidget) cw;
    int x, y, d, pointx, pointy, pointd, height, twidth;
    double alpha;
    int screen_num = DefaultScreen(XtDisplay(w));
    
    if (w->dial.label!=NULL)
    {
        height = w->core.height - (w->dial.font->ascent+w->dial.font->descent);
        twidth = XTextWidth(w->dial.font, w->dial.label, strlen(w->dial.label));
        XSetForeground(XtDisplay(w), w->dial.gc, w->dial.label_color);
        x = (int) (w->core.width - twidth)/2;
        XDrawString(XtDisplay(w), XtWindow(w), w->dial.gc, x, 
                    ( w->core.height - w->dial.font->descent ),
                    w->dial.label, strlen(w->dial.label));
    }
    else
    {
        height = w->core.height;
    }

    d = (int) MIN(w->core.width*(1.0- (4.0/32.0)), 
                  height*(1.0-(4.0/32.0)));
    x = (int) (w->core.width-d)/2;
    y = (int) (height-d)/2;
    pointd = (int) d/7;
    if (pointd<3) pointd=3; 
    alpha = (1.25*PI)-((1.5*PI)*w->dial.cur_value/MAXVALUE);
    pointx = (int) x+ ((d/2.0)-(pointd/2.0))*(cos(alpha)+1.0);
    pointy = (int) y+ ((d/2.0)-(pointd/2.0))*(-sin(alpha)+1.0);

    XSetForeground(XtDisplay(w), w->dial.gc, w->dial.dial_back_color);
    XFillArc(XtDisplay(w), XtWindow(w), w->dial.gc, x, y, d, d, 0, 360*64); 
    XSetFillStyle(XtDisplay(w), w->dial.gc, FillStippled);
    XSetForeground(XtDisplay(w), w->dial.gc, w->dial.dial_color);
    XFillArc(XtDisplay(w), XtWindow(w), w->dial.gc, x, y, d, d,   0*64,  45*64);
    XFillArc(XtDisplay(w), XtWindow(w), w->dial.gc, x, y, d, d,  90*64,  45*64);
    XFillArc(XtDisplay(w), XtWindow(w), w->dial.gc, x, y, d, d, 180*64,  45*64);
    XFillArc(XtDisplay(w), XtWindow(w), w->dial.gc, x, y, d, d, 270*64,  45*64);
    XSetFillStyle(XtDisplay(w), w->dial.gc, FillSolid);
    XFillArc(XtDisplay(w), XtWindow(w), w->dial.gc, x, y, d, d,  45*64,  45*64);
    XFillArc(XtDisplay(w), XtWindow(w), w->dial.gc, x, y, d, d, 225*64,  45*64);
    XDrawArc(XtDisplay(w), XtWindow(w), w->dial.gc, x+1, y, d, d, 240*64, 145*64); 
    XDrawArc(XtDisplay(w), XtWindow(w), w->dial.gc, x, y+1, d, d, 240*64, 145*64); 
    XDrawArc(XtDisplay(w), XtWindow(w), w->dial.gc, x+1, y+1, d, d, 285*64, 60*64); 
    XSetForeground(XtDisplay(w), w->dial.gc, w->dial.dial_border_color);
    XDrawArc(XtDisplay(w), XtWindow(w), w->dial.gc, x, y, d, d, 0, 360*64); 
    if (w->dial.pix==XtUnspecifiedPixmap)
    {
        w->dial.pix=XCreatePixmap(XtDisplay(w), 
                                  RootWindow(XtDisplay(w), screen_num),
                                  pointd+2, pointd+2, 
                                  DefaultDepth(XtDisplay(w), screen_num));
    }
    XCopyArea(XtDisplay(w), XtWindow(w), w->dial.pix, w->dial.gc, 
                  pointx-1, pointy-1, pointd+2, pointd+2, 0, 0);
    XSetForeground(XtDisplay(w), w->dial.gc, w->dial.dial_point_color);
    XFillArc(XtDisplay(w), XtWindow(w), w->dial.gc, pointx, pointy, 
             pointd, pointd, 0, 360*64); 
    if (w->dial.dial_point_color != w->dial.dial_point_border_color)
    {
        XSetForeground(XtDisplay(w), w->dial.gc, w->dial.dial_point_border_color);
        XDrawArc(XtDisplay(w), XtWindow(w), w->dial.gc, pointx, pointy, 
                 pointd, pointd, 0, 360*64); 
    }
}

/* ARGSUSED */
static Boolean SetValues(Widget current, Widget request, Widget new, 
                                     ArgList args, Cardinal *num_args)
{
    DialWidget w = (DialWidget) new;
    DialWidget wo= (DialWidget) current;
    Boolean needs_redraw = False;
    int newvalue;
    
    if (w->dial.cur_value != wo->dial.cur_value)
    {
        if (w->dial.cur_value > MAXVALUE )
        {
            w->dial.cur_value = MAXVALUE;
            XtWarning("Dial: curValue must not exceed maxValue.\n");
        }
        if (w->dial.cur_value < 0)
        {
            w->dial.cur_value = 0;
            XtWarning("Dial: curValue must not fall below zero.\n");
        }
        if XtIsRealized( (Widget) w)
        {
            newvalue=w->dial.cur_value;
            w->dial.cur_value = wo->dial.cur_value;
            UpdateDial((Widget) w, newvalue);
        }
    }
    if (( w->dial.dial_border_color != wo->dial.dial_border_color )
        || ( w->dial.dial_color != wo->dial.dial_color )
        || ( w->dial.dial_point_color != wo->dial.dial_point_color)
        || ( w->dial.dial_point_border_color!=wo->dial.dial_point_border_color )
        || ( w->dial.label_color != wo->dial.label_color )
        || ( w->dial.label != wo->dial.label ))
        needs_redraw = True;
    if ( w->dial.font != wo->dial.font )
    {
        XSetFont(XtDisplay(w), w->dial.gc, w->dial.font->fid);
        needs_redraw = True;
    }
    return needs_redraw;
}


static void Destroy(Widget cw)
{
    DialWidget w = (DialWidget) cw;

    if (w->dial.pix!=XtUnspecifiedPixmap)
    {
        XFreePixmap(XtDisplay(w), w->dial.pix);
    }

}

/* ARGSUSED */
static void Resize(Widget cw)
{
    DialWidget w = (DialWidget) cw;
    
    if (w->dial.pix!=XtUnspecifiedPixmap)
    {
        XFreePixmap(XtDisplay(w), w->dial.pix);
        w->dial.pix = XtUnspecifiedPixmap;
    }
}

static void UpdateDial(Widget cw, int newvalue)
{
    int x, y, d, pointx, pointy, pointd, oldpointx, oldpointy, oldvalue;
    int height, twidth;
    double alpha;
    DialWidget w = (DialWidget) cw;
    
    if (w->dial.label!=NULL)
    {
        height = w->core.height - (w->dial.font->ascent+w->dial.font->descent);
        twidth = XTextWidth(w->dial.font, w->dial.label, strlen(w->dial.label));
        XSetForeground(XtDisplay(w), w->dial.gc, w->dial.label_color);
        x = (int) (w->core.width - twidth)/2;
        XDrawString(XtDisplay(w), XtWindow(w), w->dial.gc, x, 
                    ( w->core.height - w->dial.font->descent ),
                    w->dial.label, strlen(w->dial.label));
    }
    else
    {
        height = w->core.height;
    }
    
    oldvalue = w->dial.cur_value;
    w->dial.cur_value = newvalue;
    if (oldvalue == w->dial.cur_value) return;
    d = (int) MIN(w->core.width*(1.0- (4.0/32.0)), 
                  height*(1.0-(4.0/32.0)));
    x = (int) (w->core.width-d)/2;
    y = (int) (height-d)/2;
    pointd = (int) d/7;
    if (pointd<3) pointd=3; 
    alpha = (1.25*PI)-((1.5*PI)*oldvalue/MAXVALUE);
    oldpointx = (int) x+ ((d/2.0)-(pointd/2.0))*(cos(alpha)+1.0);
    oldpointy = (int) y+ ((d/2.0)-(pointd/2.0))*(-sin(alpha)+1.0);
    alpha = (1.25*PI)-((1.5*PI)*w->dial.cur_value/MAXVALUE);
    pointx = (int) x+ ((d/2.0)-(pointd/2.0))*(cos(alpha)+1.0);
    pointy = (int) y+ ((d/2.0)-(pointd/2.0))*(-sin(alpha)+1.0);
    if (w->dial.pix!= XtUnspecifiedPixmap)
    {
        XCopyArea(XtDisplay(w), w->dial.pix, XtWindow(w), w->dial.gc, 
                           0, 0, pointd+2, pointd+2, oldpointx-1, oldpointy-1);
        XCopyArea(XtDisplay(w), XtWindow(w), w->dial.pix, w->dial.gc, 
                           pointx-1, pointy-1, pointd+2, pointd+2, 0, 0);
    }
    XSetForeground(XtDisplay(w), w->dial.gc, w->dial.dial_point_color);
    XFillArc(XtDisplay(w), XtWindow(w), w->dial.gc, pointx, pointy, 
             pointd, pointd, 0, 360*64); 
    if (w->dial.dial_point_color != w->dial.dial_point_border_color)
    {
        XSetForeground(XtDisplay(w), w->dial.gc, w->dial.dial_point_border_color);
        XDrawArc(XtDisplay(w), XtWindow(w), w->dial.gc, pointx, pointy, 
                 pointd, pointd, 0, 360*64); 
    }
    /* now call the callback functions */
    info.value = newvalue;
    if (oldvalue != newvalue)
        XtCallCallbacks( (Widget) w, XtNcallback, &info);  
}

static void MoveDial(Widget cw, XEvent *event)
{
    int newvalue, height;
    double alpha, beta;
    DialWidget w = (DialWidget) cw;

    if (w->dial.label!=NULL)
    {
        height = w->core.height - (w->dial.font->ascent+w->dial.font->descent);
    }
    else
    {
        height = w->core.height;
    }
    
    if (event->type==ButtonPress)
    {
        alpha = atan2(-(event->xbutton.y - (height/2)),
                        event->xbutton.x - (w->core.width/2));
    }
    else
    {
        alpha = atan2(-(event->xmotion.y - (height/2)),
                        event->xmotion.x - (w->core.width/2));
    }
    if (alpha<-(PI/2.0)) alpha+=2*PI;
    beta=(1.25*PI)-alpha;
    newvalue=beta*MAXVALUE/(1.5*PI);
    if (newvalue<0) newvalue=0;
    if (newvalue>MAXVALUE) newvalue=MAXVALUE;
    if (((abs(w->dial.cur_value-newvalue)< (MAXVALUE/4) ) &&
        (w->dial.cur_value!=newvalue)) || (event->type==ButtonPress))
    {
        UpdateDial((Widget) w, newvalue);
    }

}
