// player.cc,v 2.2 1995/06/22 18:41:15 andreas Exp

/*
**  jazz - a midi sequencer for Linux
**
**  Copyright (C) 1994 Andreas Voss (andreas@avix.rhein-neckar.de)
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "player.h"
#include "trackwin.h"

extern "C" unsigned long ntohl( unsigned long );

#define CLOCK_TO_HOST_TICKS 15 			// midinetd sends clock
						// information every 15'th
						// tick

#define FIRST_DELTACLOCK 720
						// First time send events up to
						// 720 ticks ahead in time
						// (often 6 quarter
						// notes)

#define DELTACLOCK 960				// Later send events up to
						// 960 ticks ahead in time
						// (often 8 quarter
						// notes)

#define ADVANCE_PLAY 480			// Send more events to midinetd
						// 480 ticks in time ahead of last
						// written events (often 4
						// quarternotes, which is also
						// often one bar).

char *midinethost = NULL;


#include <fcntl.h>




tPlayer *Midi = 0;

// ------------------------- tPlayLoop --------------------------

tPlayLoop::tPlayLoop()
{
  Reset();
}

void tPlayLoop::Reset()
{
  StartClock = StopClock  = 0;
}

void tPlayLoop::Set(long Start, long Stop)
{
  StartClock = Start;
  StopClock  = Stop;
}


long tPlayLoop::Ext2IntClock(long Clock)
{
  if (StopClock) {
    return (Clock - StartClock) % (StopClock - StartClock) + StartClock;
  }
  return Clock;
}

long tPlayLoop::Int2ExtClock(long Clock)
{
  return Clock;
}

void tPlayLoop::PrepareOutput(tEventArray *buf, tSong *s, long ExtFr, long ExtTo)
{
  long fr = Ext2IntClock(ExtFr);
  long delta = ExtFr - fr;
  long size = ExtTo - ExtFr;
  while (StopClock && fr + size > StopClock)
  {
    s->MergeTracks(fr, StopClock, buf, delta);
    size  -= StopClock - fr;
    fr     = StartClock;
    delta += StopClock - StartClock;
  }
  if (size > 0)
    s->MergeTracks(fr, fr + size, buf, delta);

}

// ------------------------- tPlayer ---------------------


tPlayer::tPlayer(tSong *song)
{
  Song = song;
  OutClock = 0;
  Playing = 0;
  PlayLoop = new tPlayLoop();
  ClockSource = CsInt;
  RealTimeOut = 0;
}

tPlayer::~tPlayer()
{
  delete PlayLoop;
}


void tPlayer::StartPlay(long Clock, long LoopClock)
{
  int i;

  if (LoopClock > 0)
    PlayLoop->Set(Clock, LoopClock);
  else
    PlayLoop->Reset();

  Clock = PlayLoop->Int2ExtClock(Clock);
  PlayBuffer.Clear();
  RecdBuffer.Clear();

  // Send Volume, Pan, Chorus, etc
  for (i = 0; i < Song->nTracks; i++)
  {
    tTrack *t = Song->GetTrack(i);
    if (t->Bank)
      OutNow(t->Bank);
    if (t->Patch)
      OutNow(t->Patch);
    if (t->Volume)
      OutNow(t->Volume);
    if (t->Pan)
      OutNow(t->Pan);
    if (t->Reverb)
      OutNow(t->Reverb);
    if (t->Chorus)
      OutNow(t->Chorus);
    if (t->ReverbType)
      OutNow(t->ReverbType);
    if (t->ChorusType)
      OutNow(t->ChorusType);
    if (t->VibRate)
      OutNow(t->VibRate);
    if (t->VibDepth)
      OutNow(t->VibDepth);
    if (t->VibDelay)
      OutNow(t->VibDelay);
    if (t->Cutoff)
      OutNow(t->Cutoff);
    if (t->Resonance)
      OutNow(t->Resonance);
    if (t->EnvAttack)
      OutNow(t->EnvAttack);
    if (t->EnvDecay)
      OutNow(t->EnvDecay);
    if (t->EnvRelease)
      OutNow(t->EnvRelease);
    if (t->PitchBendSens)
      OutNow(t->PitchBendSens);
    if (t->ModDepth)
      OutNow(t->ModDepth);
    if (t->ReverbType)
      OutNow(t->ReverbType);
    if (t->ChorusType)
      OutNow(t->ChorusType);
    if (t->Speed)
      OutNow(t->Speed);
  }

  OutClock = Clock + FIRST_DELTACLOCK;
  TrackWin->NewPlayPosition(PlayLoop->Ext2IntClock(Clock));
  PlayLoop->PrepareOutput(&PlayBuffer, Song, Clock, Clock + FIRST_DELTACLOCK);
  //Song->MergeTracks(Clock, Clock + FIRST_DELTACLOCK, &PlayBuffer);
  PlayBuffer.Length2Keyoff();
  Notify();
  Start(50);	// wxTimer: call Notify every 50 ms
  Playing = 1;
}


void tPlayer::StopPlay()
{
  Stop();	// stop wxTimer
  Playing = 0;
  TrackWin->NewPlayPosition(-1L);
}


void tPlayer::Notify()
{
  // called by timer
  long Now = GetRealTimeClock();

  // buffer-events eaten up by device
  if (PlayBuffer.GetFirstClock() >= OutClock)
  {
    // time to put more events
    if ( Now >= (OutClock - ADVANCE_PLAY) )
    {
      //Song->MergeTracks(OutClock, Now + DELTACLOCK, &PlayBuffer);
      PlayLoop->PrepareOutput(&PlayBuffer, Song, OutClock, Now + DELTACLOCK);
      OutClock = Now + DELTACLOCK;
      PlayBuffer.Length2Keyoff();
    }
  }
  FlushToDevice();
}


void tPlayer::FlushToDevice()
// try to send all events up to OutClock to device
{
  int BufferFull = 0;

  tEventIterator Iterator(&PlayBuffer);
  tEvent *e = Iterator.Range(0, OutClock);
  while (!BufferFull && e) 
  {
    if (OutEvent(e) != 0)
      BufferFull = 1;
    else
    {
      e->Kill();
      e = Iterator.Next();
    }
  }
  if (!BufferFull)
    OutBreak();
  PlayBuffer.Cleanup(0);
}


void tPlayer::AllNotesOff(int Reset)
{
  tControl NoteOff(0, 0, 0x7B, 0);
  tPitch   Pitch  (0, 0, 0);
  tControl CtrlRes(0, 0, 0x79, 0);

  for (int i = 0; i < 16; i++)
  {
    NoteOff.Channel = i; OutNow(&NoteOff);
    Pitch.Channel = i;   OutNow(&Pitch);
    CtrlRes.Channel = i; OutNow(&CtrlRes);
  }
  if (Reset) {
	uchar *d = SysExDT1( 0x40, 0x00, 0x7f, 0x00 );
	tSysEx GsReset( 0, d, 10 );
	OutNow( &GsReset );
	delete d;
  }
}




// ----------------------------------------------------------------------------------------------
// MPU-Player
// ----------------------------------------------------------------------------------------------

#ifndef wx_msw

tMpuPlayer::tMpuPlayer(tSong *song)
  : tPlayer(song)
{
	midinethost = getenv("MIDINETHOST");
	if (!midinethost || !strlen(midinethost)) {
		midinethost = "localhost";
	}
	dev = midinetconnect( midinethost, MIDINET_SERVICE);
}



tMpuPlayer::~tMpuPlayer()
{
  close(dev);
}


int tMpuPlayer::Installed()
{
  return dev >= 0;
}


#if DB_WRITE
int dwrite(int dev, const char *buf, int size)
{
  int i, written;
  //dev = 2;	// stderr
  written = write(dev, buf, size);
  if (written < 0)
  {
    printf("writing failed\n");
    return written;
  }
  printf("W: ");
  for (i = 0; i < written; i++)
    printf("%02x ", (unsigned char)buf[i]);
  putchar('\n');
  if (written != size)
  {
    printf("L: ");
    for (i = written; i < size; i++)
      printf("%02x ", (unsigned char)buf[i]);
    putchar('\n');
  }
  fflush(stdout);
  return written;
}
#endif


#define MAXTRIES 100000
int writenow(int dev, const void *buf, int size) {
	volatile int i;
	int written = 0;

	for (i = 0; i < MAXTRIES; i++) {
		written = written + dwrite( dev, buf + written, size - written );
		if (written == size) break;
	}
	assert( i < MAXTRIES );
	return( written );
}



void tMpuPlayer::StartPlay(long IntClock, long LoopClock)
{
  long ExtClock = PlayLoop->Int2ExtClock(IntClock);
  static const char play[] = 
  {
    CMD+1, 0x34,			    		/* timing byte always */
    CMD+1, 0x8e,					/* conductor off */
    CMD+1, 0x8c,					/* don't send measures while recording */
    CMD+1, 0xe7, DAT+1, 60,				/* clock-to-host every 15'th tick (60/4) */
    CMD+1, 0x95, 					/* send clock to host instead */
    CMD+1, 0x87,  					/* pitch+controller enabled */
    CMD+1, 0x98, CMD+1, 0x9a, CMD+1, 0x9c, CMD+1, 0x9e,	/* channel-ref-tables off */
    CMD+1, 0xec, DAT+1, ACTIVE_TRACKS_MASK, CMD+1, 0xb8, CMD+1, 0x2a  /* start record/play */
  };

  PlyBytes.Clear();

  RecBytes.Clear();
  RecBytes.Clock = ExtClock;

  playclock = ExtClock;
  clock_to_host_counter = 0;

  ActiveTrack = 0;
  for (int i = 0; i < ACTIVE_TRACKS; i++) {
	TrackClock[i] = ExtClock;
	TrackRunningStatus[i] = 0;
  }

  // Setup Timebase
  char timebase[2];
  timebase[0] = CMD+1;
  switch (Song->TicksPerQuarter)
  {
    case  48: timebase[1] = 0xc2; break;
    case  72: timebase[1] = 0xc3; break;
    case  96: timebase[1] = 0xc4; break;
    case 120: timebase[1] = 0xc5; break;
    case 144: timebase[1] = 0xc6; break;
    case 168: timebase[1] = 0xc7; break;
    case 192: timebase[1] = 0xc8; break;
    default : timebase[1] = 0xc5; break;
  }
  writenow(dev, timebase, 2);

  tPlayer::StartPlay(IntClock, LoopClock);

  // Supress realtime messages to MIDI Out port?
  if (!RealTimeOut) {
  	char realtime[2];
  	realtime[0] = CMD+1;
	realtime[1] = 0x32;
	writenow( dev, realtime, 2 );
  }

  // What is the clock source ?
  char clocksource[2];
  clocksource[0] = CMD+1;
  switch (ClockSource) {
	case CsInt: 	clocksource[1] = 0x80; break;
	case CsFsk: 	clocksource[1] = 0x81; break;
	case CsMidi: 	clocksource[1] = 0x82; break;
	default: 	clocksource[1] = 0x80; break;
  }
  writenow(dev, clocksource, 2);

  // Send start-play events one by one to try avoid TCP breaking up events
  for (i = 0; i < sizeof(play); i = i + 2 ) {
	writenow(dev, &play[i], 2 );
  }

  static const unsigned char start_play_command = START_PLAY_COMMAND;

  // Send command to inform midinetd that playing has started
  writenow(dev, &start_play_command, 1);
}



void tMpuPlayer::StopPlay()
{
  static const char stop = RES;
  static const unsigned char stop_play_command = STOP_PLAY_COMMAND;
  tPlayer::StopPlay();
  writenow(dev, &stop, 1);
  PlyBytes.Clear();
  AllNotesOff();

  // Send command to midinetd to stop play and send record buffer
  writenow(dev, &stop_play_command, 1);

  // Get record buffer
  GetRecordedData();
  RecdBuffer.Keyoff2Length();
}




int tMpuPlayer::OutEvent(tEvent *e)
{
  if (!PlyBytes.WriteFile(dev))
    return 1;	// buffer full

  int Stat = e->Stat;

  switch (Stat)
  {
    case StatKeyOff:
    case StatKeyOn:
    case StatAftertouch:
    case StatControl:
    case StatProgram:
    case StatMono:
    case StatPitch:
      {
	tGetMidiBytes midi;
	int i;
	tChannelEvent *c;

	e->Write(midi);
	Stat = midi.Buffer[0]; // Status + Channel

	OutBreak(e->Clock);

	if ( (c = e->IsChannelEvent()) != 0 ) {
		switch (c->Channel) {
			case 0:
			case 3:
				ActiveTrack = 5;
				break;
			case 1:
			case 4:
				ActiveTrack = 4;
				break;
			case 2:
			case 5:
				ActiveTrack = 3;
				break;
			case 6:
			case 10:
			case 13:
				ActiveTrack = 2;
				break;
			case 9:
				ActiveTrack = 6;
				break;
			case 7:
			case 11:
			case 14:
				ActiveTrack = 1;
				break;
			case 8:
			case 12:
			case 15:
				ActiveTrack = 0;
				break;
			default:
				ActiveTrack = 6;
				break;
		}
	}
	else {
		// Not channel event => play on track #6
		ActiveTrack = 6;
	}

	long Time = e->Clock - TrackClock[ActiveTrack];
	assert(Time < 240);

	if (Stat != TrackRunningStatus[ActiveTrack])
	{
	  PlyBytes.Put(TRK + midi.nBytes + 1 + 1);
	  PlyBytes.Put(ActiveTrack);
	  PlyBytes.Put(Time);
	  PlyBytes.Put(Stat);
	  TrackRunningStatus[ActiveTrack] = Stat;
	}
	else
	{
	  PlyBytes.Put(TRK + midi.nBytes + 1);
	  PlyBytes.Put(ActiveTrack);
	  PlyBytes.Put(Time);
	}
	for (i = 1; i < midi.nBytes; i++)
	  PlyBytes.Put(midi.Buffer[i]);

	TrackClock[ActiveTrack] = e->Clock;
	return 0;
      }

    default:	// Meterchange etc
      return 0;
      break;
  }
}


void tMpuPlayer::OutBreak()
{
  // send a break to the driver starting at PlyBytes.Clock and ending at OutClock
  if (!PlyBytes.WriteFile(dev))
    return;

  (void)OutBreak(OutClock);
  PlyBytes.WriteFile(dev);
}


long tMpuPlayer::OutBreak(long BreakOver)
{
int OverFlow = 1;

    while (OverFlow) {
	OverFlow = 0;
	for (int i = 0; i < ACTIVE_TRACKS; i++) {
		if ( (BreakOver - TrackClock[i]) >= 240 ) {
			PlyBytes.Put(TRK+1+1);
			PlyBytes.Put( i );
			PlyBytes.Put(0xf8);
			TrackClock[i] += 240;
			OverFlow = 1;
		}
	}
    }
    return BreakOver;
}


void tMpuPlayer::OutNow( tParam *r ) {
	OutNow( &r->Msb );
	OutNow( &r->Lsb );
	OutNow( &r->Data );
}


void tMpuPlayer::OutNow(tEvent *e)
{
  // send event to driver immediately regardless of events remaining
  // in the play-queue.

  char buf[20];
  int i, n = 0;
  tGetMidiBytes midi;
  if (e->Write(midi) == 0)
  {
    buf[n++] = CMD+1;
    buf[n++] = 0xd7;
    buf[n++] = DAT+midi.nBytes;
    for (i = 0; i < midi.nBytes; i++)
      buf[n++] = midi.Buffer[i];
    writenow(dev, buf, n);
  }

  else	// special event
  {
    switch (e->Stat)
    {
      case StatSetTempo:
	{
	  char cmd[20];
	  tSetTempo *s = (tSetTempo *)e;
	  int bpm = s->GetBPM();
	  cmd[0] = CMD+1;
	  cmd[1] = 0xE0;
	  cmd[2] = DAT+1;
	  cmd[3] = (char)bpm;
	  writenow(dev, cmd, 4);
	}
	break;
      case StatSysEx:
	{
		char sysex[20];
		n = 0;
		tSysEx *s = (tSysEx *) e;
    		sysex[n++] = CMD+1;
		sysex[n++] = 0xdf;
		sysex[n++] = DAT + s->Length + 1;
		sysex[n++] = StatSysEx;
		for (i = 0; i < s->Length; i++)
			sysex[n++] = s->Data[i];
		writenow(dev, sysex, n);
	}
	break;

      default:	// ignore others
        break;
    }
  }
}



long tMpuPlayer::GetRealTimeClock()
{
  int c;
  while ((c = RecBytes.Get(dev)) >= 0)
  {
    // The midinetd sends 0xfd to jazz every 15'th tick
    if (c == 0xfd) {
    // CLOCK_TO_HOST received
      playclock += CLOCK_TO_HOST_TICKS;
      clock_to_host_counter++;
#ifdef SLOW_MACHINE
      /* Update screen every 4 beats (120 ticks/beat) */
      if ( (clock_to_host_counter % 32) == 0 ) 
        TrackWin->NewPlayPosition(PlayLoop->Ext2IntClock(playclock));
#else
      /* Update screen every 8'th note (120 ticks/beat) */
      if ( (clock_to_host_counter % 4) == 0 ) 
        TrackWin->NewPlayPosition(PlayLoop->Ext2IntClock(playclock));
#endif
    }
  }
  return playclock;
}



long tMpuPlayer::GetRecordedData()
{
  int c, i, j;
  volatile int ii;
  unsigned char ch;
  unsigned char *recbuf = NULL;
  int numbytes = 0;
  unsigned char *iptr;

  // Wait (up to 10 sec) for start mark (START_OF_RECORD_BUFFER)
  ch = 0;
  for ( ii = 0; ii < 1000; ii++ ) {
	read( dev, &ch, 1 );
	if (ch == START_OF_RECORD_BUFFER) 
	  break;
	else
	  usleep(10000);
  }
  if ( ch != START_OF_RECORD_BUFFER ) { // Got to have that start mark
	printf("tMpuPlayer::GetRecordedData: Did not receive START_OF_RECORD_BUFFER\n");
	return(0);
  }

  // Read number of bytes in record buffer
  i = 0;
  iptr = (unsigned char*) &numbytes;
  while( i < sizeof(int) ) {
	if (read( dev, iptr, 1) == 1) {
		i++;
		iptr++;
	}
  }
  numbytes = ntohl( numbytes );

  if (numbytes == 0) {
	// No bytes in record buffer
	return 0;
  }

  // Allocate and read record buffer
  recbuf = (unsigned char*) malloc( numbytes );
  i = 0;
  while ( 1 ) {
	j = read( dev, recbuf + i, numbytes - i );
	if (j < 0) j = 0;
	i = i + j;
	if (i >= numbytes) break;
  }

  // Go through the record buffer and create events
  i = 0;
  while (i < numbytes)
  {
    c = recbuf[i++];
    if (c == 0xf8)
      RecBytes.Clock += 240;
    else if (c < 0xf0)
    {
      // timing byte
      RecBytes.Clock += c;
      c = recbuf[i++];
      if (c < 0xf0)  // Voice-Message
      {
        unsigned char c1, c2;
        int Channel;
        tEvent *e = 0;

        if (c & 0x80)
        {
          RecBytes.RunningStatus = c;
          c1 = recbuf[i++];
	}
	else
	  c1 = c;

	Channel = RecBytes.RunningStatus & 0x0f;
	switch (RecBytes.RunningStatus & 0xf0)
	{
	  case StatKeyOff:
	    (void)recbuf[i++];
	    e = new tKeyOff(RecBytes.Clock, Channel, c1);
	    break;

	  case StatKeyOn:
	    c2 = recbuf[i++];
	    if (!c2)
	      e = new tKeyOff(RecBytes.Clock, Channel, c1);
	    else
	      e = new tKeyOn(RecBytes.Clock, Channel, c1, c2);
	    break;
#if 0
	  case StatAftertouch:
	    c2 = recbuf[i++];
	    e  = new tAftertouch(RecBytes.Clock, Channel, c1, c2);
	    break;

	  case StatMono:
	    e  = new tMono(RecBytes.Clock, Channel, c1);
	    break;
#endif

	  case StatControl:
	    c2 = recbuf[i++];
	    e  = new tControl(RecBytes.Clock, Channel, c1, c2);
	    break;

	  case StatProgram:
	    e  = new tProgram(RecBytes.Clock, Channel, c1);
	    break;
	    break;

	  case StatPitch:
	    c2 = recbuf[i++];
	    e  = new tPitch(RecBytes.Clock, Channel, c1, c2);
	    break;

	  default:
	    printf("unrecognized MIDI Status %02x, %02x", (unsigned char)RecBytes.RunningStatus, c1);
	    break;

	}
	if (e)
	{
	  e->Clock = PlayLoop->Ext2IntClock(e->Clock);
	  RecdBuffer.Put(e);
	}
      }
      else
        printf("unrecognized Status after Timing Byte: %02x\n", c);
    }
    else
      printf("unrecognized Status (no timing byte): %02x\n", c);
  }
  if (recbuf) free( recbuf );
  return( numbytes );
}


#endif
