/*
 * Roland MPU-401 MIDI-Driver
 * 
 * Andreas Voss (andreas@avix.rni.sub.org)
 * 
 * 	and
 * 
 * Per Sigmond (Per.Sigmond@hiagder.no) (SunOS version)
 * 
 *
 * mpu_write():
 *
 *   writes Data to MPU's Registers in MPU-Format with 1 additional leading Byte:
 *     Bits 0..5: length of MPU-Data following.
 *     Bits 6..7:
 *       00 : send later on Track Data Request Interrupt
	      (0xF0-0xF6, 7 Tracks supported, track number follows leading byte)
 *       01 : write immediately to Data Register
 *       10 : write immediately to Command Register
 *       11 : driver command: 0xc0 = reset everything
 *
 * mpu_read()
 *    returns recorded data in mpu-format
 */

#include <sys/param.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/uio.h>
#include <machine/psl.h>
#include <sundev/mbvar.h>
#include <sun386/cpu.h>

#include "mpu.h"
#include "mpureg.h"


int mpuprobe(), mpuattach(), mpuintr();

struct mb_driver mpudriver = { mpuprobe, 0, mpuattach, 0, 0, mpuintr, 0, "mpu", 0, 0, 0, 0, 0 };


struct mpu_queue
{
  int rd, wr;
  unsigned char buffer[MPU_BUFFER_SIZE];
};

struct mpu_status
{
  struct mpu_queue recvq;
  struct mpu_queue sendq[ACTIVE_TRACKS];
  int running_status;
  struct proc *sleeper;
  short iobase;
  struct mb_device *md;
  int flags;
};

static struct mpu_status mpu;


/* get char from queue */
int mpu_readq(q)
struct mpu_queue *q;
{
  unsigned char c;
  if (q->wr == q->rd)
    return -1;
  c = q->buffer[q->rd];
  q->rd = (q->rd + 1) % MPU_BUFFER_SIZE;
  return c;
}

/* put char into queue */
int mpu_writeq(q,c)
struct mpu_queue *q;
unsigned char c;
{
  q->buffer[q->wr] = c;
  q->wr = (q->wr + 1) % MPU_BUFFER_SIZE;
  return 0;
}

/* true, if queue not empty */
int mpu_statq(q)
struct mpu_queue *q;
{
  return q->rd != q->wr;
}


/* # bytes free in queue */
int mpu_leftq(q)
struct mpu_queue *q;
{
  int w = q->wr;
  if (q->wr < q->rd)
    w += MPU_BUFFER_SIZE;
  return( MPU_BUFFER_SIZE-1 - (w - q->rd) );
}


void mpu_clearq(q)
struct mpu_queue *q;
{
  q->rd = q->wr = 0;
}

/*
 * MPU-Hardware
 */

#define mpu_rdd() inb(mpu.iobase)
#define mpu_wrd(a) outb(mpu.iobase,a)
#define mpu_cmd(a) outb(mpu.iobase+1,a)

#define MPU_DSR ((u_char)0x80)
#define MPU_DRR ((u_char)0x40)
#define MPU_ACK ((u_char)0xfe)

#define MAX_RESET_TRIES 5

int mpu_status()
{
  int status;
  status = inb(mpu.iobase+1);
  return( status );
}

int wait_drr()
{
  int try;
  u_char status;

  try = 0;
  status = mpu_status();
  while ( (status & MPU_DRR) && (try++ < MPU_WAIT) ) {
        status = mpu_status();
  }  
  if (try < MPU_WAIT)
	return(0);
  else
	return( ENODEV );
}

int wait_dsr()
{
  int try;
  u_char status;

  try = 0;
  status = mpu_status();
  while ( (status & MPU_DSR) && (try++ < MPU_WAIT) ) {
  	status = mpu_status();
  }
  if (try < MPU_WAIT)
	return(0);
  else
	return( ENODEV );
}


int mpu_write(a)
char a;
{
int s;

  if (wait_drr())
    return ENODEV;

  mpu_wrd(a);
  return 0;
}

int mpu_read()
{
  if (wait_dsr())
    return ENODEV;
  return( mpu_rdd() );
}


int mpu_reset()
{
int ret, s;

  if (wait_drr())
    return ENODEV;
  mpu_cmd(0xff);
  ret = mpu_read();
  return( ret != MPU_ACK );
}


int mpu_received(c)
int c;
{
  int n;
  int track;

  /* length of midi-event             8  9  A  B  C  D  E */
  static int three_bytes[8] = { 1, 1, 1, 1, 0, 0, 1 };

  if (c < 0xf0)	
  {
    /*
     * found timing byte. store it and get midi-message or mpu-mark
     */

    mpu_writeq(&mpu.recvq, c);	/* save timing byte to record-buffer */
    c = mpu_read();		/* get statusbyte (or 1st data, if running status) */
    mpu_writeq(&mpu.recvq, c);	/* store it, so we know what happend at this time */

    if (c < 0xf0)
    {
      /*
       * it's a midi-voice-message
       */

      if (c & 0x80)			/* new running status */
      {
	mpu.running_status = c;
	c = mpu_read();			/* get 1st data-byte */
	mpu_writeq(&mpu.recvq, c);
      }
      if (three_bytes[((unsigned char)mpu.running_status >> 4) - 8])
      {
	c = mpu_read();
	mpu_writeq(&mpu.recvq, c);
      }
    }

    /* 
     * else: mpu mark's are already stored
     */

  }

  else if (c < 0xf8)	
  {

    /* 
     * play track data request 
     */

    track = c - 0xf0;
    if (track >= ACTIVE_TRACKS) return( EINVAL ); /* Highest track is reserved */
						  /* for immediate data */
    if (!mpu_statq(&mpu.sendq[track]))	/* nothing to send?? */
    {
      mpu_write(0xf8);		/* wait 240 ticks */
      return EINVAL;		/* stupid since called by interrupt */
    }
    n = mpu_readq(&mpu.sendq[track]);
    while (n-- > 0)
    {
      c = mpu_readq(&mpu.sendq[track]);
      mpu_write(c);
    }

    /*
     * someone waiting for sendq empty?
     */

    if (mpu.sleeper && mpu_leftq(&mpu.sendq[track]) > MPU_BUFFER_SIZE/2)
    {
      wakeup( (caddr_t) &mpu );
    }
  } 


  else
  {
    /* 
     * got mpu-stuff
     */

    switch (c)
    {
      case 0xfe:	/* ignore ack's */
        break;

      /*
       * implemented:
       *   f8: timer overflow
       *   fc: end of data
       *   fd: clock to host
       *
       * not implemented:
       *   all others (real time messages, conductor etc). record them
       *   so the application might discover an error.
       */

      default:
	mpu_writeq(&mpu.recvq, c);
	break;
    }
  }
  return 0;
}



int mpu_command(cmd)
unsigned char cmd;
{
  int try = 0;
  unsigned char c;
  unsigned long flags;
  int s;

  if (wait_drr())
    return ENODEV;

  /*
   * i'm not shure about disabling intr's, but mpu-manual says so
   * SunOS: raise interrupt level
   */

  mpu_cmd(cmd);

  while (try++ < MPU_WAIT)
  {
    if (wait_dsr())
    {
      try = MPU_WAIT;
      break;
    }

    c = mpu_rdd();

    if (c != MPU_ACK) {
      mpu_received(c);
    }
    else
      break;
  }


  if (try < MPU_WAIT)
	return(0);
  else
	return(ENODEV);
}


/* This routine is called at loadtime and should check the existance of the
   device.
*/
int mpuprobe(reg,unit)
caddr_t reg;
int unit;
{
int i, ret;
int s;


	mpu.iobase = (short) reg;
	printf("mpuprobe(): MPU-401 driver at adress %x\n", mpu.iobase );

	for (i = 0; i < MAX_RESET_TRIES; i++) {
		ret = mpu_reset();
		if (!ret) break;
	}
	
	if (ret) {
		printf("mpuprobe(): reset of MPU-401 NOT successful\n");
		return(0);
	}

	printf("mpuprobe(): reset of MPU-401 successful\n");
	return( sizeof(struct mpuregs) );
}


/* Called at loadtime after mpuprobe and should initialize the device and 
   the driver.
*/
int mpuattach(md)
register struct mb_device *md;
{
	printf("mpuattach(): MPU-401 driver at address %x irq %d priority %d\n", (int) md->md_addr, md->md_intr, md->md_intpri );
	mpu.md = md;
}


/* The interrupt routine */
int mpuintr(cpl,irq)
int cpl;
int irq;
{
  mpu_received(mpu_read());
  selwakeup(mpu.sleeper, 0);
  mpu.sleeper = NULL;
}


/* Called when the user closes the device */
int mpuclose(dev,flags)
dev_t dev;
int flags;
{
int i, ret;

  for (i = 0; i < MAX_RESET_TRIES; i++) {
	ret = mpu_reset();
	if (!ret) break;
  }
  mpu.flags = 0;
  return(0);
}

/* Called when the user opens the device */
int mpuopen(dev,flags)
dev_t dev;
int flags;
{
int i, ret;

  for (i = 0; i < MAX_RESET_TRIES; i++) {
	ret = mpu_reset();
	if (!ret) break;
  }
  if (ret)
  {
    return ENODEV;
  }

  mpu_clearq(&mpu.recvq);
  for (i = 0; i < ACTIVE_TRACKS; i++ ) mpu_clearq(&mpu.sendq[i]);
  mpu.running_status = 0x90;
  mpu.sleeper = NULL;
  mpu.flags = flags;

  return 0;
}


/* Called when the user writes the device */
int mpuwrite(dev,uio)
dev_t dev;
struct uio *uio;
{
  char c;
  int err, n;
  int count;
  int s;

  count = uio->uio_iov->iov_len;

  while (count > 0)
  {
    c = uwritec(uio);
    count--;
    n = c & 0x3f;
    if (n > count)
      return EINVAL;

    switch ((unsigned char)c >> 6)
    {

      case 0: /* write into send-queue */

	/* Get track number */
    	c = uwritec(uio);
    	count--;
	track = c;
	if ( (track < 0) || (track >= ACTIVE_TRACKS) ) return( EINVAL );

        while (n + 1 > mpu_leftq(&mpu.sendq[track]))	
        {
	   sleep( (caddr_t) &mpu, PZERO-1 );
        }

	n--; /* Skip track number byte */
	mpu_writeq(&mpu.sendq[track], n);
	while (n-- > 0)
	{
	  c = uwritec(uio);
	  count--;
	  mpu_writeq(&mpu.sendq[track], c);
	}
	break;


      case 1: /* write to data-register */ 

	while (n-- > 0)
	{
	  c = uwritec(uio);
	  count--;
	  if (err = mpu_write(c))
	    break;
	}
	if (err) return(err);
	break;


      case 2: /* write to command-register */

	while (n-- > 0)
	{
	  c = uwritec(uio);
	  count--;
	  if (err = mpu_command(c))
	    return err;
	}
	break;


      default:	/* reset everything */
      {
	unsigned long flags;
	mpu_reset();

	s = spl6();
	mpu_clearq(&mpu.recvq);
	for (i = 0; i < ACTIVE_TRACKS; i++ ) mpu_clearq(&mpu.sendq[i]);
	mpu.running_status = 0x90;
	(void)splx(s);
      }
      break;

    }
  }
  return 0;
}


/* Called when the user reads the device */
int mpuread(dev,uio)
dev_t dev;
struct uio *uio;
{
  char c;
  int count;

  count = uio->uio_iov->iov_len;

  while (count > 0 && mpu_statq(&mpu.recvq))
  {
    c = mpu_readq(&mpu.recvq);
    ureadc(c, uio);
    count --;
  }
  return 0;
}

/* Called when the user does a select() on the device */
int mpuselect( dev, rw )
dev_t dev;
int rw;
{
int s;

	if (rw != FREAD) return(0);
		
	s = spl6();
	if (mpu_statq(&mpu.recvq)) {
		(void) splx(s);
		return(1);
	}
	mpu.sleeper = u.u_procp;
	(void) splx(s);
	return(0);
}

