/* jam.c  acb  15-1-1994
 * Program to read input from a MIDI device on /dev/midi and play
 * music on a Gravis UltraSound.
 * Contains code from Hannu Savolainen's gusload.c and midithru.c
 * (from the VoxWare utilities)
 *
 * Copyright (c) Andrew Bulhak, 1995
 * Portions copyright (c) Hannu Savolainen, 1993
 * Copyright notice from VoxWare code below:
 * -----------------------------------------
 * Copyright by Hannu Savolainen 1993
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* # define INCLUDE_TTY*/

#include <fcntl.h>
#include <sys/ioctl.h>
#include <stdarg.h>
#include <xview/xview.h>
#include <xview/frame.h>
#include <xview/panel.h>
#include <xview/notice.h>
#include <xview/notify.h>
#include <sys/soundcard.h>
#include <sys/ultrasound.h>
#include "gmidi.h"
#ifdef INCLUDE_TTY
#include <xview/tty.h>
#endif

#ifndef PATCH_PATH
#define PATCH_PATH "/C/ultrasnd/midi"
#endif

#define SEQUENCER_PATH "/dev/sequencer"

struct pat_header
  {
    char            magic[12];
    char            version[10];
    char            description[60];
    unsigned char   instruments;
    char            voices;
    char            channels;
    unsigned short  nr_waveforms;
    unsigned short  master_volume;
    unsigned long   data_size;
  };

struct sample_header
  {
    char            name[7];
    unsigned char   fractions;
    long            len;
    long            loop_start;
    long            loop_end;
    unsigned short  base_freq;
    long            low_note;
    long            high_note;
    long            base_note;
    short           detune;
    unsigned char   panning;

    unsigned char   envelope_rate[6];
    unsigned char   envelope_offset[6];

    unsigned char   tremolo_sweep;
    unsigned char   tremolo_rate;
    unsigned char   tremolo_depth;

    unsigned char   vibrato_sweep;
    unsigned char   vibrato_rate;
    unsigned char   vibrato_depth;

    char            modes;

    short           scale_frequency;
    unsigned short  scale_factor;
  };

SEQ_DEFINEBUF (2048);
SEQ_PM_DEFINES; /* patch manager stuff */

int seqfd;
struct synth_info info;
int gus_dev=-1;
unsigned char buf[100];
int bufp;

int pgm=0, num_voices, bender=0;

int gus_memsize, gus_freemem, free_voices;

/* LRU list for free operators */

unsigned char free_list[256];
int fhead=0, ftail=0, flen=0;

/* LRU list for still playing notes */

unsigned char note_list[256];
int nhead=0, ntail=0, nlen=0;
unsigned char oper_note[32];

int show_voices=1;

/* XView interface objects */
Frame frame;
Panel panel;
Panel_item patch_choice, quit, voices, memsize, memfree;

void patch_notify(Panel_item item, char *str, caddr_t client_data, 
		  Panel_list_op op, Event *event, int row);

Notify_value read_midi(Notify_client client, int fd);

/*
 * The function seqbuf_dump() must always be provided
 */

void seqbuf_dump ()
{
  if (_seqbufptr)
    if (write (seqfd, _seqbuf, _seqbufptr) == -1)
      {
	perror ("write /dev/sequencer");
	exit (-1);
      }
  _seqbufptr = 0;
}

void complain(char *complaint, ...)
{
  Xv_notice notice;
  char buffer[256];
  va_list ap;

  va_start(ap, complaint);
  vsprintf(buffer, complaint, ap);
  notice = (Xv_notice) xv_create(frame, 
				 NOTICE,
				 NOTICE_MESSAGE_STRING, buffer,
				 NOTICE_BUTTON, "Ack", 1,
				 XV_SHOW, TRUE,
				 NULL);
  xv_destroy_safe(notice);
  va_end(ap);
};

/* display a notice and die */

volatile void die(char *complaint)
{
  Xv_notice notice;

  notice = (Xv_notice) xv_create(frame, 
				 NOTICE,
				 NOTICE_MESSAGE_STRING, complaint,
				 NOTICE_BUTTON, "Ack", 1,
				 XV_SHOW, TRUE,
				 NULL);
  xv_destroy_safe(notice);
  exit(-1);
};

/* user interface element wrangling routines */

void update_memfree()
{
  xv_set(memfree, PANEL_VALUE, gus_freemem, NULL);
};

void update_voices()
{
  if(show_voices) xv_set(voices, PANEL_VALUE, free_voices, NULL);
};

/* initialise note/operator lists, etc. */
void init_lists()
{
  int i;
  for(i=0; i<16;i++) oper_note[i] = 255; /* free */

  /* this asumes that num_voices, set in init_sequencer(), is set. */
  flen = num_voices;
  for(i=0; i<num_voices; i++) {
    free_list[i]=i;
  };

  bufp = 0;
};

/* initialise the sequencer */

void init_sequencer()
{
  int i, n;

  if((seqfd = open(SEQUENCER_PATH, O_RDWR, 0)) == -1) {
    die("Cannot open " SEQUENCER_PATH ".");
  };

  if (ioctl (seqfd, SNDCTL_SEQ_NRSYNTHS, &n) == -1) {
    die("Cannot get " SEQUENCER_PATH " info.");
  };

  for (i = 0; i < n; i++) {
    info.device = i;
    
    if (ioctl (seqfd, SNDCTL_SYNTH_INFO, &info) == -1) {
      die("Cannot get " SEQUENCER_PATH " info.");
    }
    
    if (info.synth_type == SYNTH_TYPE_SAMPLE
	&& info.synth_subtype == SAMPLE_TYPE_GUS)
      gus_dev = i;
    free_voices = num_voices = (info.nr_voices>999)?999:info.nr_voices;
  }
  if(gus_dev == -1) die("Gravis UltraSound not available.");

  if (ioctl (seqfd, SNDCTL_SEQ_RESETSAMPLES, &gus_dev) == -1)
    complain("Could not reset sequencer.");

  gus_memsize = gus_dev; ioctl(seqfd, SNDCTL_SYNTH_MEMAVL, &gus_memsize);
  gus_freemem = gus_memsize;

  if (PM_LOAD_PATCH(gus_dev, 0, 0) < 0) /* Load the default instrument */
    if (errno != ESRCH)	/* No such process */
      complain("PM_LOAD_PATCH failed");
};

/* load a patch into the UltraSound. */

void load_patch(char *name)
{
  char buf[256];
  int patfd;
  struct pat_header header;
  struct sample_header sample;
  struct patch_info *patch;
  int offset;
  int i;

  if ((patfd = open (name, O_RDONLY, 0)) == -1) {
    complain ("Cannot open patch file \"%s\".", name);
    return;
  };

  if (read (patfd, buf, 0xef) != 0xef) {
    complain ("%s: Short file\n", name);
    goto exit_close;
  }
  memcpy ((char *) &header, buf, sizeof (header));

  if (strncmp (header.magic, "GF1PATCH110", 12)) {
    complain ("%s: Not a patch file\n", name);
    goto exit_close;
  }

  if (strncmp (header.version, "ID#000002", 10)) {
    complain ("%s: Incompatible patch file version\n", name);
    goto exit_close;
  }

  header.nr_waveforms = *(unsigned short *) &buf[85];
  header.master_volume = *(unsigned short *) &buf[87];

  offset = 0xef;

  ioctl (seqfd, SNDCTL_SEQ_RESETSAMPLES, &gus_dev); /* free patches */
  /* is there a better way? */

  for (i = 0; i < header.nr_waveforms; i++) {
    if (lseek (patfd, offset, 0) == -1) {
      complain ("%s: cannot lseek()", name);
      goto exit_close;
    }
    if (read (patfd, &buf, sizeof (sample)) != sizeof (sample)) {
      complain ("%s: Short file", name);
      goto exit_close;
    }

    memcpy ((char *) &sample, buf, sizeof (sample));

    /*
     * Since some fields of the patch record are not 32bit aligned, we must
     * handle them specially.
     */
    sample.low_note = *(long *) &buf[22];
    sample.high_note = *(long *) &buf[26];
    sample.base_note = *(long *) &buf[30];
    sample.detune = *(short *) &buf[34];
    sample.panning = (unsigned char) buf[36];
    
    memcpy (sample.envelope_rate, &buf[37], 6);
    memcpy (sample.envelope_offset, &buf[43], 6);
    
    sample.tremolo_sweep = (unsigned char) buf[49];
    sample.tremolo_rate = (unsigned char) buf[50];
    sample.tremolo_depth = (unsigned char) buf[51];
    
    sample.vibrato_sweep = (unsigned char) buf[52];
    sample.vibrato_rate = (unsigned char) buf[53];
    sample.vibrato_depth = (unsigned char) buf[54];
    sample.modes = (unsigned char) buf[55];
    sample.scale_frequency = *(short *) &buf[56];
    sample.scale_factor = *(unsigned short *) &buf[58];

    offset = offset + 96;
    patch = (struct patch_info *) malloc (sizeof (*patch) + sample.len);
    
    patch->key = GUS_PATCH;
    patch->device_no = gus_dev;
    patch->instr_no = 0; /* here we may substitute a program number */
    patch->mode = sample.modes | WAVE_TREMOLO | WAVE_VIBRATO | WAVE_SCALE;
    patch->len = sample.len;
    patch->loop_start = sample.loop_start;
    patch->loop_end = sample.loop_end;
    patch->base_note = sample.base_note;
    patch->high_note = sample.high_note;
    patch->low_note = sample.low_note;
    patch->base_freq = sample.base_freq;
    patch->detuning = sample.detune;
    patch->panning = (sample.panning - 7) * 16;
    
    memcpy (patch->env_rate, sample.envelope_rate, 6);
    memcpy (patch->env_offset, sample.envelope_offset, 6);
    
    patch->tremolo_sweep = sample.tremolo_sweep;
    patch->tremolo_rate = sample.tremolo_rate;
    patch->tremolo_depth = sample.tremolo_depth;
    
    patch->vibrato_sweep = sample.vibrato_sweep;
    patch->vibrato_rate = sample.vibrato_rate;
    patch->vibrato_depth = sample.vibrato_depth;
    
    patch->scale_frequency = sample.scale_frequency;
    patch->scale_factor = sample.scale_factor;
    
    patch->volume = header.master_volume;
    
    if (lseek (patfd, offset, 0) == -1) {
      complain ("%s: cannot lseek()", name);
      goto exit_close;
    }

    if (read (patfd, patch->data, sample.len) != sample.len) {
      complain ("%s: Short file", name);
      goto exit_close;
    }

    SEQ_WRPATCH (patch, sizeof (*patch) + sample.len);

    offset = offset + sample.len;
  };
  
  
  gus_freemem = gus_dev; ioctl(seqfd, SNDCTL_SYNTH_MEMAVL, &gus_freemem);
  update_memfree();
 exit_close:
  close(patfd);
};

/*
 * code adapted from midithru.c 
 */

void stop_note(int note, int velocity)
{
  int i, op=255;
  
  for (i=0;i<num_voices && op==255;i++) {
    if (oper_note[i]== note) op=i;
  }

  if (op==255) 
    {
      fprintf(stderr, "Note %d off, note not started\n", note);
      fprintf(stderr, "%d, %d\n", flen, nlen);
      return;	/* Has already been killed ??? */
    }
  
  SEQ_STOP_NOTE(gus_dev, op, note, velocity);
  SEQ_DUMPBUF();
  
  oper_note[op] = 255;
  
  free_list[ftail]=op;
  flen++;
  ftail = (ftail+1) % num_voices;
  
  for (i=0;i<16;i++)
    if (note_list[i] == op) note_list[i] = 255;
  
  while (nlen && note_list[nhead] == 255) {
      nlen--;
      nhead = (nhead+1) % 256;
    }
  free_voices++;
  update_voices();
}

void kill_one_note()
{
  int oldest;
  
  if (!nlen) {
    fprintf(stderr, "Free list empty but no notes playing\n");return;
  }	/* No notes playing */
  
  oldest = note_list[nhead];
  nlen--;
  nhead = (nhead+1) % 256;
  
  fprintf(stderr, "Killing oper %d, note %d\n", oldest, oper_note[oldest]);

  if (oldest== 255) return;	/* Was already stopped. Why? */

  stop_note(oper_note[oldest], 127);
}

void start_note(int note, int velocity)
{
  int free;

  if (!flen) kill_one_note();

  if (!flen) {printf("** no free voices\n");return;}	/* Panic??? */
  
  free = free_list[fhead];
  flen--;
  fhead = (fhead+1) % num_voices;
  
  note_list[ntail] = free;
  
  if (nlen>255) {
#if 0
    fprintf(stderr, "Note list overflow %d, %d, %d\n",
	    nlen, nhead, ntail);	
#endif
    nlen=0;	/* Overflow -> hard reset */
  }
  nlen++;
  ntail = (ntail+1) % 256;
  
  oper_note[free] = note;
  
  SEQ_SET_PATCH(gus_dev, free, pgm);
  SEQ_PITCHBEND(gus_dev, free, bender);
  SEQ_START_NOTE(gus_dev, free, note, velocity);
  SEQ_DUMPBUF();

  free_voices--;
  update_voices();
}

void channel_pressure(int ch, int pressure)
{
  int i;

  for (i=0;i<num_voices;i++) {
    if (oper_note[i] != 255) {
      SEQ_CHN_PRESSURE(gus_dev, i, pressure);
      SEQ_DUMPBUF();
    }
  }
}

void pitch_bender(int ch, int value)
{
  int i;
  
  value -= 8192;
  
  bender = value;
  
  for (i=0;i<num_voices;i++) {
    if (oper_note[i] != 255) {
      bender = value;
      SEQ_PITCHBEND(gus_dev, i, value);
      SEQ_DUMPBUF();
    }
  }
}

void do_buf()
{
  int ch = buf[0] & 0x0f;
  int value;
  
  switch (buf[0] & 0xf0) {
  case 0x90:	/* Note on */
    if (bufp < 3) break;
    if (buf[2])
      start_note(buf[1], buf[2]);
    else
      stop_note(buf[1], buf[2]);
    bufp=1;
    break;
    
  case 0xb0:	/* Control change */
    if (bufp < 3) break;
    bufp=1;
    break;
    
  case 0x80:	/* Note off */
    if (bufp < 3) break;
    stop_note(buf[1], buf[2]);
    bufp=1;
    break;
    
  case 0xe0:	/* Pitch bender */
    if (bufp < 3) break;
    value = ((buf[2] & 0x7f) << 7) | (buf[1] & 0x7f);
    pitch_bender(ch, value);
    bufp=1;
    break;
    
  case 0xc0:	/* Pgm change */
    if (bufp < 2) break;
    pgm = buf[1];
/*    xv_set(patch_choice, PANEL_LIST_SELECT, pgm, TRUE);*/
    bufp=0;
    break;
    
  case 0xd0:	/* Channel pressure */
    if (bufp < 2) break;
    channel_pressure(ch, buf[1]);
    bufp=1;
    break;
    
  default:
    bufp=0;
  }
}

void load_other_proc(Panel_item i, Event *e)
{
  do_filedlg("*.pat", &load_patch);
};

/* code for loading patches by number, from the default list */

void load_std_patch(int pn)
{
    char buffer[256]; /* should suffice */
    sprintf(buffer, "%s/%s.pat", PATCH_PATH, patch_names[pn]);
    load_patch(buffer);    
};

volatile void usage(char *whoami)
{
    fprintf(stderr, "usage: %s [-v]\n", whoami);
    exit(1);
};

main(int argc, char *argv[])
{
  char fmt_buf[64];
  Notify_client client = (Notify_client)23532;
  int n;
  int i;
#ifdef INCLUDE_TTY
  Tty ttysw;
#endif

  /* handle args, if any */
  for(i=1; i<argc; i++) {
      if(*(argv[i])=='-') {
	  switch(*(argv[i]+1)) {
	  case 'v': show_voices=0; break;
	  default: usage(argv[0]);
	  };
      };
  };

  xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, NULL);
  /* make the XView objects required for the interface */
  frame = (Frame)xv_create(NULL, FRAME, 
			   FRAME_LABEL, "Jam",
			   NULL);

  init_sequencer();
  init_lists();
  (void) notify_set_input_func(client, read_midi, seqfd);

  panel = (Panel)xv_create(frame, PANEL, 
			   PANEL_LAYOUT, PANEL_VERTICAL,
			   NULL);
  patch_choice = (Panel_item) xv_create(panel, PANEL_LIST,
					PANEL_LABEL_STRING, "Patch:",
					PANEL_LIST_STRINGS, GMIDI_NAMES, 
					NULL,
					PANEL_NOTIFY_PROC, patch_notify,
					XV_HELP_DATA, "jam:patch",
					NULL);
  (void) xv_create(panel, PANEL_BUTTON,
		   PANEL_LABEL_STRING, "Load other...",
		   PANEL_NOTIFY_PROC, load_other_proc,
		   NULL);
/*  sprintf(fmt_buf, "%d voices", num_voices);*/
  memfree = (Panel_item) xv_create(panel, PANEL_GAUGE,
				   PANEL_LABEL_STRING, "Free memory:",
				   PANEL_VALUE, gus_freemem,
				   PANEL_MIN_VALUE, 0,
				   PANEL_MAX_VALUE, gus_memsize,
				   PANEL_TICKS, 5,
				   XV_HELP_DATA, "jam:freemem",
				   PANEL_NEXT_COL, NULL,
				   NULL);
  if(show_voices) {
      voices = (Panel_item) xv_create(panel, PANEL_GAUGE,
				      PANEL_LABEL_STRING, "Free voices:",
				      PANEL_VALUE, free_voices,
				      PANEL_MIN_VALUE, 0,
				      PANEL_MAX_VALUE, num_voices,
				      PANEL_TICKS, 5,
				      XV_HELP_DATA, "jam:freevoices",
				      NULL);
  };
  window_fit(panel);
#ifdef INCLUDE_TTY
  ttysw = (Tty) xv_create(frame, TTY,
			  WIN_BELOW, panel,
			  WIN_X, 0,
			  WIN_WIDTH, xv_get(panel, WIN_WIDTH),
			  WIN_HEIGHT, 67,
			  TTY_ARGV, TTY_ARGV_DO_NOT_FORK,
			  NULL);
  /* change the standard i/o file descriptors to point into ttysw */
  dup2((int)xv_get(ttysw, TTY_TTY_FD), 0);
  dup2((int)xv_get(ttysw, TTY_TTY_FD), 1);
#endif
  window_fit(frame);

  xv_set(patch_choice, PANEL_LIST_SELECT, pgm, TRUE, NULL); load_std_patch(pgm);

  xv_main_loop(frame);
};

/* the notifier function which gets input from /dev/sequencer */
Notify_value read_midi(Notify_client client, int fd)
{
  unsigned char ev[4];
  int i,n;

  if((n=read(seqfd, ev, sizeof(ev)))==-1) die("Error reading " SEQUENCER_PATH);
  for(i=0; i<=(n/4); i++) {
    unsigned char *p = &ev[i*4];

    if(p[0]==SEQ_MIDIPUTC && p[2]==0) {
      if(p[1] & 0x80) { /* Status */
	if(bufp) do_buf();
	buf[0] = p[1]; bufp=1;
      } else
	if(bufp) {
	  buf[bufp++] = p[1];
	  if ((buf[0] & 0xf0) == 0x90 || (buf[0] & 0xf0) == 0x80 
	      || (buf[0] & 0xf0) == 0xb0 || (buf[0] & 0xf0) == 0xe0) {
	    if(bufp==3) do_buf();
	  } else if ((buf[0] & 0xf0) == 0xc0 || (buf[0] & 0xf0) == 0xd0) {
	    if(bufp==2) do_buf();	    
	  };
	    
	};
    };
  };
};

/* XView notification functions */

void patch_notify(Panel_item item, char *str, caddr_t client_data, 
		  Panel_list_op op, Event *event, int row)
{
  if(op==PANEL_LIST_OP_SELECT) load_std_patch(row);
};
