/*
   EVENTS.C
   Platform independent part of event handling.

   $Id$
 */
/*    Copyright (c) 1994.  The Regents of the University of California.
                    All rights reserved.  */

/* Read events.h for a general overview of the event handling semantics
   defined here.  */

#include "events.h"

/* The following routines constitute the semi-public interface to the
   opaque EVChannel data structure.  The ev_type and ev_context calls
   simply return the type and context as registered with ev_handler.
   EVChannels are kept in two linked lists (one for idlers, another
   for event handlers); ev_next_channel returns the next element in
   the linked list, or 0 at the end of the list.
   ev_ready and ev_set_ready read and set the ready flag:
   ready  - This is set to non-zero when unread input is available on
            the channel.  ev_handle_next resets the ready flag to zero
	    just before the Action routine is called.

   Normally, they are needed only by os_poll or other low-level routines.

   The channel argument may be 0 in any of these routines; they then
   return 0 (if they have a return value), or are no-ops.
 */
extern int ev_type(EVChannel *channel);
extern void *ev_context(EVChannel *channel);
extern EVChannel *ev_next_channel(EVChannel *channel);
extern int ev_ready(EVChannel *channel);
extern void ev_set_ready(EVChannel *channel, int ready);

/* the following platform-dependent routines are required */
extern int os_poll(EVChannel *list, long timeout, int clear);   /* ospoll.c */
extern void *os_key_context(void);        /* ospoll.c */
extern int os_key_type(void);             /* ospoll.c */
extern char *os_get_line(void);           /* osterm.c */
extern void os_free_line(void);           /* osterm.c */
extern void *os_malloc(unsigned long);    /* osmem.c */
extern double os_clock(void);             /* ostime.c */

/* The ready member is set to 1 by os_poll when input is detected;
   os_poll will not be called as long as any non-busy channel has
   ready non-zero.
   A non-zero ready member triggers the Action for that channel;
   ready is set to zero just before the Action is called.
 */
struct EVChannel {
  EVChannel *next;
  EVCallback *Action;
  void *context;
  unsigned long type;     /* long to hold timeout for timer events */
  int busy;
  int ready;
};

/* these are the linked lists for the I/O events, idlers, and timeouts */
static EVChannel *eventChannels= 0;
static EVChannel *idlerChannels= 0;
static EVChannel *timerChannels= 0;

/* these are to implement round-robin calling among the channels */
static EVChannel *eventNext= 0;
static EVChannel *idlerNext= 0;
static unsigned long timePrevious;

/* the keyboard channel is in the eventChannels list, but needs to be
   separately identified */
static EVChannel *keyChannel= 0;

static EVChannel *freeChannels= 0;   /* next free channel in block */

/* ------------------------------------------------------------------------ */

/* go until there are no non-busy input channels */
void ev_handle_forever(void)
{
  while (ev_handle_next(-1L));
}

/* handle all events which are either ready or can be read immediately */
void ev_handle_available(void)
{
  while (ev_handle_next(0L));
}

/* ------------------------------------------------------------------------ */

static int handle_ready(int clear);
static void update_timers(int force);

/* handle zero or one event (a non-busy idler counts as an event):
   1. check for any channels marked as ready
   2. check if any channels have immediately available input
   3. check if there are any idlers waiting to run
   4. wait until input becomes available on some channel

   If timeout==0, step 4 is not taken; if timeout<0, wait forever.
   If timeout>0, wait no longer than that many milliseconds.
      -- On OSs for which a non-infinite wait is impossible,
         timeout>0 is the same as timeout==0.

   If input is ready (or becomes available), the input channels are
   called in round-robin order -- that is, after one channel is called,
   all other channels have a chance to be called before the first will
   be called again.
   Idlers obey a similar round-robin rule among themselves.
 */
int ev_handle_next(long timeout)
{
  /* first see if any events are pending */
  update_timers(0);
  if (handle_ready(0)) return 1;

  /* if we're not supposed to wait, just check if anything can
     be read immediately */
  if (!timeout)
    return (os_poll(eventChannels, 0L, 0)>0 && handle_ready(0));

  /* when we've nothing else to do, maybe run an idler */
  if (!idlerNext) idlerNext= idlerChannels;
  if (idlerNext) {
    EVChannel *idler= idlerNext;
    int ready;
    do {
      if (!idler->busy) {
	/* before starting an idler, check if any immediately readable
	   event channel */
	os_poll(eventChannels, 0L, 0);
	if (!handle_ready(0)) {
	  idlerNext= idler->next;
	  idler->Action(idler->context, 0);
	}
	return 1;
      }
      idler= idler->next;
      if (!idler) idler= idlerChannels;
    } while (idler!=idlerNext);
  }

  /* reduce the timeout if there is a timer channel */
  if (timerChannels &&
      (timeout<0 || timeout<timerChannels->type))
    timeout= timerChannels->type;

  /* finally, wait until input becomes available --
     note that timers only need to be updated if no events are ready */
  if (!os_poll(eventChannels, timeout, 0)) update_timers(1);
  return handle_ready(0);
}

/* implement round-robin for channels with input ready */
static int handle_ready(int clear)
{
  if (timerChannels && (!timerChannels->type || clear)) {
    /* timers which have timed out take precedence over input channels */
    EVCallback *Action= timerChannels->Action;
    void *context= timerChannels->context;
    ev_remove(timerChannels);
    (*Action)(context, clear);
    return 1;
  }
  if (!eventNext) eventNext= eventChannels;
  if (eventNext) {
    EVChannel *chan= eventNext;
    do {
      if (!chan->busy && chan->ready) {
	chan->ready= 0;
	eventNext= chan->next;
	chan->Action(chan->context, clear);
	return 1;
      }
      chan= chan->next;
      if (!chan) chan= eventChannels;
    } while (chan!=eventNext);
  }
  return 0;
}

static double timerPrevious;

static void update_timers(int force)
{
  if (timerChannels) {
    EVChannel *chan= timerChannels;
    double elapsed= timerPrevious;
    timerPrevious= os_clock();
    elapsed= 1000.*(timerPrevious-elapsed);
    if (elapsed>1.e9) elapsed= 1.e9;
    do {
      if (elapsed>chan->type) chan->type= 0;  /* timed out */
      else chan->type-= (long)elapsed;
      chan= chan->next;
    } while (chan);
    /* the force flag forces at least the first timer channel to time out
       -- this is used to ensure that if the timeout to os_poll has been
       set to the timer timeout, and os_poll actually timed out, then the
       timer will trigger */
    if (force) timerChannels->type= 0;
  }
}

/* ------------------------------------------------------------------------ */

/* clear out any pending events and input (as after an error) */
void ev_clear(void)
{
  EVChannel *chan;
  for (chan=eventChannels ; chan ; chan=chan->next) chan->busy= 0;
  do {
    while (handle_ready(1));
  } while (os_poll(eventChannels, 0L, 1));
}

/* ------------------------------------------------------------------------ */

/* KeyCallback is the keyboard callback routine, which sets keyDataReady,
   calls EVGetLine, then calls EVKeyAction if the clear flag is not set.
 */
static void KeyCallback(void *context, int clear);
static int keyDataReady= 0;
static void (*EVKeyAction)(const char *line)= 0;

/* ev_put_back requires the previous line and a flag */
static char *keyLine= 0;
static int keyPutBack= 0;
static int keyErrors= 0;

/* call os_get_line only after keyDataReady -- until then handle others */
char *ev_get_line(void)
{
  if (!keyPutBack) os_free_line();
  while (!keyDataReady) ev_handle_next(-1L);
  if (!keyPutBack) keyLine= os_get_line();
  /* an error on keyboard input is potentially serious --
     allow only a dozen before giving up to avoid infinite loops
     note that os_get_line may have already removed keyChannel if the
     problem is really serious -- the KeyAction can check this */
  if (!keyLine && keyErrors++>12) ev_remove(keyChannel);
  else keyErrors= 0;
  keyDataReady= keyPutBack= 0;
  if (keyChannel) keyChannel->ready= 0;
  return keyLine;
}

/* rig ev_get_line to immediately return same thing as last time */
void ev_put_back(void)
{
  keyPutBack= 1;
  if (keyChannel) keyChannel->ready= 1;
}

/* this is the keyboard Action routine */
static void KeyCallback(void *context, int clear)
{
  char *line;
  keyDataReady= 1;
  line= ev_get_line();
  if (EVKeyAction && !clear) EVKeyAction(line);
}

/* install a KeyAction handler -- the os_key_type or os_key_context routines
   may perform any required platform-dependent initialization */
EVChannel *ev_key_handler(void (*KeyAction)(const char *line))
{
  if (keyChannel) return 0;
  keyChannel= ev_handler(&KeyCallback, os_key_context(), os_key_type());
  EVKeyAction = KeyAction;
  keyDataReady= keyPutBack= keyErrors= 0;
  return keyChannel;
}

EVChannel *ev_find_key(void)
{
  return keyChannel;
}

/* ------------------------------------------------------------------------ */

/* following are for use by any callback routine */

int ev_busy(EVChannel *channel)
{ return channel? channel->busy : 0; }

void ev_set_busy(EVChannel *channel, int busy)
{ if (channel) channel->busy= busy; }

/* following for semi-private use by routines like OSPoll only */

int ev_type(EVChannel *channel)
{ return channel? channel->type : 0; }

void *ev_context(EVChannel *channel)
{ return channel? channel->context : 0; }

EVChannel *ev_next_channel(EVChannel *channel)
{ return channel? channel->next : 0; }

int ev_ready(EVChannel *channel)
{ return channel? channel->ready : 0; }

void ev_set_ready(EVChannel *channel, int ready)
{ if (channel) channel->ready= ready; }

/* ------------------------------------------------------------------------ */

/* remove an event or idler channel */
void ev_remove(EVChannel *channel)
{
  EVChannel *chan= eventChannels;
  EVChannel **prev= &eventChannels;

  while (chan && chan!=channel) {
    prev= &chan->next;
    chan= *prev;
  }
  if (!chan) {
    chan= idlerChannels;
    prev= &idlerChannels;
    while (chan && chan!=channel) {
      prev= chan->next;
      chan= *prev;
    }
    if (!chan) {
      chan= timerChannels;
      prev= &timerChannels;
      while (chan && chan!=channel) {
	prev= chan->next;
	chan= *prev;
      }
    }
  }

  if (chan) {
    /* take special care with channels marked as next in round-robin */
    if (chan==eventNext) eventNext= chan->next;
    if (chan==idlerNext) idlerNext= chan->next;

    /* take care to properly remove all traces of the keyboard handler */
    if (chan==keyChannel) {
      os_free_line();
      keyLine= 0;
      keyDataReady= keyPutBack= 0;
      keyChannel= 0;
    }

    /* unlink and add to the list of free channels */
    *prev= chan->next;
    chan->next= freeChannels;
    freeChannels= chan;
  }
}

static EVChannel *NewHandler(EVCallback *Action, void *context, int type);

/* create an idler channel */
EVChannel *ev_idler(EVCallback *Action, void *context)
{
  EVChannel *chan= 0;
  if (Action && !ev_find_idler(Action, context)) {
    chan= NewHandler(Action, context, 0);
    if (chan) {
      chan->next= idlerChannels;
      idlerChannels= chan;
    }
  }
  return chan;
}

/* create a timer channel */
EVChannel *ev_timer(EVCallback *Action, void *context, long timeout)
{
  EVChannel *chan= 0;
  if (Action && timeout>=0) {
    EVChannel *next= timerChannels;
    EVChannel **prev= &timerChannels;
    if (!timerChannels) timerPrevious= os_clock();
    else update_timers(0);
    chan= NewHandler(Action, context, 0);
    if (chan) {
      /* keep the timers sorted with the one that expires first always
	 at the head of the list */
      while (next && next->type<=timeout) {
	prev= &next->next;
	next= *prev;
      }
      chan->type= timeout;
      chan->next= next;
      *prev= chan;
    }
  }
  return chan;
}

/* create an event channel */
EVChannel *ev_handler(EVCallback *Action, void *context, int type)
{
  EVChannel *chan= 0;
  if (Action && !ev_find(context, type)) {
    chan= NewHandler(Action, context, type);
    if (chan) {
      chan->next= eventChannels;
      eventChannels= chan;
    }
  }
  return chan;
}

#ifndef EV_BLOCK_SIZE
#define EV_BLOCK_SIZE 8
#endif

static EVChannel staticChannels[EV_BLOCK_SIZE];

/* manage the (small amount) of memory required to store channels
   Channels are allocated in blocks as necessary; such blocks are never
   freed.  The first block is statically allocated, any others are
   allocated using the memory manager OSmalloc which has the same
   semantics as the ANSI C malloc routine.
 */
static EVChannel *NewHandler(EVCallback *Action, void *context, int type)
{
  EVChannel *chan;
  if (!freeChannels) {
    int i= EV_BLOCK_SIZE;
    if (eventChannels | idlerChannels) {
      freeChannels= os_malloc(EV_BLOCK_SIZE*sizeof(EVChannel));
      if (!freeChannels) return 0;
    } else {
      freeChannels= staticChannels;
    }
    freeChannels->next= 0;
    while (--i) {
      freeChannels[1].next= freeChannels;
      freeChannels++;
    }
  }
  freeChannels->Action= Action;
  freeChannels->context= context;
  freeChannels->type= type;
  freeChannels->busy= freeChannels->ready= 0;
  chan= freeChannels;
  freeChannels= freeChannels->next;
  return chan;
}

/* ------------------------------------------------------------------------ */
/* routines to find an event (or idler) channel, given the context,
   and type (or Action) */

static EVChannel *FindChannel(EVCallback *Action, void *context, int type);

EVChannel *ev_find_idler(EVCallback *Action, void *context)
{
  return Action? FindChannel(Action, context, 0) : 0;
}

EVChannel *ev_find(void *context, int type)
{
  return FindChannel(0, context, type);
}

static EVChannel *FindChannel(EVCallback *Action, void *context, int type)
{
  EVChannel *chan= Action? idlerChannels : eventChannels;
  while (chan) {
    if ((!Action || chan->Action==Action) &&
	chan->context==context && chan->type==type) break;
    chan= chan->next;
  }
  return chan;
}

/* ------------------------------------------------------------------------ */
