#===========================================================================

sum -- (in)definite summation of rational functions

Calling sequence:
sum(f, x)

Parameters:
f -- an expression (rational function)
x -- an

Summary:
sum(f,x) computes the indefinite sum of f(x) with respect to x.

Copyright:
Paul Zimmermann and Inria, November 1994

--------------------------------------------------------------------------

Examples of the implementation of the "sum" function:

1) Polynomial case:

>> sum(1,k);

                                     k

>> sum(k^2,k);

                                     2    3
                                k   k    k
                                - - -- + --
                                6   2    3
 
>> sum(a*x^2+b*x+c,x);
 
                    3
                 a x      / a   b     \    2 /   a   b  \
                 ---- + x | - - - + c | + x  | - - + -  |
                  3       \ 6   2     /      \   2   2  /
 
 
2) Fractional case: 

a) Decomposition into the polynomial part and the rational part:
 
>> sum(k/(k+1),k);
 
                              k - psi(k + 1)
 
>> sum((x^2+1)/(x-1),x);
 
                              2
                         x   x     
                         - + -- + 2 psi(x - 1) 
                         2   2 
 
 
b) Reduced rational expressions:
   (degree of numerator is less than degree of denominator)

>> sum(1/k/(k-2),k);

                                1           1    
                          - --------- - ---------
                            2 (k - 1)   2 (k - 2)


>> sum(1/k/(2*k-1),k);

                          - psi(k) + psi(k - 1/2)


==========================================================================#

sum := proc(f,_x) local g,_k_,rec,u,inits,d,l,p,n,m,q,z;
begin
   if args(0)<>2 then error("two arguments expected") end_if;
   if type(_x)=DOM_IDENT then Sum::indefinite(f,_x)
   elif type(_x)="_equal" and type((_k_:=op(_x,1)))=DOM_IDENT then
      if type(op(_x,2))="_range" then
         # definite summation #
         g:=Sum::indefinite(f,_k_);
	 if type(g)="sum" then 
            userinfo(1,"indefinite summation failed");
            if Sum::ishypergeom(f,_k_) then
               userinfo(1,"summand is hypergeometric");
	       # try Zeilberger's algorithm #
               # choose in priority indets that are in the bounds #
               l:=indets(op(_x,2));
               l:=[op(l),op(indets(f) minus (l union {_k_}))];
	       for n in l do
		  if Sum::ishypergeom(f,n) then
                     userinfo(1,"trying Zeilberger's algorithm with respect to",n);
		     rec:=Sum::zeilberger(f,_k_,n,u,op(op(_x,2)));
		     if rec=FAIL then 
                        userinfo(1,"Zeilberger's algorithm fails"); next
                     else userinfo(1,"Zeilberger's algorithm succeeds") end_if;
		     d:=op(rec,[1,1])-n;
		     inits:=null();
		     for l from 0 to d-1 do
			p:=subs(f,n=l);
			inits:=inits,u(l)=_plus(eval(subs(p,_k_=m))$hold(m)=0..l);
		     end_for;
                     p:=subsop(proc(n) begin n end_proc,4=hold(_plus)(hold(_seqgen)(f,_k_=0..n)));
		     rsolve; # to load Rsolve::holonomic #
		     rec:=Rsolve::holonomic(rec,{},u,n,d);
                     # substitute initial values afterwards because
                       more can be needed due to singularities #
                     rec:=eval(subs(rec,u=proc(n) begin
			if type(n)=DOM_INT then p(n)
                        else u(n) end_if end_proc));
                     return(rec)
		  end_if
               end_for;
               # f is hypergeometric but Zeilberger's algorithm fails #
               userinfo(1,"trying definite summation of hypergeometrics");
               g:=Sum::hypergeomtype(f,_k_);
               userinfo(2,"hypergeometric type is",g);
               if g<>FAIL then
                  g:=Sum::hypergeomeval(g,op(op(_x,2)),f,_k_);
                  if g<>FAIL then
                     # normalize the result #
                     return(eval(subs(f,_k_=0))*g)
                  end_if
               end_if;
               userinfo(1,"definite summation of hypergeometrics failed");
            end_if;
            # not hypergeometric in k or in another variable #
            procname(f,_x)
	 else # indefinite summation succeeded #
            userinfo(2,"indefinite summation gave",g);
	    # eval is needed to evaluate psi(1,1) into PI^2/6 for example #
	    if traperror((g:=Sum::eval(g,_k_,op(_x,[2,2])+1)
		-Sum::eval(g,_k_,op(_x,[2,1]))))=0 then normal(level(g,2))
            else procname(f,_x)
            end_if
	 end_if
      elif type((p:=op(_x,2)))=RootOf then # sum(..., k=RootOf(q,z)) #
         userinfo(1,"sum over RootOf");
         z:=extop(p,2);
         q:=subs(extop(p,1),z=_k_);
         if testtype(f,Type::RatExpr(_k_)) then # first simplify to a sum over a polynomial #
            userinfo(2,"summand is rational");
            f:=RootOf::evala(f,q,_k_)
         end_if;
         if testtype(f,Type::PolyExpr(_k_)) then # evaluate #
            RootOf::polysum(f,q,_k_)
         else hold(sum)(args())
         end_if
      else
         error("invalid second argument")
      end_if
   else error("invalid arguments")
   end_if
end_proc:

sum:= funcattr(sum, "type", "sum"):
sum:= funcattr(sum, "print", "sum"):

Sum:= domain():
Sum::name:= "Sum":
Sum::info:= "Sum: Utilities for the sum function":
Sum::interface:= {}:

Sum::eval := proc(f,x,v) # evaluate value of f at x=v #
begin
   if v=infinity or v=infinity+1 then
      limit(f,x=infinity)
   else # do no eval here ==> problems if f contains the name "x",
          for example sum(x^k*y^(n-k),k=0..n) or sum(x^k,k=1..2) #
      subs(f,x=v)
   end_if
end_proc:

# returns h such that h(x+1)-h(x)=f(x) #
Sum::indefinite:=proc(f,x)
begin
   userinfo(1,"entering indefinite summation");
   if testtype(f,Type::PolyExpr(x,Type::AnyType)) then Sum::poly(f,x)
   elif testtype(f,Type::RatExpr(x,Type::AnyType)) then Sum::rat(f,x)
   else Sum::gosper(f,x)
   end_if
end_proc:

# p is a polynom in x, returns h such that p = subs(h,x=x+1)-h #
Sum::poly := proc(p,x)
local h,d,_a_,i,eq,c,b;
begin
  if not has(p,x) then
    p*x
  else # method of indeterminate coefficients #
   d:=degree(p,[x]);
   h:=x*_plus(_a_[i]*x^i $ i=0..d);
   eq:=subs(h,x=x+1)-h-p;
   for i from d downto 0 do
      c:=coeff(eq,[x],i);
      b[i]:=-coeff(c,[_a_[i]],0)/coeff(c,[_a_[i]],1);
      eq:=subs(eq,_a_[i]=b[i]);
   end_for;
   _plus(b[i]*x^(i+1) $ hold(i)=0..d)
  end_if
end_proc:

Sum::rat := proc(f,x)
local num,den,d,dnum,dden;
begin
   num:=numer(f); den:=denom(f);
   dnum:=degree(num,x); dden:=degree(den,x);
   if dnum>=dden then
      d:=divide(poly(num,[x]),poly(den,[x]));
      Sum::poly(expr(d[1]),x)+Sum::rat(expr(d[2])/den,x);
   else # now deg(num)<deg(den) #
      d:=Sum::rat_abramov(f,x);
      if d=FAIL then hold(sum)(f,x) else d end_if
   end_if
end_proc:

# f is minimal : sum(f,x) can not be written p+sum(r,x)
  where p is a polynomial and denom(r) has lower degree than denom(f)
#
Sum::to_psi := proc(f,x)
local ff,t,s,a,d,lin;
begin
   ff:=partfrac(f,x,full);
   userinfo(3,"full partial fraction decomposition=",ff);
   if type(ff)="_plus" then ff:=[op(ff)] else ff:=[ff] end_if;
   s:=0;
   for t in ff do
      d:=Sum::mydenom(t); # because denom expands the denominator #
      d:=d/content(d,[x]); # primpart(d,[x]) expands too !!! #
      if type(d)="_power" and testtype(op(d,2),Type::NonNegInt) then lin:=op(d,1)
      else lin:=d
      end_if;
      if testtype(lin,Type::PolyExpr(x,Type::AnyType)) then
         case degree(lin,[x])
         of 1 do a:=coeff(lin,[x],0)/coeff(lin,[x],1);
                 s:=s+subs(Sum::to_psi2(normal(subs(t,x=x-a)),x),x=x+a);
                 break;
         otherwise return(FAIL)
         end_case
      else # non polynomial #
         return(FAIL)
      end_if
   end_for;
   s
end_proc:

Sum::is_linear:=proc(p,x) # checks for a*x+b #
begin
   if testtype(p,Type::PolyExpr(x,Type::AnyType)) then
      if degree(p,x)=1 then return(TRUE) end_if
   end_if;
   FALSE
end_proc:

Sum::to_psi2 := proc(f,x) # the denominator looks like a*x^j #
# f must be in normal form #
local n,j,res,d,c,dd,t;
begin
   res:=0;
   d:=denom(f);
   c:=content(d,[x]);
   d:=normal(d/c); # should be given by primpart #
   if not ((d=x) or (type(d)="_power" and op(d,1)=x and type(op(d,2))=DOM_INT))
   then return(FAIL) # should not happen #
   end_if;
   dd:=degree(d,x)-1;
   n:=numer(f);
   if type(n)="_plus" then n:=[op(n)] else n:=[n] end_if;
   for t in n do
      j:=dd-degree(t,x);
      res:=res+content(t,[x])/c*(-1)^j/fact(j)*psi(x,j);
   end_for;
   res
end_proc:

Sum::rat_abramov := proc(f,x)
begin
   userinfo(1,"entering Abramov's algorithm");
   Sum::abr(f,Sum::dispersion(denom(normal(f)),x),x)
end_proc:

Sum::abr := proc(f,dis,x)
local p,q,cp,vwp,vwm,ba,u,newf;
begin
   userinfo(2,"dispersion=",dis);
   if f=0 then return(0) end_if;
   if dis=0 then return(Sum::to_psi(f,x)) end_if;
   p:=numer(f); q:=denom(f);
   cp:=gcd(q,subs(q,x=x+dis));
   if cp=1 then return(Sum::abr(f,dis-1,x)) end_if;
   vwp:=Sum::abr_part(q,cp);
   vwm:=Sum::abr_part(q,subs(cp,x=x-dis));
   if degree(vwm[1],x)>degree(vwp[1],x) then
      ba:=pdioe(vwp,p,x);
      u:=subs(ba[2]/vwp[1],x=x-1);
      newf:=normal(ba[1]/vwp[2]+u);
   else
      ba:=pdioe(vwm,p,x);
      u:=-ba[2]/vwm[1];
      newf:=normal(ba[1]/vwm[2]+subs(ba[2]/vwm[1],x=x+1))
   end_if;
   u+Sum::abr(newf,dis-1,x)
end_proc:

Sum::abr_part := proc(p,h)
local g,v,w;
begin
   g:=gcd(p,h); v:=1; w:=p;
   while g<>1 do w:=divide(w,g,Quo); v:=v*g; g:=gcd(w,v) end_while;
   v,w
end_proc:

Sum::dispersion := proc(q,x) # q is a polynomial #
local _k,r,i,s;
begin
   q:=normal(q/content(q,[x]));
   r:=resultant(q,subs(q,x=x+_k),x);
   r:=factor(r);
   s:={};
   for i from 2 to nops(r) step 2 do
      # this is because iroots does not like other variables #
      if indets(op(r,i))={_k} then s:=s union sharelib::iroots(op(r,i),FALSE) end_if;
   end_for;
   # no multiplicity for iroots #
   max(op(s))
end_proc:

# because denom(1/(x+1)^2) gives x^2+2*x+1 #
Sum::mydenom := proc(f)
begin
   if type(f)="_power" then
      if testtype(op(f,2),Type::NegInt) then op(f,1)^(-op(f,2))
      else 1
      end_if
   elif type(f)="_mult" then map(f,Sum::mydenom)
   else denom(f)
   end_if
end_proc:

Sum::ishypergeom := proc(f,n)
begin
   testtype(Sum::ratio(f,n),Type::RatExpr(n))
end_proc:

# f being hypergeometric in k, puts f(k+1)/f(k) in the form
  (k+a1)...(k+ap)/(k+b1)/.../(k+bq) x/(k+1)
  and returns [a1,...,ap], [b1,...,bq], x,
  cf Wilf page 5 #
Sum::hypergeomtype := proc(f,k)
local g,P,Q,a,b,x,t,d;
begin
   g:=normal(subs(Sum::ratio(f,k),k=k+1));
   P:=Factor(numer(g));
   if type(P)<>"_mult" then P:=[P] end_if;
   a:=[]; b:=[]; x:=1;
   for t in P do
      if not has(t,k) then x:=x*t
      else
         if type(t)="_power" then d:=op(t,2); t:=op(t,1) else d:=1 end_if;
         if degree(t,k)<>1 then return(FAIL) end_if;
         x:=x*coeff(t,k,1);
         a:=append(a,coeff(t,k,0)/coeff(t,k,1) $ d);
      end_if
   end_for;
   Q:=Factor(denom(g));
   if type(Q)<>"_mult" then Q:=[Q] end_if;
   for t in Q do
      if not has(t,k) then x:=x/t
      else
         if type(t)="_power" then d:=op(t,2); t:=op(t,1) else d:=1 end_if;
         if degree(t,k)<>1 then return(FAIL) end_if;
         x:=x/coeff(t,k,1);
         b:=append(b,coeff(t,k,0)/coeff(t,k,1) $ d);
      end_if
   end_for;
   t:=contains(b,1);
   if t=0 then # no factor k+1 in the denominator, add one #
      a:=append(a,1)
   else # delete it #
      b[t]:=NIL
   end_if;
   a,b,x
end_proc:

# sum(f,k=a..b) when sum(f,k=0..infinity) = pFq(P,Q,z) #
Sum::hypergeomeval := proc(P,Q,z,a,b,f,k) local init;
begin
   userinfo(1,"hypergeometric of type",nops(P),"F",nops(Q));
   if nops(P)=1 and nops(Q)=0 then # 1F0 #
      if testtype(a,Type::NonNegInt) and b=infinity then
         userinfo(2,"1F0 with right bound at infinity");
         init:=_plus(f $ k=0..a-1);
         return((1-z)^(-P[1])-init)
      end_if
   end_if;
   FAIL
end_proc:

Sum::ratio := proc(f,n) # computes f/subs(f,n=n-1) #
local p,a,b,i;
option remember;
begin
   case type(f)
   of "_mult" do return(map(f,Sum::ratio,n))
   of "_power" do
      if not has(op(f,2),n) then return(Sum::ratio(op(f,1),n)^op(f,2)) 
      else break end_if
   of "fact" do
      if not has(f,n) then return(1) 
      elif Sum::is_linear((p:=op(f)),n) then
         if type((a:=coeff(p,n,1)))=DOM_INT then
            b:=coeff(p,n,0);
            if a>0 then return(_mult(a*n+b-i$i=0..a-1))
            elif a<0 then return(1/eval(_mult(a*n+b+i$i=1..-a))) # PR-6 #
            # a cannot be 0 otherwise p would be independent of n #
            end_if
         end_if
      end_if;
   end_case;
   # other cases #
   if type(f)="function" then
      if op(f,0)=hold(rfact) then
         if not has(op(f,1),n) and Sum::is_linear(op(f,2),n) and
	    coeff(op(f,2),n,1)=1 then return(op(f,1)+op(f,2)-1)
         end_if
      end_if
   elif type(f)="binomial" then return(Sum::ratio(fact(op(f,1))/
		fact(op(f,2))/fact(op(f,1)-op(f,2)),n))
   end_if;
   normal(f/subs(f,n=n-1))
end_proc:

# now in SPECFUNC/fact.mu
fact := funcattr(fact,"expand",
   proc(e) local i,j;
   begin 
      if type(e)<>"_plus" then return(fact(e)) end_if;
      --- use the fact that integers are always at the end of a sum 
      i:=op(e,nops(e));
      if testtype(i,Type::PosInt) then
         _mult(fact(e-i),e-j$j=0..i-1)
      elif testtype(i,Type::NegInt) then
         _mult(e+j$j=1..-i);
         fact(e-i)/%
      else fact(e)
      end_if
   end_proc):
#

# from GKP pages 224-226 #
Sum::gosper := proc(f,_n_) # returns g such that f = subs(g,n=n+1)-g #
local rat,qn,rn,res;
begin
   userinfo(1,"enter Gosper");
   if not has(f,_n_) then f*_n_
   else
      # write f(k+1)/f(k) = p(k+1)/p(k) * q(k)/r(k+1) (5.117)
        where roots of q and r do not differ by an integer #
      rat:=Sum::ratio(f,_n_);
      # check for geometric summations #
      if not has(rat,_n_) then return(eval(subs(f,_n_=0))*rat^_n_/(rat-1)) end_if;
      if has(rat,{binomial,fact}) then rat:=normal(expand(rat)) end_if;
      userinfo(2,"f(",_n_,")/f(",_n_,"-1)=",rat);
      qn:=numer(rat);
      rn:=denom(rat);
      if not (testtype(qn,Type::PolyExpr(_n_,Type::AnyType)) 
         and testtype(rn,Type::PolyExpr(_n_,Type::AnyType))) then
            return(hold(sum)(f,_n_))
      end_if;
      userinfo(1,"input is hypergeometric");
      res:=Sum::gosper2(1,qn,rn,_n_,{},f);
      if res=FAIL then 
         userinfo(1,"Gosper's algorithm fails");
         hold(sum)(f,_n_)
      else normal(res*f)
      end_if
   end_if
end_proc:

# returns g/f such that f = subs(g,n=n+1)-g for f(n)/f(n-1)=p(n)/p(n-1)*qn/rn
  with pn,qn,rn polynomials,
  inds is a set of auxiliary unknowns (used only in Zeilberger) #
Sum::gosper2 := proc(pn,qn,rn,n,inds,f)
local jj,rnj,res,i,p_zeros,z,j,gn,gnj,dp,R,dR,Q,dQ,d,k0,fn,eq,F,ii,m,unk;
begin
   userinfo(3,"pn=",pn,"qn=",qn,"rn=",rn);
      rnj:=subs(rn/content(rn,[n]),n=n+jj); # do not expand ! #
      res:=resultant(qn,rnj,n);
      res:=factor(res);
      p_zeros:=[];
      for i from 1 to nops(res) div 2 do
         ff:=op(res,2*i);
         if degree(ff,jj)=1 then
            z:=-coeff(ff,jj,0)/coeff(ff,jj,1);
            if testtype(z,Type::PosInt) then p_zeros:=append(p_zeros,z) end_if
         end_if
      end_for;
      for j from nops(p_zeros) downto 1 do
         gn:=gcd(qn,expand(subs(rn,n=n+p_zeros[j])));
         qn:=divide(qn,gn,Exact);
         gnj:=gn;
         for i from 1 to p_zeros[j] do
            pn:=expand(pn*gnj);
            gnj:=expand(subs(gn,n=n-i));
         end_for;
         rn:=normal(rn/gnj); # divide(rn,gnj,Exact) ? #
      end_for;
      # make the polynoms monic (cf Prop. 3.2 of [LiPaSt93]) #
      qn:=qn/lcoeff(rn);
      rn:=rn/lcoeff(rn);
      qn:=expand(subs(qn,n=n+1));
      dp:=degree(pn,n);
      R:=expand(qn+rn); dR:=degree(R,n);
      Q:=expand(qn-rn); dQ:=degree(Q,n);
      if dR<=dQ then # case 1 #
         userinfo(2,"case 1 of Gosper's algorithm");
         if dQ=0 then # polynomial case #
            d:=dp+1
         else
            d:=dp-dQ
         end_if
      else # case 2 : deg(qn)=deg(rn) and lcoeff(qn)=lcoeff(rn) #
         m:=dR; # to have the same notation as [LiPaSt93] #
         k0:=coeff(rn-qn,n,m-1); # (5.2) of [LiPaSt93] #
         if type(k0)<>DOM_INT or m=1 # cf [LiPaSt93] page 255 #
         or testtype(f,Type::RatExpr(n,Type::AnyType)) then # case 2a #
            userinfo(2,"case 2a of Gosper's algorithm");
            d:=dp-m+1
         else # case 2b #
            userinfo(2,"case 2b of Gosper's algorithm");
            d:=max(k0,dp-m+1)
         end_if
      end_if;
      userinfo(2,"d=",d);
      if d>=0 then
         fn:=_plus(F[ii]*n^ii$ii=0..d);
         # eq. (GE) of [LiPaSt93] p. 248 #
         # where qn is q(n+1) #
         eq:=expand(pn-qn*subs(fn,n=n+1)+rn*fn);
         userinfo(2,"number of equations is",degree(eq,[n])+1);
         eq:={coeff(eq,n,i)$hold(i)=0..degree(eq,[n])};
         unk:=inds union {F[ii]$ii=0..d};
         if nops(eq)<nops(unk) then eq:=eq union {F[0]=0} end_if;
         userinfo(3,"system to solve is",eq);
         eq:=linsolve(eq,unk);
         userinfo(3,"solution is",eq);
         if eq=null() then FAIL
         else
            # do the substitution into pn too because it can
              contain some indeterminates C[l] in Zeilberger's alg. #
            fn:=Sum::normal(subs(fn/pn,op(eq))*rn);
            if has(fn,F) then FAIL
            elif inds={} then fn # for Sum::gosper #
            else fn,eq # for Sum::zeilberger #
            end_if
         end_if
      else FAIL
      end_if
end_proc:

# required because normal expands #
Sum::normal := proc(f)
begin
   Sum::factor(numer(f))/Sum::factor(denom(f))
end_proc:

Sum::factor := proc(f)
local i,l;
begin
   l:=factor(f);
   _mult(l[1],l[2*i]^l[2*i+1]$i=1..nops(l) div 2)
end_proc:

# u is the name of the output recurrence #
Sum::zeilberger:=proc(f,k,n,u,a,b) # sum(f,k=a..b) #
local ratn,d,rec,l,m,C,p,ratk,res,rnk,cert,va,vb,A,fcert,corr,i;
begin
   ratn:=Sum::ratio(f,n); # f(n)/f(n-1) #
   if not has(ratn,{n,k}) then return(FAIL) 
   elif not testtype(ratn,Type::RatExpr(n)) then return(FAIL)
   elif not testtype(ratn,Type::RatExpr(k)) then return(FAIL)
   end_if;
   d:=1;
   repeat
      rnk:=1; m:=1;
      for l from 1 to d do
         m:=m*subs(ratn,n=n+l); # now m=f(n+l)/f(n) #
         rnk:=rnk+C[l]*m;
      end_for;
      # now apply Gosper to rnk*f with respect to k #
      # i.e. search if F=f(n)+C[1]*f(n+1)+...+C[d]*f(n+d)
        satisfies F(k)=G(k+1)-G(k) #
      userinfo(3,"rnk=",rnk);
      l:=C[l]$hold(l)=1..d;
      p:=numer(rnk);
      ratk:=Sum::ratio(f/denom(rnk),k);
      res:=Sum::gosper2(p,numer(ratk),denom(ratk),k,{l},f);
      if res<>FAIL then 
         cert:=subs(res[1]*rnk,op(res[2])); # g = cert*f #
         userinfo(1,"certificate=",cert);
         # thus sum(f,k=a..b)+...+sum(subs(f,n=n+d),k=a..b) 
		= subs(f*cert,k=b+1) - subs(f*cert,k=a) #
         # only deal with the cases where a is independent from n
           and b is independent from n or b=n #
         if has(a,n) then return(FAIL) end_if;
         if has(b,n) and b<>n then return(FAIL) end_if;
         A:=_plus(u(n),C[l]*u(n+l)$hold(l)=1..d);
         fcert:=normal(f*cert);
         if traperror((va:=eval(subs(expand(subs(f*cert,k=a+k)),k=0))))<>0 then
            va:=hold(limit)(fcert,k=a)
         end_if;
         if traperror((vb:=eval(subs(expand(subs(f*cert,k=b+1+k)),k=0))))<>0 then
            vb:=hold(limit)(fcert,k=b+1)
         end_if;
         A:=A-vb+va;
         # correcting terms if b depends on n #
         if b=n then
            corr:=_plus((C[l]*eval(subs(f,n=n+l,k=n+i))$hold(i)=1..l)$hold(l)=1..d);
            A:=A-corr
         end_if;
         A:=expand(subs(A,op(res[2])));
         # to check: subs(rnk,op(res[2]))-subs(cert*Sum::ratio(f,k),k=k+1)+cert
	   should be zero #
         userinfo(1,"recurrence=",A);
         res:=numer(A);
         res:=-coeff(res,u(n+d),0)/coeff(res,u(n+d),1);
         res:=u(n+d)=collect(res,[u(n+l)$hold(l)=0..d-1],Factor);
         return(res)
      else
         userinfo(1,"Gosper's algorithm fails for order ",d);
      end_if;
      d:=d+1; # try a higher order #
   until d>Sum::maxorder end_repeat;
   FAIL
end_proc:

Sum::maxorder:=2:

sum := funcattr(sum,"diff",
proc(e) # e = sum(f,x) #
local i,l,x;
begin
  l:=[args(i)$i=2..args(0)]; # variables to differentiate #
  if not has((x:=op(e,2)),l) then # summation independent from differentiation #
     sum(diff(op(e,1),op(l)),x)
  else
     hold(diff)(e,op(l))
  end_if
end_proc
):

