# #
# $Date: 1995/06/28 15:22:54 $ $Author: frankp $ $Revision: 1.15.2.2 $ #
# #
# frankp, 25.11.1994 #

#++
  rectform.mu

    rectform -- A function for computing the rectangular form of 
		complex expressions

    rectform( x [,rl] )

    x  -- expression
    rl -- (optional) set of unknown variables


    rectform(x) tries to split x into its real and imaginary
    parts and return x - if possible - in the form x_r + I*x_i.
    If this is not possible then rectform(x) tries to find for each
    subexpression in x such a complex canonical form.

    The result of rectform is an element of the domain 'rectform'.
    Such an element consists of four operands. The first two operands
    give the real and imaginary parts of the expression. The third
    operand represents all subexpressions of x for which such a
    rectangular form can not be computed (possible 0 if there are
    not such subexpressions). The fourth operand will be described later.

    The functions 'Re' and 'Im' are overloaded. If the third operand
    is not 0 then 'Re' returns the first operand and 'Im' the
    second operand of an element of type rectform.
    Otherwise the new expressions which results by 'Re' and 'Im' 
    will be computed to their rectangular form.

    rectform maps onto sets, lists and arrays, and also onto polynomials
    and series, where rectform maps onto the coefficients.

    Arithmetical operations between ordinary expressions and elements
    of type rectform are possible, whereby the results will be of
    type 'rectform'.

    The MuPAD function 'expr' is overloaded for arguments of type
    'rectform'. If x is of type 'rectform' then 'expr(x)' will return
    the expression  extop(x,1)+I*extop(x,2)+extop(x,3), i.e. an
    expression of type DOM_EXPR.

    The general assumption of rectform is that unkown variables
    represent complex-valued quantities.
    But there are two possibilities to change this assumption:
    First -- let us call this the 'global' method -- all unknown
    variables defined in the set rectform::globalReal will represent
    real variables. For all coming operations, e.g. arithmetical
    operations, rectform will test if a corresponding unknown
    variable can be found in the set rectform::globalReal and treated 
    this as real if true.

    The second method -- let us call this the 'local' method -- 
    is to state a set rl of unknown variables by calling rectform
    which elements should represent real-valued quantities. 
    Then for this expression returned by the call of rectform
    the variables in rl will ever be treated as real.
    This set is part of any element of type 'rectform' and it is
    stored in the fourth operand of such an element.
        The fourth operand of an element x of rectform can be
    changed by calling rectform with x by stating rl. Then
    the expression will be recalculated in respect to the new variables
    in rl defined to be real. If rectform is called with x without the 
    second argument rl, then rectform will return x without changes.
	The method 'real' returns the fourth operand of an rectform
    element x.
    
    The main difference of the global method and the local method is
    that the global method will not change the internal representation
    of an element of type rectform, but will influence all further 
    operations, where this element is be a part of.

    For arithmetical operations with different sets of unknown variables
    set to be real, the result will be computed for the intersection of
    both sets (see examples). Thus the local method may not influence
    futher operations.

    For expressions, for which the 0-th operand is of type DOM_FUNC_ENV,
    rectform will used the function attribute "rectform" by calling
    the attribute with the operands of x, if it exist.

    The following example shows how the function attribute "rectform"
    should be defined:

    sin := funcattr( sin, "rectform", proc(x)
        local a, b, rl;
    begin
        if args(0) = 2 then
            x := rectform::new(x,args(2))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
        if extop(x,3) <> 0 then
            new(rectform,0,0,sin(x),rl)
        else
            a := extop(x,1);
            b := extop(x,2);
            if domtype(a+I*b) = DOM_COMPLEX
            and domtype(a) = DOM_FLOAT or domtype(b) = DOM_FLOAT
            then
                rectform::new(sin(a+I*b))
            else
                new(rectform,sin(a)*cosh(b),cos(a)*sinh(b),0,rl)
            end_if
        end_if
    end_proc):

    rectform treates expressions of type "_plus", "_mult" and "_power"
    and system functions explicitly.


    Examples:

    >> rectform( tan(x) );

                sin(2 Re(x))                 I sinh(2 Im(x))       
        ---------------------------- + ----------------------------
        cos(2 Re(x)) + cosh(2 Im(x))   cos(2 Re(x)) + cosh(2 Im(x))

    >> Im(%);

                              sinh(2 Im(x))        
                       ----------------------------
                       cos(2 Re(x)) + cosh(2 Im(x))

    >> a := rectform(x);

                              Re(x) + I Im(x)

    >> b := rectform(y,{y});

                                    y

    >> c := a+b;

                    (Re(x) + Re(y)) + I (Im(x) + Im(y))

    >> rectform::globalReal := {x,y}:
    >> a, b;

                           Re(x) + I Im(x), y 

    >> c;                    

                    (Re(x) + Re(y)) + I (Im(x) + Im(y)) 

    >> rectform(c);

                                  x + y 

    >> b::real(b);
                                   {y}
++#

#--
        Methods for rectform:

	new(z)		expand(z)	simplify(z)   
	_plus()		_mult()		iszero(z)    
	expr(z)		length(z)	testtype(z,T)
	Re(z)		Im(z)		conjugate(z) 
	convert(z)	diff()		intmult()
	globalReal	info 		abs(z)
	real(z)
--#

rectform := domain():

rectform::globalReal := {}: # default 'globalReal' is empty #

# #
# name #
# #
domattr(rectform,"name") := "rectform":

# #
# info #
# #
domattr(rectform,"info") := 
"rectform(x) -- splits x into real and imaginary part [try ?rectform for options]":

# #
# expr #
# #
domattr(rectform,"expr") := fun(
    (if domtype(args(1)) <> rectform then args(1)
    else
	_plus( extop(args(1),1),extop(args(1),2)*I,extop(args(1),3) )
    end_if)
):

# #
# float #
# #
domattr(rectform,"float") := float@expr:

# #
# real #
# #
domattr(rectform,"real") := fun( extop(args(1),4) ):

# #
# I  --  only used for ordinary output of a complex expression #
# #
domattr(rectform,"pI") := new(rectform,I,0,0,{}):

# #
# print #
# #
domattr(rectform,"print") := fun(
	(if extop(args(1),1) = 0 then
	    if extop(args(1),2) = 0 then extop(args(1),3)
	    elif extop(args(1),3) = 0 then 
		hold(_mult)(extop(args(1),2),rectform::pI)
	    else hold(_plus)(
		hold(_mult)(extop(args(1),2),rectform::pI),extop(args(1),3)
	    )
	    end_if
	elif extop(args(1),2) = 0 then 
	    if extop(args(1),3) = 0 then extop(args(1),1)
	    else hold(_plus)(extop(args(1),1),extop(args(1),3))
	    end_if
	elif extop(args(1),3) = 0 then
	    hold(_plus)(extop(args(1),1),hold(_mult)(extop(args(1),2),rectform::pI))
	else hold(_plus)(
            extop(args(1),1),hold(_mult)(extop(args(1),2),rectform::pI),extop(args(1),3)
        )
	end_if)
):

# #
# iszero #
# #
domattr(rectform,"iszero") := fun(
    bool(
	iszero(extop(args(1),1)) and iszero(extop(args(1),2))
	and iszero(extop(args(1),3))
    )
):

# #
# expand #
# #
domattr(rectform,"expand") := fun(
    (rectform::new(args(1));
    new(rectform,expand(extop(%,1)),expand(extop(%,2)),
	expand(extop(%,3)),extop(%,4))
    )
):

# #
# simplify #
# #
domattr(rectform,"simplify") := fun((
    rectform::new(args(1));
    rectform::new( simplify(rectform::expr(%)),extop(%,4) )
)):

# #
# normal #
# #
domattr(rectform,"normal") := fun((
    rectform::new(args(1));
    rectform::new( normal(rectform::expr(%)),extop(%,4) )
)):

domattr(rectform,"abs") := proc(x)
    local a, b;
begin
    x := rectform::new(x);
    if extop(x,3) = 0 then
        a := extop(x,1);
        b := extop(x,2);
        if domtype(a+I*b) = DOM_COMPLEX
        and domtype(a) = DOM_FLOAT or domtype(b) = DOM_FLOAT
        then
            rectform::new(abs(a+I*b),extop(x,4))
        elif sign(a+I*b) = 0 or sign(a+I*b) = 1 then
            new(rectform,a+I*b,0,0,extop(x,4))
        elif iszero(a) or iszero(b) then
            new(rectform,abs(a+I*b),0,0,extop(x,4))
        else
            new(rectform,sqrt(normal(a^2+b^2)),0,0,extop(x,4))
        end_if
    else
        rectform::new(0,0,abs(rectform::expr(x)),extop(x,4))
    end_if
end_proc:

# #
# diff #
# #
domattr(rectform,"diff") := fun(
    (rectform::new(args(1));
    if args(0) > 1 then
      rectform::new(diff(
	rectform::expr(%),args(i) $ hold(i)=2..args(0)),extop(%,4)
      )
    else %
    end_if) 
):

# #
# intmult (for the derivative of poly's) #
# #
domattr(rectform,"intmult") := fun( rectform::_mult(args(1),args(2)) ):

# #
# testtype #
# #
domattr(rectform,"testtype") := fun(
    (if args(2) = rectform or domtype(args(2)) = rectform then TRUE
    else testtype(rectform::expr(args(1)),args(2))
    end_if)
):

# #
# convert #
# #
domattr(rectform,"convert") := fun(rectform::new(args(1),extop(args(1),4))):

# #
# convert_to #
# #
domattr(rectform,"convert_to") := fun(
    (if domtype(args(1)) = rectform then
	domattr(args(2),"convert")(rectform::expr(args(1)))
    else
	FAIL
    end_if)
):

# #
# _plus #
# #
domattr(rectform,"_plus") := proc(x,y)
    local l1, l2;
begin
    if args(0) > 2 then
	return(
	  rectform::_plus(x,rectform::_plus(args(i) $ hold(i) = 2..args(0)))
	)
    end_if;

    if domtype(x) = rectform then l1 := extop(x,4)
    else x := rectform::new(x); l1 := {}
    end_if;
    if domtype(y) = rectform then l2 := extop(y,4)
    else y := rectform::new(y); l2 := {}
    end_if;
    if l1 <> l2 then
        x := rectform::new(x, l1 intersect l2);
        y := rectform::new(y, l1 intersect l2)
    end_if;

    new( rectform,extop(x,1)+extop(y,1),extop(x,2)+extop(y,2),
		  extop(x,3)+extop(y,3),l1 intersect l2
    )
end_proc:

# #
# _mult #
# #
domattr(rectform,"_mult") := proc(x,y)
    local l1, l2;
begin
    if args(0) > 2 then
        return(
          rectform::_mult(x,rectform::_mult(args(i) $ hold(i) = 2..args(0)))
        )
    end_if;

    if domtype(x) = rectform then l1 := extop(x,4)
    else x := rectform::new(x); l1 := {}
    end_if;
    if domtype(y) = rectform then l2 := extop(y,4)
    else y := rectform::new(y); l2 := {}
    end_if;
    if l1 <> l2 then
	x := rectform::new(x, l1 intersect l2);
        y := rectform::new(y, l1 intersect l2)
    end_if;

    new(rectform,
	extop(x,1)*extop(y,1)-extop(x,2)*extop(y,2), # Re(x*y) #
	extop(x,1)*extop(y,2)+extop(x,2)*extop(y,1), # Im(x*y) #
 	extop(x,1)*extop(y,3)+extop(x,3)*extop(y,1)+
	    extop(x,3)*extop(y,3)+I*(extop(x,2)*extop(y,3)+
	    extop(x,3)*extop(y,2)),
	l1 intersect l2
    )
end_proc:

# #
# length #
# #
domattr(rectform,"length") := fun(
    length(rectform::expr(rectform::new(args(1))))
):

# #
# Re #
# #
domattr(rectform,"Re") := fun(
    (rectform::new(args(1),extop(args(1),4));
    if extop(%,3) = 0 then
	new(rectform,extop(%,1),0,0,extop(%,4))
    else
	new(rectform,0,0,Re(rectform::expr(%)),extop(%,4))
    end_if)
):

# #
# Im #
# #
domattr(rectform,"Im") := fun(
    (rectform::new(args(1),extop(args(1),4));
    if extop(%,3) = 0 then
	new(rectform,extop(%,2),0,0,extop(%,4))
    else
	new(rectform,0,0,Im(rectform::expr(%)),extop(%,4))
    end_if)
):

# #
# conjugate #
# #
domattr(rectform,"conjugate") := fun(
    (rectform::new(args(1),extop(args(1),4));
    if extop(%,3) = 0 then
	new(rectform,extop(%,1),-extop(%,2),0,extop(%,4))
    else
	new(rectform,0,0,conjugate(rectform::expr(%)),extop(%,4))
    end_if)
):

# #
# _power #
# #
domattr(rectform,"_power") := proc(a,b)
    local t, ai, bi, l1, l2, i, tmp;
begin
    if domtype(a) = rectform then l1 := extop(a,4)
    else a := rectform::new(a); l1 := {}
    end_if;
    if domtype(b) = rectform then l2 := extop(b,4)
    else
	if testtype(sign(b),NUMERIC) then l2 := l1
	else l2 := {}
	end_if;
	b := rectform::new(b)
    end_if;
    if l1 <> l2 then
        a := rectform::new(a, l1 intersect l2);
        b := rectform::new(b, l1 intersect l2)
    end_if;

    l1 := l1 intersect l2;
    if extop(a,3) <> 0 or extop(b,3) <> 0 then
        return( new(rectform,0,0,
	    _power(rectform::expr(a),rectform::expr(b)),l1) 
	)
    end_if;
    ai := extop(a,2); a := extop(a,1);
    bi := extop(b,2); b := extop(b,1);
    t := b+I*bi;
    if domtype((a+I*ai)^t) = DOM_COMPLEX then
	return( rectform::new((a+I*ai)^t,l1) )
    elif domtype(t) = DOM_INT then
        if iszero(ai) then
	    return( new(rectform,a^b,0,0,l1) )
        elif iszero(a) then
	    return( new(rectform,0,ai^b*(-1)^(modp(b,4) div 2),0,l1) )
        elif b = 1 then
	    return( new(rectform,a,ai,0,l1) )
        elif b = -1 then
            t := a^2+ai^2;
            return( new(rectform,a/t,-ai/t,0,l1) )
        elif b < -1 then
	    t := rectform::new(expand((a+I*ai)^(-b)),l1);
	    a := extop(t,1); ai := extop(t,2);
	    t := a^2+ai^2;
	    return( new(rectform,a/t,-ai/t,0,l1) )
        else
            return( rectform::new(expand((a+I*ai)^b),l1) )
        end_if
    elif domtype(t) = DOM_RAT then
	tmp := [];
        l2 := l1 union rectform::globalReal;
        (r:=op(l2,1);tmp:=tmp.[[Re(r),Im(r)]];Re(r):=r;Im(r):=0)
	    $ i=1..nops(l2);
        if iszero(ai) then
            t := abs(a)^b;
            ai := (1/2*abs(b)-1/2*abs(b)*sign(a))*PI;
            t := new(rectform,t*cos(ai),t*sin(ai)*sign(b),0,l1)
        elif iszero(a) then
            t := abs(ai)^b;
            ai := 1/2*sign(ai)*abs(b)*PI;
            t := new(rectform,t*cos(ai),t*sin(ai)*sign(b),0,l1)
        else
            t := sqrt(a^2+ai^2)^b;
            ai := abs(b)*atan(a,ai);
            t := new(rectform,t*cos(ai),t*sin(ai)*sign(b),0,l1)
        end_if;
	(r:=op(l2,i);Re(r):=tmp[i][1];Im(r):=tmp[i][2])
            $ i=1..nops(l2)
    else
	tmp := [];
        l2 := l1 union rectform::globalReal;
        (r:=op(l2,1);tmp:=tmp.[[Re(r),Im(r)]];Re(r):=r;Im(r):=0)
            $ i=1..nops(l2);

        t := funcattr(exp,"rectform")(t*ln(a+I*ai),l1);

	(r:=op(l2,i);Re(r):=tmp[i][1];Im(r):=tmp[i][2])
            $ i=1..nops(l2)
    end_if;
    t
end_proc:

# #
# new #
# #
domattr(rectform,"new") := proc(x) 
    local t, a, rl;
begin
    if testargs() then
        if args(0) < 1 or args(0) > 2 then error("wrong no of args") end_if;
	if args(0) = 2 then
	    if domtype(args(2)) <> DOM_SET then
		error("expecting a set of indents")
	    end_if
	end_if
    end_if;
    if args(0) = 2 then rl := args(2) else rl := null() end_if;
    if x::rectform <> FAIL then return( x::rectform(x) ) end_if;

    t := domtype(x);
    case t
    of rectform    do if args(0) = 1 then rl := extop(x,4) end_if;
		      return( rectform::new(rectform::expr(x),rl) )
    of DOM_IDENT   do if domtype(rl) = DOM_NULL then rl := {} end_if;
		      if has(x,rectform::globalReal) or has(x,rl) then
		          return( new(rectform,x,0,0,rl) )
		      else
		          return( new(rectform,Re(x),Im(x),0,rl) )
		      end_if
    of DOM_INT     do 
    of DOM_RAT     do
    of DOM_FLOAT   do if domtype(rl) = DOM_NULL then rl := {} end_if;
		      return( new(rectform,x,0,0,rl) )
    of DOM_COMPLEX do if domtype(rl) = DOM_NULL then rl := {} end_if;
		      return( new(rectform,op(x,1),op(x,2),0,rl) )
    of DOM_LIST    do
    of DOM_SET     do
    of DOM_ARRAY   do return( map(x,rectform::new,rl) )
    of DOM_EXEC    do
    of DOM_PROC    do if domtype(rl) = DOM_NULL then rl := {} end_if;
		      return( new(rectform,0,0,x,rl) )
    of DOM_POLY    do return( mapcoeffs(x,rectform::new,rl) )
    of Puiseux     do return( x::map(x,rectform::new,rl) )
    of DOM_EXPR    do
	    t := op(x,0);
	    if domtype(level(t,2)) = DOM_FUNC_ENV then
	        a := funcattr(level(t,2),"rectform");
		if a <> FAIL then
		    return( a(op(x),rl) )
	        end_if
	    end_if;
	    if t = hold(_plus) or t = hold(_mult) then
		return( map(x,rectform::new,rl) )
	    elif t = hold(_power) then
		return( rectform::_power(
		    rectform::new(op(x,1),rl),rectform::new(op(x,2),rl)
		) )
	    end_if
    end_case;

    if domtype(rl) = DOM_NULL then rl := {} end_if;
    new(rectform,0,0,x,rl)
end_proc:

#--
   define the function attribute 'rectform' for some built-in functions 
--#

#-- abs --#
abs := funcattr( abs, "rectform", proc(x)
        local a, b, rl;
    begin
        if args(0) = 2 then
            x := rectform::new(x,args(2))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
	if extop(x,3) <> 0 then
	    new(rectform,0,0,hold(abs)(rectform::expr(x)),rl)
	else
	    a := extop(x,1);
	    b := extop(x,2);
	    if domtype(a+I*b) = DOM_COMPLEX 
            and domtype(a) = DOM_FLOAT or domtype(b) = DOM_FLOAT
            then rectform::new(abs(a+I*b))
	    elif sign(a+I*b) = 0 or sign(a+I*b) = 1 then
		new(rectform,a+I*b,0,0,rl)
	    elif iszero(a) or iszero(b) then
		new(rectform,abs(a+I*b),0,0,rl)
	    else
		new(rectform,sqrt(a^2+b^2),0,0,rl)
	    end_if
	end_if
    end_proc
):

#-- exp --#
exp := funcattr( exp, "rectform", proc(x)
	local a, b, rl;
    begin
        if args(0) = 2 then
            x := rectform::new(x,args(2))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
	if extop(x,3) <> 0 then
	    new(rectform,0,0,hold(exp)(rectform::expr(x)),rl)
	else
	    a := extop(x,1);
	    b := extop(x,2);
	    if domtype(a+I*b) = DOM_COMPLEX 
            and domtype(a) = DOM_FLOAT or domtype(b) = DOM_FLOAT
            then
                rectform::new(exp(a+I*b))
	    else
		a := exp(a);
		new(rectform,a*cos(b),a*sin(b),0,rl)
	    end_if
	end_if
    end_proc
):

#-- ln --#
ln := funcattr( ln, "rectform", proc(x)
	local a, b, t, rl;
    begin
        if args(0) = 2 then
            x := rectform::new(x,args(2))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
	if extop(x,3) <> 0 then
	    new(rectform,0,0,hold(ln)(rectform::expr(x)),rl)
	else
	    a := extop(x,1);
	    b := extop(x,2);
 	    if domtype(a+I*b) = DOM_COMPLEX 
            and domtype(a) = DOM_FLOAT or domtype(b) = DOM_FLOAT
            then	
		rectform::new(ln(a+I*b))
	    elif iszero(a+I*b) then
		error("singularity")
	    elif iszero(b) then
		new(rectform,ln(abs(a)),(1/2-1/2*sign(a))*PI,0,rl)
	    else
		t := a^2+b^2;
	        if iszero(a) then
		    if testtype( sqrt(t),NUMERIC ) then
			new(rectform,ln(sqrt(t)),1/2*sign(b)*PI,0,rl)
		    else
			new(rectform,1/2*ln(t),1/2*sign(b)*PI,0,rl)
		    end_if
	        elif testtype( sqrt(t),NUMERIC ) then
		    new(rectform,ln(sqrt(t)),atan(a,b),0,rl)
		else
		    new(rectform,1/2*ln(t),atan(a,b),0,rl)
		end_if
	    end_if
	end_if
    end_proc
):

#-- sin --#
sin := funcattr( sin, "rectform", proc(x)
	local a, b, rl;
    begin
        if args(0) = 2 then
	    x := rectform::new(x,args(2))
        else
	    x := rectform::new(x)
        end_if;
        rl := extop(x,4);
        if extop(x,3) <> 0 then
            new(rectform,0,0,hold(sin)(rectform::expr(x)),rl)
        else
            a := extop(x,1);
            b := extop(x,2);
	    if domtype(a+I*b) = DOM_COMPLEX 
            and domtype(a) = DOM_FLOAT or domtype(b) = DOM_FLOAT
            then
                rectform::new(sin(a+I*b))
            else
                new(rectform,sin(a)*cosh(b),cos(a)*sinh(b),0,rl)
            end_if
        end_if
    end_proc
):
sinh := funcattr( sinh, "rectform", proc(x)
        local a, b, rl;
    begin
        if args(0) = 2 then
            x := rectform::new(x,args(2))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
	if extop(x,3) <> 0 then
            new(rectform,0,0,hold(sinh)(rectform::expr(x)),rl)
        else
            a := extop(x,1);
            b := extop(x,2);
	    if domtype(a+I*b) = DOM_COMPLEX 
            and domtype(a) = DOM_FLOAT or domtype(b) = DOM_FLOAT
            then
                rectform::new(sinh(a+I*b))
            else
                new(rectform,sinh(a)*cos(b),cosh(a)*sin(b),0,rl)
            end_if
        end_if
    end_proc
):

#-- cos --#
cos := funcattr( cos, "rectform", proc(x)
        local a, b, rl;
    begin
        if args(0) = 2 then
            x := rectform::new(x,args(2))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
        if extop(x,3) <> 0 then
            new(rectform,0,0,hold(cos)(rectform::expr(x)),rl)
        else
            a := extop(x,1);
            b := extop(x,2);
	    if domtype(a+I*b) = DOM_COMPLEX 
            and domtype(a) = DOM_FLOAT or domtype(b) = DOM_FLOAT
            then
                rectform::new(cos(a+I*b))
            else
                new(rectform,cos(a)*cosh(b),-sin(a)*sinh(b),0,rl)
            end_if
        end_if
    end_proc
):
cosh := funcattr( cosh, "rectform", proc(x)
        local a, b, rl;
    begin
        if args(0) = 2 then
            x := rectform::new(x,args(2))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
	if extop(x,3) <> 0 then
            new(rectform,0,0,hold(cosh)(rectform::expr(x)),rl)
        else
            a := extop(x,1);
            b := extop(x,2);
	    if domtype(a+I*b) = DOM_COMPLEX 
            and domtype(a) = DOM_FLOAT or domtype(b) = DOM_FLOAT
            then
                rectform::new(cosh(a+I*b))
            else
                new(rectform,cosh(a)*cos(b),sinh(a)*sin(b),0,rl)
            end_if
        end_if
    end_proc
):

#-- tan --#
tan := funcattr( tan, "rectform", proc(x)
        local a, b, t, rl;
    begin
        if args(0) = 2 then
            x := rectform::new(x,args(2))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
	if extop(x,3) <> 0 then
            new(rectform,0,0,hold(tan)(rectform::expr(x)),rl)
        else
            a := extop(x,1);
            b := extop(x,2);
	    if domtype(a+I*b) = DOM_COMPLEX 
            and domtype(a) = DOM_FLOAT or domtype(b) = DOM_FLOAT
            then
                rectform::new(tan(a+I*b))
	    elif iszero(b) then
		new(rectform,tan(a),0,0,rl)
            else
		t := cos(2*a)+cosh(2*b);
                new(rectform,sin(2*a)/t,sinh(2*b)/t,0,rl)
            end_if
        end_if
    end_proc
):
tanh := funcattr( tanh, "rectform", proc(x)
        local a, b, t, rl;
    begin
        if args(0) = 2 then
            x := rectform::new(x,args(2))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
	if extop(x,3) <> 0 then
            new(rectform,0,0,hold(tanh)(rectform::expr(x)),rl)
        else
            a := extop(x,1);
            b := extop(x,2);
	    if domtype(a+I*b) = DOM_COMPLEX 
            and domtype(a) = DOM_FLOAT or domtype(b) = DOM_FLOAT
            then
                rectform::new(tanh(a+I*b))
            elif iszero(b) then 
		new(rectform,tanh(a),0,0,rl)
	    else
               	t := sinh(2*a)+cos(2*b);
          	new(rectform,sinh(2*a)/t,sin(2*b)/t,0,rl)
            end_if
        end_if
    end_proc
):

#-- cot --#
cot := funcattr( cot, "rectform", proc(x)
        local a, b, t, rl;
    begin
        if args(0) = 2 then
            x := rectform::new(x,args(2))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
        if extop(x,3) <> 0 then
            new(rectform,0,0,hold(cot)(rectform::expr(x)),rl)
        else
            a := extop(x,1);
            b := extop(x,2);
	    if domtype(a+I*b) = DOM_COMPLEX 
            and domtype(a) = DOM_FLOAT or domtype(b) = DOM_FLOAT
            then
                rectform::new(cot(a+I*b))
            elif iszero(b) then
                new(rectform,cot(a),0,0,rl)
            else
                t := sin(a)^2+sinh(b)^2;
                new(rectform,sin(a)*cos(a)/t,-sinh(b)*cosh(b)/t,0,rl)
            end_if
        end_if
    end_proc
):
coth := funcattr( coth, "rectform", proc(x)
        local a, b, t, rl;
    begin
        if args(0) = 2 then
            x := rectform::new(x,args(2))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
        if extop(x,3) <> 0 then
            new(rectform,0,0,hold(coth)(rectform::expr(x)),rl)
        else
            a := extop(x,1);
            b := extop(x,2);
            if domtype(a+I*b) = DOM_COMPLEX 
	    and domtype(a) = DOM_FLOAT or domtype(b) = DOM_FLOAT
	    then
                rectform::new(coth(a+I*b))
            elif iszero(b) then
                new(rectform,coth(a),0,0,rl)
            else
                t := sinh(a)^2+sin(b)^2;
                new(rectform,sinh(a)*cosh(a)/t,-sin(b)*cos(b)/t,0,rl)
            end_if
        end_if
    end_proc
):

#-- gamma --#
gamma := funcattr( gamma, "rectform", proc(x)
        local a, rl;
    begin
        if args(0) = 2 then
            x := rectform::new(x,args(2))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
        if extop(x,3) <> 0 or not iszero(extop(x,2)) then
            new(rectform,0,0,hold(coth)(rectform::expr(x)),rl)
        else
	    a := extop(x,1);
            if domtype(a) = DOM_FLOAT then
                rectform::new(gamma(a))
            else	
		new(rectform,gamma(a),0,0,rl)
            end_if
        end_if
    end_proc
):
igamma := funcattr( igamma, "rectform", proc(a,x)
        local t, rl;
    begin
        if args(0) = 3 then
            x := rectform::new(x,args(3))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
        if extop(x,3) <> 0 or not iszero(extop(x,2)) then
            new(rectform,0,0,hold(igamma)(a,args(1)),rl)
        else
            t := extop(x,1);
            if domtype(t) = DOM_FLOAT then 
                rectform::new(igamma(a,t))
            else        
                new(rectform,igamma(a,t),0,0,rl)
            end_if
        end_if
    end_proc
):

#-- erf --#

# sech is evaluated to 1/cosh #
# sec is evaluated to 1/cos   #
# csc is evaluated to 1/sin   #
# csch is evaluated to 1/sinh #

Re := funcattr( Re, "rectform", proc(x)
        local rl;
    begin
        if args(0) = 2 then
            x := rectform::new(x,args(2))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
        if extop(x,3) <> 0 then
            new(rectform,0,0,hold(Re)(rectform::expr(x)),rl)
        else
            new(rectform,extop(x,1),0,0,rl)
        end_if
    end_proc
):

Im := funcattr( Im, "rectform", proc(x)
        local rl;
    begin
        if args(0) = 2 then
            x := rectform::new(x,args(2))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
        if extop(x,3) <> 0 then
            new(rectform,0,0,hold(Im)(rectform::expr(x)),rl)
        else
            new(rectform,extop(x,2),0,0,rl)
        end_if
    end_proc
):

conjugate := funcattr( conjugate, "rectform", proc(x)
        local rl;
    begin
        if args(0) = 2 then
            x := rectform::new(x,args(2))
        else
            x := rectform::new(x)
        end_if;
        rl := extop(x,4);
        if extop(x,3) <> 0 then
            new(rectform,0,0,hold(conjugate)(rectform::expr(x)),rl)
        else
            new(rectform,extop(x,1),-extop(x,2),0,rl)
        end_if
    end_proc
):

null():

# end of file #
