# $Date: 1995/03/23 09:23:22 $ $Author: kg $ $Revision: 1.2 $ #

# kg, 15/03/95 #

#++
randpoly -- create random polynomial

randpoly([vars] [, dom] [, option=value...])

vars   - list of variables
dom    - coefficient domain (Expr, IntMod(n) or domain)
option - one of 'Coeffs', 'Terms', 'Degree'
value  - expression

randpoly creates a random polynomial with variables vars and coefficient
domain dom. The polynomial is created by adding a given number of terms,
where the degrees of the variables are in a given range.

- The option Terms=n with a positive integer n gives the number of terms
  (default is 6).

- The option Degree=n with a positive integer n gives the maximal
  degree of each variable (default is 5).

- The option Coeffs=f with a function f returning a random coefficient is
  used to create the coefficients.
  
If the option Coeffs=f is missing and dom is Expr the coefficients of the
single terms will be integers in the range -999..999; if dom is a
user-defined domain it must have a method "random" to create the
coefficients.

If the argument dom is missing Expr is used as default. If the argument
vars is missing [x] is used as default.
++#

randpoly:= proc()
    local cgen, egen, nt, dg, p, i, t, x, opts, v, d;
begin
    opts:= [ args() ];
    
    # get vars and coeffs domain #
    v:= [hold(x)];
    d:= hold(Expr);
    if nops(opts) <> 0 then
	if domtype(opts[1]) = DOM_LIST then
	    v:= opts[1];
	    opts[1]:= NIL;
	end_if
    end_if;
    if nops(opts) <> 0 then
	if type(opts[1]) <> "_equal" then
	    d:= opts[1];
	    opts[1]:= NIL;
	end_if
    end_if;

    # get options #
    nt:= 6;
    dg:= 5;
    cgen:= FAIL;
    for i in opts do
        if type(i) <> "_equal" then error("unknown option") end_if;
        case op(i,1)
        of hold(Terms) do
            nt:= op(i,2); break;
        of hold(Degree) do
            dg:= op(i,2); break;
        of hold(Coeffs) do
            cgen:= op(i,2); break;
        otherwise error("unknown option");
        end_case;
    end_for;
    
    # test args #
    if testargs() then
        case domtype(d)
        of DOM_IDENT do
            if d <> hold(Expr) then error("illegal coefficient ring") end_if;
            break;
        of DOM_EXPR do
            if op(d,0) <> hold(IntMod) then error("illegal coefficient ring") end_if;
            if nops(d) <> 1 then error("illegal coefficient ring") end_if;
            if not testtype(op(d,1), Type::PosInt) then
                error("illegal modulus")
            end_if;
            break;
        of DOM_DOMAIN do
            break;
        otherwise
            error("illegal coefficient ring");
        end_case;
        if not testtype(nt, Type::PosInt) then 
            error("illegal number of terms")
        end_if;
        if not testtype(dg, Type::PosInt) then 
            error("illegal degree")
        end_if;
    end_if;
    
    # random generator for coeffs #
    case domtype(d)
    of DOM_IDENT do
	if cgen = FAIL then cgen:= random(-999..999) end_if;
	p:= poly(0, v, d);
	break;
    of DOM_EXPR do
	if cgen = FAIL then cgen:= random(op(d,1)) end_if;
	p:= poly(0, v, d);
	break;
    otherwise
	if cgen = FAIL then 
	    cgen:= d::random;
	    if cgen = FAIL then 
		error("domain entry \"random\" missing") 
	    end_if;
	end_if;
	p:= poly(d::zero, v, d);
    end_case;
    
    # random generator for exponents #
    egen:= random(dg+1);
    
    # create terms #
    for i from 1 to nt do
        # create term #
        t:= 1;
        for x in v do
            t:= t * x^egen()
        end_for;
        p:= p + multcoeffs(poly(t, v, d), cgen());
    end_for;
    
    p
end_proc:

# end of file #
