# #
# $Date: 1995/07/19 12:07:07 $ $Author: frankp $ $Revision: 1.31.2.4 $ #
# #
# frankp, 11.01.1995 #

#++
   limit.mu

	limit  -- computing the limit of expressions

	limit(expr,x[=a] [,dir])

	expr -- expression in x
	x    -- limit variable x (an identifier)
	a    -- (optional) limit point (possible infinity or -infinity)
		default is 0
	dir  -- (optional) direction: one of 'Left' or 'Right'

	The limit function attempts to compute the limiting value of expr
	as x goes to a.
   
	If dir is not specified, the limit is the real bidirectional limit,
	except in the case where the limit point is infinity or -infinity.
	In this case the limit is from the left to infinity and from the 
	right to -infinity, respectivly.
   
	limit only consider real valued functions, and any assigned 
	variables in the expression are assumed to be real.

	limit returns 'undefined' iff the limit is not defined, or 
	FAIL if ORDER is not big enough for series.
	If the limit could not be computed then limit returns
	unevaluated, i.e. the result is of type "limit".

	The Algorithm based on the paper of Gaston H. Gonnet and
	Dominik Gruntz: Limit Computation in Computer Algebra,
	Nov. 1992.
++#

limit := proc(ex,point)
    name limit;
    local __LIMITx, s, t, lp, nex, _x;
begin
    if testargs() then
        if args(0) < 2 or args(0) > 3 then
            error("wrong no of args")
        end_if;
        if type(point) = "_equal" then
            if domtype(op(point,1)) <> DOM_IDENT then
        	error("invalid limit variable")
            end_if
	elif domtype(point) <> DOM_IDENT then
	    error("invalid limit variable")
	end_if;
	if args(0) = 3 then
	    if not contains({Right,Left},args(3)) then
		error("unknown direction")
	    end_if;
	    if contains({-infinity,infinity},op(point,2)) then
		error("invalid direction")
	    end_if
	end_if
    end_if;

    if type(point) = "_equal" then
        _x := op(point,1);
        lp := op(point,2)
    else
	_x := point;
	lp := 0
    end_if;

    # reset limit computation #
    domattr(stdlib::limit,"result") := FAIL;
    # force series loading (necessary for redefining Puiseux::iszero) #
    loadlib("Series"): Puiseux::create:
    domattr(stdlib::limit,"Piszero") := Puiseux::iszero;
    Puiseux::iszero := fun((
	normal(args(1));
        bool( iszero(simplify(subs(%, ____OMEGA=w),exp)) or
	iszero(simplify(subs(%, ____OMEGA=w))) )
    ));

    if args(0) = 2 then
        if lp <> -infinity and lp <> infinity then
            nex := subs(ex,_x=lp+1/__LIMITx);
            nex := (stdlib::limit)::preProcess(
		(stdlib::limit)::simp2explog(nex,__LIMITx),__LIMITx
	    );
            s := (stdlib::limit)::limit(nex,__LIMITx);
            if s <> FAIL then
                nex := subs(ex,_x=lp-1/__LIMITx);
                nex := (stdlib::limit)::preProcess(
		    (stdlib::limit)::simp2explog(nex,__LIMITx),__LIMITx
		);
                t := (stdlib::limit)::limit(nex,__LIMITx);
                if t <> FAIL then
		    Puiseux::iszero := (stdlib::limit)::Piszero;
                    if t <> s then return( hold(undefined) )
                    else return( s )
                    end_if
                end_if
	    end_if;
	    (stdlib::limit)::reset();
	    s := (stdlib::limit)::result;
	    if domtype(s) = DOM_STRING then return( text2expr(s) )
            else return( hold(limit)(args()) )
	    end_if
        end_if;
        if lp = -infinity then
            ex := subs(ex,_x=-__LIMITx)
	else
	    ex := subs(ex,_x=__LIMITx)
        end_if;
        ex := (stdlib::limit)::preProcess(
	    (stdlib::limit)::simp2explog(ex,__LIMITx),__LIMITx
	);
        s := (stdlib::limit)::limit(ex,__LIMITx);
        if s = FAIL then
	    (stdlib::limit)::reset();
	    s := (stdlib::limit)::result;
	    if domtype(s) = DOM_STRING then return( text2expr(s) )
	    else return( hold(limit)(args()) )
	    end_if
        end_if
    else
        if args(3) = hold(Right) then
	    ex := subs(ex,_x=lp+1/__LIMITx)
        else
	    ex := subs(ex,_x=lp-1/__LIMITx)
        end_if;
        ex := (stdlib::limit)::preProcess(
	    (stdlib::limit)::simp2explog(ex,__LIMITx),__LIMITx
	);
        s := (stdlib::limit)::limit(ex,__LIMITx);
        if s = FAIL then
	    (stdlib::limit)::reset();
	    s := (stdlib::limit)::result;
            if domtype(s) = DOM_STRING then return( text2expr(s) )
            else return( hold(limit)(args()) )
            end_if
        end_if
    end_if;

    Puiseux::iszero := (stdlib::limit)::Piszero;
    s
end_proc:

limit := func_env(limit,NIL,table(
    "type"="limit",
    "info"="stdlib::limit -- calculate limits [try ?limit for options]",
    "print"="limit"
)):

proc() local LIMIT, DpreProcess, eints, erfs;
begin

LIMIT := domain():
LIMIT::name := "stdlib::limit":

LIMIT::result := FAIL:  
# This string (if defined) will be returned instead of an
  unevaluated result if (stdlib::limit)::limit returns FAIL.
  A possible message is "undefined" or "FAIL".
  The string have to be converted to a mupad expression.
#

LIMIT::reset := fun((
    # clear remember tables #
    domattr(stdlib::limit,"limit") := subsop(
        domattr(stdlib::limit,"limit"),5=NIL
    );
    domattr(stdlib::limit,"lterm") := subsop(
        domattr(stdlib::limit,"lterm"),5=NIL
    );
    domattr(stdlib::limit,"getMRV") := subsop(
        domattr(stdlib::limit,"getMRV"),5=NIL
    );
    domattr(stdlib::limit,"rewrite") := subsop(
        domattr(stdlib::limit,"rewrite"),5=NIL
    );
    domattr(stdlib::limit,"maxMRV") := subsop(
        domattr(stdlib::limit,"maxMRV"),5=NIL
    );
    domattr(stdlib::limit,"sign") := subsop(
        domattr(stdlib::limit,"sign"),5=NIL
    );
    Puiseux::iszero := (stdlib::limit)::Piszero
)):

LIMIT::simp2explog := proc(ex,__LIMITx)
    name (stdlib::limit)::simp2explog;
    local t;
    option remember;
begin
    if domtype(ex) = DOM_EXPR then
        case type(ex)
        of "_power" do
            if testtype(op(ex,2),NUMERIC) then
        	return( (stdlib::limit)::simp2explog(
		    op(ex,1),__LIMITx)^op(ex,2)
		)
            else
                t := (stdlib::limit)::simp2explog( op(ex,1),__LIMITx );
                return( exp((stdlib::limit)::simp2explog(op(ex,2),__LIMITx)*ln(t)) )
            end_if;
        of "_mult"   do
        of "_plus"   do
            return( eval(map(ex,(stdlib::limit)::simp2explog,__LIMITx)) )
        end_case;
	t := op(ex,0);
	if domtype(level(t,2)) = DOM_FUNC_ENV then
            if nops(ex) = 1 then
		return( level(t,2)(
		    (stdlib::limit)::simp2explog(op(ex,1),__LIMITx)
		))
            else
		return( level(t,2)(
		    map(op(ex),(stdlib::limit)::simp2explog,__LIMITx)
		))
            end_if
	end_if
    end_if;

    ex
end_proc:

#--
        (stdlib::limit)::preProcess(ex,x)

        ex : expression
        x  : limit variable

--#

LIMIT::preProcess := proc(ex,__LIMITx)
    name (stdlib::limit)::preProcess;
    option remember;
    local t;
begin
    if domtype(ex) = DOM_EXPR then
        t := op(ex,0);
        if domtype(level(t,2)) = DOM_FUNC_ENV then
	    if domattr((stdlib::limit)::DpreProcess,type(ex)) <> FAIL then
		userinfo(2,"call (stdlib::limit::DpreProcess)::",t);
		return( domattr((stdlib::limit)::DpreProcess,type(ex))(op(ex),__LIMITx) )
	    end_if
	end_if;
	userinfo(2,"map onto ",ex);
	ex := eval(map(ex,(stdlib::limit)::preProcess,__LIMITx))
    else
	userinfo(2,"do nothing on ",ex);
    end_if;

    ex 
end_proc:

#--
	(stdlib::limit)::limit(ex,x)

	ex: expression
	x : limit variable

	Calls (stdlib::limit)::lterm and analyzes the leading
	coefficient of the asymptotic expansion (in w), i.e.
	returns the limit of ex for x -> infinity, if possible.
	Otherwise FAIL is returned.
--#

LIMIT::limit := proc(ex,__LIMITx)
    name (stdlib::limit)::limit;
    option remember;
    local lc, s;
begin
    ex := simplify(ex);
    ex := simplify(ex,exp);
    lc := (stdlib::limit)::lterm( ex,__LIMITx );
    if lc = FAIL then return( FAIL ) end_if;

# (d) Analyze the leading coefficient #

    s := sign(lc[3]);
    if s = 0 then lc[1]
    elif s = 1 then 0
    elif s = -1 then
        s := (stdlib::limit)::sign(lc[1],__LIMITx);
        if type(s) = "sign" then
            (stdlib::limit)::reset();
            error("can't determine ".expr2text(s))
        else s * infinity
        end_if
    else
        (stdlib::limit)::reset():
        error("can't decide zero equiv. of ".expr2text(lc[3]))
    end_if;
end_proc:

#--
	(stdlib::limit)::upmove(ex,x)
	(stdlib::limit)::downmove(ex,x)

	Subroutines for (stdlib::limit)::lterm.
--#

LIMIT::upmove := fun(
    eval(subs(args(1),[ln(args(2))=args(2),args(2)=exp(args(2))]))
):
LIMIT::downmove := fun(
    eval(subs(args(1),[exp(args(2))=args(2),args(2)=ln(args(2))]))
):

#--
        (stdlib::limit)::lterm(ex,x[,S])

        ex: expression
        x : limit variable
	S : (optional) mrv set

        Compute the list [c0,w,e0], whereby c0*w^e0 is the leading
	term of an asymptotic expansion of ex for x -> infinity.
	Returns FAIL if this can not be computed.
	If S is given, the mrv set will not be computed once more.
--#

LIMIT::lterm := proc(ex,__LIMITx)
    name (stdlib::limit)::lterm;
    local S, ____OMEGA, s, w, lc, le, t;
    option remember;
begin
# (a) Find the most rapidly varying subexpressions #

    if args(0) = 2 then
	userinfo(1,"(1)  Find mrv set of ",ex);
        S := (stdlib::limit)::getMRV(ex,__LIMITx);
        if S = FAIL then
	    userinfo(1,"** getMRV fails");
	    return( FAIL )
        elif nops(S) = 0 then
	    return( [ex,1,0] )
        end_if
    else S := args(3)
    end_if;

    if S intersect {__LIMITx} = {__LIMITx} then # Upward Movement #
	w := (stdlib::limit)::lterm(
	        (stdlib::limit)::upmove(ex,__LIMITx),__LIMITx,
		(stdlib::limit)::upmove(S,__LIMITx)
	);
	if w = FAIL then return( FAIL )
	else return( (stdlib::limit)::downmove(w,__LIMITx) )
	end_if
    end_if;

# (b) Choose one expression in this set and call it w. Rewrite the    #
#     other expressions in that set as g(x)=w^p h(x) and consider all #
#     expressions  independent of w as constants.                     #

    userinfo(1,"(2)  Rewrite it");

    s := (stdlib::limit)::rewrite( S,__LIMITx );
    if s = FAIL then
	userinfo(1,"** rewrite process fails");
	return( FAIL )
    end_if;
    userinfo(2,"result is ",s);
    w := op(s,1); S := op(s,2);
    ex := combine(eval(subs(ex,op(S,i)=op(s,[3,i])	
	$ hold(i)=1..nops(S),w=____OMEGA)));

# (c) Compute the leading term of the puiseux series in w around w = 0+ #

    userinfo(1,"(3) Compute series in ____OMEGA of ",ex);

    if not has(ex,____OMEGA) then 
	return( (stdlib::limit)::lterm(ex,__LIMITx) ) 
    end_if;

    s := series(ex,____OMEGA);
    if s = FAIL then
    # does not have a series expansion #
        userinfo(1,"** series computation fails");
        return( FAIL )
    end_if;

    userinfo(2,"Puiseux series is ",s);

    if type(s) = gseries then
	userinfo(3,"     type is gseries...");
	t := gseries::lterm(s);
	if type(t) = "ln" then
	    userinfo(3,"     expr has a log-singularity");
	    le := 0;
	    lc := gseries::lmonomial(s)
	elif type(t) = "_power" then
	    if op(t,1) = ____OMEGA then
		userinfo(3,"     series has real exponents");
		le := sign(op(t,2));
		if not contains({1,0,-1},le) then
		    error("can't decide zero equiv. of ".expr2text(le))
		end_if;
		lc := gseries::lcoeff(s)
	    end_if
	end_if
    else
	le := s::ldegree(s);
        lc := s::lcoeff(s)
    end_if;

    userinfo(3,"  lcoeff: ",lc, "; ldegree: ",le);
    if lc = FAIL then
	userinfo(1,"** ORDER seems to be not big enough for series");
	domattr(stdlib::limit,"result") := "FAIL";
	return( FAIL )
    end_if;

    lc := eval(subs(lc,[ln(____OMEGA)=simplify(ln(w)),____OMEGA=w]));
    if le = 0 then (stdlib::limit)::lterm(lc,__LIMITx)
    else [lc,w,le]
    end_if
end_proc:

#--
        (stdlib::limit)::getMRV(f,x)

        f: expression
        x: limit variable

        Find the most rapidly varying subexpressions at x=infinity
        (section 3.1).
--#

LIMIT::getMRV := proc(f,__LIMITx)
    name (stdlib::limit)::getMRV;
    local s, t, g;
    option remember;
begin
    if not has(f,__LIMITx) then
        return( {} )
    elif f = __LIMITx then
        return( {__LIMITx} )
    else
        case type(f)
	of "_mult" do
        of "_plus" do
	    t := [];
	    for g in f do
		s := (stdlib::limit)::getMRV(g,__LIMITx);
		if s = FAIL then return( FAIL )
		else t := t.[s]
		end_if
	    end_for;
            return( (stdlib::limit)::maxMRV({op(t)},__LIMITx) )
        of "_power" do
	    if not has(op(f,2),__LIMITx) then
                return( (stdlib::limit)::getMRV(op(f,1),__LIMITx) )
 	    else		
		return( limit::getMRV(exp(op(f,2)*ln(op(f,1))),__LIMITx) )
            end_if
 	of "ln" do
	    if f = ln(__LIMITx) then return( {__LIMITx} ) 
	    else return( (stdlib::limit)::getMRV(op(f,1),__LIMITx) )
	    end_if
	of "exp" do
            if f = exp(__LIMITx) then return( {f} ) end_if;
            t := op(f,1);
            s := (stdlib::limit)::limit(t,__LIMITx);
	    if s = FAIL then return( FAIL )
            elif s = infinity or s = -infinity then
                s := (stdlib::limit)::getMRV(t,__LIMITx);
	 	if s = FAIL then return( FAIL )
                elif contains( s,FAIL ) then return( FAIL )
                else
		    return((stdlib::limit)::maxMRV({{exp(t)},s},__LIMITx))
                end_if
            else
                return( (stdlib::limit)::getMRV(t,__LIMITx) )
            end_if
	of "sin"  do
	of "cos"  do
	    t := (stdlib::limit)::limit(op(f,1),__LIMITx);
	    if t = FAIL then return( FAIL )
            elif t <> infinity and t <> -infinity then
                return( (stdlib::limit)::getMRV(op(f,1),__LIMITx) )
            else # oscillating function #
                return( FAIL )
            end_if
	of "tan"  do
	of "asin" do
	of "acos" do
	of "cot"  do
	of "coth" do
	of "fact" do
	    t := (stdlib::limit)::limit(op(f,1),__LIMITx);
	    if t = FAIL then return( FAIL )
	    elif t <> infinity and t <> -infinity then
		return( (stdlib::limit)::getMRV(op(f,1),__LIMITx) )
	    else # essential singularity #
		return( FAIL )
	    end_if
	of "asinh" do
        of "atan" do
	of "psi" do
            return( (stdlib::limit)::getMRV(op(f,1),__LIMITx) )
	of "acosh" do
	    t := (stdlib::limit)::limit(op(f,1),__LIMITx);
	    if t = FAIL then return( FAIL )
	    else
		t := sign(t);
	        if t = 1 or t = 0 then
                    return( (stdlib::limit)::getMRV(op(f,1),__LIMITx) )
		else
		    return( FAIL )
		end_if
	    end_if;
	of "binomial" do
		return( (stdlib::limit)::getMRV(expand(f),__LIMITx) )
	otherwise
	    if domtype(f) = DOM_EXPR then
		# was f considered by the preprocessing step ? #
		if domattr((stdlib::limit)::DpreProcess,type(f)) <> FAIL then
		    return( (stdlib::limit)::getMRV(op(f,1),__LIMITx) )
		end_if
	    end_if
        end_case
    end_if;

    FAIL
end_proc:

#--
        (stdlib::limit)::rewrite(S,x)

        S: mrv set of f in respect to x
        x: limit variable

        Returns [w,S,l] where w represents ____OMEGA and l is 
        the list of elements of S rewritten in ____OMEGA.
        The i.th element of l is the i.th element of S rewritten
        in ____OMEGA (i=1..|S|).

	Returns FAIL if the rewrite process can not be performed.
--#

LIMIT::rewrite := proc(S,__LIMITx)
    name (stdlib::limit)::rewrite;
    local _S_, t, s, i, w, INVw, f, c, l, ____OMEGA;
    option remember;
begin
    userinfo(2,"Rewrite ",S);
#
  extract the set _S_ of expressions which contains subexpressions
  of the same comparability class
#
    _S_ := {};
    for i from 1 to nops(S) do
        s := op(S,i);
        w := select( S minus {s}, has, s );
        if w <> {} then
            _S_ := _S_ union w
        end_if
    end_for;
    S := S minus _S_;  # the complement of _S_ #

    w := op(sort( [op(S)],fun(
           (if length(args(1)) < length(args(2)) then TRUE
           else FALSE end_if)
         )),1);
    if type(w) <> "exp" then
        error("illegal element in mrv set: ".expr2text(w))
    end_if;

    S := (sort([op(_S_)],fun(
          (if has(args(1),args(2)) then TRUE else FALSE end_if)
         ))).[op(S minus {w})];

    s := (stdlib::limit)::sign(op(w),__LIMITx);
    if s = 1 then t := -op(w); w := 1/w
    elif s = -1 then t := op(w);
    else
        error("w = ".expr2text(w)." has illegal sign: ".expr2text(s))
    end_if;
    INVw := exp(-t);

    userinfo(2,"Selected ____OMEGA of ",S, ": ",w);

    l := [];
    for f in S do
        s := simplify(ln(f));
        c := (stdlib::limit)::limit( s/t,__LIMITx );
        if c = FAIL then
	    userinfo(1,"** unexpected FAIL of limit computation");
	    return( FAIL )
        end_if;
        l := append( l,exp(s-c*t)*____OMEGA^c );
    end_for;
    userinfo(2,"Leads to ",l);

    [w,S.[INVw],l.[1/____OMEGA]]  # append 1/w to S #
end_proc:

#-
        (stdlib::limit)::maxMRV(s,x)

        s: set containing sets of rapidly varying terms in x
        x: limit variable

        Find the more rapidly varying set (section 3.2).
--#

LIMIT::maxMRV := proc(l,__LIMITx)
    name (stdlib::limit)::maxMRV;
    local f, g, s;
    option remember;
begin
    l := l minus {{}};
    if nops(l) < 2 then return( op(l) )
    elif nops(l) > 2 then
        f := op(l,1);
        return( (stdlib::limit)::maxMRV({f,(stdlib::limit)::maxMRV(l minus {f},__LIMITx)},__LIMITx) )
    else
        f := op(op(l,1),1); g := op(op(l,2),1); # choose representatives #
        s := (stdlib::limit)::limit( ln(f)/ln(g),__LIMITx );
        if s = FAIL then return( FAIL )
        elif s = infinity or s = -infinity then return( op(l,1) )
        elif iszero(s) then return( op(l,2) )
        else return( op(l,1) union op(l,2) )
        end_if
    end_if
end_proc:

#--
        (stdlib::limit)::sign(ex,x)

        ex: expression in x
        x : ident

        Compute the sign of ex, assuming that x goes to infinity.
	Returns sign(ex) if this can not be determined.
--#

LIMIT::sign := proc(ex,__LIMITx)
    name (stdlib::limit)::sign;
    local s, t;
    option remember;
begin
    userinfo(2,"compute the sign of ",ex);

    if ex = __LIMITx then return( 1 ) end_if;
    if not has(ex,__LIMITx) then
	s := sign(ex);
        if s = 0 or s = 1 then return( 1 )
        elif s = -1 then return( -1 )
	else return(hold(sign)(ex))
	end_if
    end_if;

    case type(ex)
    of "_mult"  do
	return( map(ex,(stdlib::limit)::sign,__LIMITx) )
    of "_power" do
	if domtype(op(ex,2)) = DOM_INT then
            if modp(op(ex,2),2) = 0 then return( 1 )
            else
		return((stdlib::limit)::sign(op(ex,1),__LIMITx))
            end_if
        elif (stdlib::limit)::sign(op(ex,1),__LIMITx) = 1 then
	    return( 1 )
	else
	    return( hold(sign)(ex) )
        end_if
    of "_plus"do
	t := (stdlib::limit)::limit(ex,__LIMITx);
	if t = FAIL then return( hold(sign)(ex) ) end_if;
	t := sign(t,__LIMITx);
	if not has(t,hold(sign)) then return( t )
	else return( hold(sign)(ex) )
	end_if
    of "ln" do
	t := (stdlib::limit)::limit(op(ex,1),__LIMITx);
	if t = FAIL or t = -infinity then return( hold(sign)(ex) )
	elif t = infinity then return( 1 )
	elif iszero(t) then return( -1 )
	end_if;
	t := sign(ln(t));
	if has(t,hold(sign)) then return( hold(sign)(ex) )
	else return( t )
	end_if
    of "exp" do
	return( 1 )
    of "erf" do
    of "cot" do
	t := (stdlib::limit)::limit(op(ex,1),__LIMITx);
	if t = FAIL then return( hold(sign)(ex) ) end_if;
	t := sign(t);
	if t = 1 or t = 0 then return( 1 )
	elif t = -1 then return( -1 )
	else return( hold(sign)(ex) )
	end_if
    otherwise
	if domtype(ex) = DOM_EXPR then
	    t := op(ex,0);
	    if domtype(level(t,2)) = DOM_FUNC_ENV then
	        t := funcattr(level(t,2),"(stdlib::limit)::sign");
	        if t <> FAIL then
	            t := t(op(ex),__LIMITx);
	            if t <> FAIL then return( t ) end_if
	        end_if
	    end_if
        end_if
    end_case;

    hold(sign)(ex)
end_proc:

#--
        (stdlib::limit)::subsFUNC(ex,f1,f2)

        ex: expression
        f1: string (type of function)
	f2: function

	Substitutes all function calls f(x) of type f1 to 
	function calls f2(x).
--#

LIMIT::subsFUNC := proc(ex,f1,f2) 
    name (stdlib::limit)::subsFUNC;
    option remember;
begin
    case domtype(ex) 
    of Puiseux do
	userinfo(2,"map onto the coefficients of ",ex);
	return( eval(map(ex,(stdlib::limit)::subsFUNC,f1,f2)) )
    of DOM_EXPR do
	if type(ex) = f1 then ex := f2(op(ex)) end_if;
        userinfo(2,"map onto ",ex);
	return( eval(map(ex,(stdlib::limit)::subsFUNC,f1,f2)) )
    otherwise
	userinfo(2,"do nothing on ",ex);
	return( ex )
    end_case
end_proc:
	
# --
        Define the pre-processing algorithms for the
	smooth functions:

        erf:            erf  = 1-exp(-x^2)*limit::erfs(x)
        gamma

	The algorithms are defined in the domain

	    (stdlib::limit)::preProcess

	The methods are called by the type of the regarded function.
	Two pre-pocessing procedures must be defined, one named by the 
	type of the original function, the second by the type of the 
	corresponding smooth function (for example "erf" and 
	"limit::erfs").
--#

LIMIT::DpreProcess := domain():
DpreProcess := LIMIT::DpreProcess:

DpreProcess::erf := proc(ex,x)
    local s;
begin
    s := limit(ex,x=infinity);
    if s = FAIL then return( FAIL )
    elif s = infinity then
        1-exp(-ex^2)*domattr(stdlib::limit,"erfs")(ex)
    elif s = -infinity then
	-1+exp(-ex^2)*domattr(stdlib::limit,"erfs")(-ex)
    else hold(erf)(ex)
    end_if
end_proc:
domattr(DpreProcess,"limit::erfs") := proc(ex,x)
    local s;
begin
    s := limit(ex,x=infinity);
    if s = FAIL then return( FAIL )
    elif s <> infinity and s <> -infinity then
	s := sign(s);
	if s = 1 then (1-erf(ex))*exp(ex^2)
	elif s = -1 then (1+erf(-ex))*exp(ex^2)
	else FAIL
	end_if
    else subsop((stdlib::limit,"erfs")(X),1=ex)
    end_if
end_proc:

stdlib::limit := LIMIT:

# --
        Define the smooth functions:
        erfs  (satisfies  erf(x) = 1-exp(-x^2)*limit::erfs(x) )
        gammas
--#

domattr(stdlib::limit,"erfs") :=
    proc() begin subsop(x(x),0=level(procname,2),1=args()) end_proc:

domattr(stdlib::limit,"erfs") := funcattr(domattr(stdlib::limit,"erfs"),
"series", proc(f,x,n)
        local t, s, k;
    begin
        t := Series::series(f,x,n);
	if not testtype(t,Type::Series(Puiseux)) then
	    error("unable to compute series expansion")
	end_if;
        k := t::ldegree(t);
        if k = 0 then
	    t := t::expr(t);
            s := Series::series((1-erf(t))*exp(t^2),x,n);
	    (stdlib::limit)::subsFUNC(s,"erf",
		fun(1-exp(-args(1)^2)*domattr(stdlib::limit,"erfs")(args(1))
	    ))
        elif k < 0 then # f goes to +/- infinity #
            s := Puiseux::create(-1,1,n,
	[(1/sqrt(PI)*(-1/4)^s*fact(2*s)/fact(s),0) $ s=0..iquo(n-1,2)]
	    );
            Puiseux::_fconcat(Series::series(1/f,x,n),s);
        else
            Series::unknown(domattr(stdlib::limit,"erfs")(f),x,n)
        end_if
   end_proc
):

domattr(stdlib::limit,"erfs") := funcattr(domattr(stdlib::limit,"erfs"),
"type","limit::erfs"):
domattr(stdlib::limit,"erfs") := funcattr(domattr(stdlib::limit,"erfs"),
"print","limit::erfs");

domattr(stdlib::limit,"erfs") := funcattr(domattr(stdlib::limit,"erfs"),
"diff", proc(x)
    begin
        if args(0) = 1 then x
        elif args(0) = 2 then
            x := op(x);
            if has(x,args(2)) then
		(-2/sqrt(PI)+2*domattr(stdlib::limit,"erfs")(x)*x)*diff(x,args(2))
	    else 0
	    end_if
        else
            diff( diff(x,args(2)),args(3..args(0)) )
        end_if
    end_proc
):

domattr(stdlib::limit,"erfs") := funcattr(domattr(stdlib::limit,"erfs"),
"(stdlib::limit)::sign", proc(ex,x)
	local s;
    begin
	s := (stdlib::limit)::limit(ex,x);
	if s = FAIL then FAIL
	else
	    s := sign(s);
	    if s = 0 or s = 1 then 1 else FAIL end_if
	end_if
    end_proc
):

end_proc():

# end of file #
