#--
rational - convert a floating point numbers to an exact 
           or approximate rational number

Calling sequence:
rational(expr);
rational(expr , Approx [, digits])


Parameters:
expr   -- any expression
digits -- optional integer 


Summary:
The function rational converts all floating-point numbers contained
in the expression expr to exact rational numbers. Lists, sets, tables
and arrays are also converted.
An exact conversion of a floating-point number f will be performed 
by default. The meaning of an exact conversion of f is m*10^e, where
m is the mantissa (0 < m < 1) and e the exponent of the normalized 
representation of f. 
The flag Approx indicates, that an approximation of f should be computed.
The accuracy of the conversion will depend on the value of the global
variable DIGITS, or the value of digits if specified as an integer.

Examples:
rational(0.3333333333);		-> 3333333333/10000000000
rational(0.3333333333, Approx);	-> 1/3
rational(float(PI), Approx, 1);	-> 3	
rational(float(PI), Approx, 5);	-> 333/106
rational(0.2*a+b^(0.2*I))	-> a/5 + b^(1/5*I)
--#

sharelib::rational := proc ( f )
   local flag, digits;
begin
      digits := DIGITS; 
      flag := Exact;
      case args(0) 
        of 1 do break
        of 3 do if not testtype(args(3), DOM_INT) then
                   error("digits must be a positive integer")
                elif args(3) < 0 then
                   error("digits must be a positive integer")
                else
                   digits := args(3)
                end_if;
        of 2 do if args(2) = Approx then 
                   flag := Approx
                else
                   error("unknown flag")
                end_if;
                break
        otherwise error("wrong number of arguments")
      end_case;
      sharelib::rec_rat(f, digits, flag)
end_proc:


sharelib::rec_rat := proc(f, digits, flag)
begin
   case type(f)
     of DOM_RAT do
     of DOM_POLY do
     of DOM_PROC do
        return(f)
     of DOM_FLOAT do 
        return(sharelib::float2rat( f, digits, flag))
     of DOM_COMPLEX do
        # Because map don't works on complex numbers #
        return(sharelib::rec_rat( op(f,1), digits, flag ) +
               sharelib::rec_rat( op(f,2), digits, flag ) * I )
     otherwise 
        if nops(f) > 1 or contains({DOM_LIST, DOM_SET}, type(f)) then
           map(f, sharelib::rec_rat, digits, flag)
        else
           f
        end_if
   end_case
end_proc:

   
sharelib::float2rat := proc( f, digits, flag)
   local c,p0,p1,p2,q0,q1,q2,nu,de,r,inu,ide;
begin
   ops := sharelib::get_op(f, digits);
   nu := ops[1]; de := ops[2];
   if de >= 0 or flag = Exact then 
      return(nu*10^de) 
   end_if;
   de := 10^(-de);
   inu := nu;  ide := de;
   p0:=0;  p1:=1;  q0:=1;  q1:=0;
   while TRUE do
       c := nu div de; r := nu mod de;
       p2 := c*p1+p0;  q2 := c*q1+q0;
       # termination criteria: it is exact or the approximation #
       # is | p2/q2 - f | < 10 ^ (de-1) * f			#
       if r=0 or 10^(digits-1) * abs( inu*q2-ide*p2 ) < ide*p2 then
          return(p2/q2) 
       end_if;
       p0:=p1; p1:=p2; q0:=q1; q1:=q2; nu := de; de := r;
   end_while;
end_proc:


# x - floating-point number 				  #
# get_op(x) yields a list of two integers m and e,	  #
# where x = m * 10^e					  #
# Every floating-point number is normalized, e.g.	  #
# x is of the form <mantissa>e<exp> or <mantissa>	  #
# where <exp> is an integer and -10.0 < <mantissa> < 10.0 #

sharelib::get_op := proc(x)
  local i, expo, len, mantissa, s;
begin
  # Convert x to a string. expr2text depends on DIGITS! #
  s := expr2text(x); len := strlen(s);
  expo := 0;
  for i from 0 to len-1 do
      if substring(s, i, 1) = "e" then
         # Scientific notation #
         expo := text2expr(substring(s, i+1, len-1-i));
         len := i;
         s := substring(s, 0, len);
         break;
      end_if
  end_for;
  # s contains only the mantissa      #
  # Delete trailing zeros in mantissa #
  i := len-1;
  while substring(s, i, 1) = "0" do
      len := len-1; i := i-1;
  end_while;
  mantissa := text2expr(substring(s, 0, 1).substring(s, 2, len-2));
  expo := expo-len+2;
  [ mantissa, expo ]
end_proc:
