
# kg, 08/07/94 #

#
Lsys -- the domain of Lindenmeyer systems
#

Lsys:= domain():
Lsys::name:= "Lsys":

#
new -- create new L-system

new(deg,start,rule...)

deg   - turtle degree (real number)
start - starting rule (non-empty string)
rule  - rule of the form '<lhs>=<rhs>'

The <lhs> must be a string of length 1. The <rhs> may be a string,
a turtle primitive or a color value. A turtle primitive is one of the
identifiers 'line', 'move', 'left', 'right', 'push' or 'pop'. A color value
is a list with 3 floats in the range 0..1 which define the
red-, green- and blue value of the color.
#

Lsys::new:= proc(deg,start)
    local i, j, p, sp, pr, rule, prim, r, lhs, rhs, prims, rules;
    option hold;
begin
    # parse degree #
    if args(0) < 3 then error("wrong no of args") end_if;
    if not testtype(deg, Type::RealNum) then
	error("illegal degree")
    end_if;

    # parse start rule #
    if domtype(start) <> DOM_STRING then
	error("illegal start rule")
    end_if;
    if strlen(start) = 0 then
	error("illegal start rule")
    end_if;
    p:= [];
    for j from 0 to strlen(start)-1 do
	p:= append(p, hold(r)(substring(start,j,1)))
    end_for;
    sp:= subsop(proc(T) local r; begin
	    gen:=gen-1; if gen = 0 then r:= prim else r:= rule end_if;
	    T:= f(T); gen:=gen+1; T
	end_proc, [4,3,2,0] = hold(_fconcat)(op(p)));

    # parse rules #
    prim:= proc() begin id end_proc;
    rule:= proc() begin prim(args()) end_proc;
    pr:= {};
    prims:= {};
    rules:= {};

    for i from 3 to args(0) do
	r:= args(i);
	if type(r) <> "_equal" then
	    error("illegal rule")
	end_if;
	if not testtype(op(r), Type::Product(DOM_STRING,
		    Type::Union(DOM_STRING, DOM_IDENT,
			Type::ListProduct(DOM_INT, DOM_INT, DOM_INT)))) then
	    error("illegal rule")
	end_if;
	lhs:= op(r,1);
	if strlen(lhs) <> 1 then
	    error("illegal left hand side of rule")
	end_if;
	rhs:= op(r,2);
	case domtype(rhs)
	of DOM_IDENT do
	    if contains(prims, lhs) then
		error("multiple turtle lhs")
	    end_if;
	    prims:= prims union {lhs};
	    # rule for drawing primitive #
	    case (rhs)
	    of hold(line) do
		pr:= pr union {"F"};
		prim(lhs):= proc(T) begin Turtle::line(T,1.0) end_proc;
		break;
	    of hold(move) do
		pr:= pr union {"f"};
		prim(lhs):= proc(T) begin Turtle::move(T,1.0) end_proc;
		break;
	    of hold(push) do
		pr:= pr union {"["};
		prim(lhs):= proc(T) begin Turtle::push(T) end_proc;
		break;
	    of hold(pop) do
		pr:= pr union {"]"};
		prim(lhs):= proc(T) begin Turtle::pop(T) end_proc;
		break;
	    of hold(left) do
		pr:= pr union {"+"};
		prim(lhs):= subs(proc(T) begin Turtle::left(T,deg) end_proc,
				hold(deg)=float(deg));
		break;
	    of hold(right) do
		pr:= pr union {"-"};
		prim(lhs):= subs(proc(T) begin Turtle::right(T,deg) end_proc,
				hold(deg)=float(deg));
		break;
	    otherwise error("illegal right hand side of rule")
	    end_case;
	    break;

	of DOM_LIST do
	    if contains(prims, lhs) then
		error("multiple turtle lhs")
	    end_if;
	    prims:= prims union {lhs};
	    if min(op(rhs)) < 0.0 or max(op(rhs)) > 1.0 then
		error("illegal color")
	    end_if;
	    prim(lhs):= subsop(proc(T) begin Turtle::color(T,r,g,b) end_proc,
			    [4,2]=rhs[1], [4,3]=rhs[2], [4,4]=rhs[3]);
	    break;

	otherwise
	    if contains(rules, lhs) then
		error("multiple rule lhs")
	    end_if;
	    rules:= rules union {lhs};
	    # grammar rule #
	    p:= [];
	    for j from 0 to strlen(rhs)-1 do
		p:= [ hold(r)(substring(rhs,j,1)) ].p
	    end_for;
	    rule(lhs):= subsop(proc(T) local r; begin
		    gen:=gen-1; if gen = 0 then r:= prim else r:= rule end_if;
		    T:= f(T); gen:=gen+1; T
		end_proc, [4,3,2,0] = hold(_fconcat)(op(p)));
	end_case;
    end_for;

    # insert default drawing primitive rules #
    if not contains(pr, "F") then
	prim("F"):= proc(T) begin Turtle::line(T,1.0) end_proc;
	prims:= prims union {"F"};
    end_if;
    if not contains(pr, "f") then
	prim("f"):= proc(T) begin Turtle::move(T,1.0) end_proc;
	prims:= prims union {"f"};
    end_if;
    if not contains(pr, "[") then
	prim("["):= proc(T) begin Turtle::push(T) end_proc;
	prims:= prims union {"["};
    end_if;
    if not contains(pr, "]") then
	prim("]"):= proc(T) begin Turtle::pop(T) end_proc;
	prims:= prims union {"]"};
    end_if;
    if not contains(pr, "+") then
	prim("+"):= subs(proc(T) begin Turtle::left(T,deg) end_proc,
			hold(deg)=float(deg));
	prims:= prims union {"+"};
    end_if;
    if not contains(pr, "-") then
	prim("-"):= subs(proc(T) begin Turtle::right(T,deg) end_proc,
			hold(deg)=float(deg));
	prims:= prims union {"-"};
    end_if;
    
    # create string printing primitives #
    sprim:= proc(s) begin s end_proc;
    for p in prims do
    	sprim(p):= subsop(proc(s) begin s.p end_proc, [4,2]=p)
    end_for;
    
    # deg, start, args(i) are used for printing only #
    new(Lsys, sp, rule, prim, sprim, deg, start, args(i) $ hold(i)=3..args(0));
end_proc:

#
print -- return expression to print instead of L-system l
#

Lsys::print:= proc(l) local i; begin
    hold(Lsys)(extop(l,i) $ i=5..extnops(l))
end_proc:

#
plot -- execute L-system l for gen generations and plot turtle path
#

Lsys::plot:= proc(l,gen)
    local rule, prim, t;
begin
    if args(0) <> 2 then error("wrong no of args") end_if;
    if domtype(l) <> Lsys then error("no L-system") end_if;
    if not testtype(gen, Type::PosInt) then
	error("illegal generation count")
    end_if;
    t:= Turtle();
    rule:= extop(l,2);
    prim:= extop(l,3);
    Turtle::plot(extop(l,1)(t));
end_proc:

#
path -- execute L-system l for gen generations and return turtle path
#

Lsys::path:= proc(l,gen)
    local rule, prim, t;
begin
    if args(0) <> 2 then error("wrong no of args") end_if;
    if domtype(l) <> Lsys then error("no L-system") end_if;
    if not testtype(gen, Type::PosInt) then
	error("illegal generation count")
    end_if;
    t:= Turtle();
    rule:= extop(l,2);
    prim:= extop(l,3);
    Turtle::path(extop(l,1)(t));
end_proc:

#
gen -- execute L-system l for gen generations and return generated string
#

Lsys::gen:= proc(l,gen)
    local rule, prim;
begin
    if args(0) <> 2 then error("wrong no of args") end_if;
    if domtype(l) <> Lsys then error("no L-system") end_if;
    if not testtype(gen, Type::PosInt) then
	error("illegal generation count")
    end_if;
    rule:= extop(l,2);
    prim:= extop(l,4);
    extop(l,1)("")
end_proc:

# end of file #
