# main file of the ODE library solver. This solver works as follows:
  the user calls solve(ode(...)) which calls in turn ode::solve(...)
  by the overloading mechanism of solve, and which calls ode::dsolve(...)
  (see below for details).

  The main reference for the methods used in this library is the book
  of Daniel Zwillinger, Handbook of Differential Equations, Academic Press.
#
ode::solve:=proc(o) begin ode::dsolve(extop(o)) end_proc:

# ode::dsolve(sys,yofz) solve the system (or equation) sys with
  respect to yofz.
  Input: sys is either a single equation (or expression) or a set of equations
         yofz is either of the form y(z) or a set of such unknown functions
  Output: a (possibly empty) list of solutions. In the case of one unknown function,
          each solution is an expression. In the case of several unknown functions,
          each solution is a list of assignments for these functions.

  Examples: ode::dsolve(diff(y(x),x,x)=0,y(x));
            ode::dsolve({diff(y(x),x,x)=0,y(0)=a,D(y)(0)=b},y(x));
  ode::dsolve({diff(y1(t),t)=9*y1(t)+2*y2(t),diff(y2(t),t)=y1(t)+8*y2(t)},{y1(t),y2(t)});
#
ode::dsolve := proc(sys,yofz)
local y,z,eqs,inits,l;
begin
   case type(yofz)
   of "function" do break;
   of DOM_SET do 
      if nops(yofz)=1 then yofz:=op(yofz)
      else return(ode::solve_sys(args()))
      end_if
   otherwise error("invalid arguments")
   end_case;
   y:=op(yofz,0);
   if not has(sys,y) then error("differential function does not appear") end_if;
   if nops(yofz)<>1 then
      l:=ode::diffvars(sys,y);
      if nops(l)=1 then # only one variable is "differential #
         z:=op(l);
         l:=[op(yofz)];
         sys:=ode::subsvars(sys,y,contains(l,z),l);
         return(ode::dsolve(sys,y(z)))
      else error("resolution of partial differential equations not yet implemented") 
      end_if
   end_if;
   z:=op(yofz);
   if {type(y),type(z)}<>{DOM_IDENT} then error("invalid arguments") end_if;
   if type(sys)=DOM_SET then
      eqs:=ode::split(sys,z); inits:=eqs[2]; eqs:=eqs[1]
   else eqs:={sys}; inits:={}
   end_if;
   if nops(eqs)=1 then ode::solve_eq(op(eqs),y,z,inits)
   else error("number of equations differs from number of functions")
   end_if
end_proc:
