# #
# $Date: 1995/06/16 13:40:00 $  $Author: frankp $   $Revision: 1.38.2.1 $ #
# #
# frankp, 24/02/94 #

#++
  linsolve.mu

      linalg::linearSolve  --  solve a linear system of equations

      linearSolve( A,B )

      A,B: matrices

      linearSolve(A,B) solves the systems A*x=col(B,i), i=1..ncols(B) 
      and returns a matrix representing the special solutions of the 
      ncols(B) systems.

      linearSolve( A,v,l )

      A: matrix
      l: array or list

      linearSolve(A,v,l) solves the system A*x=B. If there do 
      not exist a unique solution, the general solution of the system 
      will be expressed by the sum of a special solution and a
      linear combination of the general solution of the corresponding
      homogeneus system with scalars in l. The solution a an element
      of the domain of Matrix(A::coeffRing).

      linearSolve( A,v,{Special,Unique} )

      A                 : matrix
      v                 : vector
      'Special','Unique': options

      linearSolve(A,B,Special) returns a special solution of the system.
      linearSolve(A,B,Unique) returns an unique solution if one exits,
      otherwise [] will be returned.
	

      linearSolve returns [] if the system has no solution.
++#

linalg::linearSolve := proc(A,B,opt)
    name linalg::linearSolve;
    local v, ys, y, t, r, c, SwapList, i, j, k, cB, rank, d, R, 
	  Ssol, Rmult, Rplus, Riszero, Rzero, Rdivex, Rnegate, MnewThis;
begin
    if testargs() then
	case args(0)
	of 3 do
	    if not contains( {hold(Unique),hold(Special)},opt ) and 
              not contains({DOM_ARRAY,DOM_LIST},domtype(opt))
	    then 
		error("invalid argument specified")
	    end_if;
	    if domtype(opt) = DOM_ARRAY then
		if op(opt,[0,1]) > 1 then
		    error("array has invalid dimension")
	    	end_if
	    end_if
	of 2 do
	    if A::hasProp( MatrixCat ) <> TRUE
	      or B::hasProp( MatrixCat ) <> TRUE
	    then
		error("expecting a matrix")
	    end_if;
	    R := A::coeffRing;
	    if not R::hasProp( IntegralDomain ) then
		error("expecting matrices over an integral domain")
	    end_if;
	    if A::constructor = SquareMatrix then
		if not testtype( B,A::getSuperDomain(A) ) then
		    error("types of matrices don't match")
		end_if
	    elif not testtype( B,type(A) ) then 
		error("types of matrices don't match")
	    end_if;
	    t := B::dimen(B);
	    if op( t,1 ) <> op( A::dimen(A),1 ) then
		error("invalid operands")
	    end_if;
	    if op(t,1) <> 1 and op(t,2) <> 1 and args(0) > 2 then
		error("wrong no of args")
	    end_if;
	    break
	otherwise
	    error("wrong no of args")
	end_case
    end_if;

    R := A::coeffRing;
    if R <> B::coeffRing then
	if A::constructor = SquareMatrix then
	    B := domattr(A::getSuperDomain(A),"convert")( B )
	else
	    B := A::convert( B )
	end_if
    end_if;

    if A::constructor = SquareMatrix then
	MnewThis := domattr(A::getSuperDomain(A),"newThis")
    else
	MnewThis := A::newThis
    end_if;

    Riszero := R::iszero; 
    d := A::dimen(A); r := op(d,1); c := op(d,2);
    cB := op( B::dimen(B),2 );

    userinfo(1,"compute an upper triangular matrix");
    userinfo(2,"of ", A::concatMatrix(A,B));

    t := A::gaussElim( A::concatMatrix(A,B) );
    A := op(t,1); rank := op(t,2);

    # test, if the system is overdetermined #
    for i from 1 to c do
	if not Riszero( A[rank,i] ) then break end_if
    end_for;
    if i > c or (opt = hold(Unique) and rank < c) then 
	return( [] ) 
    end_if; 

    userinfo(1,"swap columns until A[1..rank,1..rank] is regular");

    SwapList := [];
    for i from 1 to rank do
	j := i;
	while Riszero(A[i,j]) do j := j + 1 end_while;
	if j > i then
	    A := A::swapCol(A,i,j); 
	    SwapList := append( SwapList,[i,j] )
	end_if
    end_for;

    Rplus := R::_plus;
    Rmult := R::_mult;
    Rzero := R::zero;
    Rnegate := R::negate;
    Rdivex := R::divex;

    userinfo(1,"compute a special or unique solution by backsolving");

    Ssol := array( 1..cB );
    d := A[rank,rank];
    for k from 1 to cB do
	ys := [ Rzero $ c ]; ys[rank] := A[rank,c+k];
	for i from rank-1 downto 1 do
	    t := Rplus( Rzero,Rmult(A[i,j],ys[j]) $ hold(j)=i+1..rank );
	    ys[i] := Rdivex( Rplus( Rmult( d, A[i,c+k] ), 
		     Rnegate( t ) ),A[i,i] )
        end_for;
        ys := map( ys,fun(Rdivex(args(1),d)) );
        if contains( ys,FAIL ) > 0 then return( FAIL ) end_if;
	Ssol[k] := ys
    end_for;
    if opt = hold(Special) or rank = c or cB > 1 then
        # reswap the cols #
	userinfo(1,"reswap the columns of A");

	for k from 1 to cB do
	    ys := Ssol[k];
	    for i from nops(SwapList) downto 1 do
		v := SwapList[i];
		d := ys[v[1]]; ys[v[1]] := ys[v[2]]; ys[v[2]] := d
	    end_for;
	    Ssol[k] := ys
	end_for;
	t := MnewThis( c,1,Ssol[1] );
	for k from 2 to cB do
	    t := t::concatMatrix( t,MnewThis( c,1,Ssol[k] ) )
	end_for;

	return( t )
    end_if; 

    userinfo(1,"compute the nullspace (dimen = ",c-rank,")");
    if not R::hasProp( Field ) then
	error("Sorry, can not yet handle basis of modules")
    end_if;

    y := []; t := [ Rzero $ c ];
    for i from 1 to c-rank do
	t[rank+i] := R::one; y := append( y,t ); t[rank+i] := Rzero
    end_for;
    for k from 1 to c-rank do
	v := op(y,k);
	for i from rank downto 1 do
	    d := Rplus( Rzero, Rmult( A[i,j],v[j] ) $ hold(j)=i+1..rank );
	    d := Rplus( d,Rmult( A[i,rank+k],v[rank+k] ) );
	    v[i] := Rdivex( Rnegate(d),A[i,i] )
	end_for;
	y := subsop( y,k=v )
    end_for;

    userinfo(1,"reswap the columns");

    for i from nops(SwapList) downto 1 do
	v := SwapList[i];
	d := ys[v[1]]; ys[v[1]] := ys[v[2]]; ys[v[2]] := d;
	for j from 1 to c-rank do
	    t := op(y,j);
	    d := t[v[1]]; t[v[1]] := t[v[2]]; t[v[2]] := d;
	    y := subsop( y,j=t )
	end_for
    end_for;

    # return the correct type of solution #
    if opt = hold(opt) then
	t := [ MnewThis( c,1,y[i] ) $ hold(i)=1..c-rank ];
	return( [MnewThis( c,1,ys ),t] )
    end_if;

    if testargs() then
	case domtype(opt)
	of DOM_LIST  do if nops(opt) < c-rank then
			    error("missing elements in list")
			end_if;
			break
	of DOM_ARRAY do d := op(opt,[0,2]);
			if op(d,1) <> 1 or op(d,2) < c-rank then
			    error("array has invalid dimension")
			end_if
	end_case
    end_if;

    for i from 1 to c-rank do
	d := opt[i];
	if domtype(opt[i]) = R then next end_if;
	t := R::convert( d );
	if t = FAIL then 
	    t := d::convert_to( d,R );
	    if t = FAIL then
		error("type of coefficients don't match")
	    end_if
	end_if;
	opt[i] := t
    end_for;

     MnewThis( array(1..c,1..1,
	[ [Rplus(ys[j],Rmult(opt[i],op(y,[i,j])) $ hold(i)=1..c-rank)]
	     $ hold(j)=1..c ])
    )
end_proc:

# end of file #
