# #
# $Date: 1995/06/16 13:39:52 $  $Author: frankp $   $Revision: 1.1.2.1 $ #
# #
# frankp, 16/05/95 #

#++
  cholesky.mu

        linalg::cholesky -- computes the Cholesky decomposition

        cholesky(S [,isPosDef])
	cholesky(H [,isPosDef])

	S	: symmetric matrix
        H       : Hermitian matrix
        isPosDef: (optional) ident

	cholesky(H) computes the Cholesky decomposition for a 
	positive definite matrix H over a field.
	This is a decomposition in a product H=G*transpose(G) such that
	G is lower triangular and has positive entries on the main
	diagonal (the Cholesky factor of H).
	If H is not positive definit (i.e. such a decomposition does
	not exist), then an error occures.
	
	The call cholesky(H,isPosDef) just tests if the Hermitian matrix
	H is really positive definit and returns TRUE and FALSE respectivly.

	The same holds if the coefficient ring of the regarded
	matrix does not provide a method "conjugate". Then the matrix
	must be symmetric.
++#

linalg::cholesky := proc(A)
    name linalg::cholesky;
    local R, Rplus, Rzero, Rmult, Rconvert, Rdivex, Rnegate, Rconjugate,
	  t, i, j, k, n;
begin
    if testargs() then
	if args(0) < 1 or args(0) > 2 then
	    error("wrong no of args")
	end_if;
	if A::hasProp( MatrixCat ) <> TRUE then
	    error("first argument must be a matrix")
	end_if;
	R := A::coeffRing;
	if not R::hasProp( Field ) then
	    error("expecting matrix over a field")
	end_if;
	if R::conjugate = FAIL then
	    if not bool( A = A::transpose(A) ) then
		error("expecting a symmetric matrix")
	    end_if
	elif not bool( A = A::transpose(A::conjugate(A)) ) then
	    error("expecting a Hermitian matrix")
	end_if;
	if args(0) = 2 then
	    if args(2) <> hold(isPosDef) then
		error("option 'isPosDef' expected")
	    end_if
	end_if
    end_if;

    t := args(0);
    R := A::coeffRing;
    Rzero := R::zero;
    Rplus := R::_plus;
    Rmult := R::_mult;
    Rconvert := R::convert;
    Rnegate := R::negate;
    Rdivex := R::divex;
    Rconjugate := R::conjugate;
    if Rconjugate = FAIL then Rconjugate := id end_if;

    n := op(A::dimen(A),1);
    for j from 1 to n do
	for k from 1 to j-1 do
	    A[j,j] := Rplus( A[j,j],Rnegate(Rmult(A[j,k],A[j,k])) );
	end_for;

	k := A[j,j];
	userinfo(1,"check if ",A[j,j]," is real and positive");
	if Rconjugate(k) <> k then
	    # A[j,j] is not real, hence A is not pos. def. #
	    if t = 1 then
                error("given matrix is not positive definit")
            else
                A := FAIL;
                break # j #
            end_if
	elif k <= Rzero then
	    # A[j,j] is not positive, hence A is not pos. def. #
	    if t = 1 then
		error("given matrix is not positive definit")
	    else
		A := FAIL; 
		break # j #
	    end_if
	end_if;

	# compute G #
	(A[j,k] := Rzero) $ hold(k)=j+1..n; # upper triangular of G is 0 #

	A[j,j] := Rconvert( sqrt( A[j,j] ) );
	if A[j,j] = FAIL then
	    error("unable to compute the Cholesky factor over ".expr2text(R))
	end_if;

	for i from j+1 to n do
	    for k from 1 to j-1 do
		A[i,j] := Rplus( A[i,j],Rnegate(Rmult(A[i,k],A[j,k])) )
	    end_for;
	    A[i,j] := Rdivex( A[i,j],A[j,j] )
	end_for
    end_for;

    if t = 2 then bool( A <> FAIL ) else A end_if
end_proc:

# end of file #
