# Adaptive 8th-order Newton-Cotes quadrature				#
# 									#
# Reference:								#
# [1] Graf Finck zu Finckenstein, Einf"uhrung in die numerische		#
#     Mathematik, Band 1, Carl Hanser Verlag, 1977.			#
#									#
# Input:                                                                #
# f      - a function or procedure, the integrand.                      #
# a, b   - finite closed interval on which integral is taken.           #
# abserr - absolute error                                               #
# relerr - requested relative accuracy					#
#                                                                       #
# * If f is an expression, for example sin(x)+cos(x)+Pi, you have to    #
#   use func(f) instead of f.                                           #
#                                                                       #
# Output:                                                               #
# [ result, esterr, nofun, flag ]					#
#  									#
# result - the value of the integral					#
# esterr - an estimate of the absolute error actually committed		#
# nofun  - number of function/procedure evaluations			#
#                                                                       #
# Note: 								#
# * Newton-Cotes quadrature is not effective for high precisions,	#
#   because it is a fixed-order method.          			#
# * f must be continuous in [a, b]. No check is done.               	#
#									#
# Use traperror to trap an error condition during evaluation of f:      #
#									#
# abserr := 0.01*eps; 							#
# if traperror((l := ncquad(f, a, b, abserr, eps))) <> 0 then           #
#    userinfo(1, "singularity in interval of integration");             #
#    return(FAIL)                                                       #
# end_if;                                                               #
# r := l[1]; err := l[2]; nofun := l[3]; f := l[4];                     #
# if f <> 0 then                                       			#
#    userinfo(1, "unable to handle singularity");			#
#    return(FAIL)							#
# end_if;                                                               #
#                                                                       #
# How to use the estimated error:                                       #
# Calculate error tolerance, with eps as a relative error criterion;    #
# if too small then use absolute error.                                 #
# tol := max(eps*abs(r), abserr);                                       #
# userinfo(1, "error = ", err, ", error tolerance = ", tol);            #
# if err <= tol then                                                    #
#    return( r )                                                        #
# else                                                                  #
#    userinfo(1, "numerical integration failed to converge");           #
#    return(FAIL)                                                       #
# end_if 								#


intlib::ncquad := proc(g,a,b,abserr,relerr)
     local result,errest,nofun,flag,w0,w1,w2,w3,w4,area,x0,f0,stone,
           Step,cor11,temp,qprev,qnow,qdiff,qleft,esterr,tolerr,qright,f,
           x,fsave,xsave,levmin,levmax,levout,nomax,nofin,lev,nim,k,j,go;
begin
    qright := array(1 .. 31);
    f := array(1 .. 16);
    x := array(1 .. 16);
    fsave := array(1 .. 8,1 .. 30);
    xsave := array(1 .. 8,1 .. 30);
    levmin := 1;
    levmax := 30;
    levout := 6;
    nomax := 5000;
    nofin := nomax-8*levmax+8*levout-8*2^(levout+1);
    w0 := float(3956/14175);
    w1 := float(23552/14175);
    w2 := float(-3712/14175);
    w3 := float(41984/14175);
    w4 := float(-3632/2835);
    flag := 0;
    result := 0;
    cor11 := 0;
    errest := 0;
    area := 0;
    nofun := 0;
    if a = b then
        return( [ result, errest, nofun, flag ] );
    end_if;
    a := float(a); b := float(b);
    lev := 0;
    nim := 1;
    x0 := a;
    x[16] := b;
    qprev := 0;
    f0 := float(g(x0));
    stone := float(1/16*b-1/16*a);
    x[8] := float(1/2*x0+1/2*x[16]);
    x[4] := float(1/2*x0+1/2*x[8]);
    x[12] := float(1/2*x[8]+1/2*x[16]);
    x[2] := float(1/2*x0+1/2*x[4]);
    x[6] := float(1/2*x[4]+1/2*x[8]);
    x[10] := float(1/2*x[8]+1/2*x[12]);
    x[14] := float(1/2*x[12]+1/2*x[16]);
    for j from 2 to 16 step 2 do  f[j] := float(g(x[j])) end_for;
    nofun := 9;
    go := 1;
    while go = 1 do
        x[1] := float(1/2*x0+1/2*x[2]);
        f[1] := float(g(x[1]));
        for j from 3 to 15 step 2 do
            x[j] := float(1/2*x[j-1]+1/2*x[j+1]); f[j] := float(g(x[j]))
        end_for;
        nofun := nofun+8;
        Step := float(1/16*x[16]-1/16*x0);
        qleft :=
            (w0*(f0+f[8])+w1*(f[1]+f[7])+w2*(f[2]+f[6])+w3*(f[3]+f[5])+w4*f[4])
            *Step;
        qright[lev+1] := (w0*(f[8]+f[16])+w1*(f[9]+f[15])+w2*(f[10]+f[14])+
            w3*(f[11]+f[13])+w4*f[12])*Step;
        qnow := qleft+qright[lev+1];
        qdiff := qnow-qprev;
        area := area+qdiff;
        esterr := float(1/1023*abs(qdiff));
        tolerr := max(abserr,relerr*abs(area))*Step/stone;
        if (lev < levmin) or (tolerr < esterr) then
            nim := 2*nim;
            lev := lev+1;
            for k from 1 to 8 do  
                fsave[k,lev] := f[k+8]; xsave[k,lev] := x[k+8] 
            end_for;
            qprev := qleft;
            for k from 1 to 8 do
                j := -k; f[2*j+18] := f[j+9]; x[2*j+18] := x[j+9]
            end_for
        else
            if (lev < levmax) and (nofin < nofun) then
                nofin := 2*nofin; levmax := levout; flag := flag+(b-x0)/(b-a)
            end_if;
            if levmax <= lev then
                flag := flag+1;
                return( [ result, errest, nofun, flag ] )
            end_if;
            result := result+qnow;
            errest := errest+esterr;
            cor11 := cor11+float(1/1023*qdiff);
            while nim <> 2*(nim div 2) do
                nim := nim div 2; lev := lev-1
            end_while;
            nim := nim+1;
            if lev <= 0 then go := 0
            else
                qprev := qright[lev];
                x0 := x[16];
                f0 := f[16];
                for k from 1 to 8 do
                    f[2*k] := fsave[k,lev]; x[2*k] := xsave[k,lev]
                end_for
            end_if
        end_if
    end_while;
    result := result+cor11;
    if errest = 0 then
        return( [ result, errest, nofun, flag ] )
    end_if;
    while TRUE do
        temp := abs(result)+errest;
        if temp <> abs(result) then
            return( [ result, errest, nofun, flag ] )
        end_if;
        errest := 2*errest
    end_while
end_proc:
