# $Date: 1995/06/22 14:52:28 $ $Author: kg $ $Revision: 1.1.2.1 $ #

# kg, 24/05/95 #

#++
faclib::sqrfree_ffield -- square-free factorization for finite fields

faclib::sqrfree_ffield(p, n, m)

p    - univariate polynomial over finite field GF(n^m)
n, m - n^m is the number of field elements, n is a prime

faclib::sqrfree_ffield(p,n,m) returns a list [f1,e1,...,fk,ek] where the fi's
are the square-free factors with order ei.

The algorithm used is described in K.O.Geddes et.al. "Algorithms for Computer
Algebra", Kluwer 1992, p345 (SquareFreeFF).

NOTE:
It is assumed that for each indeterminate x of p the following holds:
p is primitive when viewed as polynomial in x. This is true if p was
'pre-factorized' by faclib::pre_factor.
++#

faclib::sqrfree_ffield:= proc(p, n, m)
    local X, T, deg, res, i, d, w, y, z;
begin
    X:= op(p,2);
    T:= op(p,3);
    
    # make p monic #
    p:= multcoeffs(p, 1/lcoeff(p));
    
    deg:= 1;
    res:= [];
    
    d:= Dpoly(p);
    if iszero(d) then
        p:= faclib::sqrfree_ffield(faclib::recip_pow(p, n, m), n, m);
        res:= [ (p[i], p[i+1]*n) $ i=1..(nops(p) div 2) ]
    else
        d:= gcd(p, d);
        w:= divide(p, d, Exact);
        while degree(w) <> 0 do
            y:= gcd(w, d);
            z:= divide(w, y, Exact);
            if degree(z) <> 0 then
		res:= append(res, z, deg)
            end_if;
            deg:= deg + 1;
            w:= y;
            d:= divide(d, y, Exact)
        end_while;
        if degree(d) <> 0 then
            d:= faclib::sqrfree_ffield(faclib::recip_pow(d, n, m), n, m);
            res:= append(res, (d[i], d[i+1]*n) $ i=1..(nops(d) div 2))
        end_if;
    end_if;
    res
end_proc:

#--
faclib::recip_pow -- recipocal power

faclib::recip_pow(p, n, m)

p    - univariate non-zero polynomial over GF(n^m)
n, m - n^m is the number of field elements, n is a prime

Returns p^(1/n).
--#

faclib::recip_pow:= proc(p, n, m)
    local x, i;
begin
    if m <> 1 then
        p:= mapcoeffs(p, _power, n^(m-1))
    end_if;
    x:= poly(op(p,[2,1]), op(p, 2..3));
    
    _plus(multcoeffs(x^(degree(nthterm(p,i)) / n), nthcoeff(p,i))
          $ i=1..nterms(p))
end_proc:

# end of file #
