# $Date: 1995/06/28 10:01:51 $  $Author: kg $  $Revision: 1.36.2.5 $ #

#++
ExpressionField -- domain of arithmetical MuPAD expressions
		   supposed to be a field. 

ExpressionField( [ Normal [, IsZero ]] )

Normal - function used to normalize intermediate expressions,
	 must be of type Expression -> Expression
IsZero - function used to detect 0, must be of type Expression -> Boolean
	 and return TRUE iff the argument is zero

This domain assumes that the expressions occuring are valid representations
of field elements; the field is assumed to be of characteristic 0. The
function 'Normal' is used to normalize an expression, the function 'IsZero'
is used to test an expression for 0.

The call 'ExpressionField()' creates a domain which is similar to 
the domain 'ArithmeticalExpression'. In this case only the standard 
simplifications are done. Some methods are defined additionaly (see
below).

If 'IsZero' is missing then 'iszero @ Normal' is used instead. 

If a normalizing function other than 'id' is given it is assumed that
this functions returns a normal form where the zero element is uniquely
represented by the constant 0.

The elements of this domain are created by 'new' and have one operand:
the expression representing the element of the field.

Examples:

ExpressionField( normal ) creates a field of rational expressions over
the rationals where the intermediate expressions occuring are normalized
by 'normal'.

In ExpressionField( id, iszero @ normal ) the intermediate expressions
are not normalized, but 0 is detected correctly.


The following methods are defined additionaly:

conjugate(x)	-- computes the complex conjugate of x
Re(x)		-- computes the real part of x
Im(x)		-- computes the imaginary part of x
norm(x)		-- computes the norm of x 
numer(x)	-- return the numerator of x
denom(x)	-- return the denominator of x
sqrfree(x)	-- returns a sqrfree representation of x
factor(x)	-- factorize x
diff(...)	-- computes the derivative of an expression
++#

ExpressionField := DomainConstructor(
    ExpressionField,
    [ Normal, IsZero ],
    [ ],
    ( case args(0)
      of 0 do
	  Normal:= id;
	  IsZero:= iszero;
	  break;
      of 1 do 
	  if Normal = id then 
	      IsZero:= iszero 
	  else 
	      IsZero:= iszero @ Normal 
	  end_if; 
	  break;
      of 2 do 
          break;
      otherwise error("wrong no of args");
      end_case ),
    (if Normal = id and IsZero = iszero then ArithmeticalExpression
     else BaseDomain 
     end_if),
    [ Field, DifferentialRing ],
    [ ( if Normal = id and IsZero = iszero then systemRep
	elif Normal <> id then normalRep 
	end_if ) ],

    "new" = proc(x)
    begin
	if args(0) <> 1 then error("wrong no of args") end_if;
	x:= this::convert(x);
	if x = FAIL then error("illegal argument") end_if;
	x
    end_proc, 

    "characteristic" = 0,
    
    "convert" = (if not this::hasProp(systemRep) then
		    fun((
			if domtype(args(1)) = this then return(args(1)) end_if;
			ArithmeticalExpression::convert(args(1));
			if % = FAIL then FAIL else
	    		    Normal(%);
	    		    if % = FAIL then FAIL else new(this, %) end_if
			end_if
		    ))
		 else 
		    ArithmeticalExpression::convert
		 end_if),

    "length" = (if not this::hasProp(systemRep) then
		   fun(length(extop(args(1),1)))
               else length end_if),

    "expr" = (if not this::hasProp(systemRep) then
	         fun(extop(args(1),1))
	     else id end_if),

    "conjugate" = (if not this::hasProp(systemRep) then
		      fun(new( this,Normal(conjugate(extop(args(1),1))) ))
		  else conjugate end_if),

    "Re" = (if not this::hasProp(systemRep) then
	       fun(new( this,Normal(Re(extop(args(1),1))) ))
	   else Re end_if),

    "Im" = (if not this::hasProp(systemRep) then
	       fun(new( this,Normal(Im(extop(args(1),1))) ))
           else Im end_if),

    "norm" = (if not this::hasProp(systemRep) then 
	         fun(new( this,Normal(abs(extop(args(1),1))) ))
	      else abs end_if),

    "zero" = (if not this::hasProp(systemRep) then new(this, 0) else 0 end_if),

    "one" = (if not this::hasProp(systemRep) then new(this, 1) else 1 end_if),

    "iszero" = (if not this::hasProp(systemRep) then
	           fun( IsZero(extop(args(1),1)) ) 
	       else iszero end_if),
	
    "equal" =  (if not this::hasProp(systemRep) then
		    fun(IsZero(extop(args(1),1)-extop(args(2),1)))
		else bool @ _equal end_if),

    "_plus" =  (if not this::hasProp(systemRep) then 
    	fun((if map({args()}, domtype) = {this} then
    	       new(this, Normal(_plus(extop(args(i),1) $ hold(i)=1..args(0))))
    	     else
    	       map([args()], this::convert);
    	       if contains(%, FAIL) = 0 then
    	           new(this, Normal(_plus(op(map(%, extop, 1)))))
    	       else
    	           FAIL
    	       end_if
    	     end_if))
    else _plus end_if),

    "negate" = (if not this::hasProp(systemRep) then
	           fun(new(this, Normal(-extop(args(1), 1))))
	       else -id end_if),

    "minus" =  (if not this::hasProp(systemRep) then
		   fun( new(this, Normal(extop(args(1),1)-extop(args(2),1))) )
	       else
	           fun( args(1)-args(2) )
	       end_if),

    "_mult" =  (if not this::hasProp(systemRep) then
    	fun((if map({args()}, domtype) = {this} then
    	       new(this, Normal(_mult(extop(args(i),1) $ hold(i)=1..args(0))))
    	     else
    	       map([args()], this::convert);
    	       if contains(%, FAIL) = 0 then
    	           new(this, Normal(_mult(op(map(%, extop, 1)))))
    	       else
    	           FAIL
    	       end_if
    	     end_if))
    else _mult end_if),

    "invert" = (if not this::hasProp(systemRep) then
	           fun( new(this,Normal(1/extop(args(1),1))) )
	       else 1/id end_if),

    "divex" =  (if not this::hasProp(systemRep) then 
	           fun(new(this, Normal(extop(args(1),1) / extop(args(2),1))))
	       else
	           fun( args(1) / args(2) )
	       end_if),

    "intmult" = (if not this::hasProp(systemRep) then
    	fun(new(this, Normal(extop(args(1),1) * args(2))))
    else _mult end_if),

    "_power" = (if not this::hasProp(systemRep) then
    	fun(new(this, Normal(extop(args(1),1) ^ args(2))))
    else _power end_if),

    "D" = (if not this::hasProp(systemRep) then
    	fun((
    	    case args(0)
    	    of 1 do
    		new(this, Normal(D(extop(args(1),1)))); break;
    	    of 2 do
    		new(this, Normal(D(args(1), extop(args(2),1)))); break;
    	    otherwise error("wrong no of args");
    	    end_case
    	))
    else D end_if),

    "diff" = (if not this::hasProp(systemRep) then
        fun(new(this, Normal(diff(extop(args(1),1), 
        	  args(i) $ hold(i)=2..args(0)))))
    else diff end_if),

    "normal" = (if not this::hasProp(systemRep) then
    	fun(new(this, Normal(normal(extop(args(1),1)))))
    else normal end_if),

    "simplify" = (if not this::hasProp(systemRep) then
	fun(new(this, Normal(simplify(extop(args(1),1)))))
    else simplify end_if),

    "radsimp" = (if not this::hasProp(systemRep) then
	fun(new(this, Normal(radsimp(extop(args(1),1)))))
    else radsimp end_if),

    "numer" =  (if not this::hasProp(systemRep) then
    	fun(new(this, Normal(numer(extop(args(1),1)))))
    else numer end_if),

    "denom" =  (if not this::hasProp(systemRep) then
    	fun(new(this, Normal(denom(extop(args(1),1)))))
    else denom end_if),

    "gcd" = (if not this::hasProp(systemRep) then 
		fun((
		    gcd(map(args, extop, 1));
		    if % = FAIL then FAIL else new(this, Normal(%)) end_if
		))
	     else gcd end_if),

    "factor" = (if not this::hasProp(systemRep) then
		    proc(x) local i, r; begin
			r := factor(extop(x,1));
		 	if r = FAIL then return(FAIL) end_if;
			r:= [ this::convert(r[1]),
	      		(this::convert(r[2*i]), r[2*i+1]) $ i=1..(nops(r) div 2) ];
			if contains(r, FAIL) = 0 then r else FAIL end_if
		    end_proc
		else factor end_if),

    "sqrfree" = (if not this::hasProp(systemRep) then
		    proc(x) local i, r; begin
			r:= sqrfree(extop(x,1));
			if r = FAIL then return(FAIL) end_if;
			r:= [ this::convert(r[1]),
	      		(this::convert(r[2*i]), r[2*i+1]) $ i=1..(nops(r) div 2) ];
			if contains(r, FAIL) = 0 then r else FAIL end_if
		    end_proc
		 else sqrfree end_if),

    "random" = (if not this::hasProp(systemRep) then
        fun(new(this, Normal((
            randpoly([hold(x)], hold(Expr));
            if iszero(%) then 0
            else expr(randpoly([hold(x)], hold(Expr))) / expr(%)
            end_if
        ))))
    end_if),

    "subs" = (if not this::hasProp(systemRep) then
    	fun(new(this, Normal(
		subs(extop(args(1),1), args(i) $ hold(i)=2..args(0)))))
    else subs end_if),

    "subsop" = (if not this::hasProp(systemRep) then
    	fun(new(this, Normal(
		subsop(extop(args(1),1), args(i) $ hold(i)=2..args(0)))))
    else subsop end_if),

    "subsex" = (if not this::hasProp(systemRep) then
    	fun(new(this, Normal(
		subsex(extop(args(1),1), args(i) $ hold(i)=2..args(0)))))
    else subsex end_if),

    "pivotSize" = (if not this::hasProp(systemRep) then
	fun(length(extop(args(1),1)))
    else length end_if)
):

# end of file #
