/*
 * Takes udp requests and processes them
 */
#include <arpa/inet.h> 
#include <linux/ax25.h> 
#include <linux/if.h>
#include <netinet/in.h> 
#include <sys/socket.h> 
#include <sys/time.h>
#include <sys/types.h> 
#include <netdb.h>
#include <stdio.h>
#include <syslog.h>
#include <string.h>
#include <unistd.h>

#include "rspfax25.h"
#include "rspfudp.h"
#include "queue.h"
#include "rspfd.h"
#include "rspfif.h"

extern int errno;
extern char *sys_errlist[];
extern struct queue *routerq;
extern struct queue *adj_queue;
extern struct queue *linkq;
extern struct queue *ifqueue;
extern struct queue *rt_queue;
extern struct queue *nodgrpq;
extern const char *StatusName[];
extern const char *version;
extern struct rspf_mib rspf_stats;

#define RETURN_STR "\255\128"

void send_adjs(int s, struct sockaddr sa, int len);
void send_links(int s, struct sockaddr sa, int slen);
void send_routers(int s, struct sockaddr sa, int slen);
void send_interfaces(int s, struct sockaddr sa, int slen);
void send_stats(int s, struct sockaddr sa, int slen);
void send_routes(int s, struct sockaddr sa, int slen);
void send_nodegroups(int s, struct sockaddr sa, int slen);

char* in_ntoa(u_long addr);

int init_udp(int port)
{
	struct sockaddr_in asin;
	struct protoent *ppe;
	int s;
	
	bzero((char*)&asin, sizeof(asin));
	asin.sin_family = AF_INET;
	asin.sin_addr.s_addr = INADDR_ANY;
	asin.sin_port = htons(port);
	
	if ( (ppe = getprotobyname("udp")) == 0)
	{
		syslog(LOG_DAEMON | LOG_ERR, "init_udp(): getprotobyname failed (%m)");
		return -1;
	}
	
	if ( (s= socket(PF_INET, SOCK_DGRAM, ppe->p_proto)) < 0)
	{
		syslog(LOG_DAEMON | LOG_ERR, "init_udp(): socket failed (%m)");
		(void)close(s);
		return -1;
	}
	
	if (bind(s, (struct sockaddr *)&asin, sizeof(asin)) < 0)
	{
		syslog(LOG_DAEMON | LOG_ERR, "init_udp(): bind failed (%m)");
		return -1;
	}
	
	return s;
}

void do_udp(int s)
{
	struct sockaddr sa;
	char buf[512];
	char outbuf[512];
	int slen;
	int size;
	
	slen = sizeof(sa);
	if ( (size = recvfrom(s, buf, sizeof(buf), 0, &sa, &slen)) < 0)
	{
		syslog(LOG_DAEMON | LOG_ERR,"do_udp(): recvfrom failed (%m)");
		return;
	}
	
	/* Work out the reply */
	strcpy(outbuf, "RSPF Daemon admin port on ");
	gethostname(outbuf + strlen(outbuf), 300);
	strcpy(outbuf + strlen(outbuf), ".\n\0");
	sendto(s, outbuf, strlen(outbuf)+1, 0, &sa, slen);
	
	if (strncmp(buf, "HELO", 4) == 0)
	{
		strcpy(outbuf + strlen(outbuf), ".\n\0");
		sendto(s, outbuf, strlen(outbuf)+1, 0, &sa, slen);
		sendto(s, version, strlen(version)+1, 0, &sa, slen);
		sendto(s, RETURN_STR, 2, 0 ,&sa, slen);
		return;
	}
	if (strncmp(buf, "HELP", 4) == 0) 
	{
		strcpy(outbuf, "Valid commands: ADJS, HELO, HELP, IFCS, LNKS, NGRP, ROUT, RTRS, STAT.\n");
		sendto(s, outbuf, strlen(outbuf)+1, 0, &sa, slen);
		sendto(s, RETURN_STR, 2, 0, &sa, slen);
		return;
	}
	if (strncmp(buf, "ADJS", 4) == 0)
	{
		send_adjs(s, sa, slen);
		return;
	}
	
	if (strncmp(buf, "LNKS", 4) == 0)
	{
		send_links(s, sa, slen);
		return;
	}
	
	if (strncmp(buf, "RTRS", 4) == 0)
	{
		send_routers(s, sa, slen);
		return;
	}
	
	if ( strncmp(buf, "ROUT", 4) == 0)
	{
		send_routes(s, sa, slen);
		return;
	}
	
	if (strncmp(buf, "NGRP", 4) == 0)
	{
		send_nodegroups(s, sa, slen);
		return;
	}
	
	if (strncmp(buf, "IFCS", 4) == 0)
	{
		send_interfaces(s, sa, slen);
		return;
	}
	
	if (strncmp(buf, "STAT", 4) == 0)
	{
		send_stats(s, sa, slen);
		return;
	}
	/* else */
	strcpy(outbuf, "Unknown Command\n");
	sendto(s, outbuf, strlen(outbuf)+1, 0, &sa, slen);
	sendto(s, RETURN_STR, 2, 0, &sa, slen);	
}

	
void send_adjs(int s, struct sockaddr sa, int slen)
{
	struct rspf_adj *ptr;
	char tbuf[30];
	char outbuf[128];
	time_t elapsed;
	qmark adj_qm;
	

	ptr = (struct rspf_adj*)qmove_first(adj_queue, &adj_qm);

	sprintf(outbuf, "Callsign IP Address       Port seq #  %% hrd Lastheard");
	sendto(s, outbuf, strlen(outbuf)+1, 0, &sa, slen);
      	while(ptr != NULL) 
      	{
      		bzero(outbuf, strlen(outbuf));
		if (ptr->rrhtime != 0) 
      		{
			elapsed = time(NULL) -  ptr->rrhtime;
			strftime(tbuf, 30, "%a %H:%M:%S", localtime(&(ptr->rrhtime)));
		} else
			strcpy(tbuf, "Not Router");
		sprintf(outbuf,"%8s %15s %5s %#6x %3d %14s %10s", ax25_ntoa(&ptr->dladdr), in_ntoa(ptr->addr), ptr->port, ptr->tx_pkts, ptr->rx_ratio, tbuf, StatusName[ptr->status]);
		sendto(s, outbuf, strlen(outbuf)+1, 0, &sa, slen);		
		
		ptr = (struct rspf_adj*)qmove_next(adj_queue,&adj_qm);
	}
	sendto(s, RETURN_STR, 2, 0, &sa, slen);
} /* print_adjs */	

void send_links(int s, struct sockaddr sa, int slen)
{
	struct link *ptr;
	char buf[200];
	qmark link_qm;
	
	ptr = (struct link*)qmove_first(linkq, &link_qm);
	sprintf(buf, "Link States\nSource Address   Destinatn Address\tCost Horizon Seq # ");
	sendto(s, buf, strlen(buf)+1, 0, &sa, slen);
	while (ptr != NULL)
	{
		sprintf(buf, "%15s->", in_ntoa(ptr->saddr));
		sprintf(buf+strlen(buf), "%15s/%2d\t%4d (%4d)  %5u",
			in_ntoa(ptr->daddr), ptr->sigbits, ptr->cost, ptr->horizon, ptr->seq_no);
		sendto(s, buf, strlen(buf)+1, 0, &sa, slen);		
		ptr = (struct link*)qmove_next(linkq, &link_qm);
	}
	sendto(s, RETURN_STR, 2, 0, &sa, slen);		
} /* send_links */

void send_routers(int s, struct sockaddr sa, int slen)
{
	struct router *rtr;
	qmark rt_qm;
	char buf[200];
	char tbuf[30];
	
	sprintf(buf, "Router List\nIP Address Sequence sub-seq  Bull Time");
	sendto(s, buf, strlen(buf)+1, 0, &sa, slen);
	rtr = (struct router*)qmove_first(routerq, &rt_qm);
	while(rtr != NULL)
	{
		strftime(tbuf, 30, "%a %H:%M:%S", localtime(&(rtr->bulltime)));
		sprintf(buf, "%15s %4u %3u %s", in_ntoa(rtr->addr), rtr->seq_no, rtr->sub_seq_no, tbuf);
		sendto(s, buf, strlen(buf)+1, 0, &sa, slen);
		rtr = (struct router*)qmove_next(routerq, &rt_qm);
	}
	sendto(s, RETURN_STR, 2, 0, &sa, slen);				
} /*send routers */

void send_interfaces(int s, struct sockaddr sa, int slen)
{
	char buf[128];
	struct rspf_if *ifptr;
	qmark if_qm;
	int adjcnt;
	struct rspf_adj *aptr;
	qmark adj_qm;
	
	
	sprintf(buf, "Interfaces\n   Name    Cost Nodes\n");
	sendto(s, buf, strlen(buf)+1, 0, &sa, slen);
	ifptr = (struct rspf_if*)qmove_first(ifqueue, &if_qm);
	while(ifptr != NULL)
	{
		adjcnt = 0;
		aptr = (struct rspf_adj*)qmove_first(adj_queue, &adj_qm);
		while(aptr != NULL)
		{
			if (strcmp(aptr->port, ifptr->name) == 0)
				adjcnt++;
			aptr = (struct rspf_adj*)qmove_next(adj_queue, &adj_qm);
		}
		sprintf(buf, "%10s %4d %4d", ifptr->name, ifptr->cost, adjcnt);
		sendto(s, buf, strlen(buf)+1, 0, &sa, slen);
		ifptr = (struct rspf_if*)qmove_next(ifqueue, &if_qm);
	}
	
	sendto(s, RETURN_STR, 2, 0, &sa, slen);
}

void send_stats(int s, struct sockaddr sa, int slen)
{
	char buf[128];
	
	sprintf(buf, "System Statistics\nIncoming              \t Outgoing");
	sendto(s, buf, sizeof(buf)+1, 0, &sa, slen);
	
	sprintf(buf, "Messages       %6lu \t %6lu", rspf_stats.rspfInMsgs, rspf_stats.rspfOutMsgs);
	sendto(s, buf, sizeof(buf)+1, 0, &sa, slen);
	
	sprintf(buf, "Rtr-Rtr Hellos %6lu \t %6lu", rspf_stats.rspfInRrhs, rspf_stats.rspfOutRrhs);
	sendto(s, buf, sizeof(buf)+1, 0, &sa, slen);
	
	sprintf(buf, "Rtr Envelopes  %6lu \t %6lu", rspf_stats.rspfInRouteEnvs, rspf_stats.rspfOutRouteEnvs);
	sendto(s, buf, sizeof(buf)+1, 0, &sa, slen);
	
	sprintf(buf, "Header Errors  %6lu", rspf_stats.rspfInHdrErrors);
	sendto(s, buf, sizeof(buf)+1, 0, &sa, slen);
	
	sprintf(buf, "Unknown Types  %6lu", rspf_stats.rspfInUnknownTypes);
	sendto(s, buf, sizeof(buf)+1, 0, &sa, slen);
	
	sprintf(buf, "Not RSPF iface %6lu\n", rspf_stats.rspfInNotIfaces);
	sendto(s, buf, sizeof(buf)+1, 0, &sa, slen);
	
	sprintf(buf, "RRH Timer     %6lu\tSus Pings     %6lu", rspf_stats.rspfRrhTimer,  rspf_stats.rspfSusPings);
	sendto(s, buf, sizeof(buf)+1, 0, &sa, slen);
	
	sprintf(buf, "Sus Timeout   %6lu\tPing Timer    %6lu\tBullitn Timer %6lu", rspf_stats.rspfSusTimeout, rspf_stats.rspfPingTimer, rspf_stats.rspfBullTimer);
	sendto(s, buf, sizeof(buf)+1, 0, &sa, slen);
	
	sprintf(buf, "Bull Timeout  %6lu", rspf_stats.rspfBullTimeout);
	sendto(s, buf, sizeof(buf)+1, 0, &sa, slen);
	
	sprintf(buf, "Current Adjs  %6lu\tCurrent Iface %6lu", rspf_stats.rspfCurrAdjacencies, rspf_stats.rspfCurrIfaces);
	sendto(s, buf, sizeof(buf)+1, 0, &sa, slen);

	sprintf(buf, "Envelope No   %6u\tSequence No   %5u\tSub-Seq No    %6u", rspf_stats.EnvelopeNumber, rspf_stats.SequenceNumber, rspf_stats.SubSequenceNumber);
	sendto(s, buf, sizeof(buf)+1, 0, &sa, slen); 
	
	sprintf(buf, "Link Horizon  %6u\tGroup Horizon %6u\tLocal Horizon %6u", rspf_stats.rspfLinkHorizon, rspf_stats.rspfGroupHorizon, rspf_stats.rspfLocalHorizon);
	sendto(s, buf, sizeof(buf)+1, 0, &sa, slen);
	
	sprintf(buf, "Port Horizon  %6u", rspf_stats.rspfPortableHorizon);
	sendto(s, buf, sizeof(buf)+1, 0, &sa, slen);
	
	sendto(s, RETURN_STR, 2, 0, &sa, slen);
}

void send_routes(int s, struct sockaddr sa, int slen)
{
	char buf[128];
	struct rspf_route *rt;
	qmark rtqm;
	
	sprintf(buf, "Routes");
	sendto(s, buf, strlen(buf)+1, 0, &sa, slen);
	
	sprintf(buf, "Destination    Sigbits  Cost  Iface");
	sendto(s, buf, strlen(buf)+1, 0 , &sa, slen);
	
	rt = (struct rspf_route*)qmove_first(rt_queue, &rtqm);
	while(rt != NULL)
	{
		sprintf(buf, "%15s %7u  %4u  %5s", in_ntoa(rt->addr), rt->sigbits, rt->cost, rt->port);
		sendto(s, buf, strlen(buf)+1, 0, &sa, slen);
		rt = (struct rspf_route*)qmove_next(rt_queue, &rtqm);
	}
	
	sprintf(buf, "------");
	sendto(s, buf, strlen(buf)+1, 0, &sa, slen);
	
	sendto(s, RETURN_STR, 2, 0, &sa, slen);
}

void send_nodegroups(int s, struct sockaddr sa, int slen)
{
	char buf[128];
	struct nodegroup *ng;
	qmark ngqm;
	
	sprintf(buf, "Node Groups");
	sendto(s, buf, strlen(buf)+1, 0, &sa, slen);
	
	sprintf(buf, "Group               Cost");
	sendto(s, buf, strlen(buf)+1, 0, &sa, slen);
	
	ng = (struct nodegroup*)qmove_first(nodgrpq, &ngqm);
	while(ng != NULL)
	{
		sprintf(buf, "%15s/%2u  %u", in_ntoa(ng->addr), ng->sigbits, ng->cost);
		sendto(s, buf, strlen(buf)+1, 0, &sa, slen);
		ng = (struct nodegroup*)qmove_next(nodgrpq, &ngqm);
	}
	
	sprintf(buf, "------");
	sendto(s, buf, strlen(buf)+1, 0, &sa, slen);
	
	sendto(s, RETURN_STR, 2, 0, &sa, slen);
}