/*
 * tkTkwm.c -- provides the core TKWM commands.
 *
 * Copyright (c) 1993 Eric Schenk.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL ERIC SCHENK BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ERIC
 * SCHENK HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * ERIC SCHENK SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND ERIC SCHENK HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 * Portions of this code are modified from TK 3.6, copyright (c) 1990-1993
 * The Regents of the University of California.
 * Other portions of this code are modified from ctwm, copyright (c) 1988
 * Evans & Sutherland Computer Corporation, portions copyright (c) 1989
 * by the Massachusetts Institute of Technology.
 * The applicable copyright notices appear below.
 *
 * Copyright (c) 1990-1993 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *       Copyright 1988 by Evans & Sutherland Computer Corporation,
 *                          Salt Lake City, Utah
 *  Portions Copyright 1989 by the Massachusetts Institute of Technology
 *                        Cambridge, Massachusetts
 *
 *                           All Rights Reserved
 *
 *    Permission to use, copy, modify, and distribute this software and
 *    its documentation  for  any  purpose  and  without  fee is hereby
 *    granted, provided that the above copyright notice appear  in  all
 *    copies and that both  that  copyright  notice  and  this  permis-
 *    sion  notice appear in supporting  documentation,  and  that  the
 *    names of Evans & Sutherland and M.I.T. not be used in advertising
 *    in publicity pertaining to distribution of the  software  without
 *    specific, written prior permission.
 *
 *    EVANS & SUTHERLAND AND M.I.T. DISCLAIM ALL WARRANTIES WITH REGARD
 *    TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES  OF  MERCHANT-
 *    ABILITY  AND  FITNESS,  IN  NO  EVENT SHALL EVANS & SUTHERLAND OR
 *    M.I.T. BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL  DAM-
 *    AGES OR  ANY DAMAGES WHATSOEVER  RESULTING FROM LOSS OF USE, DATA
 *    OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 *    TORTIOUS ACTION, ARISING OUT OF OR IN  CONNECTION  WITH  THE  USE
 *    OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifndef lint
static char rcsid[] = "$Header: /usr/src/local/tcl/tkwm/RCS/tkTkwm.c,v 1.27 1994/12/20 17:05:22 schenk Exp schenk $";
#endif /* not lint */


#include<X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xproto.h>
#include <X11/Xresource.h>
#include <X11/Xutil.h>
#include "tk.h"
#include "tkTkwm.h"
#include <stdlib.h>
#include <ctype.h>
#include "tkShape.h"


/*
 * ANSI Function prototypes.
 */

extern char *strdup _ANSI_ARGS_((const char *));
extern int strncmp _ANSI_ARGS_((const char *, const char *, size_t));
extern int sscanf _ANSI_ARGS_((const char *, const char *, ...));
extern int fprintf _ANSI_ARGS_((FILE *, const char *, ...));

static void EatFocusInEvents _ANSI_ARGS_((Display *, Window));
static void InternUsefulAtoms _ANSI_ARGS_((void));
static int WindowExists _ANSI_ARGS_((Window));
static void GetSizeHints _ANSI_ARGS_((Plug *));
static Plug *TkWinPlug _ANSI_ARGS_((Tk_Window));
static Plug *WindowPlug _ANSI_ARGS_((Window));
static void GetColormapWindows _ANSI_ARGS_((Window, Colormaps *));
static void FreeColormapWindows _ANSI_ARGS_((Colormaps *));
static void InstallWindowColormaps _ANSI_ARGS_((Colormaps *));
static ColormapWindow *NewColormapWindow _ANSI_ARGS_((Window));

static void ComputePlugGeometry _ANSI_ARGS_((Plug *));
static void PlugInitializeGeometry _ANSI_ARGS_((Plug *));
static void PlugPositionProc _ANSI_ARGS_((ClientData, XEvent *));
static void PlugEventProc _ANSI_ARGS_((ClientData, XEvent *));
static void PlugDisplay _ANSI_ARGS_((ClientData));
static void PlugDestroy _ANSI_ARGS_((ClientData));
static int PlugConfigure _ANSI_ARGS_((Tcl_Interp *, Plug *, int, char **, int));
static void PlugSendConfig _ANSI_ARGS_((ClientData));
static int PlugWidgetCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **));
static int PlugCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **));

static int FocusCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **));
static int ColormapCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **));
static int PointerCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **));
static int RestackCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **));
static int ReparentCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **));
static int BindCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **));
static int OutlineCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **));
static int GrabserverCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **));
static int GrabeventCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **));
static int GrabpointerCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **));
static int UnmanagedCmd _ANSI_ARGS_((ClientData, Tcl_Interp *));
static int IconSizeCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **));
static int InfoCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **));

static char *ExpandPercents _ANSI_ARGS_((char *, XEvent *, char *, int));
void doBinding _ANSI_ARGS_((int, XEvent *));
static void DrawOutline _ANSI_ARGS_((void));
static void MoveOutline _ANSI_ARGS_((ClientData));
static char *GetField _ANSI_ARGS_((char *, char *, int));

int HandleColormapNotify _ANSI_ARGS_((XEvent *));
int HandleVisiblityNotify _ANSI_ARGS_((XEvent *));
int HandleLeaveNotify _ANSI_ARGS_((XEvent *));
int HandleEnterNotify _ANSI_ARGS_((XEvent *));
int HandlePropertyNotify _ANSI_ARGS_((XEvent *));
int HandleClientMessage _ANSI_ARGS_((XEvent *));
int HandleConfigureRequest _ANSI_ARGS_((XEvent *));
int HandleMapRequest _ANSI_ARGS_((XEvent *));
int HandleUnmapNotify _ANSI_ARGS_((XEvent *));
int HandleDestroyNotify _ANSI_ARGS_((XEvent *));
int TkwmEventProc _ANSI_ARGS_((ClientData, XEvent *));

static int DoInfoCmd _ANSI_ARGS_((Tcl_Interp *, Plug *, char *));
int MappedNotOverride _ANSI_ARGS_((Window));
int TkwmCmd _ANSI_ARGS_((ClientData, Tcl_Interp *, int, char **));
static int AlreadyManagedError _ANSI_ARGS_((Display *, XErrorEvent *));
void Tk_Substitute _ANSI_ARGS_((Tk_Window, Window));
int Tkwm_Init _ANSI_ARGS_((Tcl_Interp *));

/*
 * Save the original X error handler.
 */

static int (* defaultErrorHandler)();

/* 
 * The current state of the window placement/move/resize outline.
 */

static int outlinepending = 0;
static int lastx = 0;
static int lasty = 0;
static int lastWidth = 0;
static int lastHeight = 0;

/*
 * The next focus change. Focus must be delayed because the focus window
 * may not exist when the focus change is requested.
 */

static Window pending_focus = None;

static ServerGrabbed = 0;


/*
 * Binding tables for Tkwm events.
 */

#define TKWM_ICONIFY 0
#define TKWM_MAPREQ 1
#define TKWM_PROPERTY 2
#define TKWM_CONFIGURE 3
#define TKWM_FOCUSIN 4
#define TKWM_SHAPENOTIFY 5

static struct {
  char *name;
  char *command;
  Tcl_Interp *interp;
} bindings[] = {
  {"<Iconify>", "", 0},
  {"<MapRequest>", "", 0},
  {"<Property>", "", 0},
  {"<Configure>", "", 0},
  {"<FocusIn>", "", 0},
  {"<ShapeNotify>", "", 0},
  {0,0}
};


/*
 * Colormap variables.
 */

struct {
  Colormaps *cmaps;
  int maxCmaps;
  unsigned long first_req;
  int root_pushes;
} cmapInfo;

Colormaps rootcmap;		   /* Colormap chain for the root window */
static Plug *colormapFocusPlug;    /* The plug with the colormap focus */
static Tk_Window colormapwin;	   /* The window with the colormap focus */
int ColortableThrashing = 0;       /* The current set of colortables needs
                                    * to be reinstalled */

/*
 * Configuration variables.
 */

/* FIXME. The state should not be initialized. It should read the property. */
static Tk_ConfigSpec configSpecs[] = {
  {TK_CONFIG_WINDOW, "-frame", "frame", "Frame",
     0, Tk_Offset(Plug, frame), TK_CONFIG_DONT_SET_DEFAULT},
  {TK_CONFIG_INT, "-iconwin", "iconWin", "IconWin",
     "0", Tk_Offset(Plug, iconwin), 0},
  {TK_CONFIG_STRING, "-state", "state", "State",
     0, Tk_Offset(Plug, state), TK_CONFIG_DONT_SET_DEFAULT},
  {TK_CONFIG_BOOLEAN, "-setgrid", "setGrid", "SetGrid",
     "1", Tk_Offset(Plug, setGrid), 0},
  {TK_CONFIG_BORDER, "-background", "background", "Background",
     "#cdb79e", Tk_Offset(Plug, background), TK_CONFIG_COLOR_ONLY},
  {TK_CONFIG_BORDER, "-background", "background", "Background",
     "white", Tk_Offset(Plug, background), TK_CONFIG_MONO_ONLY},
  {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
     (char *) NULL, 0, 0}
};


/*
 * The following data structures are borrowed from tkBind.c.
 * We use the to parse keyboard and button event grabs.
 */

/*
 * A hash table is kept to map from the string names of event
 * modifiers to information about those modifiers.  The structure
 * for storing this information, and the hash table built at
 * initialization time, are defined below.
 */

typedef struct {
  char *name;                 /* Name of modifier. */
  int mask;                   /* Button/modifier mask value,                  
			       * such as Button1Mask. */
  int flags;                  /* Various flags;  see below for
			       * definitions. */
} ModInfo;

/*
 * Flags for ModInfo structures:
 *
 * ANY -                Non-zero means that this event allows
 *                      any unspecified modifiers.
 */

#define ANY             4

static ModInfo modArray[] = {
  {"Control",         ControlMask,    0},
  {"Shift",           ShiftMask,      0},
  {"Lock",            LockMask,       0},
  {"B1",              Button1Mask,    0},
  {"Button1",         Button1Mask,    0},
  {"B2",              Button2Mask,    0},
  {"Button2",         Button2Mask,    0},
  {"B3",              Button3Mask,    0},
  {"Button3",         Button3Mask,    0},
  {"B4",              Button4Mask,    0},
  {"Button4",         Button4Mask,    0},
  {"B5",              Button5Mask,    0},
  {"Button5",         Button5Mask,    0},
  {"Mod1",            Mod1Mask,       0},
  {"M1",              Mod1Mask,       0},
  {"Meta",            Mod1Mask,       0},
  {"M",               Mod1Mask,       0},
  {"Mod2",            Mod2Mask,       0},
  {"M2",              Mod2Mask,       0},
  {"Alt",             Mod2Mask,       0},
  {"Mod3",            Mod3Mask,       0},
  {"M3",              Mod3Mask,       0},
  {"Mod4",            Mod4Mask,       0},
  {"M4",              Mod4Mask,       0},
  {"Mod5",            Mod5Mask,       0},
  {"M5",              Mod5Mask,       0},
  {"Any",             0,              ANY},
  {"Double",          0,              0},
  {"Triple",          0,              0},
  {NULL,              0,              0}
};
static Tcl_HashTable modTable;

/*
 * Some atoms we need to know about. (mostly taken from ctwm).
 */

Atom _XA_WM_CHANGE_STATE;
Atom _XA_WM_STATE;
Atom _XA_WM_COLORMAP_WINDOWS;
Atom _XA_WM_PROTOCOLS;
Atom _XA_WM_TAKE_FOCUS;
Atom _XA_WM_SAVE_YOURSELF;
Atom _XA_WM_DELETE_WINDOW;
Atom _XA_WM_ROOT;

/*
 * Client management utility routines.
 */

/*
 * Return true if the window still exists. Sync's the server connection.
 */

static int WindowExists(win)
     Window win;
{
  int d;

  XSync(dpy,False);
  return XGetGeometry(dpy,win,(Window *)&d,&d,&d,(unsigned int *)&d,
                      (unsigned int *)&d,(unsigned int *)&d,(unsigned int *)&d);
}

/*
 * Return the plug associated with a Tk window.
 */

static Plug *TkWinPlug(tkwin)
     Tk_Window tkwin;
{
  Plug *plug;
  
  if (XFindContext(dpy, (XID)tkwin, plugWinContext, (caddr_t *)&plug) == XCNOENT)
    return NULL;
  return plug;
}

/*
 * Return the plug associated with a given X window id.
 */

static Plug *WindowPlug(win)
     Window win;
{
  Plug *plug;
  
  if (XFindContext(dpy, (XID)win, plugContext, (caddr_t *)&plug) == XCNOENT)
    return NULL;
  return plug;
}

/*
 * Get the size hint for a plug
 */

static void GetSizeHints(plug)
     Plug *plug;
{
  plug->old_normalhints = plug->normalhints;
  if (!XGetWMNormalHints(dpy, plug->window, &plug->normalhints, &plug->supplied_return))
    plug->normalhints.flags = 0;
  
  if (!(plug->normalhints.flags & PResizeInc))
    plug->normalhints.width_inc = plug->normalhints.height_inc = 1;
  if (plug->normalhints.width_inc == 0)
    plug->normalhints.width_inc = 1;
  if (plug->normalhints.height_inc == 0)
    plug->normalhints.height_inc = 1;
  
  if (!(plug->normalhints.flags & (PBaseSize | PMinSize))) {
    plug->normalhints.base_width = plug->normalhints.base_height = 
      plug->normalhints.min_width = plug->normalhints.min_height = 0;
  }
  if (!(plug->normalhints.flags & PBaseSize)) {
    if (plug->normalhints.flags & PResizeInc) {
      /* ICCCM says to assume this if not supplied, but we have
       * to be carefull only to do this if the client declared a grid.
       * Otherwise we are imposing our own grid and this would be wrong.
       */
      plug->normalhints.base_width = plug->normalhints.min_width,
      plug->normalhints.base_height = plug->normalhints.min_height;
    } else {
      plug->normalhints.base_width = 0;
      plug->normalhints.base_height = 0;
    }
  }
  if (!(plug->normalhints.flags & PMinSize)) {
    if (plug->normalhints.flags & PResizeInc) {
      plug->normalhints.min_width = plug->normalhints.base_width,
      plug->normalhints.min_height = plug->normalhints.base_height;
    } else {
      plug->normalhints.min_width = 0;
      plug->normalhints.min_height = 0;
    }
  }
  if (!(plug->normalhints.flags & PWinGravity))
    plug->normalhints.win_gravity = NorthWestGravity;
}

static void PlugInitializeGeometry(plug)
     Plug *plug;
{
  char gx,gy;
  int x,y;
  /*
   * Store the plug's initial position as the initial final position.
   */
  plug->fx = plug->winatts.x;
  plug->fy = plug->winatts.y;
  
  /*
   * Compute the initial requested location of the window in TK geometry terms.
   */
  switch(plug->normalhints.win_gravity) {
  case NorthWestGravity:
    gx = '+'; gy = '+';
    x = plug->winatts.x;
    y = plug->winatts.y;
    break;
  case SouthWestGravity:
    gx = '+'; gy = '-';
    x = plug->winatts.x;
    y = screenHeight - (plug->winatts.y + plug->winatts.height + 2*plug->winatts.border_width);
    break;
  case NorthEastGravity:
    gx = '-'; gy = '+';
    x = screenWidth - (plug->winatts.x + plug->winatts.width + 2*plug->winatts.border_width);
    y = plug->winatts.y;
    break;
  case SouthEastGravity:
    gx = '-'; gy = '-';
    x = screenWidth - (plug->winatts.x + plug->winatts.width + 2*plug->winatts.border_width);
    y = screenHeight - (plug->winatts.y + plug->winatts.height + 2*plug->winatts.border_width);
    break;
  default:
    /* Some gravity we don't deal with... */
    gx = '+'; gy = '+';
    x = plug->winatts.x;
    y = plug->winatts.y;
    break;
  }
  
  sprintf(plug->geometry,"%dx%d%c%d%c%d",
	  plug->winatts.width,plug->winatts.height,gx,x,gy,y);
}

static void ComputePlugGeometry(plugPtr)
     Plug *plugPtr;
{
  int fwidth = Tk_Width(plugPtr->frame);
  int fheight = Tk_Height(plugPtr->frame);
  int width = plugPtr->winatts.width;
  int height = plugPtr->winatts.height;
  int px,py;
  int x,y;
  char gx,gy;

  /*
   * Compute the position this window will be placed into on
   * termination of the window manager.
   */
  
  Tk_GetRootCoords(plugPtr->frame, &x, &y);
  switch(plugPtr->normalhints.win_gravity) {
  case NorthWestGravity:
    gx = '+', gy = '+';
    px = x;
    py = y;
    plugPtr->fx = x;
    plugPtr->fy = y;
    break;
  case SouthWestGravity:
    gx = '+', gy = '-';
    px = x;
    py = screenHeight - (y + fheight);
    plugPtr->fx = x;
    plugPtr->fy = y + fheight - height - 2*plugPtr->winatts.border_width;
    break;
  case NorthEastGravity:
    gx = '-', gy = '+';
    px = screenWidth - (x + fwidth);
    py = y;
    plugPtr->fx = x + fwidth - width - 2*plugPtr->winatts.border_width;
    plugPtr->fy = y;
    break;
  case SouthEastGravity:
    gx = '-', gy = '-';
    px = screenWidth - (x + fwidth);
    py = screenHeight - (y + fheight);
    plugPtr->fx = x + fwidth - width - 2*plugPtr->winatts.border_width;
    plugPtr->fy = y + fheight - height - 2*plugPtr->winatts.border_width;
    break;
  default:
    gx = '+', gy = '+';
    px = x;
    py = y;
    plugPtr->fx = x;
    plugPtr->fy = y;
    break;
  }
  sprintf(plugPtr->geometry,"%dx%d%c%d%c%d",
	  plugPtr->winatts.width,plugPtr->winatts.height,gx,px,gy,py);
}

/*
 * Procedure to watch frame movement and resize events.
 */

static void
PlugPositionProc(clientData, eventPtr)
     ClientData clientData;	/* Information about window. */
     XEvent *eventPtr;		/* Information about event. */
{
  Plug *plugPtr = (Plug *) clientData;
  
  if (eventPtr->type == ConfigureNotify)
    ComputePlugGeometry(plugPtr);
}


/*
 *--------------------------------------------------------------
 *
 * PlugEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various
 *	events on plugs.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

static void
PlugEventProc(clientData, eventPtr)
     ClientData clientData;	/* Information about window. */
     XEvent *eventPtr;		/* Information about event. */
{
  Plug *plugPtr = (Plug *) clientData;
  
  if (eventPtr->type == Expose) {
    if ((eventPtr->xexpose.count == 0) && !plugPtr->updatePending) {
      Tk_DoWhenIdle(PlugDisplay, (ClientData) plugPtr);
      plugPtr->updatePending = 1;
    }
  } else if (eventPtr->type == ConfigureNotify) {
    int width = Tk_Width(plugPtr->tkwin);
    int height = Tk_Height(plugPtr->tkwin);
    
    /* Munge width and height to nearest smaller grid fit here (?) */

    if (plugPtr->winatts.width != width || plugPtr->winatts.height != height) {
      /* resized, feed back a real event */
      XWindowChanges chg;
      unsigned int value_mask;
      
      plugPtr->winatts.width = width;
      plugPtr->winatts.height = height;
      value_mask = CWWidth | CWHeight | CWX | CWY;
      chg.x = 0;
      chg.y = 0;
      chg.width = plugPtr->winatts.width;
      chg.height = plugPtr->winatts.height;
      XConfigureWindow(dpy, plugPtr->window, value_mask, &chg);
      ComputePlugGeometry(plugPtr);
    }
    
    if (!plugPtr->updatePending) {
      Tk_DoWhenIdle(PlugDisplay, (ClientData) plugPtr);
      plugPtr->updatePending = 1;
    }
  } else if (eventPtr->type == DestroyNotify) {
    /*
     * I should consider adding in a "destroyed" field to plugs
     * so that if a command is received after destruction it can
     * be ignored. I'm not sure that there can ever be a problem
     * the way things currently work, but better safe than sorry.
     */
    /* When a parent is destroyed there is a race between an unmap
       on the plug, and the real destroy event. How can I avoid this?
       */
    Tcl_DeleteCommand(plugPtr->interp, Tk_PathName(plugPtr->tkwin));
    /* Free the plug structure up and remove from context listings */
    /* If the plug is not already destroyed, then reparent it back
       to the root. */
    XDeleteContext(dpy,(XID)plugPtr->window,plugContext);
    XDeleteContext(dpy,(XID)plugPtr->tkwin,plugWinContext);
    if (plugPtr->tkwin != plugPtr->frame)
      XDeleteContext(dpy,(XID)plugPtr->frame,plugWinContext);
    if (!plugPtr->destroyed && plugPtr->reparented) {
      /*
       * Getting here indicates a destroy initiated by TCL code.
       * Return the window, make sure it is mapped to the screen,
       * and remove it from the save set.
       */
      XReparentWindow(dpy,plugPtr->window,rootWindow,plugPtr->fx,plugPtr->fy);
      XMapWindow(dpy,plugPtr->window);
      XRemoveFromSaveSet(dpy,plugPtr->window);
      XSetWindowBorderWidth(dpy, plugPtr->window, plugPtr->winatts.border_width);
      plugPtr->reparented = 0;
    }
    plugPtr->tkwin = NULL;
    /* Clear colormap focus if need be */
    if (colormapFocusPlug == plugPtr)
      InstallWindowColormaps(&rootcmap);
    /* Free this windows colormaps list */
    FreeColormapWindows(&plugPtr->cmaps);
    if (plugPtr->updatePending) {
      Tk_CancelIdleCall(PlugDisplay, (ClientData) plugPtr);
    }
    if (plugPtr->sendconfigPending) {
      Tk_CancelIdleCall(PlugSendConfig, (ClientData) plugPtr);
    }
    Tk_EventuallyFree((ClientData) plugPtr, PlugDestroy);
  } else if (eventPtr->type == MapNotify) {
    /* Map the plug child as well */
    XMapWindow(plugPtr->display,plugPtr->window);
    /* Change the focus to this window if it was the pending focus */
    plugPtr->ismapped = 1;
    if (pending_focus == plugPtr->window) {
      XSetInputFocus(dpy, plugPtr->window, RevertToPointerRoot, CurrentTime);
      EatFocusInEvents(dpy, plugPtr->window);
      pending_focus = None;
    }
  } else if (eventPtr->type == UnmapNotify) {
    /* Unmap the plug child as well, but make sure that we don't cause a delete */
    XSelectInput(dpy,Tk_WindowId(plugPtr->tkwin),SubstructureRedirectMask);
    XUnmapWindow(plugPtr->display,plugPtr->window);
    XSelectInput(dpy,Tk_WindowId(plugPtr->tkwin),SubstructureRedirectMask|SubstructureNotifyMask);
    plugPtr->ismapped = 0;
  }
}

/*
 *--------------------------------------------------------------
 *
 * PlugDisplay --
 *
 *	This procedure redraws the contents of a plug window.
 *	It is invoked as a do-when-idle handler, so it only runs
 *	when there's nothing else for the application to do.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information appears on the screen.
 *
 *--------------------------------------------------------------
 */

static void
PlugDisplay(clientData)
     ClientData clientData;	/* Information about window. */
{
  Plug *plugPtr = (Plug *) clientData;
  Tk_Window tkwin = plugPtr->tkwin;
  
  plugPtr->updatePending = 0;
  if (!Tk_IsMapped(tkwin)) {
    return;
  }

/*
 * What should I be doing here to make the matte show
 * through on shaped clients? I really don't understand
 * what is going on here.
 */

}

static void
PlugDestroy(clientData)
     ClientData clientData;	/* Info about plug widget. */
{
  Plug *plugPtr = (Plug *) clientData;
  
  ckfree((char *) plugPtr);
}


static int
PlugConfigure(interp, plugPtr, argc, argv, flags)
     Tcl_Interp *interp;         /* Used for error reporting. */
     register Plug *plugPtr;     /* Information about widget;  may or may
				  * not already have values for some fields. */
     int argc;                   /* Number of valid entries in argv. */
     char **argv;                /* Arguments. */
     int flags;                  /* Flags to pass to Tk_ConfigureWidget. */
{
  Tk_Window old_frame = plugPtr->frame;
  unsigned long data[2];
  
  if (Tk_ConfigureWidget(interp, plugPtr->tkwin, configSpecs,
			 argc, argv, (char *) plugPtr, flags) != TCL_OK) {
    return TCL_ERROR;
  }

  /*
   * Check to be sure the state is valid, and if so change the property.
   */

  if (strcmp(plugPtr->state,"normal") == 0) {
    data[0] = NormalState;
  } else if (strcmp(plugPtr->state,"iconic") == 0) {
    data[0] = IconicState;
  } else if (strcmp(plugPtr->state,"withdrawn") == 0) {
    data[0] = WithdrawnState;
  } else {
    Tcl_AppendResult(interp, "\"", plugPtr->state, "\" not a valid state. Must be one of: normal, iconic, or withdrawn", (char *) NULL);
    return TCL_ERROR;
  }
  data[1] = plugPtr->iconwin;
  XChangeProperty(dpy, plugPtr->window, _XA_WM_STATE, _XA_WM_STATE, 32,
                  PropModeReplace, (unsigned char *) data, 2);

  /*
   * Register griding if desired.
   */
  
  if (plugPtr->setGrid && !plugPtr->isgridded) {
    /* Make the underlying window request size 0x0 */
    Tk_GeometryRequest(plugPtr->tkwin, 0, 0);
    Tk_SetGrid(plugPtr->tkwin, 0, 0, 1, 1);
    plugPtr->isgridded = 1;
  }
  
  /*
   * Make "frame" a referent for the plug in tkwm commands that that want
   * the plug to watch to obtain its geometry on the screen.
   */
  
  if (plugPtr->frame != old_frame) {
    if (plugPtr->tkwin != old_frame)
      XDeleteContext(dpy,(XID)old_frame,plugWinContext);
    Tk_DeleteEventHandler(old_frame, StructureNotifyMask,
			  PlugPositionProc, (ClientData) plugPtr);
    if (plugPtr->tkwin != plugPtr->frame)
      XSaveContext(dpy,(XID)plugPtr->frame,plugWinContext, (caddr_t) plugPtr);
    Tk_CreateEventHandler(plugPtr->frame, StructureNotifyMask,
			  PlugPositionProc, (ClientData) plugPtr);
  }
  
  plugPtr->updatePending = 1;
  Tk_DoWhenIdle(PlugDisplay, (ClientData) plugPtr);
  
  return TCL_OK;
}

/*
 * Send a Configuration event to a client.
 */

static void PlugSendConfig(clientData)
ClientData clientData;
{
  XEvent event;
  Plug *plugPtr = (Plug *)clientData;
  
  Tk_GetRootCoords(plugPtr->tkwin, &plugPtr->winatts.x, &plugPtr->winatts.y);
  plugPtr->winatts.x -= plugPtr->winatts.border_width;
  plugPtr->winatts.y -= plugPtr->winatts.border_width;
  event.type = ConfigureNotify;
  event.xconfigure.display = dpy;
  event.xconfigure.event = plugPtr->window;
  event.xconfigure.window = plugPtr->window;
  event.xconfigure.x = plugPtr->winatts.x;
  event.xconfigure.y = plugPtr->winatts.y;
  event.xconfigure.width = plugPtr->winatts.width;
  event.xconfigure.height = plugPtr->winatts.height;
  event.xconfigure.border_width = plugPtr->winatts.border_width;
  event.xconfigure.above = Tk_WindowId(plugPtr->tkwin);
  event.xconfigure.override_redirect = False;
  XSendEvent(dpy, plugPtr->window, False, StructureNotifyMask, &event);
  plugPtr->sendconfigPending = 0;
}

static int
PlugWidgetCmd(clientData, interp, argc, argv)
     ClientData clientData;      /* Information about plug widget. */
     Tcl_Interp *interp;         /* Current interpreter. */
     int argc;                   /* Number of arguments. */
     char **argv;                /* Argument strings. */
{
  register Plug *plugPtr = (Plug *) clientData;
  int result = TCL_OK;
  int length;
  char c;
  
  if (argc < 2) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " option ?arg arg ...?\"", (char *) NULL);
    return TCL_ERROR;
  }
  Tk_Preserve((ClientData) plugPtr);
  c = argv[1][0];
  length = strlen(argv[1]);
  
  if (c == 'c' && strncmp(argv[1], "configure", length) == 0) {
    if (argc == 2) {
      result = Tk_ConfigureInfo(interp, plugPtr->tkwin, configSpecs,
				(char *) plugPtr, (char *) NULL, 0);
    } else if (argc == 3) {
      result = Tk_ConfigureInfo(interp, plugPtr->tkwin, configSpecs,
				(char *) plugPtr, argv[2], 0);
    } else {
      result = PlugConfigure(interp, plugPtr, argc-2, argv+2,
			     TK_CONFIG_ARGV_ONLY);
    }
  } else if (c == 'g' && strncmp(argv[1], "geometry", length) == 0) {
    if (argc == 2) {
      interp->result = plugPtr->geometry;
    } else if (argc == 3) {
      /* FIXME */
      Tcl_AppendResult(interp, "setting plug geometry not yet implemented",
		       (char *) NULL);
      result = TCL_ERROR;
    } else {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " geometry ?pos?\"", (char *) NULL);
      result = TCL_ERROR;
    }
  } else if (c == 'h' && strncmp(argv[1], "hasShape", length) == 0) {
	  return hasShape(interp, plugPtr);
  } else if (c == 'i' && strncmp(argv[1], "info", length) == 0) {
    if (argc == 3) {
      result = DoInfoCmd(interp, plugPtr,argv[2]);
    } else {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " info ?query?\"", (char *) NULL);
      result = TCL_ERROR;
    }
  } else if (c == 'd' && strncmp(argv[1], "delete", length) == 0) {
    int i;
    XClientMessageEvent ev;
    for (i = 0; i < plugPtr->num_protocols; i++)
      if (plugPtr->protocols[i] == _XA_WM_DELETE_WINDOW)
        goto delete_message;

    /* No delete protocol. Just kill the window. */
    if (XDestroyWindow(dpy,plugPtr->window) == 0) {
	Tcl_AppendResult(interp, "Destroy failed", (char *) NULL);
	result = TCL_ERROR;
    }
    goto done_delete;
delete_message:
    ev.type = ClientMessage;
    ev.window = plugPtr->window;
    ev.message_type = _XA_WM_PROTOCOLS;
    ev.format = 32;
    ev.data.l[0] = _XA_WM_DELETE_WINDOW;
    ev.data.l[1] = CurrentTime;
    if (XSendEvent (dpy, plugPtr->window, False, 0L, (XEvent *) &ev) == 0) {
      Tcl_AppendResult(interp, "failed to send message", (char *) NULL);
      result = TCL_ERROR;
    }
done_delete:;
  } else if (c == 's' && strncmp(argv[1], "save_yourself", length) == 0) {
    XClientMessageEvent ev;
    ev.type = ClientMessage;
    ev.window = plugPtr->window;
    ev.message_type = _XA_WM_PROTOCOLS;
    ev.format = 32;
    ev.data.l[0] = _XA_WM_SAVE_YOURSELF;
    ev.data.l[1] = CurrentTime;
    if (XSendEvent (dpy, plugPtr->window, False, 0L, (XEvent *) &ev) == 0) {
      Tcl_AppendResult(interp, "failed to send message", (char *) NULL);
      result = TCL_ERROR;
    }
  } else if (c == 's' && strncmp(argv[1], "shapify", length) == 0) {
	  return shapify(interp, plugPtr);
  } else if (c == 'u' && strncmp(argv[1], "unshapify", length) == 0) {
	  return unshapify(interp, plugPtr);
  } else if (c == 't' && strncmp(argv[1], "take_focus", length) == 0) {
    XClientMessageEvent ev;
    ev.type = ClientMessage;
    ev.window = plugPtr->window;
    ev.message_type = _XA_WM_PROTOCOLS;
    ev.format = 32;
    ev.data.l[0] = _XA_WM_TAKE_FOCUS;
    ev.data.l[1] = CurrentTime;
    if (XSendEvent (dpy, plugPtr->window, False, 0L, (XEvent *) &ev) == 0) {
      Tcl_AppendResult(interp, "failed to send message", (char *) NULL);
      result = TCL_ERROR;
    }
  } else if (c == 's' && strncmp(argv[1], "sendconfig", length) == 0) {
    if (argc == 2) {
      if (!plugPtr->sendconfigPending)
	Tk_DoWhenIdle(PlugSendConfig, (ClientData) plugPtr);
      plugPtr->sendconfigPending = 1;
    } else {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " sendconfig\"", (char *) NULL);
      result = TCL_ERROR;
    }
  } else {
    Tcl_AppendResult(interp, "bad option \"", argv[1],
		     "\":  must be configure",
		     (char *) NULL);
    result = TCL_ERROR;
  }
  
  Tk_Release((ClientData) plugPtr);
  return result;
}

SetInitialState(plug)
Plug *plug;
{
    Atom actual_type;
    int actual_format;
    unsigned long nitems, bytesafter;
    unsigned long *datap = NULL;
    Bool retval = False;

    if (XGetWindowProperty (dpy, plug->window, _XA_WM_STATE, 0L, 2L, False,
      _XA_WM_STATE, &actual_type, &actual_format, &nitems, &bytesafter,
      (unsigned char **) &datap) != Success || !datap) {
        plug->state = strdup("withdrawn");
	return;
    }
    switch (datap[0]) {
      case NormalState:
	plug->state = strdup("normal");
	break;
      case IconicState:
	plug->state = strdup("iconic");
	break;
      case WithdrawnState:
      default:
	plug->state = strdup("withdrawn");
	break;
    }
}

static int
PlugCmd(clientData, interp, argc, argv)
     ClientData clientData;      /* Main window associated with
				  * interpreter. */
     Tcl_Interp *interp;         /* Current interpreter. */
     int argc;                   /* Number of arguments. */
     char **argv;                /* Argument strings. */
{
  Plug *plug;
  Tk_Window main = (Tk_Window) clientData;
  Tk_Window tkwin;
  XWMHints *wmhints;
  int window;
  
  if (argc < 4) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " ", argv[1], " pathName win ?options?\"", (char *) NULL);
    return TCL_ERROR;
  }
  
  XGrabServer(dpy);
  sscanf(argv[3],"%i",&window);
  
  if (!WindowExists(window)) {
    Tcl_AppendResult(interp, "No window ", argv[3], "exists", (char *)NULL);
    XUngrabServer(dpy);
    return TCL_ERROR;
  }
  
  if (WindowPlug(window)) {
    Tcl_AppendResult(interp, "Window ", argv[3], "already managed", (char *)NULL);
    if (!ServerGrabbed) XUngrabServer(dpy);
    return TCL_ERROR;
  }
  
  tkwin = Tk_CreateWindowFromPath(interp, main, argv[2], (char *) NULL);
  if (tkwin == NULL) {
    if (!ServerGrabbed) XUngrabServer(dpy);
    return TCL_ERROR;
  }
  Tk_SetClass(tkwin, "Plug");
  
  /*
   * Allocate and initialize the widget record.
   */
  
  plug = (Plug *) ckalloc(sizeof(Plug));
  plug->tkwin = tkwin;
  plug->display = Tk_Display(tkwin);
  plug->window = window;
  plug->interp = interp;
  plug->background = NULL;
  plug->frame = tkwin;
  plug->destroyed = 0;
  plug->ismapped = 0;
  plug->updatePending = 0;
  plug->sendconfigPending = 0;
  plug->reparented = 0;
  plug->state = 0;
  plug->iconwin = 0;
  plug->isgridded = 0;
  SetInitialState(plug);

  XSaveContext(dpy,(XID)plug->window,plugContext, (caddr_t) plug);
  XSaveContext(dpy,(XID)plug->tkwin,plugWinContext, (caddr_t) plug);

  Tk_CreateEventHandler(plug->tkwin, ExposureMask|StructureNotifyMask,
			PlugEventProc, (ClientData) plug);
  Tk_CreateEventHandler(plug->frame, StructureNotifyMask,
			PlugPositionProc, (ClientData) plug);
  
  /*
   *  Get the client window hints structures.
   */
  
  XGetWindowAttributes(Tk_Display(tkwin), plug->window, &plug->winatts);

  wmhints = XGetWMHints(Tk_Display(tkwin), plug->window);
  if (wmhints != 0) {
    plug->WMhints = *wmhints;
    XFree((char *)wmhints);
  } else {
    plug->WMhints.flags = 0;
  }
  GetSizeHints(plug);
  PlugInitializeGeometry(plug);
  plug->name = 0;
  XFetchName(Tk_Display(tkwin), plug->window, &plug->name);
  plug->icon_name = 0;
  XGetIconName(Tk_Display(tkwin), plug->window, &plug->icon_name);
  plug->class.res_name = 0;
  plug->class.res_class = 0;
  XGetClassHint(Tk_Display(tkwin), plug->window, &plug->class);
  plug->transient_for = None;
  XGetTransientForHint(Tk_Display(tkwin),plug->window,&plug->transient_for);
  plug->argv = 0;
  plug->argc = 0;
  XGetCommand(dpy,plug->window,&plug->argv,&plug->argc);
  plug->client_machine.nitems = 0;
  plug->num_client_machine = 0;
  plug->client_machine_l = 0;
  XGetWMClientMachine(dpy,plug->window,&plug->client_machine);
  XTextPropertyToStringList(&plug->client_machine,&plug->client_machine_l,&plug->num_client_machine);
  plug->protocols = 0;
  plug->num_protocols = 0;
  XGetWMProtocols(dpy,plug->window,&plug->protocols,&plug->num_protocols);
  plug->cmaps.cwins = 0;
  plug->cmaps.number_cwins = 0;
  plug->cmaps.scoreboard = 0;

  
  /* Match the initial TK window geometry to the underlying window */
  Tk_GeometryRequest(tkwin,plug->winatts.width,plug->winatts.height);
  Tk_MakeWindowExist(tkwin);
  /* Reparent the client to the plug */
  XAddToSaveSet(dpy,plug->window);
  XSetWindowBorderWidth(dpy,plug->window,0);
  XReparentWindow(dpy,plug->window,Tk_WindowId(plug->tkwin),0,0);
  plug->reparented = 1;

  /* Must set intial selections before getting the colormap windows,
     since the colormap windows may change selections for the client! */
  XShapeSelectInput(dpy, plug->window, ShapeNotifyMask);
  XSelectInput(dpy,plug->window,PropertyChangeMask|FocusChangeMask);
  /* Idea: don't set redirect masks on internal windows??? */
  XSelectInput(dpy,Tk_WindowId(plug->tkwin),SubstructureRedirectMask|SubstructureNotifyMask);
  GetColormapWindows(plug->window,&plug->cmaps);

  if (!ServerGrabbed) XUngrabServer(dpy);
  
  Tcl_CreateCommand(interp, Tk_PathName(plug->tkwin),
		    PlugWidgetCmd, (ClientData) plug, (void (*)()) NULL);

  if (PlugConfigure(interp, plug, argc-4, argv+4, 0) != TCL_OK) {
    Tk_DestroyWindow(plug->tkwin);
    return TCL_ERROR;
  }
  
  interp->result = Tk_PathName(plug->tkwin);
  return TCL_OK;
}

/*
 * BINDINGS SUPPORT PROCEDURES.
 */

/*
 *--------------------------------------------------------------
 *
 * ExpandPercents --
 *
 *      Given a command and an event, produce a new command
 *      by replacing % constructs in the original command
 *      with information from the X event.
 *
 * Results:
 *      The return result is a pointer to the new %-substituted
 *      command.  If the command fits in the space at after, then
 *      the return value is after.  If the command is too large
 *      to fit at after, then the return value is a pointer to
 *      a malloc-ed buffer holding the command;  in this case it
 *      is the caller's responsibility to free up the buffer when
 *      finished with it.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

static char *
ExpandPercents(before, eventPtr, after, afterSize)
     register char *before;      /* Command containing percent
				  * expressions to be replaced. */
     register XEvent *eventPtr;  /* X event containing information
				  * to be used in % replacements. */
     char *after;                /* Place to generate new expanded
				  * command.  Must contain at least
				  * "afterSize" bytes of space. */
     int afterSize;              /* Number of bytes of space available at
				  * after. */
{
  register char *buffer;      /* Pointer to buffer currently being used
			       * as destination. */
  register char *dst;         /* Pointer to next place to store character
			       * in substituted string. */
  int spaceLeft;              /* Indicates how many more non-null bytes
			       * may be stored at *dst before space
			       * runs out. */
  int spaceNeeded, cvtFlags;  /* Used to substitute string as proper Tcl
			       * list element. */
  int number;
#define NUM_SIZE 40
  register char *string;
  char numStorage[NUM_SIZE+1];
  Plug *plug;
  
  dst = buffer = after;
  spaceLeft = afterSize - 1;
  while (*before != 0) {
    if (*before != '%') {
      
      /*
       * Expand the destination string if necessary.
       */
      
      if (spaceLeft <= 0) {
	char *newSpace;
	
	newSpace = (char *) ckalloc((unsigned) (2*afterSize));
	memcpy((VOID *) newSpace, (VOID *) buffer, afterSize);
	afterSize *= 2;
	dst = newSpace + (dst - buffer);
	if (buffer != after) {
	  ckfree(buffer);
	}
	buffer = newSpace;
	spaceLeft = afterSize - (dst-buffer) - 1;
      }
      *dst = *before;
      dst++;
      before++;
      spaceLeft--;
      continue;
    }
    number = 0;
    string = "??";
    switch (before[1]) {
    case 'x':
      if (eventPtr->xany.type == ConfigureRequest
	  && eventPtr->xconfigurerequest.value_mask & CWX) {
	number = eventPtr->xconfigurerequest.x;
	goto doNumber;
      }
      if (eventPtr->xany.type == shapeEventType) {
	      number = ((XShapeEvent *)eventPtr)->x;
	      goto doNumber;
      }
      break;
    case 'y':
      if (eventPtr->xany.type == shapeEventType) {
	      number = ((XShapeEvent *)eventPtr)->y;
	      goto doNumber;
      }
      if (eventPtr->xany.type == ConfigureRequest
	  && eventPtr->xconfigurerequest.value_mask & CWY) {
	number = eventPtr->xconfigurerequest.y;
	goto doNumber;
      }
      break;
    case 'w':
      if (eventPtr->xany.type == shapeEventType) {
	      number = ((XShapeEvent *)eventPtr)->width;
	      goto doNumber;
      }
      if (eventPtr->xany.type == ConfigureRequest
	  && XFindContext(dpy, eventPtr->xconfigurerequest.window,
			  plugContext, (caddr_t *)&plug) != XCNOENT
	  && eventPtr->xconfigurerequest.value_mask & (CWHeight | CWWidth)) {
	if (!(eventPtr->xconfigurerequest.value_mask & CWWidth))
	  number = plug->winatts.width;
	else
	  number = eventPtr->xconfigurerequest.width;
      	goto doNumber;
      }
      break;
    case 'h':
      if (eventPtr->xany.type == shapeEventType) {
	      number = ((XShapeEvent *)eventPtr)->height;
	      goto doNumber;
      }
      if (eventPtr->xany.type == ConfigureRequest
	  && XFindContext(dpy, eventPtr->xconfigurerequest.window,
                          plugContext, (caddr_t *)&plug) != XCNOENT
          && eventPtr->xconfigurerequest.value_mask & (CWHeight | CWWidth)) {
	if (!(eventPtr->xconfigurerequest.value_mask & CWHeight))
	  number = plug->winatts.height;
	else
	  number = eventPtr->xconfigurerequest.height;
      	goto doNumber;
      }
      break;
    case 'd':
      /* detail field in ConfigureRequest events */
      if (eventPtr->xany.type == shapeEventType) {
	      number = ((XShapeEvent *)eventPtr)->shaped;
	      goto doNumber;
      }
      if (eventPtr->xany.type == ConfigureRequest
	  && eventPtr->xconfigurerequest.value_mask & CWStackMode) {
	switch(eventPtr->xconfigurerequest.detail) {
	case Above:
	  string = "Above"; break;
	case Below:
	  string = "Below"; break;
	case TopIf:
	  string = "TopIf"; break;
	case BottomIf:
	  string = "BottomIf"; break;
	case Opposite:
	 string = "Opposite"; break;
        }
      }
      break;
    case 's':
      if (eventPtr->xany.type == shapeEventType) {
	      switch (((XShapeEvent *)eventPtr)->kind) {
	      case ShapeClip:
		      string = "ShapeClip";
		      break;
	      case ShapeBounding:
		      string = "ShapeBounding";
		      break;
	      }
      }
      break;
    case 'S':
      /* the sibling detail field in ConfigureRequest events */
      if (eventPtr->xany.type == ConfigureRequest
	  && eventPtr->xconfigurerequest.value_mask & CWStackMode) {
	Plug *sib;
        if (eventPtr->xconfigurerequest.above == None) {
	   string = "None";
           goto doString;
        } else {
	   sib = WindowPlug(eventPtr->xconfigurerequest.above);
	   if (sib)
	     number = eventPtr->xconfigurerequest.above;
           else
	     number = Tk_WindowId(sib->tkwin);
           goto doNumber;
        }
      }
      break;
    case 'e':
      number = eventPtr->xany.window;
      goto doNumber;
    case 'P':
      if (eventPtr->xany.type == PropertyNotify) {
	switch ((int)eventPtr->xproperty.atom) {
	case XA_WM_CLASS:
	  string = "WM_CLASS";
	  break;
	case XA_WM_TRANSIENT_FOR:
	  string = "WM_TRANSIENT_FOR";
	  break;
	case XA_WM_HINTS:
	  string = "WM_HINTS";
	  break;
	case XA_WM_ICON_NAME:
	  string = "WM_ICON_NAME";
	  break;
	case XA_WM_NAME:
	  string = "WM_NAME";
	  break;
	case XA_WM_ICON_SIZE:
	  string = "WM_ICON_SIZE";
	  break;
	case XA_WM_NORMAL_HINTS:
	  string = "WM_NORMAL_HINTS";
	  break;
	case XA_WM_SIZE_HINTS:
	  string = "WM_SIZE_HINTS";
	  break;
	case XA_WM_CLIENT_MACHINE:
	  string = "WM_CLIENT_MACHINE";
	  break;
	case XA_WM_COMMAND:
	  string = "WM_COMMAND";
	  break;
	default:
	  if (eventPtr->xproperty.atom == _XA_WM_COLORMAP_WINDOWS)
	    string = "WM_COLORMAP_WINDOWS";
	  else if (eventPtr->xproperty.atom == _XA_WM_PROTOCOLS)
	    string = "WM_PROTOCOLS";
	  else if (eventPtr->xproperty.atom == _XA_WM_STATE)
	    string = "WM_STATE";
	  break;
	}
      }
      break;
    case 'W':
      /* find the window that the event is inside of */
      if (XFindContext(dpy, eventPtr->xany.window, plugContext,
                       (caddr_t *)&plug) != XCNOENT) {
	string = Tk_PathName(plug->tkwin);
      }
      break;
    default:
      numStorage[0] = before[1];
      numStorage[1] = '\0';
      string = numStorage;
      break;
    }
  goto doString;
    
  doNumber:
    sprintf(numStorage, "%d", number);
    string = numStorage;
    
  doString:
    spaceNeeded = Tcl_ScanElement(string, &cvtFlags);
    if (spaceNeeded >= spaceLeft) {
      char *newSpace;
      
      newSpace = (char *) ckalloc((unsigned)
				  (afterSize + spaceNeeded + 50));
      memcpy((VOID *) newSpace, (VOID *) buffer, afterSize);
      afterSize += spaceNeeded + 50;
      dst = newSpace + (dst - buffer);
      if (buffer != after) {
	ckfree(buffer);
      }
      buffer = newSpace;
      spaceLeft = afterSize - (dst-buffer) - 1;
    }
    spaceNeeded = Tcl_ConvertElement(string, dst,
				     cvtFlags | TCL_DONT_USE_BRACES);
    dst += spaceNeeded;
    spaceLeft -= spaceNeeded;
    before += 2;
  }
  *dst = '\0';
  return buffer;
}

void doBinding(index,eventPtr)
     int index;
     XEvent *eventPtr;
{
  char *p;
  char shortSpace[200];
  int result;
  
  if (bindings[index].interp) {
      p = ExpandPercents(bindings[index].command,eventPtr,shortSpace,200);
      result = Tcl_GlobalEval(bindings[index].interp, p);
      if (p != shortSpace) {
        ckfree(p);
      }
  }
  /* What should I do with a TCL error here? Yipes!*/
}

/*
 * OUTLINE DRAWING SUPPORT PROCEDURES
 */

/* From fvwm */
static void DrawOutline()
{
  XRectangle rects[5];

  if (lastWidth || lastHeight) {
    rects[0].x = lastx;
    rects[0].y = lasty;
    rects[0].width = lastWidth;
    rects[0].height = lastHeight;
    rects[1].x = lastx+1;
    rects[1].y = lasty+1;
    rects[1].width = lastWidth-2;
    rects[1].height = lastHeight-2;
    rects[2].x = lastx+2;
    rects[2].y = lasty+2;
    rects[2].width = lastWidth-4;
    rects[2].height = lastHeight-4;
    rects[3].x = lastx+3;
    rects[3].y = lasty+3 + (lastHeight-6)/3;
    rects[3].width = lastWidth-6;
    rects[3].height = (lastHeight-6)/3;
    rects[4].x = lastx+3 + (lastWidth-6)/3;
    rects[4].y = lasty+3;
    rects[4].width = (lastWidth-6)/3;
    rects[4].height = (lastHeight-6);
    XDrawRectangles(dpy,rootWindow,DrawGC,rects,5);
  }
}

static void
MoveOutline(clientData)
     ClientData clientData;	/* Information about window. */
{
  struct TkwmOutline *Outline = (struct TkwmOutline *)clientData;
  int x = Outline->x, y = Outline->y, width = Outline->w, height = Outline->h;
  
  outlinepending = 0;
  
  if (x == lastx && y == lasty && width == lastWidth && height == lastHeight)
    return;
  
  /* undraw the old one, if any */
  DrawOutline();
  
  lastx = x;
  lasty = y;
  lastWidth = width;
  lastHeight = height;
  
  /* draw the new one, if any */
  DrawOutline();
}


/*
 * COLORMAP SUPPORT PROCEDURES.
 */

static void FreeColormapWindows(cmaps)
Colormaps *cmaps;
{
  XWindowAttributes atts;
  int i;

  if (cmaps->scoreboard)
    free(cmaps->scoreboard);
  for (i = 0; i < cmaps->number_cwins; i++) {
    cmaps->cwins[i]->refcnt--;
    if (cmaps->cwins[i]->refcnt == 0) {
      int dummy;

      /* Decrement ref count on colormap, and free if ok */
      cmaps->cwins[i]->colormap->refcnt--;
#ifdef DEBUG_COLORMAPS
      printf("colormap %d, refcnt = %d\n",cmaps->cwins[i]->colormap,cmaps->cwins[i]->colormap->refcnt);
#endif
      if (cmaps->cwins[i]->colormap->refcnt == 0) {
	XDeleteContext(dpy, cmaps->cwins[i]->colormap->c, colormapContext);
#ifdef DEBUG_COLORMAPS
        printf("Removed %d from context\n",cmaps->cwins[i]->colormap->c);
#endif
	free((char *)cmaps->cwins[i]->colormap);
      }
      XDeleteContext(dpy, cmaps->cwins[i]->w, colormapContext);
#ifdef DEBUG_COLORMAPS
      printf("Removed %d from context\n",cmaps->cwins[i]->w);
#endif
      ckfree((char *)cmaps->cwins[i]);

      if (XFindContext(dpy, cmaps->cwins[i]->w, tkWindowContext, (caddr_t *)&dummy) != XCNOENT
	  && XGetWindowAttributes(dpy, cmaps->cwins[i]->w, &atts)
	  && (atts.your_event_mask & (ColormapChangeMask|VisibilityChangeMask))) {
	/* Establish that we are no longer interested in events */
#ifdef DEBUG_COLORMAPS
        printf("Clearing watch on window %d\n",cmaps->cwins[i]->w);
#endif
	XSelectInput(dpy, cmaps->cwins[i]->w, atts.your_event_mask &
		     ~(ColormapChangeMask|VisibilityChangeMask));
      }
    }
  }
  if (cmaps->cwins)
    ckfree(cmaps->cwins);
}

/*
 * Get the WMColormapWindows property and update the colormap structs
 * as needed.
 */

static void GetColormapWindows(window,cmaps)
     Window window;
     Colormaps *cmaps;
{
  XWindowAttributes atts;
  Colormaps oldcmaps;
  ColormapWindow **new = 0;
  ColormapWindow *cwin;
  Window *windows;
  int num;
  int i,j;
  int found;
  
  windows = 0;
  num = 0;
  XGetWMColormapWindows(dpy,window,&windows,&num);
  
  oldcmaps = *cmaps;
  
  /* check if the plug itself is in the list of windows */
  found = -1;
  for (i = 0; i < num; i++)
    if (windows[i] == window)
      found = i;
  
  if (num > 0 || found == -1) {
    new = (ColormapWindow **)ckalloc(sizeof(ColormapWindow *)*(num+((found == -1)?1:0)));
    for (i = 0; i < num; i++) {
      if (XFindContext(dpy, windows[i], colormapContext, (caddr_t *)&cwin) == XCNOENT)
        cwin = NewColormapWindow(windows[i]);
      if (cwin)
        cwin->refcnt++;
      new[i+((found==-1)?1:0)] = cwin;
    }
    
    /* If the plug is not in the list, add it to the front */
    if (found == -1) {
      if (XFindContext(dpy, window, colormapContext, (caddr_t *)&cwin) == XCNOENT)
	cwin = NewColormapWindow(window);
      if (cwin)
        cwin->refcnt++;
      new[0] = cwin;
      num++;
    }
  }
  
  /* squeeze out NULL entries (windows that were deleted on us) */
  for (j = 0, i = 0; i < num; i++) {
    if (new[i] != 0)
      new[j++] = new[i];
  }
  num = j;
  
  if (windows) XFree((char *)windows);
  
#ifdef DEBUG_COLORMAPS
  printf("Num = %d\n",num);
#endif
  cmaps->number_cwins = num;
  cmaps->cwins = new;
  cmaps->scoreboard = (char *) calloc(1, ColormapsScoreboardLength(cmaps));
  
  FreeColormapWindows(&oldcmaps);
}

TkwmColormap *
CreateTkwmColormap(c)
     Colormap c;
{
  TkwmColormap *cmap;
  cmap = (TkwmColormap *) malloc(sizeof(TkwmColormap));
  if (!cmap || XSaveContext(dpy, c, colormapContext, (caddr_t) cmap)) {
    if (cmap) free((char *) cmap);
    return (NULL);
  }
#ifdef DEBUG_COLORMAPS
  printf("Added %d to context\n",c);
#endif
  cmap->c = c;
  cmap->state = 0;
  cmap->install_req = 0;
  cmap->w = None;
  cmap->refcnt = 1;
  return (cmap);
}

static ColormapWindow *NewColormapWindow(win)
     Window win;
{
  XWindowAttributes atts;
  TkwmColormap *cmap;
  ColormapWindow *new = (ColormapWindow *)ckalloc(sizeof(ColormapWindow));
  
  if (new) {
    if (!XGetWindowAttributes(dpy, win, &atts) ||
	XSaveContext(dpy, win, colormapContext, (caddr_t) new)) {
      ckfree((char *)new);
      return (NULL);
    }
#ifdef DEBUG_COLORMAPS
    printf("Added %d to context\n",win);
#endif
    
    if (XFindContext(dpy, atts.colormap,  colormapContext,
		     (caddr_t *)&new->colormap) == XCNOENT) {
      new->colormap = cmap = CreateTkwmColormap(atts.colormap);
      if (!cmap) {
        XDeleteContext(dpy, win, colormapContext);
#ifdef DEBUG_COLORMAPS
        printf("Removed %d from context\n",win);
#endif
        free((char *) new);
        return (NULL);
      }
    } else {
      new->colormap->refcnt++;
    }
    
    new->w = win;
    /* Initially we must assume the window is visible, since we can't check */
    new->visibility = VisibilityUnobscured;
    new->refcnt = 0;
    
    /* Establish that we are interested in events on this window */
    if ((atts.your_event_mask & (ColormapChangeMask|VisibilityChangeMask))
	!= (ColormapChangeMask|VisibilityChangeMask))
#ifdef DEBUG_COLORMAPS
      printf("Setting watch on window %d\n",win);
#endif
    XSelectInput(dpy, win, atts.your_event_mask
		 | ColormapChangeMask|VisibilityChangeMask);
  }
  return new;
}

static void InstallWindowColormaps(change)
     Colormaps *change;
{
  int i, j, n, number_cwins, state;
  ColormapWindow **cwins, *cwin, **maxcwin = NULL;
  TkwmColormap *cmap;
  char *row, *scoreboard;
  
  if (change) {
#ifdef DEBUG_COLORMAPS
    printf("New colormap windows list %d\n",change);
#endif
    /* Don't reload the current colormap */
    if (cmapInfo.cmaps == change)
      return;
    if (cmapInfo.cmaps)
      for (i = cmapInfo.cmaps->number_cwins,
	   cwins = cmapInfo.cmaps->cwins; i-- > 0; cwins++)
	(*cwins)->colormap->state &= ~CM_INSTALLABLE;
    cmapInfo.cmaps = change;
  }
  
#ifdef DEBUG_COLORMAPS
  printf("Doing colormap install\n");
#endif
  number_cwins = cmapInfo.cmaps->number_cwins;
  cwins = cmapInfo.cmaps->cwins;
  scoreboard = cmapInfo.cmaps->scoreboard;
  
  ColortableThrashing = 0; /* in case installation aborted */
  
  state = CM_INSTALLED;
  
  for (i = n = 0; i < number_cwins; i++) {
    cwin = cwins[i];
    cmap = cwin->colormap;
    cmap->state |= CM_INSTALLABLE;
    cmap->state &= ~CM_INSTALL;
    cmap->w = cwin->w;
  }
  
  for (i = n = 0; i < number_cwins; i++) {
    cwin = cwins[i];
    cmap = cwin->colormap;
    if (cwin->visibility != VisibilityFullyObscured &&
	n < cmapInfo.maxCmaps) {
      row = scoreboard + (i*(i-1)/2);
      for (j = 0; j < i; j++)
	if (row[j] && (cwins[j]->colormap->state & CM_INSTALL))
	  break;
      if (j != i)
	continue;
      n++;
      maxcwin = &cwins[i];
      state &= (cmap->state & CM_INSTALLED);
      cmap->state |= CM_INSTALL;
    }
  }
  
  cmapInfo.first_req = NextRequest(dpy);
  
  for ( ; n > 0; maxcwin--) {
    cmap = (*maxcwin)->colormap;
    if (cmap->state & CM_INSTALL) {
      cmap->state &= ~CM_INSTALL;
      if (!(state & CM_INSTALLED)) {
	cmap->install_req = NextRequest(dpy);
#ifdef DEBUG_COLORMAPS
	printf("XInstallColormap %d\n",cmap->c);
#endif
	XInstallColormap(dpy, cmap->c);
      }
      cmap->state |= CM_INSTALLED;
      n--;
    }
  }
}

/* CORE COMMANDS (other than the PLUG widget) */

/*
 * Tkwm core command driver.
 */

int TkwmCmd(clientData, interp, argc, argv)
     ClientData clientData;      /* Main window associated with
				  * interpreter. */
     Tcl_Interp *interp;         /* Current interpreter. */
     int argc;                   /* Number of arguments. */
     char **argv;                /* Argument strings. */
{
  char c;
  int length;
  
  if (argc < 2) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " command ?options?\"", (char *) NULL);
    return TCL_ERROR;
  }
  
  c = argv[1][0];
  length = strlen(argv[1]);
  if (c == 'b' && strncmp(argv[1],"bind",length) == 0) {
    return BindCmd(clientData, interp, argc, argv); 
  } else if (c == 'o' && strncmp(argv[1],"outline",length) == 0) {
    return OutlineCmd(clientData, interp, argc, argv); 
  } else if (c == 'f' && strncmp(argv[1],"focus",length) == 0) {
    return FocusCmd(clientData, interp, argc, argv); 
  } else if (c == 'c' && strncmp(argv[1],"colormap",length) == 0) {
    return ColormapCmd(clientData, interp, argc, argv); 
  } else if (c == 'c' && strncmp(argv[1],"compressMotion",length) == 0) {
    return compressMotion(clientData, interp, argc, argv); 
  } else if (c == 'p' && strncmp(argv[1],"pointer",length) == 0) {
    return PointerCmd(clientData, interp, argc, argv); 
  } else if (c == 'r' && strncmp(argv[1],"restack",length) == 0) {
    return RestackCmd(clientData, interp, argc, argv); 
  } else if (c == 'r' && strncmp(argv[1],"reparent",length) == 0) {
    return ReparentCmd(clientData, interp, argc, argv); 
  } else if (c == 'g' && strncmp(argv[1],"grabserver",length) == 0) {
    return GrabserverCmd(clientData, interp, argc, argv); 
  } else if (c == 'u' && strncmp(argv[1],"ungrabserver",length) == 0) {
    return GrabserverCmd(clientData, interp, argc, argv); 
  } else if (c == 'g' && strncmp(argv[1],"grabpointer",length) == 0) {
    return GrabpointerCmd(clientData, interp, argc, argv); 
  } else if (c == 'u' && strncmp(argv[1],"ungrabpointer",length) == 0) {
    return GrabpointerCmd(clientData, interp, argc, argv); 
  } else if (c == 'g' && strncmp(argv[1],"grabevent",length) == 0) {
    return GrabeventCmd(clientData, interp, argc, argv); 
  } else if (c == 'u' && strncmp(argv[1],"ungrabevent",length) == 0) {
    return GrabeventCmd(clientData, interp, argc, argv); 
  } else if (c == 'i' && strncmp(argv[1],"iconsize",length) == 0) {
    return IconSizeCmd(clientData, interp, argc, argv); 
  } else if (c == 'i' && strncmp(argv[1],"info",length) == 0) {
    return InfoCmd(clientData, interp, argc, argv); 
  } else if (c == 'u' && strncmp(argv[1],"unmanaged",length) == 0) {
    if (argc == 2) {
      return UnmanagedCmd(clientData,interp);
    } else {
      Tcl_AppendResult(interp, "\"tkwm unmanaged\" takes no arguments", 
		       (char *) NULL);
      return TCL_ERROR;
    }
  } else if (c == 'p' && strncmp(argv[1],"plug",length) == 0) {
    return PlugCmd(clientData, interp, argc, argv); 
  } else {
    Tcl_AppendResult(interp, "bad command \"", argv[1],
		     "\": must be one of bind, outline, restack, focus, colormap, pointer, iconsize, grabkey, ungrabkey, grabbutton, ungrabbutton, unmanaged, info, or plug", (char *) NULL);
    return TCL_ERROR;
  }
}

static int
FocusCmd(clientData, interp, argc, argv)
     ClientData clientData;      /* Main window associated with
				  * interpreter. */
     Tcl_Interp *interp;         /* Current interpreter. */
     int argc;                   /* Number of arguments. */
     char **argv;                /* Argument strings. */
{
  static char *focuswin = 0;
  Plug *plug;
  Tk_Window tkwin = (Tk_Window) clientData;
  Tk_Window new;
  
  if (argc < 2 || argc > 3) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " ", argv[1], " ?focusWindow?\"", (char *) NULL);
    return TCL_ERROR;
  }
  
  if (argc == 2) {
    interp->result = focuswin;
    return TCL_OK;
  } else {
    if (strncmp(argv[2],"None",strlen(argv[2])) == 0) {
      XSetInputFocus(dpy, PointerRoot, RevertToPointerRoot, CurrentTime);
      if (focuswin) free(focuswin);
      focuswin = strdup(argv[2]);
      pending_focus = None;
    } else {
      if ((new = Tk_NameToWindow(interp, argv[2], tkwin)) == NULL)
        return TCL_ERROR;
      
      if ((plug = TkWinPlug(new)) == NULL) {
        Tcl_AppendResult(interp, argv[2], " not a plug window", (char *) NULL);
        return TCL_ERROR;
      }
      
      if (focuswin) free(focuswin);
      focuswin = strdup(argv[2]);
      if (plug->ismapped) {
	XSetInputFocus(dpy, plug->window, RevertToPointerRoot, CurrentTime);
	EatFocusInEvents(dpy, plug->window);
      } else
      	pending_focus = plug->window;
    }
  }
  return TCL_OK;
}

/*
 *      This procedure is called to eliminate at most one FocusIn event
 *      in a queue that has a window number matching the named window.
 *      DispPtr's display gets sync-ed, and some of the events get
 *      removed from its queue.  Unaffected events are initially
 *      removed from the queue but they are eventually put back again
 *      in the right order.
 */

static void
EatFocusInEvents(dpy, window)
    Display *dpy;               /* Display from which to consume events. */
    Window window;		/* Window for which to eat a focus in */
{
    int numEvents, i, once;
    XEvent *events, *eventPtr;

    XSync(dpy, False);
    numEvents = QLength(dpy);
    if (numEvents == 0) {
        return;
    }
    events = (XEvent *) ckalloc((unsigned) (numEvents * sizeof(XEvent)));
    for (i = 0; i < numEvents; i++) {
        XNextEvent(dpy, &events[i]);
    }
    once = 0;
    for (i = numEvents-1, eventPtr = &events[i]; i >= 0; i--, eventPtr--) {
        if (once || eventPtr->type != FocusIn
	    || eventPtr->xfocus.window != window) {
          if (eventPtr->type == FocusIn && eventPtr->xfocus.window == window)
		once = 1;
          XPutBackEvent(dpy, eventPtr);
        }
    }
    ckfree((char *) events);
}

static int
ColormapCmd(clientData, interp, argc, argv)
     ClientData clientData;      /* Main window associated with
				  * interpreter. */
     Tcl_Interp *interp;         /* Current interpreter. */
     int argc;                   /* Number of arguments. */
     char **argv;                /* Argument strings. */
{
  Tk_Window tkwin = (Tk_Window) clientData;
  Tk_Window new;
  Plug *plug;
  
  if (argc < 2 || argc > 3) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " ", argv[1], " ?colormapWindow?\"", (char *) NULL);
    return TCL_ERROR;
  }
  
  if (argc == 3) {
    if ((new = Tk_NameToWindow(interp, argv[2], tkwin)) == NULL)
      return TCL_ERROR;
    colormapwin = new;
    /* Check if this is a plug or plug's frame, if so install the
       colormap list for the corresponding plug, otherwise install
       the colormap corresponding to the TK window */
    if (plug = TkWinPlug(new)) {
      colormapFocusPlug = plug;
      InstallWindowColormaps(&plug->cmaps);
    } else {
      colormapFocusPlug = 0;
      InstallWindowColormaps(&rootcmap);
    }
  }

  interp->result = Tk_PathName(colormapwin);
  return TCL_OK;
}

static int
PointerCmd(clientData, interp, argc, argv)
     ClientData clientData;      /* Main window associated with
				  * interpreter. */
     Tcl_Interp *interp;         /* Current interpreter. */
     int argc;                   /* Number of arguments. */
     char **argv;                /* Argument strings. */
{
  if (argc != 2 && argc != 5) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " ", argv[1], " ?[absolute|relative] x y?\"", (char *) NULL);
    return TCL_ERROR;
  }
  
  if (argc == 2) {
    int x, y, dummy;
    static char buffer[20];
    XQueryPointer(dpy, rootWindow, (Window *)&dummy, (Window *)&dummy, &x,
        &y, &dummy, &dummy, (unsigned int *)&dummy);
    sprintf(buffer,"%d %d",x,y);
    interp->result = buffer;
  } else {
    int x,y;
    int len;
    Window dest;
    
    len = strlen(argv[2]);
    if (strncmp(argv[2],"absolute",len) == 0 && len >= 3) {
      dest = rootWindow;
    } else if (strncmp(argv[2],"relative",len) == 0 && len >= 3) {
      dest = None;
    } else {
      Tcl_AppendResult(interp,"positioning argument \"", argv[2],
                       "\" must be one of: absolute or relative",
                       (char *) NULL);
      return TCL_ERROR;
    }

    /* warp the pointer to this location??? */
    if (Tcl_GetInt(interp,argv[3],&x) == TCL_ERROR) {
      Tcl_AppendResult(interp,"x argument \"", argv[3],
                       "\" is not an integer", (char *) NULL);
      return TCL_ERROR;
    }
    if (Tcl_GetInt(interp,argv[4],&y) == TCL_ERROR) {
      Tcl_AppendResult(interp,"y argument \"", argv[4],
                       "\" is not an integer", (char *) NULL);
      return TCL_ERROR;
    }
    XWarpPointer(dpy, None, dest, 0, 0, 0, 0, x, y);
  }
  return TCL_OK;
}

static int
RestackCmd(clientData, interp, argc, argv)
     ClientData clientData;      /* Main window associated with
				  * interpreter. */
     Tcl_Interp *interp;         /* Current interpreter. */
     int argc;                   /* Number of arguments. */
     char **argv;                /* Argument strings. */
{
  Tk_Window tkwin = (Tk_Window) clientData;
  Tk_Window new;
  XWindowChanges changes;
  int mask;
  
  if (argc != 5) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " ", argv[1], " pathName stackingmode sibling\"",
                     (char *) NULL);
    return TCL_ERROR;
  }

  if ((new = Tk_NameToWindow(interp, argv[2], tkwin)) == NULL)
    return TCL_ERROR;
  if (!Tk_IsTopLevel(new)) {
    Tcl_AppendResult(interp, "\"", argv[2], "\" not a toplevel window",
                     (char *) NULL);
    return TCL_ERROR;
  }
  if (strcmp(argv[4],"None") == 0) {
     changes.sibling = None;
     mask = CWStackMode;
  } else {
     if (Tcl_GetInt(interp,argv[4],(int *)&changes.sibling) == TCL_ERROR) {
         Tcl_AppendResult(interp,"sibling argument \"", argv[4],
                          "\" is not an integer", (char *) NULL);
         return TCL_ERROR;
     }
     mask = CWStackMode|CWSibling;
  }
  if (strcmp(argv[3],"Above") == 0) {
    changes.stack_mode = Above;
  } else if (strcmp(argv[3],"Below") == 0) {
    changes.stack_mode = Below;
  } else if (strcmp(argv[3],"TopIf") == 0) {
    changes.stack_mode = TopIf;
  } else if (strcmp(argv[3],"BottomIf") == 0) {
    changes.stack_mode = BottomIf;
  } else if (strcmp(argv[3],"Opposite") == 0) {
    changes.stack_mode = Opposite;
  } else {
    Tcl_AppendResult(interp,"\"", argv[3], "\" is not a valid stacking order.",
                     " Must be one of: Above, Below, TopIf, BottomIf,",
                     " or Opposite", (char *) NULL);
    return TCL_ERROR;
  }

  if (XConfigureWindow(Tk_Display(new), Tk_WindowId(new), mask, &changes) == 0) {
    Tcl_AppendResult(interp,"stacking request failed", (char *) NULL);
    return TCL_ERROR;
  }
  
  return TCL_OK;
}

static int
ReparentCmd(clientData, interp, argc, argv)
     ClientData clientData;      /* Main window associated with
				  * interpreter. */
     Tcl_Interp *interp;         /* Current interpreter. */
     int argc;                   /* Number of arguments. */
     char **argv;                /* Argument strings. */
{
  Tk_Window tkwin = (Tk_Window) clientData;
  Tk_Window child;
  Tk_Window parent;
  XWindowChanges changes;
  int mask;
  
  if (argc != 4) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " ", argv[1], " childSpec parentSpec\"",
                     (char *) NULL);
    return TCL_ERROR;
  }

  if ((child = Tk_NameToWindow(interp, argv[2], tkwin)) == NULL)
    return TCL_ERROR;
  if (!Tk_IsTopLevel(child)) {
    Tcl_AppendResult(interp, "\"", argv[2], "\" not a toplevel window",
                     (char *) NULL);
    return TCL_ERROR;
  }
  if ((parent = Tk_NameToWindow(interp, argv[3], tkwin)) == NULL)
    return TCL_ERROR;

  if (Tk_Display(child) != Tk_Display(parent)) {
    Tcl_AppendResult(interp, "parent and child not on same display", (char *) NULL);
    return TCL_ERROR;
  }

  if (XReparentWindow(Tk_Display(child), Tk_WindowId(child), Tk_WindowId(parent),0,0) == 0) {
    Tcl_AppendResult(interp,"reparenting request failed", (char *) NULL);
    return TCL_ERROR;
  }
  
  /* Set the __WM_ROOT property */

  XChangeProperty(Tk_Display(child), Tk_WindowId(child),
	_XA_WM_ROOT, XA_WINDOW, 32,  PropModeReplace, (unsigned char *)&Tk_WindowId(parent), 1);

  return TCL_OK;
}

static int
BindCmd(clientData, interp, argc, argv)
     ClientData clientData;      /* Main window associated with
				  * interpreter. */
     Tcl_Interp *interp;         /* Current interpreter. */
     int argc;                   /* Number of arguments. */
     char **argv;                /* Argument strings. */
{
  int i;

  /* FIXME */
  /*
   * I should add reporting of the currently bound events and their
   * current bindings.
   */

  if (argc != 4) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " ", argv[1], " event command\"", (char *) NULL);
    return TCL_ERROR;
  }
  for (i = 0; bindings[i].name; i++) {
    if (strcmp(bindings[i].name,argv[2]) == 0) {
      /* should I be copying argv[3]? YES! */
      if (bindings[i].command != 0 && bindings[i].command[0] != 0)
	free(bindings[i].command);
      bindings[i].command = strdup(argv[3]);
      bindings[i].interp = interp;
      return TCL_OK;
    }
  }
  Tcl_AppendResult(interp, "\"", argv[2],
		   "\" not a valid event: should be one of <Iconify>,",
		   " <MapRequest>, <Property>, <Configure>", (char *) NULL);
  return TCL_ERROR;
}

static int
OutlineCmd(clientData, interp, argc, argv)
     ClientData clientData;      /* Main window associated with
				  * interpreter. */
     Tcl_Interp *interp;         /* Current interpreter. */
     int argc;                   /* Number of arguments. */
     char **argv;                /* Argument strings. */
{
  static struct TkwmOutline outline;

  if (argc != 6) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " ", argv[1], " x y width height\"",
		     (char *) NULL);
    return TCL_ERROR;
  }

  if (Tcl_GetInt(interp,argv[2],&outline.x) == TCL_ERROR) {
    Tcl_AppendResult(interp,"x argument \"", argv[2],
                     "\" is not an integer", (char *) NULL);
    return TCL_ERROR;
  }
  if (Tcl_GetInt(interp,argv[3],&outline.y) == TCL_ERROR) {
    Tcl_AppendResult(interp,"y argument \"", argv[3],
                     "\" is not an integer", (char *) NULL);
    return TCL_ERROR;
  }
  if (Tcl_GetInt(interp,argv[4],&outline.w) == TCL_ERROR) {
    Tcl_AppendResult(interp,"width argument \"", argv[4],
                     "\" is not an integer", (char *) NULL);
    return TCL_ERROR;
  }
  if (Tcl_GetInt(interp,argv[5],&outline.h) == TCL_ERROR) {
    Tcl_AppendResult(interp,"height argument \"", argv[5],
                     "\" is not an integer", (char *) NULL);
    return TCL_ERROR;
  }
  if (outlinepending)
    Tk_CancelIdleCall(MoveOutline, (ClientData) &outline);
  Tk_DoWhenIdle(MoveOutline, (ClientData) &outline);
  outlinepending = 1;
  return TCL_OK;
}

static int
GrabserverCmd(clientData, interp, argc, argv)
     ClientData clientData;      /* Main window associated with
				  * interpreter. */
     Tcl_Interp *interp;         /* Current interpreter. */
     int argc;                   /* Number of arguments. */
     char **argv;                /* Argument strings. */
{
  if (argc != 2) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " ", argv[1], (char *) NULL);
    return TCL_ERROR;
  }

  if (strncmp("grabserver",argv[1],strlen(argv[1])) == 0) {
    XGrabServer(dpy);
    ServerGrabbed = 1;
  } else {
    XUngrabServer(dpy);
    ServerGrabbed = 0;
  }
  return TCL_OK;
}

static int
GrabpointerCmd(clientData, interp, argc, argv)
     ClientData clientData;      /* Main window associated with
                                  * interpreter. */
     Tcl_Interp *interp;         /* Current interpreter. */
     int argc;                   /* Number of arguments. */
     char **argv;                /* Argument strings. */
{
  Tk_Window tkwin = (Tk_Window) clientData;
  Tk_Window main = Tk_MainWindow(interp);
  static Cursor current_cursor = None;
  
  
  /* FIXME: add in code to allow a cursor to be specified by the grab. */
  if (strncmp("grabpointer",argv[1],strlen(argv[1])) == 0) {
    if (!(argc == 2 || (argc == 4 && strcmp(argv[2],"-cursor")==0))) {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		       argv[0], " ", argv[1], " ?-cursor cursorName?\"", (char *) NULL);
      return TCL_ERROR;
    }
    if (argc == 4) {
      if ((current_cursor = Tk_GetCursor(interp, main, argv[3])) == None)
	return TCL_ERROR;
    } else {
      current_cursor = None;
    }
    if (XGrabPointer(dpy, rootWindow, False,
	ButtonPressMask|ButtonReleaseMask|ButtonMotionMask|PointerMotionMask,
	GrabModeAsync, GrabModeAsync, None, current_cursor, CurrentTime)) {
      Tcl_AppendResult(interp, "XGrabPointer failed", (char *) NULL);
      return TCL_ERROR;
    }
  } else {
    if (current_cursor != None) {
	Tk_FreeCursor(dpy, current_cursor);
    }
    if (argc != 2) {
      Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " ", argv[1], "\"", (char *) NULL);
      return TCL_ERROR;
    }
    if (!XUngrabPointer(dpy, CurrentTime)) {
      Tcl_AppendResult(interp, "XUngrabPointer failed", (char *) NULL);
      return TCL_ERROR;
    }
  }
  return TCL_OK;
}

/* The following code is borrowed from tkBind.c (highly modified) */

static char *
GetField(p, copy, size)
     register char *p;           /* Pointer to part of pattern. */
     register char *copy;        /* Place to copy field. */
     int size;                   /* Maximum number of characters to
				  * copy. */
{
  while ((*p != '\0') && !isspace(((unsigned char)*p)) && (*p != '>')
	 && (*p != '-') && (size > 1)) {
    *copy = *p;
    p++;
    copy++;
    size--;
  }
  *copy = '\0';
  return p;
}

static int
GrabeventCmd(clientData, interp, argc, argv)
     ClientData clientData;      /* Main window associated with
                                  * interpreter. */
     Tcl_Interp *interp;         /* Current interpreter. */
     int argc;                   /* Number of arguments. */
     char **argv;                /* Argument strings. */
{
  static int initialized = 0;
  Tk_Window tkwin = (Tk_Window) clientData;
  Tk_Window new;
  Plug *plug;
  int type;
  KeyCode keycode = 0;
  int button = 0;
  unsigned int modifiers = 0;
  int any = 0;
  register Tcl_HashEntry *hPtr;
  register ModInfo *modPtr;
  int dummy;
  char *p;
#define FIELD_SIZE 20
  char field[FIELD_SIZE];
  
  
  if (argc != 4) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " ", argv[1], " pathName <event> \"", (char *) NULL);
    return TCL_ERROR;
  }
  
  if ((new = Tk_NameToWindow(interp, argv[2], tkwin)) == NULL)
    return TCL_ERROR;
  
  plug = TkWinPlug(new);

  if (!initialized) {
    
    initialized = 1;
    
    Tcl_InitHashTable(&modTable, TCL_STRING_KEYS);
    for (modPtr = modArray; modPtr->name != NULL; modPtr++) {
      hPtr = Tcl_CreateHashEntry(&modTable, modPtr->name, &dummy);
      Tcl_SetHashValue(hPtr, modPtr);
    }
  }
  
  p = argv[3];
  
  if (*p != '<') {
    interp->result = "missing \"<\" in event description";
    return TCL_ERROR;
  }
  p++;
  while (1) {
    p = GetField(p, field, FIELD_SIZE);
    hPtr = Tcl_FindHashEntry(&modTable, field);
    if (hPtr == NULL) break;
    modPtr = (ModInfo *) Tcl_GetHashValue(hPtr);
    modifiers |= modPtr->mask;
    if (modPtr->flags & ANY) any = 1;
    while ((*p == '-') || isspace(((unsigned char)*p))) p++;
  }
  
  if (strcmp(field,"Button") == 0 || strcmp(field,"ButtonPress") == 0
  || strcmp(field,"ButtonRelease") == 0 || strcmp(field,"Motion") == 0) {
    type = 0;
    while ((*p == '-') || isspace(((unsigned char)*p))) p++;
    p = GetField(p, field, FIELD_SIZE);
  } else if (strcmp(field,"Key") == 0 || strcmp(field,"KeyPress") == 0
  || strcmp(field,"KeyRelease") == 0) {
    type = 1;
    while ((*p == '-') || isspace(((unsigned char)*p))) p++;
    p = GetField(p, field, FIELD_SIZE);
  } else {
    type = 1;
  }
  
  if (*field != '\0') {
    if (type == 0) {
      if ((*field >= '1') && (*field <= '5') && (field[1] == '\0')) {
	static int masks[] = {~0, ~Button1Mask, ~Button2Mask,
				~Button3Mask, ~Button4Mask, ~Button5Mask};
	
	button = (*field - '0');
	
	/*
	 * Ignore this button as a modifier:  its state is already
	 * fixed.
	 */
	
	modifiers &= masks[button];
      } else {
	interp->result = "invalid button # descriptor";
	return TCL_ERROR;
      }
    } else {
      KeySym keysym = XStringToKeysym((char *)field);
      if (keysym == (KeySym)NULL) {
	Tcl_AppendResult(interp, "bad event type or keysym \"",
			 field, "\"", (char *) NULL);
	return TCL_ERROR;
      }
      keycode = XKeysymToKeycode(dpy,keysym);
      if (keycode == 0) {
	Tcl_AppendResult(interp, "the keysym \"", field,
                         "\" is not defined on this keyboard", (char *) NULL);
        return TCL_OK;
      }
    }
  } else {
    interp->result = "no button # or keysym";
    return TCL_ERROR;
  }
  while ((*p == '-') || isspace(((unsigned char)*p))) p++;
  if (*p != '>') {
    interp->result = "missing \">\" in event description";
    return TCL_ERROR;
  }
  p++;
  
  if (any) modifiers = AnyModifier;
  
  if (strncmp("grabevent",argv[1],strlen(argv[1])) == 0) {
    if (type == 1) {
      if (!XGrabKey(dpy, keycode, modifiers, (plug)?plug->window:Tk_WindowId(new), True, GrabModeAsync , GrabModeAsync)) {
        Tcl_AppendResult(interp, "XGrabKey failed", (char *) NULL);
      }
    } else {
      if (!XGrabButton(dpy, button, modifiers, (plug)?plug->window:Tk_WindowId(new), True, PointerMotionMask|ButtonPressMask|ButtonReleaseMask|ButtonMotionMask, GrabModeAsync, GrabModeAsync, None, None)) {
        Tcl_AppendResult(interp, "XGrabButton failed", (char *) NULL);
      }
    }
  } else {
    if (type == 1) {
      if (!XUngrabKey(dpy, keycode, modifiers, (plug)?plug->window:Tk_WindowId(new))) {
        Tcl_AppendResult(interp, "XUngrabKey failed", (char *) NULL);
      }
    } else {
      if (!XUngrabButton(dpy, button, modifiers, (plug)?plug->window:Tk_WindowId(new))) {
        Tcl_AppendResult(interp, "XUngrabButton failed", (char *) NULL);
      }
    }
  }
  return TCL_OK;
}


static int
InfoCmd(clientData, interp, argc, argv)
     ClientData clientData;      /* Main window associated with
				  * interpreter. */
     Tcl_Interp *interp;         /* Current interpreter. */
     int argc;                   /* Number of arguments. */
     char **argv;                /* Argument strings. */
{
  Plug *plug;
  Tk_Window tkwin = (Tk_Window) clientData;
  Tk_Window new;
  
  if (argc != 4) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " ", argv[1], " query pathName\"", (char *) NULL);
    return TCL_ERROR;
  }
  if ((new = Tk_NameToWindow(interp, argv[3], tkwin)) == NULL)
    return TCL_ERROR;
  
  if ((plug = TkWinPlug(new)) == NULL) {
    Tcl_AppendResult(interp, argv[3], " not a plug window or associated frame", (char *) NULL);
    return TCL_ERROR;
  }
  
  return DoInfoCmd(interp, plug,argv[2]);
}


static int UnmanagedCmd(clientData,interp)
     ClientData clientData;      /* Main window associated with
				  * interpreter. */
     Tcl_Interp *interp;         /* Current interpreter. */
{
  Tk_Window tkwin;
  Window root, parent, *children;
  unsigned int nchildren;
  int i,j;
  char buf[20];
  
  XSync(dpy, 0);
  XQueryTree(dpy, rootWindow, &root, &parent, &children, &nchildren);
  
  /*
   * weed out icon windows
   */
  for (i = 0; i < nchildren; i++) {
    if (children[i]) {
      XWMHints *wmhintsp = XGetWMHints (dpy, children[i]);
      if (wmhintsp) {
	if (wmhintsp->flags & IconWindowHint) {
	  for (j = 0; j < nchildren; j++) {
	    if (children[j] == wmhintsp->icon_window) {
	      children[j] = None;
	      break;
	    }
	  }
	}
	XFree ((char *) wmhintsp);
      }
    }
  }
  /*
   * return a list of all of the non-override windows that are
   * not currently managed
   */
  for (i = 0; i < nchildren; i++) {
    if (children[i]
	&& MappedNotOverride(children[i])
	&& !WindowPlug(children[i])
	&& XFindContext(dpy, (XID)children[i], tkWindowContext, (caddr_t *)&tkwin) == XCNOENT) {
      sprintf(buf,"%d",children[i]);
      Tcl_AppendElement(interp,buf);
    }
  }
  
  XFree((char *)children);
  return TCL_OK;
}

static int
IconSizeCmd(clientData, interp, argc, argv)
     ClientData clientData;      /* Main window associated with
				  * interpreter. */
     Tcl_Interp *interp;         /* Current interpreter. */
     int argc;                   /* Number of arguments. */
     char **argv;                /* Argument strings. */
{
  Plug *plug;
  Tk_Window tkwin = (Tk_Window) clientData;
  XIconSize *sizes;
  int count;
  int i,j;
  
  if ((argc-2)%6 != 0) {
    Tcl_AppendResult(interp, "wrong # args: should be \"",
		     argv[0], " ", argv[1], " ?min_width min_height max_width max_height width_inc height_inc ...?\"", (char *) NULL);
    return TCL_ERROR;
  }
  if (argc == 2) {
    char buf[20];
    /* return the property */
    if (XGetIconSizes(dpy, rootWindow, &sizes, &count) == 0)
	return TCL_OK;
    for (i = 0; i < count; i++) {
      sprintf(buf,"%d",sizes[i].min_width); Tcl_AppendElement(interp, buf);
      sprintf(buf,"%d",sizes[i].min_height); Tcl_AppendElement(interp, buf);
      sprintf(buf,"%d",sizes[i].max_width); Tcl_AppendElement(interp, buf);
      sprintf(buf,"%d",sizes[i].max_height); Tcl_AppendElement(interp, buf);
      sprintf(buf,"%d",sizes[i].width_inc); Tcl_AppendElement(interp, buf);
      sprintf(buf,"%d",sizes[i].height_inc); Tcl_AppendElement(interp, buf);
    }
  } else {
    /* set the property */
    int count = (argc-2)/6;
    sizes = (XIconSize *) ckalloc(sizeof(XIconSize)*count);

    for (i = 2, j = 0; j < count; j++, i += 6) {
      if (Tcl_GetInt(interp,argv[i+0],&sizes[j].min_width) == TCL_ERROR) {
        Tcl_AppendResult(interp,"min_width argument \"", argv[i+0],
                         "\" is not an integer", (char *) NULL);
        return TCL_ERROR;
      }
      if (Tcl_GetInt(interp,argv[i+1],&sizes[j].min_height) == TCL_ERROR) {
        Tcl_AppendResult(interp,"min_height argument \"", argv[i+1],
                         "\" is not an integer", (char *) NULL);
        return TCL_ERROR;
      }
      if (Tcl_GetInt(interp,argv[i+2],&sizes[j].max_width) == TCL_ERROR) {
        Tcl_AppendResult(interp,"max_width argument \"", argv[i+2],
                         "\" is not an integer", (char *) NULL);
        return TCL_ERROR;
      }
      if (Tcl_GetInt(interp,argv[i+3],&sizes[j].max_height) == TCL_ERROR) {
        Tcl_AppendResult(interp,"max_height argument \"", argv[i+3],
                         "\" is not an integer", (char *) NULL);
        return TCL_ERROR;
      }
      if (Tcl_GetInt(interp,argv[i+4],&sizes[j].width_inc) == TCL_ERROR) {
        Tcl_AppendResult(interp,"width_inc argument \"", argv[i+4],
                         "\" is not an integer", (char *) NULL);
        return TCL_ERROR;
      }
      if (Tcl_GetInt(interp,argv[i+5],&sizes[j].height_inc) == TCL_ERROR) {
        Tcl_AppendResult(interp,"height_inc argument \"", argv[i+5],
                         "\" is not an integer", (char *) NULL);
        return TCL_ERROR;
      }
    }

    if (XSetIconSizes(dpy, rootWindow, sizes, count) == 0) {
      Tcl_AppendResult(interp,"X error: failed to set icon size",
                       (char *) NULL);
    }

    ckfree((char *) sizes);
  }
  return TCL_OK;
}

/* Info query core */

static int DoInfoCmd(interp, plug, query)
     Tcl_Interp *interp;         /* Current interpreter. */
     Plug *plug;
     char *query;
{
  int c, length, i;
  char buf[40];
  
  c = query[0];
  length = strlen(query);
  
  /* window attributes */
  if (c == 'x' && query[1] == 0) {
    sprintf(interp->result, "%d", plug->winatts.x);
  } else if (c == 'y' && query[1] == 0) {
    sprintf(interp->result, "%d", plug->winatts.y);
  } else if (c == 'w' && strncmp("width",query,length) == 0) {
    sprintf(interp->result, "%d", plug->winatts.width);
  } else if (c == 'h' && strncmp("height",query,length) == 0) {
    sprintf(interp->result, "%d", plug->winatts.height);
  } else if (c == 'b' && strncmp("border_width",query,length) == 0) {
    sprintf(interp->result, "%d", plug->winatts.border_width);
  } else if (c == 'd' && strncmp("depth",query,length) == 0) {
    sprintf(interp->result, "%d", plug->winatts.depth);

  /* MISC */
  } else if (c == 'w' && strncmp("window_id",query,length) == 0) {
    sprintf(interp->result,"%d",plug->window);

  /* WM_NAME */
  } else if (c == 'n' && strncmp("name",query,length) == 0) {
    if (plug->name)
      interp->result = plug->name;
    else
      interp->result = "NoName";

  /* WM_ICON_NAME */
  } else if (c == 'i' && strncmp("icon_name",query,length) == 0) {
    if (plug->icon_name)
      interp->result = plug->icon_name;
    else
      interp->result = "NoName";

  /* WM_NORMAL_HINTS (a.k.a. WM_SIZE_HINTS) */
  } else if (c == 'm' && strncmp("min_width",query,length) == 0) {
    sprintf(interp->result, "%d", plug->normalhints.min_width);
  } else if (c == 'm' && strncmp("min_height",query,length) == 0) {
    sprintf(interp->result, "%d", plug->normalhints.min_height);
  } else if (c == 'm' && strncmp("max_width",query,length) == 0) {
    if (plug->normalhints.flags & PMaxSize)
      sprintf(interp->result, "%d", plug->normalhints.max_width);
    else
      interp->result = "Infinity";
  } else if (c == 'm' && strncmp("max_height",query,length) == 0) {
    if (plug->normalhints.flags & PMaxSize)
      sprintf(interp->result, "%d", plug->normalhints.max_height);
    else
      interp->result = "Infinity";
  } else if (c == 'w' && strncmp("width_inc",query,length) == 0) {
    sprintf(interp->result, "%d", plug->normalhints.width_inc);
  } else if (c == 'h' && strncmp("height_inc",query,length) == 0) {
    sprintf(interp->result, "%d", plug->normalhints.height_inc);
  } else if (c == 'b' && strncmp("base_width",query,length) == 0) {
    sprintf(interp->result, "%d", plug->normalhints.base_width);
  } else if (c == 'b' && strncmp("base_height",query,length) == 0) {
    sprintf(interp->result, "%d", plug->normalhints.base_height);
  } else if (c == 'm' && strncmp("min_aspect",query,length) == 0) {
    if (plug->normalhints.flags & PAspect)
      sprintf(interp->result, "%f", (double)plug->normalhints.min_aspect.x
	      / (double)plug->normalhints.min_aspect.y);
    else
      interp->result = "None";
  } else if (c == 'm' && strncmp("max_aspect",query,length) == 0) {
    if (plug->normalhints.flags & PAspect)
      sprintf(interp->result, "%f", (double)plug->normalhints.max_aspect.x
	      / (double)plug->normalhints.max_aspect.y);
    else
      interp->result = "None";
  } else if (c == 'w' && strncmp("win_gravity",query,length) == 0) {
    /* make this a string later */
    sprintf(interp->result, "%d", plug->normalhints.win_gravity);
  } else if (c == 'U' && strncmp("USPosition",query,length) == 0) {
    if (plug->normalhints.flags & USPosition)
      interp->result = "1";
    else
      interp->result = "0";

  /* WM_HINTS */
  } else if (c == 'i' && strncmp("input",query,length) == 0) {
    if (plug->WMhints.flags & InputHint)
      interp->result = (plug->WMhints.input)?"1":"0";
    else
      interp->result = "??";
  } else if (c == 'i' && strncmp("initial_state",query,length) == 0) {
    if (plug->WMhints.flags & StateHint) {
      switch(plug->WMhints.initial_state) {
      case WithdrawnState:
	interp->result = "withdrawn";
	break;
      case NormalState:
	interp->result = "normal";
	break;
      case IconicState:
	interp->result = "iconic";
	break;
      default:
        interp->result = "normal";
        break;
      }
    } else
      interp->result = "normal";
  } else if (c == 'i' && strncmp("icon_pixmap",query,length) == 0) {
    if (plug->WMhints.flags & IconPixmapHint)
      sprintf(interp->result,"%d",plug->WMhints.icon_pixmap);
    else
      interp->result = "??";
  } else if (c == 'i' && strncmp("icon_window",query,length) == 0) {
    if (plug->WMhints.flags & IconWindowHint)
      sprintf(interp->result,"%d",plug->WMhints.icon_window);
    else
      interp->result = "??";
  } else if (c == 'i' && strncmp("icon_x",query,length) == 0) {
    if (plug->WMhints.flags & IconPositionHint)
      sprintf(interp->result,"%d",plug->WMhints.icon_x);
    else
      interp->result = "??";
  } else if (c == 'i' && strncmp("icon_y",query,length) == 0) {
    if (plug->WMhints.flags & IconPositionHint)
      sprintf(interp->result,"%d",plug->WMhints.icon_y);
    else
      interp->result = "??";
  } else if (c == 'i' && strncmp("icon_mask",query,length) == 0) {
    if (plug->WMhints.flags & IconMaskHint)
      sprintf(interp->result,"%d",plug->WMhints.icon_mask);
    else
      interp->result = "??";
  } else if (c == 'w' && strncmp("window_group",query,length) == 0) {
    if (plug->WMhints.flags & WindowGroupHint)
      sprintf(interp->result,"%d",plug->WMhints.window_group);
    else
      interp->result = "None";

  /* WM_CLASS */
  } else if (c == 'r' && strncmp("res_name",query,length) == 0) {
    if (plug->class.res_name)
      interp->result = plug->class.res_name;
    else
      interp->result = "NoName";
  } else if (c == 'r' && strncmp("res_class",query,length) == 0) {
    if (plug->class.res_class)
      interp->result = plug->class.res_class;
    else
      interp->result = "NoName";

  /* WM_TRANSIENT_FOR */
  } else if (c == 't' && strncmp("transient_for",query,length) == 0) {
    sprintf(interp->result,"%d",plug->transient_for);

  /* WM_PROTOCOLS */
  } else if (c == 'p' && strncmp("protocols",query,length) == 0) {
    for (i = 0; i < plug->num_protocols; i++) {
      if (plug->protocols[i] == _XA_WM_TAKE_FOCUS)
	sprintf(buf,"TAKE_FOCUS");
      if (plug->protocols[i] == _XA_WM_SAVE_YOURSELF)
	sprintf(buf,"SAVE_YOURSELF");
      if (plug->protocols[i] == _XA_WM_DELETE_WINDOW)
	sprintf(buf,"DELETE_WINDOW");
      else
      	sprintf(buf,"%d",plug->protocols[i]);
      Tcl_AppendElement(interp,buf);
    }

  /* WM_COLORMAP_WINDOWS */
  } else if (c == 'c' && strncmp("colormap_windows",query,length) == 0) {
    for (i = 0; i < plug->cmaps.number_cwins; i++) {
      sprintf(buf,"%d",plug->cmaps.cwins[i]->w);
      Tcl_AppendElement(interp,buf);
    }

  /* WM_COMMAND */
  } else if (c == 'c' && strncmp("command",query,length) == 0) {
    for (i = 0; i < plug->argc; i++) {
      sprintf(buf,"%s",plug->argv[i]);
      Tcl_AppendElement(interp,buf);
    }

  /* WM_CLIENT_MACHINE */
  } else if (c == 'c' && strncmp("client_machine",query,length) == 0) {
    for (i = 0; i < plug->num_client_machine; i++) {
      sprintf(buf,"%s",plug->client_machine_l[i]);
      Tcl_AppendElement(interp,buf);
    }

  /* ERROR */
  } else {
    Tcl_AppendResult(interp, "query must be one of: ",
	"x, y, width, height, border_width, depth, window_id, name, icon_name, ",
	"min_width, min_height, max_width, max_height, width_inc, height_inc, ",
	"base_width, base_height, min_aspect, max_aspect, win_gravity, USPosition, ",
	"input, initial_state, icon_pixmap, icon_window, icon_x, icon_y, ",
	"icon_mask, window_group, res_name, res_class, transient_for, ",
	"protocols, colormap_windows, command, or client_machine", (char *) NULL);
    return TCL_ERROR;
  }
  return TCL_OK;
}

/*
 * UTILITY ROUTINES.
 */

int MappedNotOverride(w)
     Window w;
{
  XWindowAttributes wa;
  
  XGetWindowAttributes(dpy, w, &wa);
  return ((wa.map_state != IsUnmapped) && (wa.override_redirect != True));
}

/*
 *  EVENT HANDLING CODE.
 */

/*
 * Tkwm core event dispatcher.
 */

int TkwmEventProc(clientData,eventPtr)
     ClientData clientData;
     XEvent *eventPtr;
{
  switch (eventPtr->xany.type) {
  case KeyPress:
  case KeyRelease:
  case ButtonPress:
  case ButtonRelease:
  case MotionNotify:
    return HandleInput(eventPtr);
  case UnmapNotify:
    return HandleUnmapNotify(eventPtr);
  case ColormapNotify:
    return HandleColormapNotify(eventPtr);
  case VisibilityNotify:
    return HandleVisibilityNotify(eventPtr);
  case DestroyNotify:
    return HandleDestroyNotify(eventPtr);
  case PropertyNotify:
    return HandlePropertyNotify(eventPtr);
  case FocusIn:
    return HandleFocusIn(eventPtr);
  case ClientMessage:
    return HandleClientMessage(eventPtr);
  case ConfigureRequest:
    return HandleConfigureRequest(eventPtr);
  case MapRequest:
    return HandleMapRequest(eventPtr);
  }
  if (eventPtr->xany.type == shapeEventType)
    return HandleShapeNotify(eventPtr);
  return 0;
}

static void IdleInstall(clientData)
ClientData clientData;
{
    InstallWindowColormaps(0);
}

/*
 * Handle changes in colormaps. This is a hacked version of the
 * ctwm code to handle colormaps.
 */

int HandleColormapNotify(eventPtr)
     XEvent *eventPtr;
{
  XColormapEvent *cevent = (XColormapEvent *)eventPtr;
  ColormapWindow *cwin, **cwins;
  TkwmColormap *cmap;
  int lost, won, n, number_cwins;

  /* Find out if we are watching this window's colormap */
  if (XFindContext(dpy, cevent->window, colormapContext, (caddr_t *)&cwin) == XCNOENT)
    return 0;
  cmap = cwin->colormap;

#ifdef DEBUG_COLORMAPS
  printf("cevent: state = %d, window = %d, colormap = %d, new = %d\n",
	cevent->state,cevent->window,cevent->colormap,cevent->new);
#endif

  if (cevent->new) {
    /* The colormap has changed or is new */
#ifdef DEBUG_COLORMAPS
    printf("Colormap change on window %d: old = %d\n",cwin,cwin->colormap);
#endif
    if (XFindContext(dpy, cevent->colormap, colormapContext,
                     (caddr_t *)&cwin->colormap) == XCNOENT)
      cwin->colormap = CreateTkwmColormap(cevent->colormap);
    else
      cwin->colormap->refcnt++;
#ifdef DEBUG_COLORMAPS
    printf("Colormap change: new = %d\n",cwin->colormap);
#endif

    cmap->refcnt--;

    if (cevent->state == ColormapUninstalled)
      cmap->state &= ~CM_INSTALLED;
    else
      cmap->state |= CM_INSTALLED;

    if (cmap->state & CM_INSTALLABLE)
      InstallWindowColormaps((Colormaps *)NULL);

    if (cmap->refcnt == 0) {
      XDeleteContext(dpy, cmap->c, colormapContext);
#ifdef DEBUG_COLORMAPS
      printf("Removed %d from context\n",cmap->c);
#endif
      free((char *) cmap);
    }

    return 0;
  }

  if (cevent->state == ColormapUninstalled && (cmap->state & CM_INSTALLABLE)) {
    if (!(cmap->state & CM_INSTALLED))
      return 0;
    cmap->state &= ~CM_INSTALLED;

    if (!ColortableThrashing) {
#ifdef DEBUG_COLORMAPS
      printf("THRASHING!\n");
#endif
      ColortableThrashing = 1;
      /* Get the rest of the events that are causing thrashing */
      XSync(dpy, 0);
      Tk_DoWhenIdle(IdleInstall,0);
    }

    if (cevent->serial >= cmapInfo.first_req) {
      number_cwins = cmapInfo.cmaps->number_cwins;

      /*
       * Find out which colortables collided.
       */

      cwins = cmapInfo.cmaps->cwins;
      for (lost = won = -1, n = 0;
           (lost == -1 || won == -1) && n < number_cwins;
           n++) {
        if (lost == -1 && cwins[n] == cwin) {
          lost = n;   /* This is the window which lost its colormap */
          continue;
        }

        if (won == -1 && cwins[n]->colormap->install_req == cevent->serial) {
          won = n;    /* This is the window whose colormap caused */
          continue;   /* the de-install of the previous colormap */
        }
      }

      /*
       ** Cases are:
       ** Both the request and the window were found:
       **          One of the installs made honoring the WM_COLORMAP
       **          property caused another of the colormaps to be
       **          de-installed, just mark the scoreboard.
       **
       ** Only the request was found:
       **          One of the installs made honoring the WM_COLORMAP
       **          property caused a window not in the WM_COLORMAP
       **          list to lose its map.  This happens when the map
       **          it is losing is one which is trying to be installed,
       **          but is getting getting de-installed by another map
       **          in this case, we'll get a scoreable event later,
       **          this one is meaningless.
       **
       ** Neither the request nor the window was found:
       **          Somebody called installcolormap, but it doesn't
       **          affect the WM_COLORMAP windows.  This case will
       **          probably never occur.
       **
       ** Only the window was found:
       **          One of the WM_COLORMAP windows lost its colormap
       **          but it wasn't one of the requests known.  This is
       **          probably because someone did an "InstallColormap".
       **          The colormap policy is "enforced" by re-installing
       **          the colormaps which are believed to be correct.
       */

      if (won != -1)
        if (lost != -1) {
          /* lower diagonal index calculation */
          if (lost > won)
            n = lost*(lost-1)/2 + won;
          else
            n = won*(won-1)/2 + lost;
            cmapInfo.cmaps->scoreboard[n] = 1;
        } else {
            /*
             ** One of the cwin installs caused one of the cwin
             ** colormaps to be de-installed, so I'm sure to get an
             ** UninstallNotify for the cwin I know about later.
             ** I haven't got it yet, or the test of CM_INSTALLED
             ** above would have failed.  Turning the CM_INSTALLED
             ** bit back on makes sure we get back here to score
             ** the collision.
             */
            cmap->state |= CM_INSTALLED;
        }
        else if (lost != -1)
          InstallWindowColormaps((Colormaps *)NULL);
    }
  } else if (cevent->state == ColormapUninstalled)
    cmap->state &= ~CM_INSTALLED;
  else if (cevent->state == ColormapInstalled)
    cmap->state |= CM_INSTALLED;

#ifdef DEBUG_COLORMAPS
  printf("cmap: state = %d, colormap = %d\n",cmap->state,cmap->c);
#endif

  return 0;
}

int HandleVisibilityNotify(eventPtr)
     XEvent *eventPtr;
{
  XVisibilityEvent *vevent = (XVisibilityEvent *)eventPtr;
  ColormapWindow *cwin;
  TkwmColormap *cmap;

  if (XFindContext(dpy, vevent->window, colormapContext, (caddr_t *)&cwin) == XCNOENT)
    return 0;

  cmap = cwin->colormap;
  if ((cmap->state & CM_INSTALLABLE) &&
      vevent->state != cwin->visibility &&
      (vevent->state == VisibilityFullyObscured ||
      cwin->visibility == VisibilityFullyObscured) &&
      cmap->w == cwin->w) {
    cwin->visibility = vevent->state;
    InstallWindowColormaps((Colormaps *)NULL);
  } else
    cwin->visibility = vevent->state;
  return 0;
}

int HandlePropertyNotify(eventPtr)
     XEvent *eventPtr;
{
  Plug *plug = WindowPlug(eventPtr->xany.window);

  if (plug) {
    switch((int)eventPtr->xproperty.atom) {
    case XA_WM_NAME:
      XFetchName(dpy, plug->window, &plug->name);
      break;
    case XA_WM_ICON_NAME:
      XGetIconName(dpy, plug->window, &plug->icon_name);
      break;
    case XA_WM_NORMAL_HINTS:
      GetSizeHints(plug);
      break;
    case XA_WM_HINTS: {
      XWMHints *wmhints;
      
      wmhints = XGetWMHints(dpy, plug->window);
      if (wmhints != 0) {
      	plug->WMhints = *wmhints;
        XFree((char *)wmhints);
      } else
	plug->WMhints.flags = 0;
      break;
    }
    case XA_WM_CLASS:
      XGetClassHint(dpy, plug->window, &plug->class);
      break;
    case XA_WM_TRANSIENT_FOR:
      XGetTransientForHint(dpy,plug->window,&plug->transient_for);
      break;
    case XA_WM_COMMAND:
      if (plug->argv) XFreeStringList(plug->argv);
      XGetCommand(dpy,plug->window,&plug->argv,&plug->argc);
      break;
    case XA_WM_CLIENT_MACHINE:
      if (plug->client_machine.value) XFree((char *)plug->client_machine.value);
      if (plug->client_machine_l) XFreeStringList(plug->client_machine_l);
      plug->client_machine_l = 0;
      XGetWMClientMachine(dpy,plug->window,&plug->client_machine);
      XTextPropertyToStringList(&plug->client_machine,&plug->client_machine_l,&plug->num_client_machine);
      break;
    default:
      if (eventPtr->xproperty.atom == _XA_WM_PROTOCOLS) {
	XGetWMProtocols(dpy,plug->window,&plug->protocols,&plug->num_protocols);
      } else if (eventPtr->xproperty.atom == _XA_WM_COLORMAP_WINDOWS) {
        GetColormapWindows(plug->window,&plug->cmaps);
        /* If this plug is the current focus then we better reinstall colormaps */
        if (colormapFocusPlug == plug)
          InstallWindowColormaps((Colormaps *)NULL);
      }
      break;
    }
    
    doBinding(TKWM_PROPERTY,eventPtr);
    return 1;
  }
  return 0;
}

int HandleClientMessage(eventPtr)
     XEvent *eventPtr;
{
  doBinding(TKWM_ICONIFY,eventPtr);
  return 0;
}

/*
 * Handle plugs changing shape
 */
int HandleShapeNotify(eventPtr)
     XEvent *eventPtr;
{
	doBinding(TKWM_SHAPENOTIFY, eventPtr);
	return 0;
}


/* generate focus events on plugs so the window manager can watch
   focus requests from clients */
int HandleFocusIn(eventPtr)
     XEvent *eventPtr;
{
    doBinding(TKWM_FOCUSIN,eventPtr);
    return 0;
}

int HandleConfigureRequest(eventPtr)
     XEvent *eventPtr;
{
  Plug *plug = WindowPlug(eventPtr->xconfigurerequest.window);
  
  eventPtr->xany.window = eventPtr->xconfigurerequest.window;
  if (!plug) {
    XWindowChanges chg;
    unsigned int value_mask;
    
    value_mask = eventPtr->xconfigurerequest.value_mask;
    chg.x = eventPtr->xconfigurerequest.x;
    chg.y = eventPtr->xconfigurerequest.y;
    chg.width = eventPtr->xconfigurerequest.width;
    chg.height = eventPtr->xconfigurerequest.height;
    chg.border_width = eventPtr->xconfigurerequest.border_width;
    chg.sibling = eventPtr->xconfigurerequest.above;
    chg.stack_mode = eventPtr->xconfigurerequest.detail;

    XConfigureWindow(dpy, eventPtr->xconfigurerequest.window, value_mask, &chg);
    return 0;
  } else {
    /*
     * Record the border width since we need to know it for synthetic geometry
     * events. Subtle point: in order to be ICCCM compliant,
     * the user must either make a "plugPath sendconfig" or perform a
     * resize on the plug (and that resize must actually change the size).
     */
    if (eventPtr->xconfigurerequest.value_mask & CWBorderWidth) {
      plug->winatts.border_width = eventPtr->xconfigurerequest.border_width;
      ComputePlugGeometry(plug);
    }
    doBinding(TKWM_CONFIGURE,eventPtr);
    return 0;
  }
}

int HandleMapRequest(eventPtr)
     XEvent *eventPtr;
{
  eventPtr->xany.window = eventPtr->xmaprequest.window;
  doBinding(TKWM_MAPREQ,eventPtr);
  return 1;
}

/*
 * The plug is withdrawing itself. Synthesize a destroy event for
 * the toplevel window after removing the child from the plug and marking
 *  it as destroyed
 */

int HandleUnmapNotify(eventPtr)
     XEvent *eventPtr;
{
  Plug *plug;
  unsigned long data[2];
  plug = WindowPlug(eventPtr->xunmap.window);
  /* Reparenting can cause unmap events, but only when the new client is
     already mapped. I think this only happens on startup when the parent
     is the root.
     */
  if (plug && !plug->destroyed && !(eventPtr->xunmap.event == rootWindow && eventPtr->xunmap.send_event == 0)) {
    plug->destroyed = 1;
    if (plug->reparented) {
      /*
       * Must deal with this seperately from destroy on plug, since
       * we don't want to remap a window that is just getting withdrawn!
       */
      XReparentWindow(dpy,plug->window,rootWindow,plug->winatts.x,plug->winatts.y);
      XRemoveFromSaveSet(dpy,plug->window);
      XSelectInput(dpy, plug->window, NoEventMask);
      XShapeSelectInput(dpy, plug->window, 0);
      XSetWindowBorderWidth(dpy, plug->window, plug->winatts.border_width);
      plug->reparented = 0;
      /*
       * Must set withdrawn state here, since the plug command will no
       * longer exist when the TCL side gets notifcation of this withdrawl.
       */
      data[0] = WithdrawnState;
      data[1] = plug->iconwin;
      XChangeProperty(dpy, plug->window, _XA_WM_STATE, _XA_WM_STATE, 32,
                  PropModeReplace, (unsigned char *) data, 2);
    }
    /* munge into destroy event for plug window frame  */
    Tk_DestroyWindow(plug->frame);
    return 1;
  }
  return 0;
}

int HandleDestroyNotify(eventPtr)
     XEvent *eventPtr;
{
  Plug *plug;
  plug = WindowPlug(eventPtr->xdestroywindow.window);
  if (plug && !plug->destroyed) {
    plug->destroyed = 1;
    /* munge into destroy event for plug window frame */
    Tk_DestroyWindow(plug->frame);
    return 1;
  }
  return 0;
}

int HandleInput(eventPtr)
     XEvent *eventPtr;
{
  Plug *plug;
  plug = WindowPlug(eventPtr->xany.window);
  if (plug && !plug->destroyed) {
    /* munge into an event for plug window frame */
    eventPtr->xany.window = Tk_WindowId(plug->tkwin);
  }
  return 0;
}

/*
 * ERROR HANDLING.
 */

/*
 * X Error handler. This should probably be done with TK's mechanism instead.
 */

static int TkwmErrorHandler(dpy, event)
     Display *dpy;
     XErrorEvent *event;
{
  if ((event->error_code == BadWindow)
      || (event->error_code = X_GetGeometry)
      || (event->error_code == BadDrawable))
    return 0;
  fprintf(stderr,"tkwm: Internal error.\n");
  (*defaultErrorHandler)(dpy,event);
}

/*
 * INITIALIZATION CODE.
 */

int Tkwm_Init(interp)
     Tcl_Interp *interp;
{
  Tk_Window main;
  XGCValues gcv;
  unsigned long gcm;

  main = Tk_MainWindow(interp);
  
  dpy = Tk_Display(main);
  rootWindow = RootWindow(dpy, Tk_ScreenNumber(main));
  screenHeight = DisplayHeight(dpy, Tk_ScreenNumber(main));
  screenWidth = DisplayWidth(dpy, Tk_ScreenNumber(main));
  cmapInfo.cmaps = NULL;
  cmapInfo.maxCmaps = MaxCmapsOfScreen(ScreenOfDisplay(dpy, Tk_ScreenNumber(main)));
  cmapInfo.root_pushes = 0;
  colormapFocusPlug = 0;
  colormapwin = 0;
  GetColormapWindows(rootWindow,&rootcmap);
  InstallWindowColormaps(&rootcmap);

  Tk_GeometryRequest(main, screenWidth, screenHeight);
  
  InternUsefulAtoms();

  /*
   * Creat a graphics context for drawing window outlines.
   */
  
  gcm = GCFunction|GCLineWidth|GCForeground|GCSubwindowMode;
  gcv.function = GXxor;
  gcv.line_width = 1;
  gcv.foreground =
    (((unsigned long) 1) << DefaultDepth(dpy, Tk_ScreenNumber(main))) - 1;
  gcv.subwindow_mode = IncludeInferiors;
  DrawGC = XCreateGC(dpy, rootWindow, gcm, &gcv);
  
  /* Force the main window to be the root window */
  Tk_Substitute(main,rootWindow);
  
  plugContext = XUniqueContext();
  plugWinContext = XUniqueContext();
  
  Tk_CreateGenericHandler(TkwmEventProc,0);
  
  Tcl_CreateCommand(interp, "tkwm", TkwmCmd,
		    (ClientData) main, (Tcl_CmdDeleteProc *) NULL);
  if (initShape(interp, dpy) != TCL_OK)
	  return TCL_ERROR;
  return TCL_OK;
}

/* HACK ALERT! mucking about to make TK think the root is its own window */
/* HACK ALERT 2! Force context memorization that TK expects */
/* This might be better replaced by writing a real root window widget... */
void Tk_Substitute(tkwin, win)
     Tk_Window tkwin;
     Window win;
{
  defaultErrorHandler = XSetErrorHandler(AlreadyManagedError);
  Tk_MakeWindowExist(tkwin);
  XDeleteContext(dpy,Tk_WindowId(tkwin),tkWindowContext);
  XDestroyWindow(dpy,Tk_WindowId(tkwin));
  Tk_WindowId(tkwin) = win;
  /* force the main window to be in the "mapped" state. */
  ((Tk_FakeWin *)tkwin)->flags |= TK_MAPPED;
  XSaveContext(dpy,Tk_WindowId(tkwin), tkWindowContext, (caddr_t) tkwin);
  XSelectInput(dpy,Tk_WindowId(tkwin),
       KeyPressMask|KeyReleaseMask|ButtonPressMask|ButtonReleaseMask|
       EnterWindowMask|LeaveWindowMask|PointerMotionMask|ExposureMask|
       VisibilityChangeMask|SubstructureNotifyMask|SubstructureRedirectMask|
       FocusChangeMask|PropertyChangeMask|ColormapChangeMask);
  XSync(dpy, 0);
  XSetErrorHandler(TkwmErrorHandler);
}

static int AlreadyManagedError(dpy,event)
     Display *dpy;
     XErrorEvent *event;
{
  fprintf(stderr, "tkwm: Can't get control of display. Perhaps another window manager is running?\n");
  /* I should really call the TK exit routines... */
  exit(1);
}

/*
 * Modified from ctwm.
 */

static void InternUsefulAtoms ()
{
  /* 
   * Create priority colors if necessary.
   */

  _XA_WM_CHANGE_STATE = XInternAtom (dpy, "WM_CHANGE_STATE", False);
  _XA_WM_STATE = XInternAtom (dpy, "WM_STATE", False);
  _XA_WM_COLORMAP_WINDOWS = XInternAtom (dpy, "WM_COLORMAP_WINDOWS", False);
  _XA_WM_PROTOCOLS = XInternAtom (dpy, "WM_PROTOCOLS", False);
  _XA_WM_TAKE_FOCUS = XInternAtom (dpy, "WM_TAKE_FOCUS", False);
  _XA_WM_SAVE_YOURSELF = XInternAtom (dpy, "WM_SAVE_YOURSELF", False);
  _XA_WM_DELETE_WINDOW = XInternAtom (dpy, "WM_DELETE_WINDOW", False);
  _XA_WM_ROOT = XInternAtom (dpy, "__WM_ROOT", False);
}
