/* getblk.c, DCT block decoding                                             */

/* Copyright (C) 1994, MPEG Software Simulation Group. All Rights Reserved. */

/*
 * Disclaimer of Warranty
 *
 * These software programs are available to the user without any license fee or
 * royalty on an "as is" basis.  The MPEG Software Simulation Group disclaims
 * any and all warranties, whether express, implied, or statuary, including any
 * implied warranties or merchantability or of fitness for a particular
 * purpose.  In no event shall the copyright-holder be liable for any
 * incidental, punitive, or consequential damages of any kind whatsoever
 * arising from the use of these programs.
 *
 * This disclaimer of warranty extends to the user of these programs and user's
 * customers, employees, agents, transferees, successors, and assigns.
 *
 * The MPEG Software Simulation Group does not represent or warrant that the
 * programs furnished hereunder are free of infringement of any third-party
 * patents.
 *
 * Commercial implementations of MPEG-1 and MPEG-2 video, including shareware,
 * are subject to royalty fees to patent holders.  Many of these patents are
 * general enough such that they are unavoidable regardless of implementation
 * design.
 *
 */

#include <stdio.h>

#include "config.h"
#include "global.h"


/* defined in getvlc.h */
typedef struct {
  char run, level, len;
} DCTtab;

extern DCTtab DCTtabfirst[],DCTtabnext[],DCTtab0[],DCTtab1[];
extern DCTtab DCTtab2[],DCTtab3[],DCTtab4[],DCTtab5[],DCTtab6[];
extern DCTtab DCTtab0a[],DCTtab1a[];


/* decode one intra coded MPEG-1 block */

void getintrablock(comp,dc_dct_pred)
int comp;
int dc_dct_pred[];
{
  int val, i, j, sign;
  unsigned int code;
  int tval;
  DCTtab *tab;
  short *bp;

  bp = ld->block[comp];
  /* decode DC coefficients */
  if (comp<4)
    bp[0] = (dc_dct_pred[0]+=getDClum()) << 3;
  else if (comp==4)
    bp[0] = (dc_dct_pred[1]+=getDCchrom()) << 3;
  else
    bp[0] = (dc_dct_pred[2]+=getDCchrom()) << 3;

  if (fault) return;

  /* decode AC coefficients */
  for (i=1; ; i++)
  {
    code = showbits(16);
    if (code>=16384)
      tab = &DCTtabnext[(code>>12)-4];
    else if (code>=1024)
      tab = &DCTtab0[(code>>8)-4];
    else if (code>=512)
      tab = &DCTtab1[(code>>6)-8];
    else if (code>=256)
      tab = &DCTtab2[(code>>4)-16];
    else if (code>=128)
      tab = &DCTtab3[(code>>3)-16];
    else if (code>=64)
      tab = &DCTtab4[(code>>2)-16];
    else if (code>=32)
      tab = &DCTtab5[(code>>1)-16];
    else if (code>=16)
      tab = &DCTtab6[code-16];
    else
    {
      if (!quiet)
        fprintf(stderr,"invalid Huffman code in getintrablock()\n");
      fault = 1;
      return;
    }

    flushbits(tab->len);

    if (tab->run==64) /* end_of_block */
      return;

    if (tab->run==65) /* escape */
    {
      i+= getbits(6);

      val = getbits(8);
      if (val==0)
        val = getbits(8);
      else if (val==128)
        val = getbits(8) - 256;
      else if (val>128)
        val -= 256;

      if (sign = (val<0))
        val = -val;
    }
    else
    {
      i+= tab->run;
      val = tab->level;
      sign = getbits(1);
    }

    if (i>=64)
    {
      if (!quiet)
        fprintf(stderr,"DCT coeff index (i) out of bounds (intra)\n");
      fault = 1;
      return;
    }

    j = zig_zag_scan[i];
    val = (val*ld->quant_scale*ld->intra_quantizer_matrix[j]) >> 3;

    /* mismatch control ('oddification') */
    if (val!=0) /* should always be true, but it's not guaranteed */
      val = (val-1) | 1; /* equivalent to: if ((val&1)==0) val = val - 1; */

    /* saturation */
    if (!sign)
      bp[j] = (val>2047) ?  2047 :  val; /* positive */
    else
      bp[j] = (val>2048) ? -2048 : -val; /* negative */
  }
}


/* decode one non-intra coded MPEG-1 block */

void getinterblock(comp)
int comp;
{
  int val, i, j, sign;
  unsigned int code;
  int tval;
  DCTtab *tab;
  short *bp;

  bp = ld->block[comp];

  /* decode AC coefficients */
  for (i=0; ; i++)
  {
    code = showbits(16);
    if (code>=16384)
    {
      if (i==0)
        tab = &DCTtabfirst[(code>>12)-4];
      else
        tab = &DCTtabnext[(code>>12)-4];
    }
    else if (code>=1024)
      tab = &DCTtab0[(code>>8)-4];
    else if (code>=512)
      tab = &DCTtab1[(code>>6)-8];
    else if (code>=256)
      tab = &DCTtab2[(code>>4)-16];
    else if (code>=128)
      tab = &DCTtab3[(code>>3)-16];
    else if (code>=64)
      tab = &DCTtab4[(code>>2)-16];
    else if (code>=32)
      tab = &DCTtab5[(code>>1)-16];
    else if (code>=16)
      tab = &DCTtab6[code-16];
    else
    {
      if (!quiet)
        fprintf(stderr,"invalid Huffman code in getinterblock()\n");
      fault = 1;
      return;
    }

    flushbits(tab->len);

    if (tab->run==64) /* end_of_block */
      return;

    if (tab->run==65) /* escape */
    {
      i+= getbits(6);

      val = getbits(8);
      if (val==0)
        val = getbits(8);
      else if (val==128)
        val = getbits(8) - 256;
      else if (val>128)
        val -= 256;

      if (sign = (val<0))
        val = -val;
    }
    else
    {
      i+= tab->run;
      val = tab->level;
      sign = getbits(1);
    }

    if (i>=64)
    {
      if (!quiet)
        fprintf(stderr,"DCT coeff index (i) out of bounds (inter)\n");
      fault = 1;
      return;
    }

    j = zig_zag_scan[i];
    val = (((val<<1)+1)*ld->quant_scale*ld->non_intra_quantizer_matrix[j]) >> 4;

    /* mismatch control ('oddification') */
    if (val!=0) /* should always be true, but it's not guaranteed */
      val = (val-1) | 1; /* equivalent to: if ((val&1)==0) val = val - 1; */

    /* saturation */
    if (!sign)
      bp[j] = (val>2047) ?  2047 :  val; /* positive */
    else
      bp[j] = (val>2048) ? -2048 : -val; /* negative */
  }
}


/* decode one intra coded MPEG-2 block */

void getmpg2intrablock(comp,dc_dct_pred)
int comp;
int dc_dct_pred[];
{
  int val, i, j, sign, nc, cc, run;
  unsigned int code;
  DCTtab *tab;
  short *bp;
  int *qmat;
  struct layer_data *ld1;

  /* with data partitioning, data always goes to base layer */
  ld1 = (ld->scalable_mode==SC_DP) ? &base : ld;
  bp = ld1->block[comp];

  if (base.scalable_mode==SC_DP)
    if (base.pri_brk<64)
      ld = &enhan;
    else
      ld = &base;

  cc = (comp<4) ? 0 : (comp&1)+1;

  qmat = (comp<4 || chroma_format==CHROMA420)
         ? ld1->intra_quantizer_matrix
         : ld1->chroma_intra_quantizer_matrix;

  /* decode DC coefficients */
  if (cc==0)
    val = (dc_dct_pred[0]+= getDClum());
  else if (cc==1)
    val = (dc_dct_pred[1]+= getDCchrom());
  else
    val = (dc_dct_pred[2]+= getDCchrom());

  if (fault) return;

  bp[0] = val << (3-dc_prec);

  nc=0;

  if (trace)
    printf("DCT(%d)i:",comp);

  /* decode AC coefficients */
  for (i=1; ; i++)
  {
    code = showbits(16);
    if (code>=16384 && !intravlc)
      tab = &DCTtabnext[(code>>12)-4];
    else if (code>=1024)
    {
      if (intravlc)
        tab = &DCTtab0a[(code>>8)-4];
      else
        tab = &DCTtab0[(code>>8)-4];
    }
    else if (code>=512)
    {
      if (intravlc)
        tab = &DCTtab1a[(code>>6)-8];
      else
        tab = &DCTtab1[(code>>6)-8];
    }
    else if (code>=256)
      tab = &DCTtab2[(code>>4)-16];
    else if (code>=128)
      tab = &DCTtab3[(code>>3)-16];
    else if (code>=64)
      tab = &DCTtab4[(code>>2)-16];
    else if (code>=32)
      tab = &DCTtab5[(code>>1)-16];
    else if (code>=16)
      tab = &DCTtab6[code-16];
    else
    {
      if (!quiet)
        fprintf(stderr,"invalid Huffman code in getmpg2intrablock()\n");
      fault = 1;
      return;
    }

    flushbits(tab->len);

    if (trace)
    {
      printf(" (");
      printbits(code,16,tab->len);
    }

    if (tab->run==64) /* end_of_block */
    {
      if (trace)
        printf("): EOB\n");
      return;
    }

    if (tab->run==65) /* escape */
    {
      if (trace)
      {
        putchar(' ');
        printbits(showbits(6),6,6);
      }

      i+= run = getbits(6);

      if (trace)
      {
        putchar(' ');
        printbits(showbits(12),12,12);
      }

      val = getbits(12);
      if ((val&2047)==0)
      {
        if (!quiet)
          fprintf(stderr,"invalid signed_level (escape) in getmpg2intrablock()\n");
        fault = 1;
        return;
      }
      if (sign = (val>=2048))
        val = 4096 - val;
    }
    else
    {
      i+= run = tab->run;
      val = tab->level;
      sign = getbits(1);

      if (trace)
        printf("%d",sign);
    }

    if (i>=64)
    {
      if (!quiet)
        fprintf(stderr,"DCT coeff index (i) out of bounds (intra2)\n");
      fault = 1;
      return;
    }

    if (trace)
      printf("): %d/%d",run,sign ? -val : val);

    j = (ld1->altscan ? alternate_scan : zig_zag_scan)[i];
    val = (val * ld1->quant_scale * qmat[j]) >> 4;
    bp[j] = sign ? -val : val;
    nc++;

    if (base.scalable_mode==SC_DP && nc==base.pri_brk-63)
      ld = &enhan;
  }
}


/* decode one non-intra coded MPEG-2 block */

void getmpg2interblock(comp)
int comp;
{
  int val, i, j, sign, nc, run;
  unsigned int code;
  DCTtab *tab;
  short *bp;
  int *qmat;
  struct layer_data *ld1;

  /* with data partitioning, data always goes to base layer */
  ld1 = (ld->scalable_mode==SC_DP) ? &base : ld;
  bp = ld1->block[comp];

  if (base.scalable_mode==SC_DP)
    if (base.pri_brk<64)
      ld = &enhan;
    else
      ld = &base;

  qmat = (comp<4 || chroma_format==CHROMA420)
         ? ld1->non_intra_quantizer_matrix
         : ld1->chroma_non_intra_quantizer_matrix;

  nc = 0;

  if (trace)
    printf("DCT(%d)n:",comp);

  /* decode AC coefficients */
  for (i=0; ; i++)
  {
    code = showbits(16);
    if (code>=16384)
    {
      if (i==0)
        tab = &DCTtabfirst[(code>>12)-4];
      else
        tab = &DCTtabnext[(code>>12)-4];
    }
    else if (code>=1024)
      tab = &DCTtab0[(code>>8)-4];
    else if (code>=512)
      tab = &DCTtab1[(code>>6)-8];
    else if (code>=256)
      tab = &DCTtab2[(code>>4)-16];
    else if (code>=128)
      tab = &DCTtab3[(code>>3)-16];
    else if (code>=64)
      tab = &DCTtab4[(code>>2)-16];
    else if (code>=32)
      tab = &DCTtab5[(code>>1)-16];
    else if (code>=16)
      tab = &DCTtab6[code-16];
    else
    {
      if (!quiet)
        fprintf(stderr,"invalid Huffman code in getmpg2interblock()\n");
      fault = 1;
      return;
    }

    flushbits(tab->len);

    if (trace)
    {
      printf(" (");
      printbits(code,16,tab->len);
    }

    if (tab->run==64) /* end_of_block */
    {
      if (trace)
        printf("): EOB\n");
      return;
    }

    if (tab->run==65) /* escape */
    {
      if (trace)
      {
        putchar(' ');
        printbits(showbits(6),6,6);
      }

      i+= run = getbits(6);

      if (trace)
      {
        putchar(' ');
        printbits(showbits(12),12,12);
      }

      val = getbits(12);
      if ((val&2047)==0)
      {
        if (!quiet)
          fprintf(stderr,"invalid signed_level (escape) in getmpg2intrablock()\n");
        fault = 1;
        return;
      }
      if (sign = (val>=2048))
        val = 4096 - val;
    }
    else
    {
      i+= run = tab->run;
      val = tab->level;
      sign = getbits(1);

      if (trace)
        printf("%d",sign);
    }

    if (i>=64)
    {
      if (!quiet)
        fprintf(stderr,"DCT coeff index (i) out of bounds (inter2)\n");
      fault = 1;
      return;
    }

    if (trace)
      printf("): %d/%d",run,sign?-val:val);

    j = (ld1->altscan ? alternate_scan : zig_zag_scan)[i];
    val = (((val<<1)+1) * ld1->quant_scale * qmat[j]) >> 5;
    bp[j] = sign ? -val : val;
    nc++;

    if (base.scalable_mode==SC_DP && nc==base.pri_brk-63)
      ld = &enhan;
  }
}
