/*
 * TkTreeNode.cc - member routines for class TkTreeNode
 * 
 * Class TkTreeNode redefines virtual functions from its base class,
 * DrawTreeNode, to allow drawing nodes on a tk canvas.
 * 
 * -----------------------------------------------------------------------------
 * Copyright 1993 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  Allan
 * Brighton make no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied
 * warranty.
 * -----------------------------------------------------------------------------
 */

#include <TkTreeNode.h>
#include <TkTree.h>
#include <Point.h>
#include <strstream.h>


/*
 * construct a node with given tag, pos and size
 */
TkTreeNode::TkTreeNode(TkTree* tree, const String& tag, const String& line, 
    int x, int y, int w, int h, int border)
: DrawTreeNode(tag, x, y, w, h, border), tree_(tree), line_(line)
{
}


/*
 * return distance from parent node
 */
int TkTreeNode::parentDistance() const
{
    return tree_->parentDistance();
}
    

/*
 * Return true if the tree's orientation is horizontal
 */
int TkTreeNode::horizontal() const 
{
    return tree_->horizontal();
}

    
/*
 * Draw the tree node using Tk 
 */
void TkTreeNode::drawNode() const
{
    if (tag().length())
        tree_->tcl << tree_->canvas().string() << " move " 
	    << "{" << tag().string() << "}" 
		<< " " << pos() - prev_pos() << eval;
}


/*
 * Remove a tree node using Tk 
 */
void TkTreeNode::removeNode() const
{
    if (tag().length())
        Tcl_VarEval(tree_->interp(), tree_->canvas().string(), 
		    " delete {", tag().string(), "}", 0);
    
    if (removeCmd_.length())
        Tcl_VarEval(tree_->interp(), removeCmd_.string(), 0);
}


/*
 * Draw the line between this node and its parent 
 */
void TkTreeNode::drawLine(const Point& p1, const Point& p2) const
{
    // make sure the lines are close enough to the nodes in tk 
    // (I'm not sure why this is needed...)
    if (horizontal()) 
        tree_->tcl << tree_->canvas().string() << " coords " << line_.string() << " " 
           << p1.x-3 << " " << p1.y << " " << p2.x-1 << " " << p2.y << eval;
    else 
        tree_->tcl << tree_->canvas().string() << " coords " << line_.string() << " " 
           << p1.x << " " << p1.y-3 << " " << p2.x << " " << p2.y-1 << eval;
        
}


/*
 * Remove the line from this nodes parent  
 */
void TkTreeNode::removeLine() const
{
    Tcl_VarEval(tree_->interp(), tree_->canvas().string(), " delete ", line_.string(), 0);
}


/*
 * This routine is called with the bounding box of the tree before drawing
 * We use this information to change the scrollregion of the tree's canvas
 */
void TkTreeNode::setDrawArea(int x1, int y1, int x2, int y2) const
{
    int b = tree_->borderWidth();	// border width around tree
    tree_->tcl << tree_->canvas().string() << " config -scrollregion {" 
        << x1-b << " " << y1-b << " " << x2+b << " " << y2+b << "}" << eval;
}
