////////////////////////////////////////////////////////////////////////////////
//                      Universal matrix class.                               //   
//  LAST EDIT: Fri Aug  5 08:55:06 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //
//              yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>  
#include <stdlib.h>  
#include <math.h>   
#include <string.h>  
#include <values.h>
#include "rs_matrx.h"
#include "rs_io.h"
#include "../global.h"
#include "../error.h"

extern int MatEqSys(RT_RS_Matrix& A, RT_RS_Vector& X,
                      RT_RS_Vector& B, int flag = 0 ); 


///////////////////  THE RT_RS_Matrix MEMBER FUNCTIONS ///////////////////////
void RT_RS_Matrix::alloch(int anr, int anc, char atemp)
{
  nr = anr;
  nc = anc;
  temp = atemp;

  m = new float* [nr];
  for(int i = 0; i < nr ; i++)
    m[i] = new float [nc];
}

void RT_RS_Matrix::freeh()
{
  if(m == NULL) return;
  for(int i = 0; i < nr; i++)
    delete m[i];
  delete m;
  m = NULL;
}

RT_RS_Matrix::RT_RS_Matrix(char *filename, FILE *fp)
{
  m = NULL;
  if(!read(filename, fp)) freeh();
}

RT_RS_Matrix::RT_RS_Matrix(RT_RS_Matrix& a)
{
#ifdef RS_DEBUG
  if(a.m == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in copy-constructor", NULL);
#endif
  nr = a.nr;
  nc = a.nc;
  temp = a.temp;
  if(a.temp == 'y'){
    m = a.m;
    a.m = 0;
  }
  else{
    int i;
    m = new float* [nr];
    for(i = 0; i < nr; i++)
      m[i] = new float [nc];
#ifdef RS_USE_MEM_FUNCS
    int sz = nc * sizeof(float);
     for(i = 0; i < nr; i++)
      memcpy(m[i], a.m[i], sz);
#else
    int j;
     for(i = 0; i < nr; i++)
       for(j = 0; j < nc; j++)
        m[i][j] = a.m[i][j];
#endif
  }
}

void RT_RS_Matrix::set(float f)
{
#ifdef RS_DEBUG
  if(m == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in set()", NULL);
#endif
  int i, j;
  for(i = 0; i < nr; i++)
    for(j = 0; j < nc; j++)
      m[i][j] = f;
}

RT_RS_Matrix& RT_RS_Matrix::operator = (RT_RS_Matrix& a)
{
#ifdef RS_DEBUG
  if(m == NULL || a.m == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator = ", NULL);
  if(nr != a.nr || nc != a.nc)
    rt_Output->fatalVar(get_class(), ":Equating matrices of unequal size", NULL);
#endif
#ifdef RS_USE_MEM_FUNCS
  int sz = nc * sizeof(float);
  int i;
  for(i = 0; i < nr; i++)
    memcpy(m[i], a.m[i], sz);
#else
  int j;
  for(i = 0; i < nr; i++)
    for(j = 0; j < nc; j++)
      m[i][j] = a.m[i][j];
#endif
  a.freet();
  return *this;
}

RT_RS_Matrix RT_RS_Matrix::operator ~ () //Transposition
{
  int i,j;
#ifdef RS_DEBUG
  if(m == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator ~", NULL);
#endif
  RT_RS_Matrix Temp(nc, nr, 'y'); //nc <-> nr
  for(i = 0; i < nr; i++){
    for(j = 0; j < nc; j++)
      Temp.m[j][i] = m[i][j];
  }
  return Temp;
}

float& RT_RS_Matrix::operator ()(int i, int j) //range checked element
{
#ifdef RS_DEBUG
  if(m == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator ()", NULL);
  if(i < 0 || i >= nr )
    rt_Output->fatalVar(get_class(), ":Illegal row index in operator ()", NULL);
  if(j < 0 || j >= nc )
    rt_Output->fatalVar(get_class(), ":Illegal column index in operator ()", NULL);
#endif
  return m[i][j];
}

RT_RS_Matrix RT_RS_Matrix::operator ! () // Inversion
{
  if(temp == 'n') {
    RT_RS_Matrix Temp(*this);
    Temp.temp = 'y';
    Temp.invert();
    return Temp;
  }
  else { // If *this is temporary, turn it into its inverse.
    invert();
    return *this;
  }
}

// Turn a RT_RS_Matrix into its inverse.
boolean RT_RS_Matrix::invert()
{
  RT_RS_Vector X(RS_MAX_EQSYS), B(RS_MAX_EQSYS);
  RT_RS_Matrix I(nr, nc, 'y');
  unsigned i = 0;
  int flag = 0;
  int j;

#ifdef RS_DEBUG
  if(nr != nc)
    rt_Output->fatalVar(get_class(), " is not square in function invert()", NULL);
  if(nr > RS_MAX_EQSYS)
    rt_Output->fatalVar(get_class(), " is too large in function invert()", NULL);
#endif
  X.set(0.0);
  B.set(0.0);
  while(i < nc)
  {
    B(i) = 1.0;
    if(MatEqSys(*this, X, B, flag) < 0) {
       rt_Output->errorVar(get_class(), ":Cannot invert", NULL);
      return FALSE;
    }
    for(j = 0; j < nr; j++)
      I(j, i) = X(j);
    B(i) = 0;
    flag = 1;
    i++;
  }
  freeh();
  m = I.m;
  I.m = NULL;
  return TRUE;
}

RT_RS_Matrix RT_RS_Matrix::operator + (RT_RS_Matrix& a)
{
  int i, j;
#ifdef RS_DEBUG
  if(m == NULL || a.m == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator +", NULL);
  if(nc != a.nc || nr != a.nr)
    rt_Output->fatalVar(get_class(), ":Dimensions do not match in operator +", NULL);
#endif

  RT_RS_Matrix Temp(nr, nc, 'y');
  for(i = 0; i < nr; i++)
    for(j = 0; j < nc; j++)
      Temp.m[i][j] = m[i][j] + a.m[i][j];
  freet();
  a.freet();
  return Temp;
}

RT_RS_Matrix RT_RS_Matrix::operator - (RT_RS_Matrix& a)
{
  int i, j;
#ifdef RS_DEBUG
  if(m == NULL || a.m == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator -", NULL);
  if(nc != a.nc || nr != a.nr)  
    rt_Output->fatalVar(get_class(), ":Dimensions do not match in operator -", NULL);
#endif

  RT_RS_Matrix Temp(nr, nc, 'y');
  for(i = 0; i < nr; i++)
    for(j = 0; j < nc; j++)
      Temp.m[i][j] = m[i][j] - a.m[i][j];
  freet();
  a.freet();
  return Temp;
}

RT_RS_Matrix RT_RS_Matrix::operator * (float f)
{
  int i,j;

#ifdef RS_DEBUG
  if(m == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator * (float)", NULL);
#endif
  RT_RS_Matrix Temp(nr, nc, 'y');
  for(i = 0; i < nr; i++)
    for(j = 0; j < nc; j++)
      Temp.m[i][j] = f * m[i][j];
  freet();
  return Temp;
}

RT_RS_Matrix RT_RS_Matrix::operator / (float f)
{
  int i, j;

#ifdef RS_DEBUG
  if(m == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), " in operator /", NULL);
#endif
  if(f < epsilon)
    rt_Output->fatalVar(get_class(), ":Division by zero in operator /", NULL);
  float recip = 1./f;
  RT_RS_Matrix Temp(nr, nc, 'y');
  for(i = 0; i < nr; i++)
    for(j = 0; j < nc; j++)
      Temp.m[i][j] = recip * m[i][j];
  freet();
  return(Temp);
}

boolean RT_RS_Matrix::read(char *filename, FILE *fp)
{
  int _nr, _nc, i, j;
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = getint(filename, fp, _nr, RS_s, clmn);
  if(Ok) Ok = getint(filename, fp, _nc, RS_s, clmn);
  if(Ok) {
    freeh();
    alloch(_nr, _nc, 'n');
    for(i = 0; i < nr && Ok; i++)
      for(j = 0; j < nc && Ok; j++)
        Ok = getfloat(filename, fp, m[i][j], RS_s, clmn);
  }
  if(aln) fclose(fp);
  return Ok;
}

boolean RT_RS_Matrix::write(char *filename, FILE *fp)
{
#ifdef RS_DEBUG
  if(m == NULL)
    rt_Output->fatalVar("Reference to a deleted ", get_class(), NULL);
#endif
  int i, j;
  boolean aln = fp == NULL;
  boolean Ok = TRUE;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
    fprintf(fp,"%s %d %d \n", get_class(), nr, nc);
    Ok = !ferror(fp);
  }
  for(i = 0; i < nr && Ok; i++)
    for(j = 0; j < nc && Ok; j++) {
      fprintf(fp, "% .6f\n", m[i][j]);
      Ok = !ferror(fp);
    }
  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

void RT_RS_Matrix::print(FILE *f, char *n, int width, int decimalPlaces)
{
  int i, j, nperline;

  if(m == NULL) return;
  if(n) fprintf(f, "%s\n", n);
  nperline = 79 / width;
  sprintf(fmt, "%s%d.%df", "%", width, decimalPlaces);
  for(i = 0; i < nr; i++){
    for(j = 0; j < nc; j++){
      fprintf(f, fmt, m[i][j]);
      if((j % nperline == nperline -1) && j < nc - 1)
        fprintf(f, "\n");
    }
    fprintf(f, "\n");
  }
}

