////////////////////////////////////////////////////////////////////////////////
//                         Surface related classes.                           //
//  LAST EDIT: Thu Jul 12 12:41:00 1994 by H. Fischer
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //
//              ekki@prakinf.tu-ilmenau.de                                    //
//                                                                            //
// (C) Copyright 1993 - 1995 YART team                                        //
////////////////////////////////////////////////////////////////////////////////

#include <stdlib.h>
#include <string.h>
#include "rs_areas.h"
#include "rs_io.h"
#include "../global.h"
#include "../error.h"


//////////////////////  THE RT_RS_Area MEMBER FUNCTIONS /////////////////////////
RT_RS_Area::RT_RS_Area(long v0, long v1, long v2, long v3,
                      long n0, long n1, long n2, long n3,
                      long amesh_nr, RT_RS_Scene* ascene) {
    scene = ascene;
  vidx[0] = v0; vidx[1] = v1; vidx[2] = v2; vidx[3] = v3;
  neighbors[0] = n0; neighbors[1] = n1; neighbors[2] = n2; neighbors[3] = n3;
  mesh_nr = amesh_nr;
  anchor_type = 'a';
}

void RT_RS_Area::set_ar0() {
  for(int i = 0; i < 4; i++) get_vtx(vidx[i])->set_ar(0.);
}

void RT_RS_Area::comp_ar() {
  int i, j = vidx[2] == vidx[3] ? 3 : 4;
  float dar, n1, n2 = .0;
  RT_RS_Vtx* v[4];

  RT_RS_3DVector p[3];

  for(i = 0; i < j; i++)
    v[i] = get_vtx(vidx[i]);
  for(i = 0; i < j - 1; i++)
    p[i] = v[i + 1]->pnt - v[0]->pnt;
  n1 = (p[0].cross(p[1])).vabs();
  if(j == 4)
    n2 = (p[1].cross(p[2])).vabs();

  ar = (n1 + n2) * .5;
  dar = ar / (float)j;
  for(i = 0; i < j; i++)
    v[i]->set_ar(v[i]->get_ar() + dar);
}

void RT_RS_Area::sub_ar()
{
  int i, j;
  float dar;
  RT_RS_Vtx *vtx;

  if(vidx[2] == vidx[3]) {
    j = 3; dar = ar / 3.;
  } else {
    j = 4; dar = ar * .25;
  }
  for(i = 0; i < j; i++) {
    vtx = get_vtx(vidx[i]);
    vtx->set_ar(vtx->get_ar() - dar);
  }
}

RT_RS_Vtx RT_RS_Area::comp_center()
{
  int i, j = vidx[2] == vidx[3] ? 3 : 4;
  float recip_j = 1. / (float)j;
  RT_RS_Vtx* v;
  RT_RS_3DVector center_pnt, center_nrm;

  for(i = 0; i < j; i++) {
    v = get_vtx(vidx[i]);
    center_pnt = center_pnt + v->pnt;
    center_nrm = center_nrm + v->nrm;
  }
  RT_RS_Vtx center(center_pnt * recip_j, center_nrm * recip_j);
  return center;
}

boolean RT_RS_Area::read(char *filename, FILE *fp)
{
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = getfloat(filename, fp, ar, RS_s, clmn);
  if(Ok) Ok = getlong(filename, fp, mesh_nr, RS_s, clmn);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  anchor_type = RS_s1[0];
  for(int i = 0; i < 4 && Ok; i++)
    Ok = getlong(filename, fp, vidx[i], RS_s, clmn);
  if(Ok) Ok = getlong(filename, fp, neighbors[0], RS_s, clmn);
  if(Ok) Ok = getlong(filename, fp, neighbors[1], RS_s, clmn);
  if(Ok) Ok = getlong(filename, fp, neighbors[2], RS_s, clmn);
  if(Ok) Ok = getlong(filename, fp, neighbors[3], RS_s, clmn);
  if(aln) fclose(fp);
  return Ok;
}

boolean RT_RS_Area::write(char *filename, FILE *fp)
{
  boolean Ok = TRUE;
  boolean aln = fp == NULL;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
     fprintf(fp,"%s\n % .6f %ld %c\n%ld %ld %ld %ld\n%ld %ld %ld %ld\n",
            get_class(), ar,
            mesh_nr, anchor_type, vidx[0], vidx[1], vidx[2], vidx[3],
            neighbors[0], neighbors[1], neighbors[2], neighbors[3]);
    Ok = !ferror(fp);
  }
  if(!Ok) {
    rt_Output->errorVar(get_class(), ": Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

#pragma argsused
void RT_RS_Area::print(FILE *f, char *n, int, int) {
  if(n) fprintf(f, "%s\n", n);
  sprintf(RS_s,
  "area: %f mesh nr: %ld anchor_type: %c\nvertices: %ld %ld %ld %ld  neighbors: %ld %ld %ld %ld\n",
    ar, mesh_nr, anchor_type, vidx[0], vidx[1], vidx[2], vidx[3], neighbors[0], neighbors[1],
    neighbors[2], neighbors[3]);
  fprintf(f, RS_s);
}

//////////////////////  THE RT_RS_DiffArea MEMBER FUNCTIONS /////////////////////////
void RT_RS_DiffArea::insert_ip(int n, long* vtxidx, RT_RS_Vtx*& new_vtx,
                               long& new_vtxidx)
{
  RT_RS_DiffVtx* vtx;
  scene->Points->insert_ip(n, vtxidx, vtx, new_vtxidx);
  new_vtx = vtx;
}

RT_RS_DiffVtx* RT_RS_DiffArea::get_dvtx(long i)
{
  RT_RS_DiffVtx* vtx;
  scene->Points->get(i, vtx);
  return vtx;
}

RT_RS_Area* RT_RS_DiffArea::get_area(long i)
{
  RT_RS_DiffArea* area;
  scene->Areas->get(i, area);
  return area;
}
//////////////////////  THE RT_RS_InsArea MEMBER FUNCTIONS /////////////////////////
void RT_RS_InsArea::insert_ip(int n, long* vtxidx, RT_RS_Vtx*& new_vtx,
                              long& new_vtxidx)
{
  RT_RS_InsVtx* vtx;
  scene->Points->insert_ip(n, vtxidx, vtx, new_vtxidx);
  new_vtx = vtx;
}

RT_RS_InsVtx* RT_RS_InsArea::get_ivtx(long i)
{
  RT_RS_InsVtx* vtx;
  scene->Points->get(i, vtx);
  return vtx;
}

RT_RS_Area* RT_RS_InsArea::get_area(long i)
{
  RT_RS_InsArea* area;
  scene->Areas->get(i, area);
  return area;
}

//////////////////////  THE RT_RS_Areas MEMBER FUNCTIONS /////////////////////////
void RT_RS_Areas::dallc()
{
  if(dspns >= 1024)
    rt_Output->fatalVar(get_class(), ":Too many areas in dallc()", NULL);
  dArs[dspns++] = new RT_RS_DiffArea* [1024];
}

void RT_RS_Areas::iallc()
{
  if(ispns >= 1024)
    rt_Output->fatalVar(get_class(), ":Too many areas in iallc()", NULL);
  iArs[ispns++] = new RT_RS_InsArea* [1024];
}

void RT_RS_Areas::freeh()
{
  int spns;
  long cnt;

  for(cnt = 0; cnt < dcnt; cnt++)  delete get_diff(cnt);
  for(spns = 0; spns < dspns; spns++) delete dArs[spns];
  for(cnt = 0; cnt < icnt; cnt++)  delete get_ins(cnt);
  for(spns = 0; spns < ispns; spns++) delete iArs[spns];
  dcnt = icnt = dspns = ispns = 0;
}

RT_RS_Areas::RT_RS_Areas(RT_RS_Scene* ascene)
{
  scene = ascene; dcnt = icnt = max_mesh_nr = 0;
  dspns = ispns = 0;
  diff_min_area = ins_min_area = edge_min_area =
  shadow_min_area = light_min_area =
  grad_min_area = 0;    //skip 
  grad_max_delta = RS_GRAD_MAX_DELTA;
  auto_mesh = TRUE;
}

void RT_RS_Areas::insert(RT_RS_DiffArea* a)
{
  a->set_scene(scene);
  a->comp_ar();

  int sn = (int)(dcnt >> 10);
  if(sn >= dspns) dallc();
  dArs[sn][(int)(dcnt - (sn << 10))] = a;
  dcnt++;
}

void RT_RS_Areas::insert(RT_RS_InsArea* a)
{
  a->set_scene(scene);
  a->comp_ar();

  int sn = (int)(icnt >> 10);
  if(sn >= ispns) iallc();
  iArs[sn][(int)(icnt - (sn << 10))] = a;
  icnt++;
}

void RT_RS_Areas::get(long n, RT_RS_DiffArea*& darea)
{
#ifdef RS_DEBUG
  if(n >= dcnt)
    rt_Output->fatalVar(get_class(), ":dcnt <= n in get_diff()", NULL);
  if(n < 0)
    rt_Output->fatalVar(get_class(), ":n < 0 in get_diff()", NULL);
#endif
  int sn = (int)(n >> 10);
  darea = dArs[sn][(int)(n - (sn << 10))];
}

void RT_RS_Areas::get(long n, RT_RS_InsArea*& iarea)
{
#ifdef RS_DEBUG
  if(n >= icnt)
    rt_Output->fatalVar(get_class(), ":icnt <= n in get_ins()", NULL);
  if(n < 0)
    rt_Output->fatalVar(get_class(), ":n < 0 in get_ins()", NULL);
#endif
  int sn = (int)(n >> 10);
  iarea = iArs[sn][(int)(n - (sn << 10))];
}

void RT_RS_Areas::make_square()
{
  long i, cnt;
  RT_RS_Area* area;

  cnt = get_diffcnt();
  for(i = 0; i < cnt; i++) {
    area = get_diff(i);
    if(area->neighbors[0] < 0 && area->neighbors[1] < 0 &&
       area->neighbors[2] < 0 && area->neighbors[3] < 0)
       make_square_diff(i);
  }
  cnt = get_inscnt();
  for(i = 0; i < cnt; i++) {
    area = get_ins(i);
    if(area->neighbors[0] < 0 && area->neighbors[1] < 0 &&
       area->neighbors[2] < 0 && area->neighbors[3] < 0)
      make_square_ins(i);
  }
}

void RT_RS_Areas::subdivide_init() {
  enum { init, visible, not_visible, divide, horizon } test = init;
  int m, n, o, k2, i2;
  long i, j;
  float r, min_area, dmy;
  RT_RS_Area *area1, *area2;
  RT_RS_Vtx center1, center2;
  RT_RS_Vtx *vtx2[4], *vtx;

#ifdef RS_DEBUG
  check_mesh();
#endif

  scene->Points->comp_max_scale();

  if(auto_mesh) {  //compute default values 
    diff_min_area = (float)(pow(scene->Points->SpatPar->get_volume(), 0.66) / 50.);
    ins_min_area = diff_min_area * .3;
    edge_min_area = diff_min_area * .2;
    shadow_min_area = edge_min_area * .1;
    light_min_area = edge_min_area * .2;

    grad_min_area = edge_min_area;
  }

  if (rt_RSYMsg) 
      rt_Output->message("****************** Processing polygons ...\n");

  make_square();
#ifdef RS_DEBUG
  check_mesh();
#endif

  min_area = diff_min_area;
  for(m = 0; m < 2; m++) {
    for(i = 0; i < (m ? icnt : dcnt); i++) {
      if(min_area < epsilon) break;
      if(m) area1 = get_ins(i); else area1 = get_diff(i);
      if(area1->ar < min_area  && area1->anchor_type == 'a') continue;

      if (rt_RSYMsg) {
	  if (m) 
	      sprintf(RS_s, "Creating a regular mesh - Dividing non-diffuse patch nr.: %ld ...", i);
	  else
	      sprintf(RS_s, "Creating a regular mesh - Dividing diffuse patch nr.: %ld ...", i);
	  rt_Output->message(RS_s);
      }
 
      subdivide(area1, i);

      if (rt_RSYMsg) {
	  sprintf(RS_s, "Diffuse patches now: %ld.  Non-diffuse patches now: %ld.\n", 
		  get_diffcnt(), get_inscnt());
	  rt_Output->message(RS_s);
      }

      i--;    //repeat
    }
    min_area = ins_min_area;
  }
#ifdef RS_DEBUG
  check_mesh();
#endif

  if(light_min_area > 0) {
     if (rt_RSYMsg) rt_Output->message("****************** Processing light sources ...\n ");
    for(m = 0; m < 2; m++) {
      for(i = 0; i < (m ? icnt : dcnt); i++) {
        if(m) area1 = get_ins(i); else area1 = get_diff(i);
        if(area1->ar < light_min_area && area1->anchor_type == 'a') continue;
        vtx = area1->get_vtx(area1->vidx[0]);
        if(vtx->get_acc() < epsilon) continue;

	if (rt_RSYMsg) {
	    if (m) 
		sprintf(RS_s, "Creating a regular mesh - Dividing non-diffuse (light source) patch nr.: %ld ...", i);
	    else
		sprintf(RS_s, "Creating a regular mesh - Dividing diffuse (light source) patch nr.: %ld ...", i);
	    rt_Output->message(RS_s);
	}	    
        subdivide(area1, i);

	if (rt_RSYMsg) {
	    sprintf(RS_s, "Diffuse patches now: %ld.  Non-diffuse patches now: %ld.\n", 
		    get_diffcnt(), get_inscnt());
	    rt_Output->message(RS_s);
	}
        i--;    //repeat
      }
    }
#ifdef RS_DEBUG
  check_mesh();
#endif
  }

  if(edge_min_area > 0) {
      if (rt_RSYMsg) 
	 rt_Output->message("****************** Processing edges ...\n ");
      for(m = 0; m < 2; m++) {
	  for(o = 0; o < 2; o++) {
        for(i = 0; i < (m ? icnt : dcnt); i++) {
          if(m) area1 = get_ins(i); else area1 = get_diff(i);
          if(area1->get_vtx(area1->vidx[0])->get_vtx_type() != curved)
            if(area1->neighbors[0] >= 0 && area1->neighbors[1] >= 0 &&
               area1->neighbors[2] >= 0 && area1->neighbors[3] >= 0) {
              if(!o) continue;
            } else
              if(o) continue;
          center1 = area1->comp_center();
          for(n = 0; n < 2; n++) {
            for(j = 0; j < (n ? icnt : dcnt); j++) {
              if(n) area2 = get_ins(j); else area2 = get_diff(j);
              if(area1->mesh_nr == area2->mesh_nr) continue;
              if(area2->get_vtx(area2->vidx[0])->get_vtx_type() == curved) continue;
              if(area2->neighbors[0] >= 0 && area2->neighbors[1] >= 0 &&
                area2->neighbors[2] >= 0 && area2->neighbors[3] >= 0) {
                if(o) continue;
                else
                  if(area2->ar < edge_min_area && area2->anchor_type == 'a') continue;
              }
              else
                if(area2->ar < edge_min_area && area2->anchor_type == 'a') continue;
              center2 = area2->comp_center();
              //is area2 is backfacing or behind area2 ?
              if(!scene->Points->is_visible(&center1, &center2, dmy, dmy, FALSE))
                continue;
              r = (center1.pnt - center2.pnt).vabs();
              if (r / sqrt(area2->ar) < 1.) {

		  if (rt_RSYMsg) { 
		      sprintf(RS_s, "Processing edges - Dividing ");
		      if (m) 
			  sprintf(RS_s1, "non-diffuse patch nr.: %ld ...", j);
		      else
			  sprintf(RS_s1, "diffuse patch nr.: %ld ...", j);
		      rt_Output->messageVar(RS_s, RS_s1, NULL);
		  }
		  subdivide(area2, j);

	       if (rt_RSYMsg) {
		   sprintf(RS_s, "Diffuse patches now: %ld.  Non-diffuse patches now: %ld.\n Processing ...\n", 
			   get_diffcnt(), get_inscnt());
		   rt_Output->message(RS_s);
	       }
	      
	      j--;    //repeat
              }
            }
          }
        }
      }
    }
#ifdef RS_DEBUG
  check_mesh();
#endif
  }

  if(shadow_min_area > 0) {
      if (rt_RSYMsg) rt_Output->message("****************** Processing shadow boundaries\n ");
    for(m = 0; m < 2; m++) {
      for(i = 0; i < (m ? scene->Points->get_inscnt() : scene->Points->get_diffcnt()); i++) {
        if(m) vtx = scene->Points->get_ins(i); else vtx = scene->Points->get_diff(i);
        if(vtx->get_vtx_type() == inner) continue;    //skip inner vertices of planar surfaces
        if(vtx->get_acc() < epsilon) continue;     // lightsource ?

	if (rt_RSYMsg) {
	    sprintf(RS_s, "Precomputing shadow boundaries - Testing ");    
	    if (m) 
		sprintf(RS_s1, "non-diffuse light source point nr.: %ld ...", i);
	    else
		sprintf(RS_s1, "diffuse light source point nr.: %ld ...", i);
	    
	    rt_Output->messageVar(RS_s, RS_s1, NULL);
	}

        for (n = 0; n < 2; n++) {
	    for (j = 0; j < (n ? icnt : dcnt); j++) {
		if (n) area2 = get_ins(j); else area2 = get_diff(j);
            if(area2->ar < shadow_min_area && area2->anchor_type == 'a') continue;
            if(area2->get_vtx(area2->vidx[0])->get_acc() > epsilon) continue;  // lightsource?
            k2 = area2->vidx[2] == area2->vidx[3] ? 3 : 4;

            //is area2 (partial ?) backfacing or behind area2 ?
            test = init;
            k2 = area2->vidx[2] == area2->vidx[3] ? 3 : 4;
            for(i2 = 0; i2 < k2; i2++) {
              vtx2[i2] = area2->get_vtx(area2->vidx[i2]);
              if(scene->Points->is_visible(vtx, vtx2[i2], dmy, dmy, FALSE)) {
                if(test == init )
                  test = visible;
                else
                  if(test == not_visible) {
                    test = horizon;
                    break;
                  }
              } else {
                if(test == init )
                  test = not_visible;
                else
                  if(test == visible) {
                    test = horizon;
                    break;
                  }
              }
            }
            if(test != visible) continue;

            //is area2 partially occluded from the light source ?

            test = init;
            for(i2 = 0; i2 < k2; i2++) {
              vtx2[i2] = area2->get_vtx(area2->vidx[i2]);
              if(scene->Points->is_occluded(vtx, vtx2[i2])) {
                if(test == init )
                  test = not_visible;
                else
                  if(test == visible) {
                    test = divide;
                    break;
                  }
              } else {
                if(test == init )
                  test = visible;
                else
                  if(test == not_visible) {
                    test = divide;
                    break;
                  }
              }
            }
		if (test == divide) {
		if (rt_RSYMsg) {
		    sprintf(RS_s, "Precomputing shadow boundaries - Dividing ");    
		    if(m) 
			sprintf(RS_s1, "non-diffuse patch nr.: %ld ...", j);
		    else
			sprintf(RS_s1, "diffuse patch nr.: %ld ...", j);
		    rt_Output->messageVar(RS_s, RS_s1, NULL);
		}
		subdivide(area2, j);
		
		if (rt_RSYMsg) {
		    sprintf(RS_s, "Diffuse patches now: %ld.  Non-diffuse patches now: %ld.\n Processing ...\n", 
			    get_diffcnt(), get_inscnt());
		    rt_Output->message(RS_s);
		}
		j--;
            }
	    }
        }
    }
  }
#ifdef RS_DEBUG
      check_mesh();
#endif
  }
}

void RT_RS_Areas::subdivide_grad() {
  RT_RS_Area *area;
  RT_RS_Vtx *vtx[4];

  if(grad_min_area < epsilon || grad_max_delta < epsilon) return;
  for (int m = 0; m < 2; m++) {
    for( long i = 0; i < (m ? icnt : dcnt); i++) {
      if(m) area = get_ins(i); else area = get_diff(i);
      if(area->ar < grad_min_area  && area->anchor_type == 'a') continue;
      int k = area->vidx[2] == area->vidx[3] ? 3 : 4;
      for(int j = 0; j < k; j++) vtx[j] = area->get_vtx(area->vidx[j]);
      for(j = 0; j < k; j++) {
        if(fabs(vtx[j]->get_acc() - vtx[(j + 1) % k]->get_acc()) > grad_max_delta) {
	    if (rt_RSYMsg) {
		if(m) 
		    sprintf(RS_s, 
			    "Intensity-gradient of non-diffuse patch nr.: %ld \n exceeds the specified value", i);
		else
		    sprintf(RS_s, 
			    "Intensity-gradient of diffuse patch nr.: %ld \n exceeds the specified value", i);
		sprintf(RS_s1, " - Dividing the patch ...");
		rt_Output->messageVar(RS_s, RS_s1, NULL);
	    }
	    subdivide(area, i);
	    
           if (rt_RSYMsg) {
	       sprintf(RS_s, "Diffuse patches now: %ld.  Non-diffuse patches now: %ld.\n", 
		       get_diffcnt(), get_inscnt());
	       rt_Output->message(RS_s);
	   }
	    i--;
        }
        break;
      }
    }
  }
#ifdef RS_DEBUG
  check_mesh();
#endif
}

void RT_RS_Areas::make_square_diff(long n)
{
  int  i, max_idx, max_idx1, max_idx2, max_idx3;
  long new_v_idx[2];
  float l[4], lmax, lomax;
  RT_RS_DiffVtx *old_vtx[4], *new_vtx[2];
  RT_RS_DiffArea *old_area, *new_area;

  // get the original patch
  old_area = get_diff(n);

  if(old_area->neighbors[0] >= 0 || old_area->neighbors[1] >= 0 ||
     old_area->neighbors[2] >= 0 || old_area->neighbors[3] >= 0)
    return; // divide only single polygons

  for(i = 0; i < 4; i++)
    old_vtx[i] = scene->Points->get_diff(old_area->vidx[i]);
  l[0] = (old_vtx[1]->pnt - old_vtx[0]->pnt).vabs();
  l[1] = (old_vtx[2]->pnt - old_vtx[1]->pnt).vabs();
  l[2] = (old_vtx[3]->pnt - old_vtx[2]->pnt).vabs();
  l[3] = (old_vtx[0]->pnt - old_vtx[3]->pnt).vabs();
  lmax = l[0]; max_idx = 0;
  for(i = 1; i < 4; i++)
    if(l[i] > lmax) {
      lmax = l[i];
      max_idx = i;
    }
  max_idx1 = (max_idx + 1) % 4;
  max_idx2 = (max_idx + 2) % 4;
  max_idx3 = (max_idx + 3) % 4;
  lomax = l[max_idx1];
  int ld = (int)floor((lmax / lomax) + .5);
  if(ld < 2) return;

  // subtract the area from the associated vertices
  old_area->sub_ar();

  float recip_ld = 1. / (float)ld;
  // the increments:
  RT_RS_3DVector d_pnt((old_vtx[max_idx1]->pnt - old_vtx[max_idx]->pnt) * recip_ld);
  RT_RS_3DVector d_nrm((old_vtx[max_idx1]->nrm - old_vtx[max_idx]->nrm) * recip_ld);
  RT_RS_3DVector do_pnt((old_vtx[max_idx2]->pnt - old_vtx[max_idx3]->pnt) * recip_ld);
  RT_RS_3DVector do_nrm((old_vtx[max_idx2]->nrm - old_vtx[max_idx3]->nrm) * recip_ld);

  long old_v_idx1 = old_area->vidx[max_idx1];
  long old_v_idx2 = old_area->vidx[max_idx2];

  RT_RS_DiffVtx d_vtx(*old_vtx[max_idx]);
  RT_RS_DiffVtx d_vtx3(*old_vtx[max_idx3]);
  d_vtx.set_vtx_type(inner); d_vtx3.set_vtx_type(inner);

  for(i = 0; i < ld; i++) {
    // compute the new vertices
    if(i < ld-1) {
      d_vtx.pnt = d_vtx.pnt + d_pnt;
      d_vtx.nrm = d_vtx.nrm + d_nrm;
      new_vtx[0] = new RT_RS_DiffVtx(d_vtx);
      new_v_idx[0] = scene->Points->get_diffcnt();
      scene->Points->insert(new_vtx[0]);

      d_vtx3.pnt = d_vtx3.pnt + do_pnt;
      d_vtx3.nrm = d_vtx3.nrm + do_nrm;
      new_vtx[1] = new RT_RS_DiffVtx(d_vtx3);
      new_v_idx[1] = scene->Points->get_diffcnt();
      scene->Points->insert(new_vtx[1]);
    }
    if(i == 0) {
      old_area->vidx[max_idx1] = new_v_idx[0];
      old_area->vidx[max_idx2] = new_v_idx[1];
      old_area->comp_ar();
      old_area->neighbors[max_idx1] = get_diffcnt();
    } else {
      if(i < ld - 1) {
        //create the new patch
        new_area = new RT_RS_DiffArea(
          old_area->vidx[i == 1 ? max_idx1 : 1], new_v_idx[0],
          new_v_idx[1], old_area->vidx[i == 1 ? max_idx2 : 2],
          -1, get_diffcnt() + 1, -1, n, old_area->mesh_nr);
        old_area = new_area; n = get_diffcnt();
        insert(new_area);
      } else { // the last new patch
        new_area = new RT_RS_DiffArea(
          old_area->vidx[i == 1 ? max_idx1 : 1], old_v_idx1,
          old_v_idx2, old_area->vidx[i == 1 ? max_idx2 : 2],
          -1, -1, -1, n, old_area->mesh_nr);
        insert(new_area);
      }
    }
  }
}

void RT_RS_Areas::make_square_ins(long n)
{
  int  i, max_idx, max_idx1, max_idx2, max_idx3;
  long new_v_idx[2];
  float l[4], lmax, lomax;
  RT_RS_InsVtx *old_vtx[4], *new_vtx[2];
  RT_RS_InsArea *old_area, *new_area;

  // get the original patch
  old_area = get_ins(n);

  if(old_area->neighbors[0] >= 0 || old_area->neighbors[1] >= 0 ||
    old_area->neighbors[2] >= 0 || old_area->neighbors[3] >= 0)
    return;

  for(i = 0; i < 4; i++)
    old_vtx[i] = scene->Points->get_ins(old_area->vidx[i]);
  l[0] = (old_vtx[1]->pnt - old_vtx[0]->pnt).vabs();
  l[1] = (old_vtx[2]->pnt - old_vtx[1]->pnt).vabs();
  l[2] = (old_vtx[3]->pnt - old_vtx[2]->pnt).vabs();
  l[3] = (old_vtx[0]->pnt - old_vtx[3]->pnt).vabs();
  lmax = l[0]; max_idx = 0;
  for(i = 1; i < 4; i++)
    if(l[i] > lmax) {
      lmax = l[i];
      max_idx = i;
    }
  max_idx1 = (max_idx + 1) % 4;
  max_idx2 = (max_idx + 2) % 4;
  max_idx3 = (max_idx + 3) % 4;
  lomax = l[max_idx1];
  int ld = (int)floor((lmax / lomax) + .5);
  if(ld < 2) return;

  // subtract the area from the associated vertices
  old_area->sub_ar();

  float recip_ld = 1. / (float)ld;
  // the increments:
  RT_RS_3DVector d_pnt((old_vtx[max_idx1]->pnt - old_vtx[max_idx]->pnt) * recip_ld);
  RT_RS_3DVector d_nrm((old_vtx[max_idx1]->nrm - old_vtx[max_idx]->nrm) * recip_ld);
  RT_RS_3DVector d_tgn((old_vtx[max_idx1]->tgn - old_vtx[max_idx]->tgn) * recip_ld);
  RT_RS_3DVector do_pnt((old_vtx[max_idx2]->pnt - old_vtx[max_idx3]->pnt) * recip_ld);
  RT_RS_3DVector do_nrm((old_vtx[max_idx2]->nrm - old_vtx[max_idx3]->nrm) * recip_ld);
  RT_RS_3DVector do_tgn((old_vtx[max_idx2]->tgn - old_vtx[max_idx3]->tgn) * recip_ld);

  long old_v_idx1 = old_area->vidx[max_idx1];
  long old_v_idx2 = old_area->vidx[max_idx2];

  RT_RS_InsVtx d_vtx(*old_vtx[max_idx]);
  RT_RS_InsVtx d_vtx3(*old_vtx[max_idx3]);
  d_vtx.set_vtx_type(inner); d_vtx3.set_vtx_type(inner);

  for(i = 0; i < ld; i++) {
    // compute the new vertices
    if(i < ld-1) {
      d_vtx.pnt = d_vtx.pnt + d_pnt;
      d_vtx.nrm = d_vtx.nrm + d_nrm;
      d_vtx.tgn = d_vtx.tgn + d_tgn;
      new_vtx[0] = new RT_RS_InsVtx(d_vtx);
      new_v_idx[0] = scene->Points->get_inscnt();
      scene->Points->insert(new_vtx[0]);

      d_vtx3.pnt = d_vtx3.pnt + do_pnt;
      d_vtx3.nrm = d_vtx3.nrm + do_nrm;
      d_vtx3.tgn = d_vtx3.tgn + do_tgn;
      new_vtx[1] = new RT_RS_InsVtx(d_vtx3);
      new_v_idx[1] = scene->Points->get_inscnt();
      scene->Points->insert(new_vtx[1]);
    }
    if(i == 0) {
      old_area->vidx[max_idx1] = new_v_idx[0];
      old_area->vidx[max_idx2] = new_v_idx[1];
      old_area->comp_ar();
      old_area->neighbors[max_idx1] = get_inscnt();
    } else {
      if(i < ld - 1) {
        //create the new patch
        new_area = new RT_RS_InsArea(
          old_area->vidx[i == 1 ? max_idx1 : 1], new_v_idx[0],
          new_v_idx[1], old_area->vidx[i == 1 ? max_idx2 : 2],
          -1, get_inscnt() + 1, -1, n, old_area->mesh_nr);
        old_area = new_area; n = get_inscnt();
        insert(new_area);
      } else { // the last new patch
        new_area = new RT_RS_InsArea(
          old_area->vidx[i == 1 ? max_idx1 : 1], old_v_idx1,
          old_v_idx2, old_area->vidx[i == 1 ? max_idx2 : 2],
          -1, -1, -1, n, old_area->mesh_nr);
        insert(new_area);
      }
    }
  }
}

#ifdef RS_DEBUG
void RT_RS_Areas::check_mesh()
{
  long i, n_a, n_p;
  int j, k, m, nidx;
  boolean tri;
  float ar, par;
  RT_RS_Area *a, *na;
  RT_RS_Vtx* p;

  n_a = get_diffcnt(); n_p = scene->Points->get_diffcnt();
  for(m = 0; m < 2; m++) {
    ar = par = .0;
    for(i = 0; i < n_a; i++) {
      if(m) a = get_ins(i); else a = get_diff(i);
/*    if(a->ar < epsilon) {
        sprintf(RS_s, "%s:Area of patch %ld is %f!", get_class(), i, a->ar);
        rt_Output->fatal(RS_s);
      }
*/    ar += a->ar;
      tri = a->vidx[2] == a->vidx[3];
      if(tri && (a->anchor_type == 'a' || a->anchor_type == 'i' ||
                 a->anchor_type == 'l')) {
        a->print(stderr, "patch:");
        sprintf(RS_s, "%s:Anchor type of patch %ld is %c!", get_class(), i, a->anchor_type);
        rt_Output->fatal(RS_s);
      }
      for(k = 0; k < 4; k++) {
        if(a->neighbors[k] > 0) {
          if(tri && k > 2) break;
          if(m) na = get_ins(a->neighbors[k]);
          else na = get_diff(a->neighbors[k]);
          nidx = -1;
          for(j = 0; j < 4; j++) if(na->neighbors[j] == i) nidx = j;
          if(nidx < 0) {
            a->print(stderr, "patch:"); na->print(stderr, "neighbor:");
            sprintf(RS_s, "%s:nidx < 0! patch: %ld  neighbor: %d  (%ld)",
                    get_class(), i, k, a->neighbors[k]);
            rt_Output->fatal(RS_s);
          }
          if(k > 1 && tri) k++;
          if(a->vidx[k] != na->vidx[(nidx + 1) % 4]) {
            a->print(stderr, "patch:"); na->print(stderr, "neighbor:");
            sprintf(RS_s, "%s:vidx[%d] of patch %ld != neighbor[%d]->vidx[%d] (%ld)!",
                    get_class(), k, i, k, (nidx + 1) % 4, a->neighbors[k]);
            rt_Output->fatal(RS_s);
          }
          if(a->vidx[(k + 1) % 4] != na->vidx[nidx]) {
            a->print(stderr, "patch:"); na->print(stderr, "neighbor:");
            sprintf(RS_s, "%s:vidx[%d] of patch %ld != neighbor[%d]->vidx[%d] (%ld)!",
                    get_class(), (k + 1) % 4, i, k, nidx, a->neighbors[k]);
            rt_Output->fatal(RS_s);
          }
        }
      }
    }
    for(i = 0; i < n_p; i++) {
      if(m) p = scene->Points->get_ins(i);
      else p = scene->Points->get_diff(i);
      if(p->get_ar() < epsilon) {
        p->print(stderr, "vertex:");
        sprintf(RS_s, "%s:Area of vertex %ld is %f!", get_class(), i, p->get_ar());
        rt_Output->fatal(RS_s);
      }
      par += p->get_ar();
    }
    if(fabs(ar - par) > fabs(ar * 0.001)) {
      sprintf(RS_s, "%s:Total area of the vertices is %f. Total area of the patches is %f.",
              get_class(), par, ar);
      rt_Output->fatal(RS_s);
    }
    n_a = get_inscnt(); n_p = scene->Points->get_inscnt();
  }
}
#endif

void RT_RS_Areas::subdivide(RT_RS_Area* a, long n)
{
  long dmy;
  RT_RS_Area* na;

  // anchored ?
  switch(a->anchor_type) {

    case 'b': re_anchor1(a, n, a->neighbors[1], a->neighbors[2]);
              return;

    case 'c': na = a->get_area(a->neighbors[2]);
              re_anchor1(na, a->neighbors[2], n, na->neighbors[2]);
              return;

    case 'd': na = a->get_area(a->neighbors[2]);
              re_anchor1(na, a->neighbors[2], na->neighbors[1], n);
              return;

    case 'e': na = a->get_area(a->neighbors[2]);
              re_anchor2a(a, n, na->neighbors[0], na->neighbors[1],
                a->neighbors[2]);
              return;

    case 'f': na = a->get_area(a->neighbors[2]);
              re_anchor2a(a->get_area(na->neighbors[2]), na->neighbors[2], n,
                               na->neighbors[1], a->neighbors[2]);
              return;

    case 'g': na = a->get_area(a->neighbors[2]);
              re_anchor2a(a->get_area(na->neighbors[2]), na->neighbors[2],
                na->neighbors[0], n, a->neighbors[2]);
              return;

    case 'h': re_anchor2a(a->get_area(a->neighbors[2]), a->neighbors[2], a->neighbors[0],
                a->neighbors[1], n);
              return;

    case 'i': re_anchor2b(a, n, a->neighbors[3], a->neighbors[1]);
              return;

    case 'j': na = a->get_area(a->neighbors[2]);
              re_anchor2b(na, a->neighbors[2], n, na->neighbors[1]);
              return;

    case 'k': na = a->get_area(a->neighbors[2]);
              re_anchor2b(na, a->neighbors[2], na->neighbors[3], n);
              return;

    case 'l': na = a->get_area(a->neighbors[3]);
              re_anchor3(a, n, na->neighbors[0], na->neighbors[1],
                a->neighbors[3], dmy, dmy);
              return;

    case 'm': na = a->get_area(a->neighbors[2]);
              re_anchor3(a->get_area(na->neighbors[2]), na->neighbors[2], n,
                na->neighbors[1], a->neighbors[2], dmy, dmy);
              return;

    case 'n': na = a->get_area(a->neighbors[2]);
              re_anchor3(a->get_area(na->neighbors[2]), na->neighbors[2],
                na->neighbors[0], n, a->neighbors[2], dmy, dmy);
              return;

    case 'o': re_anchor3(a->get_area(a->neighbors[2]), a->neighbors[2],
                a->neighbors[0], a->neighbors[1], n, dmy, dmy);
              return;
  }

  //type is "a" (not anchored):
  int i, j, k;
  long old_neighbors[4], new_ar_idx[4], new_v_idx[5], vtx[2];
  RT_RS_Vtx* new_vtx[5];
  RT_RS_Area* new_areas[4];

  // subtract the area from the associated vertices
  a->sub_ar();

  // get the original vertices and neighbors
  for(i = 0; i < 4; i++)
    old_neighbors[i] =  a->neighbors[i];

  // compute the center vertex
  a->insert_ip(4, a->vidx, new_vtx[0], new_v_idx[0]);

  // compute the new outer vertices
  for(i = 0; i < 4; i++) {
    j = i + 1; k = j % 4;
    vtx[0] = a->vidx[i]; vtx[1] = a->vidx[k];
    a->insert_ip(2, vtx, new_vtx[j], new_v_idx[j]);
  }

  //create the new patches
  //0: (the original patch)
  new_areas[0] = a;
  new_ar_idx[0] = n;
  //1:
  new_areas[1] = a->new_area(
    new_v_idx[1], new_areas[0]->vidx[1], new_v_idx[2], new_v_idx[0],
    -1, -1, a->get_cnt() + 1, new_ar_idx[0], new_areas[0]->mesh_nr);
  new_ar_idx[1] = a->get_cnt();
  a->insert(new_areas[1]);
  //2:
  new_areas[2] = a->new_area(
    new_v_idx[2], new_areas[0]->vidx[2], new_v_idx[3], new_v_idx[0],
    -1, -1, a->get_cnt() + 1, new_ar_idx[1], new_areas[0]->mesh_nr);
  new_ar_idx[2] = a->get_cnt();
  a->insert(new_areas[2]);
  //3:
  new_areas[3] = a->new_area(
    new_v_idx[3], new_areas[0]->vidx[3], new_v_idx[4], new_v_idx[0],
    -1, -1, new_ar_idx[0], new_ar_idx[2], new_areas[0]->mesh_nr);
  new_ar_idx[3] = a->get_cnt();
  a->insert(new_areas[3]);

  //update the original patch
  new_areas[0]->vidx[1] = new_areas[0]->vidx[0];
  new_areas[0]->vidx[0] = new_v_idx[4];
  new_areas[0]->vidx[2] = new_v_idx[1];
  new_areas[0]->vidx[3] = new_v_idx[0];
  new_areas[0]->neighbors[2] = new_ar_idx[1];
  new_areas[0]->neighbors[3] = new_ar_idx[3];
  new_areas[0]->comp_ar();

  new_areas[1]->anchor_type = new_areas[2]->anchor_type =
  new_areas[3]->anchor_type = 'a';

  //anchor the neighbors
  for(i = 0; i < 4; i++)  {
    int cy_idx1 = (i + 1) % 4;
    if(old_neighbors[i] >= 0) {
      anchor1(a->get_area(old_neighbors[i]), old_neighbors[i], n, new_ar_idx[i],
        new_ar_idx[cy_idx1], new_v_idx[i + 1],
        new_areas[i]->neighbors[1], new_areas[cy_idx1]->neighbors[0]);
    } else {
      new_areas[i]->neighbors[1] = -1;
      new_areas[cy_idx1]->neighbors[0] = -1;
    }
  }
}

void RT_RS_Areas::anchor1(RT_RS_Area* a, long n, long to_old_area,
       long to_new_area1, long to_new_area2,
       long shared_vtxidx, long& area1_nb, long& area2_nb)
{
  RT_RS_Area *na, *na1;

  // anchored ?
  switch(a->anchor_type) {

    case 'f':
    case 'm':
    case 'n': subdivide(a, n); break;

    case 'i': subdivide(a, n);
              na = a->get_area(a->neighbors[2]);
              na1 = a->get_area(na->neighbors[2]);
              if(to_old_area == na1->neighbors[0]) {
                n = na->neighbors[2];
                a = na1;
              };
              break;

    case 'b': anchor1to2b(a, n, a->neighbors[1], a->neighbors[2],
                to_new_area1, to_new_area2, shared_vtxidx);
              area1_nb = n; area2_nb = a->neighbors[1];
              return;

    case 'c': if(to_old_area == a->neighbors[0]) {
                na = a->get_area(a->neighbors[2]);
                anchor1to2a(na, a->neighbors[2], n, na->neighbors[2],
                  to_old_area, to_new_area1, to_new_area2,
                  shared_vtxidx, area1_nb, area2_nb);
                return;
              }
              subdivide(a, n); break;

    case 'd': if(to_old_area == a->neighbors[1]) {
                na = a->get_area(a->neighbors[2]);
                anchor1to2a(na, a->neighbors[2], na->neighbors[1], n,
                  to_old_area, to_new_area1, to_new_area2,
                  shared_vtxidx, area1_nb, area2_nb);
                return;
              }
              subdivide(a, n); break;

    case 'e': if(to_old_area == a->neighbors[0]) {
                na = a->get_area(a->neighbors[2]);
                anchor2ato3(a, n, na->neighbors[0], na->neighbors[1],
                  a->neighbors[2], to_old_area, to_new_area1, to_new_area2,
                  shared_vtxidx, area1_nb, area2_nb);
                return;
              }
              subdivide(a, n); break;

    case 'g': if(to_old_area == a->neighbors[1]) {
                na = a->get_area(a->neighbors[2]);
                anchor2ato3(a->get_area(na->neighbors[2]), na->neighbors[2],
                  na->neighbors[0], n, a->neighbors[2], to_old_area,
                  to_new_area1, to_new_area2,
                  shared_vtxidx, area1_nb, area2_nb);
                return;
              }
              subdivide(a, n); break;

    case 'j': if(to_old_area == a->neighbors[0]) {
                na = a->get_area(a->neighbors[2]);
                anchor2bto3(na, a->neighbors[2], n, na->neighbors[1],
                  to_old_area, to_new_area1, to_new_area2,
                  shared_vtxidx, area1_nb, area2_nb);
                return;
              }
              subdivide(a, n); break;

    case 'k': if(to_old_area == a->neighbors[0]) {
                na = a->get_area(a->neighbors[2]);
                anchor2bto3(na, a->neighbors[2], na->neighbors[3], n,
                  to_old_area, to_new_area1, to_new_area2,
                  shared_vtxidx, area1_nb, area2_nb);
                return;
              }
              subdivide(a, n); break;

    case 'l': na = a->get_area(a->neighbors[3]);
              if(to_old_area == a->neighbors[1]) {
                re_anchor3(a, n, na->neighbors[0], na->neighbors[1],
                  a->neighbors[3], area1_nb, area2_nb,
                  to_old_area, to_new_area1, to_new_area2,
                  shared_vtxidx);
                return;
              }
              subdivide(a, n);
              if(to_old_area == na->neighbors[0]) {
                n = a->neighbors[3];
                a = na;
              }
  }

  //type is "a" (not anchored):
  int i, n_idx;
  long ar1_vtxidx, ar2_vtxidx, area1_vtxidx, area2_vtxidx,
       ac_idx, ad_idx;
  RT_RS_Area *ac, *ad;

  // subtract the area from the associated vertices
  a->sub_ar();

  area1_vtxidx = area2_vtxidx = ar1_vtxidx = ar2_vtxidx = -1;
  for(i = 0; i < 4; i++) {
    if(a->neighbors[i] == to_old_area) {
      area1_vtxidx = a->vidx[(i + 1) % 4];
      area2_vtxidx = a->vidx[i];
      ar1_vtxidx = a->vidx[(i + 2) % 4];
      ar2_vtxidx = a->vidx[(i + 3) % 4];
      n_idx = i;
    }
  }

  //create the new patches "c" and "d"
  //c
  ac = a->new_area(
    ar2_vtxidx, area2_vtxidx, shared_vtxidx, shared_vtxidx,
    a->neighbors[(n_idx + 3) % 4], to_new_area2, n, n, a->mesh_nr);
  ac_idx = area2_nb = a->get_cnt();
  a->insert(ac);
  //d
  ad = a->new_area(
    shared_vtxidx, area1_vtxidx, ar1_vtxidx, ar1_vtxidx,
    to_new_area1, a->neighbors[(n_idx + 1) % 4], n, n, a->mesh_nr);
  ad_idx = area1_nb = a->get_cnt();
  a->insert(ad);

  a->anchor_type = 'b'; ac->anchor_type = 'c'; ad->anchor_type = 'd';

  //update the neighbors of the original patch  (if any)
  if(ac->neighbors[0] >= 0) {
    na = a->get_area(ac->neighbors[0]);
    for(i = 0; i < 4; i++)
      if(na->neighbors[i] == n) {
        na->neighbors[i] = ac_idx;
        break;
      }
  }
  if(ad->neighbors[1] >= 0) {
    na = a->get_area(ad->neighbors[1]);
    for(i = 0; i < 4; i++)
      if(na->neighbors[i] == n) {
        na->neighbors[i] = ad_idx;
        break;
      }
  }

  //update the original patch
  //a->b
  a->vidx[2] = a->vidx[3] = shared_vtxidx;
  a->vidx[0] = ar1_vtxidx;
  a->vidx[1] = ar2_vtxidx;
  a->neighbors[0] = a->neighbors[(n_idx + 2) % 4];
  a->neighbors[1] = ac_idx;
  a->neighbors[2] = a->neighbors[3] = ad_idx;
  a->comp_ar();
}

void RT_RS_Areas::anchor1to2a(RT_RS_Area* ab, long ab_idx, long ac_idx, long ad_idx,
       long to_old_area, long to_new_area1, long to_new_area2,
       long shared_vtxidx, long& area1_nb, long& area2_nb)
{
  long ah_idx;
  RT_RS_Area *ac, *ad, *ah;

  // get the 3 patches of the anchored structure
  ac = ab->get_area(ac_idx); ad = ab->get_area(ad_idx);

  // subtract the area from the associated vertices
  ab->sub_ar(); ac->sub_ar(); ad->sub_ar();

  ah_idx = ab->get_cnt();
  if(to_old_area >= 0 && to_old_area == ac->neighbors[0]) {
    //create new patch "h"
    ah = ab->new_area(
      shared_vtxidx, ad->vidx[0], ad->vidx[2], ad->vidx[2],
      ac_idx, ad_idx, ab_idx, ab_idx, ab->mesh_nr);
    //update the original patches
    //b->e
    ab->vidx[2] = ab->vidx[3] = shared_vtxidx;
    ab->neighbors[1] = to_new_area2;
    ab->neighbors[2] = ab->neighbors[3] = ah_idx;
    ab->comp_ar(); ab->anchor_type = 'e';
    //c->f
    ac->vidx[0] = shared_vtxidx;
    ac->neighbors[0] = to_new_area1;
    ac->neighbors[2] = ac->neighbors[3] = ah_idx;
    ac->comp_ar(); ac->anchor_type = 'f';
    //d->g
    ad->neighbors[2] = ad->neighbors[3] = ah_idx;
    ad->comp_ar(); ad->anchor_type = 'g';

    area1_nb = ac_idx; area2_nb = ab_idx;

  } else {
    //create new patch "h"
    ah = ab->new_area(
      ad->vidx[0], shared_vtxidx, ab->vidx[1], ab->vidx[1],
      ad_idx, ab_idx, ac_idx, ac_idx, ab->mesh_nr);
    //update the original patches
    //b->g
    ab->vidx[2] = ab->vidx[3] = ab->vidx[1];
    ab->vidx[1] = ab->vidx[0];
    ab->vidx[0] = shared_vtxidx;
    ab->neighbors[1] = ab->neighbors[0];
    ab->neighbors[0] = to_new_area1;
    ab->neighbors[2] = ab->neighbors[3] = ah_idx;
    ab->comp_ar(); ab->anchor_type = 'g';
    //c->e
    ac->neighbors[2] = ac->neighbors[3] = ah_idx;
    ac->comp_ar(); ac->anchor_type = 'e';
    //d->f
    ad->vidx[2] = ad->vidx[3] = shared_vtxidx;
    ad->neighbors[1] = to_new_area2;
    ad->neighbors[2] = ad->neighbors[3] = ah_idx;
    ad->comp_ar(); ad->anchor_type = 'f';

    area1_nb = ab_idx; area2_nb = ad_idx;
  }
  ab->insert(ah);
  ah->anchor_type = 'h';
}

void RT_RS_Areas::anchor1to2b(RT_RS_Area* ab, long ab_idx, long ac_idx, long ad_idx,
       long to_new_area1, long to_new_area2, long shared_vtxidx)
{
  int i;
  RT_RS_Area *ac, *ad, *na;

  // get the 3 patches of the anchored structure
  ac = ab->get_area(ac_idx); ad = ab->get_area(ad_idx);

  // subtract the area from the associated vertices
  ab->sub_ar(); ad->sub_ar();

  //update the original patches
  //b->i
  ab->vidx[0] = ad->vidx[0];
  ab->vidx[1] = ad->vidx[1];
  ab->vidx[2] = shared_vtxidx;
  ab->vidx[3] = ac->vidx[0];
  ab->neighbors[0] = ad->neighbors[0];
  ab->neighbors[1] = ad_idx;
  ab->neighbors[2] = to_new_area1;
  ab->neighbors[3] = ac_idx;
  ab->comp_ar(); ab->anchor_type = 'i';
  //c->j
  ac->anchor_type = 'j';
  //d->k
  ad->vidx[0] = ad->vidx[1];
  ad->vidx[1] = ad->vidx[2];
  ad->vidx[2] = ad->vidx[3] = shared_vtxidx;
  ad->neighbors[0] = ad->neighbors[1];
  ad->neighbors[1] = to_new_area2;
  ad->comp_ar(); ad->anchor_type = 'k';

  //update the neighbors
  if(ab->neighbors[0] >= 0) {
    na = ab->get_area(ab->neighbors[0]);
    for(i = 0; i < 4; i++)
      if(na->neighbors[i] == ad_idx) {
        na->neighbors[i] = ab_idx;
        break;
      }
  }
}

void RT_RS_Areas::anchor2ato3(RT_RS_Area* ae, long ae_idx, long af_idx,
       long ag_idx, long ah_idx,
       long to_old_area, long to_new_area1, long to_new_area2,
       long shared_vtxidx, long& area1_nb, long& area2_nb)
{
  RT_RS_Area *af, *ag, *ah;

  // get the 3 patches of the anchored structure
  af = ae->get_area(af_idx); ag = ae->get_area(ag_idx); ah = ae->get_area(ah_idx);

  // subtract the area from the associated vertices
  ae->sub_ar(); ag->sub_ar(); ah->sub_ar();

  //update the original patches
  if(to_old_area >= 0 && to_old_area == ae->neighbors[0]) {
    //e->m
    ae->vidx[0] = shared_vtxidx;
    ae->neighbors[0] = to_new_area1;
    ae->comp_ar(); ae->anchor_type = 'm';
    //f->n
    af->anchor_type = 'n';
    //g->l
    ag->vidx[3] = shared_vtxidx;
    ag->neighbors[2] = to_new_area2;
    ag->comp_ar(); ag->anchor_type = 'l';
    //h->o
    ah->vidx[2] = ah->vidx[3] = ah->vidx[1];
    ah->vidx[1] = ah->vidx[0];
    ah->vidx[0] = shared_vtxidx;
    ah->neighbors[0] = ae_idx;
    ah->neighbors[1] = af_idx;
    ah->neighbors[2] = ah->neighbors[3]  = ag_idx;
    ah->comp_ar(); ah->anchor_type = 'o';

    area1_nb = ae_idx; area2_nb = ag_idx;

  } else {
    //e->l
    ae->vidx[2] = ae->vidx[1];
    ae->vidx[1] = ae->vidx[0];
    ae->vidx[0] = shared_vtxidx;
    ae->neighbors[2] = ae->neighbors[1];
    ae->neighbors[1] = ae->neighbors[0];
    ae->neighbors[0] = to_new_area1;
    ae->comp_ar(); ae->anchor_type = 'l';
    //f->m
    af->anchor_type = 'm';
    //g->n
    ag->vidx[2] = ag->vidx[3] = shared_vtxidx;
    ag->neighbors[1] = to_new_area2;
    ag->comp_ar(); ag->anchor_type = 'n';
    //h->o
    ah->vidx[2] = ah->vidx[3] = shared_vtxidx;
    ah->comp_ar(); ah->anchor_type = 'o';

    area1_nb = ae_idx; area2_nb = ag_idx;
  }
}

void RT_RS_Areas::anchor2bto3(RT_RS_Area* ai, long ai_idx, long aj_idx, long ak_idx,
       long to_old_area, long to_new_area1, long to_new_area2,
       long shared_vtxidx, long& area1_nb, long& area2_nb)
{
  int i;
  long ao_idx;
  RT_RS_Area *aj, *ak, *ao, *na;

  // get the 3 patches of the anchored structure
  aj = ai->get_area(aj_idx); ak = ai->get_area(ak_idx);

  // subtract the area from the associated vertices
  ai->sub_ar(); aj->sub_ar(); ak->sub_ar();

  ao_idx = ai->get_cnt();
  if(to_old_area >= 0 && to_old_area == aj->neighbors[0]) {
    //create new patch "o"
    ao = ai->new_area(
      ai->vidx[2], shared_vtxidx, ai->vidx[0], ai->vidx[0],
      ai_idx, aj_idx, ak_idx, ak_idx, ai->mesh_nr);
    //update the original patches
    //k->l
    ak->vidx[2] = ak->vidx[1];
    ak->vidx[1] = ak->vidx[0];
    ak->vidx[0] = ai->vidx[0];
    ak->neighbors[3] = ao_idx;
    ak->neighbors[2] = ak->neighbors[1];
    ak->neighbors[1] = ak->neighbors[0];
    ak->neighbors[0] = ai->neighbors[0];
    ak->comp_ar(); ak->anchor_type = 'l';
    //i->m
    ai->vidx[0] = ai->vidx[2];
    ai->vidx[1] = ai->vidx[3];
    ai->vidx[2] = ai->vidx[3] = shared_vtxidx;
    ai->neighbors[0] = ai->neighbors[2];
    ai->neighbors[1] = to_new_area2;
    ai->neighbors[2] = ai->neighbors[3] = ao_idx;
    ai->comp_ar(); ai->anchor_type = 'm';
    //j->n
    aj->vidx[0] = shared_vtxidx;
    aj->neighbors[0] = to_new_area1;
    aj->neighbors[2] = aj->neighbors[3] = ao_idx;
    aj->comp_ar(); aj->anchor_type = 'n';

    area1_nb = aj_idx; area2_nb = ai_idx;

    //update the neighbors
    if(ak->neighbors[0] >= 0) {
      na = ai->get_area(ak->neighbors[0]);
      for(i = 0; i < 4; i++)
        if(na->neighbors[i] == ai_idx) {
          na->neighbors[i] = ak_idx;
          break;
        }
    }
  } else {
    //create new patch "o"
    ao = ai->new_area(
      ai->vidx[0], shared_vtxidx, ai->vidx[2], ai->vidx[2],
      ai_idx, ak_idx, aj_idx, aj_idx, ai->mesh_nr);
    //update the original patches
    //j->l
    aj->vidx[2] = aj->vidx[1];
    aj->vidx[1] = aj->vidx[0];
    aj->vidx[0] = ai->vidx[2];
    aj->neighbors[3] = ao_idx;
    aj->neighbors[2] = aj->neighbors[1];
    aj->neighbors[1] = aj->neighbors[0];
    aj->neighbors[0] = ai->neighbors[2];
    aj->comp_ar(); aj->anchor_type = 'l';
    //i->m
    ai->vidx[2] = ai->vidx[3] = shared_vtxidx;
    ai->neighbors[1] = to_new_area2;
    ai->neighbors[2] = ai->neighbors[3] = ao_idx;
    ai->comp_ar(); ai->anchor_type = 'm';
    //k->n
    ak->vidx[0] = shared_vtxidx;
    ak->neighbors[0] = to_new_area1;
    ak->neighbors[2] = ak->neighbors[3] = ao_idx;
    ak->comp_ar(); ak->anchor_type = 'n';

    area1_nb = ak_idx; area2_nb = ai_idx;

    //update the neighbors
    if(aj->neighbors[0] >= 0) {
      na = ai->get_area(aj->neighbors[0]);
      for(i = 0; i < 4; i++)
        if(na->neighbors[i] == ai_idx) {
          na->neighbors[i] = aj_idx;
          break;
        }
    }
  }
  ai->insert(ao);
  ao->anchor_type = 'o';
}

void RT_RS_Areas::re_anchor1(RT_RS_Area* ab, long ab_idx, long ac_idx, long ad_idx)
{
  long shared_vtxidx, area1_nb, area2_nb, dmy;
  RT_RS_Vtx* shared_vtx;
  RT_RS_Area* ad = ab->get_area(ad_idx);

  // compute the new outer vertex
  ab->insert_ip(2, &ad->vidx[1], shared_vtx, shared_vtxidx);

  if(ad->neighbors[1] >= 0)
    anchor1(ab->get_area(ad->neighbors[1]), ad->neighbors[1], ad_idx, ad_idx, ab_idx,
      shared_vtxidx, area1_nb, area2_nb);
  else area1_nb = area2_nb = -1;

  anchor1to2a(ab, ab_idx, ac_idx, ad_idx, ad->neighbors[1], area2_nb, area1_nb,
              shared_vtxidx, dmy, dmy);

  re_anchor2a(ab->get_area(ac_idx), ac_idx, ad_idx, ab_idx, ad->neighbors[2]);
}

void RT_RS_Areas::re_anchor2a(RT_RS_Area* ae, long ae_idx, long af_idx,
                              long ag_idx, long ah_idx)
{
  long shared_vtxidx, area1_nb, area2_nb, dmy;
  RT_RS_Vtx* shared_vtx;
  RT_RS_Area* ag = ae->get_area(ag_idx);

  // compute the new outer vertex
  ae->insert_ip(2, &ag->vidx[1], shared_vtx, shared_vtxidx);

  if(ag->neighbors[1] >= 0)
    anchor1(ae->get_area(ag->neighbors[1]), ag->neighbors[1], ag_idx, ag_idx, ae_idx,
      shared_vtxidx, area1_nb, area2_nb);
  else area1_nb = area2_nb = -1;

  anchor2ato3(ae, ae_idx, af_idx, ag_idx, ah_idx,
              ag->neighbors[1], area2_nb, area1_nb,
              shared_vtxidx, dmy, dmy);
  re_anchor3(ae, ae_idx, af_idx, ag_idx, ah_idx, dmy, dmy);
}

void RT_RS_Areas::re_anchor2b(RT_RS_Area* ai, long ai_idx, long aj_idx, long ak_idx)
{
  long shared_vtxidx, area1_nb, area2_nb, dmy;
  RT_RS_Vtx* shared_vtx;
  RT_RS_Area* ak = ai->get_area(ak_idx);

  // compute the new outer vertex
  ai->insert_ip(2, &ak->vidx[0], shared_vtx, shared_vtxidx);

  if(ak->neighbors[0] >= 0)
    anchor1(ai->get_area(ak->neighbors[0]), ak->neighbors[0], ak_idx, ai_idx, ak_idx,
      shared_vtxidx, area1_nb, area2_nb);
  else area1_nb = area2_nb = -1;

  anchor2bto3(ai, ai_idx, aj_idx, ak_idx,
              ak->neighbors[0], area2_nb, area1_nb,
              shared_vtxidx, dmy, dmy);

  re_anchor3(ai->get_area(aj_idx), aj_idx, ai_idx, ak_idx, ak->neighbors[2], dmy, dmy);
}


void RT_RS_Areas::re_anchor3(RT_RS_Area* al, long al_idx, long am_idx,
       long an_idx, long ao_idx,
       long& area1_nb, long& area2_nb, long to_old_area,
       long to_new_area1, long to_new_area2, long shared_vtxidx)
{
  int i;
  long center_idx, old_neighbor, old_vtx[4];
  RT_RS_Vtx *center, *shared_vtx;
  RT_RS_Area *am, *an, *ao, *na;

  // get the 3 patches of the anchored structure
  am = al->get_area(am_idx); an = al->get_area(an_idx); ao = al->get_area(ao_idx);

  // subtract the area from the associated vertices
  al->sub_ar(); am->sub_ar(); an->sub_ar(); ao->sub_ar();

  // get the "bounding vertices"
  old_vtx[0] = al->vidx[1]; old_vtx[1] = al->vidx[2];
  old_vtx[2] = am->vidx[1]; old_vtx[3] = an->vidx[1];
  old_neighbor = al->neighbors[1];

  // compute the center vertex
  al->insert_ip(4, old_vtx, center, center_idx);

  if(shared_vtxidx < 0)
    // compute the new outer vertex
    al->insert_ip(2, old_vtx, shared_vtx, shared_vtxidx);

  //update the original patches (->a)
  //o->a
  ao->vidx[0] = an->vidx[2];
  ao->vidx[1] = al->vidx[1];
  ao->vidx[2] = shared_vtxidx;
  ao->vidx[3] = center_idx;
  ao->comp_ar();
  ao->neighbors[0] = al->neighbors[0];
  ao->neighbors[1] = to_new_area2;
  ao->neighbors[2] = al_idx;
  ao->neighbors[3] = an_idx;
  //l->a
  al->vidx[0] = shared_vtxidx;
  al->vidx[1] = al->vidx[2];
  al->vidx[2] = am->vidx[0];
  al->vidx[3] = center_idx;
  al->comp_ar();
  al->neighbors[0] = to_new_area1;
  al->neighbors[1] = al->neighbors[2];
  al->neighbors[2] = am_idx;
  al->neighbors[3] = ao_idx;
  //m->a
  am->vidx[3] = center_idx;
  am->comp_ar();
  am->neighbors[2] = an_idx;
  am->neighbors[3] = al_idx;
  //n->a
  an->vidx[3] = center_idx;
  an->comp_ar();
  an->neighbors[2] = ao_idx;
  an->neighbors[3] = am_idx;

  al->anchor_type = am->anchor_type = an->anchor_type = ao->anchor_type = 'a';

  //update the neighbors
  if(ao->neighbors[0] >= 0) {
    na = al->get_area(ao->neighbors[0]);
    for(i = 0; i < 4; i++)
      if(na->neighbors[i] == al_idx) {
        na->neighbors[i] = ao_idx;
        break;
      }
  }
  if(to_old_area < 0 && old_neighbor >= 0)
    anchor1(al->get_area(old_neighbor), old_neighbor, al_idx, ao_idx, al_idx,
      shared_vtxidx, ao->neighbors[1], al->neighbors[0]);
  else {
    area1_nb = al_idx;
    area2_nb = ao_idx;
  }
}


boolean RT_RS_Areas::read(char *filename, FILE *fp)
{
  long _dcnt, _icnt, i;
  RT_RS_DiffArea* diff_ar;
  RT_RS_InsArea* ins_ar;
  boolean aln = fp == NULL;
  boolean Ok = TRUE;
  int clmn = 0;

  if(aln) Ok = openfile(filename, fp, "rt");
  if(Ok) Ok = getline(filename, fp, RS_s);
  if(Ok) Ok = getstr(filename, fp, RS_s, RS_s1, clmn);
  if(Ok) Ok = isclass(filename, RS_s1);
  if(Ok) Ok = getlong(filename, fp, _dcnt, RS_s, clmn);
  if(Ok) Ok = getlong(filename, fp, _icnt, RS_s, clmn);
  if(Ok) Ok = getint(filename, fp, auto_mesh, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, diff_min_area, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, ins_min_area, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, edge_min_area, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, shadow_min_area, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, light_min_area, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, grad_min_area, RS_s, clmn);
  if(Ok) Ok = getfloat(filename, fp, grad_max_delta, RS_s, clmn);
  if(Ok) {
    scene->Points->set_ar0();
    freeh();
    for(i = 0; i < _dcnt; i++) {
      diff_ar = new RT_RS_DiffArea(scene);
      Ok = diff_ar->read(filename, fp);
      if(!Ok) break;
      if(diff_ar->mesh_nr > max_mesh_nr) max_mesh_nr = diff_ar->mesh_nr;
      insert(diff_ar);
    }
  }
  if(Ok) {
    for(i = 0; i < _icnt; i++) {
      ins_ar = new RT_RS_InsArea(scene);
      Ok = ins_ar->read(filename, fp);
      if(!Ok) break;
      if(ins_ar->mesh_nr > max_mesh_nr) max_mesh_nr = ins_ar->mesh_nr;
      insert(ins_ar);
    }
  }
  if(aln) fclose(fp);
  return Ok;
}

boolean RT_RS_Areas::write(char *filename, FILE *fp)
{
  long i;
  boolean Ok = TRUE;
  boolean aln = fp == NULL;

  if(aln) Ok = openfile(filename, fp, "wt");
  if(Ok) {
     fprintf(fp,"%s %ld %ld\n%d\n%f %f %f %f %f\n%f %f\n", get_class(), dcnt, icnt,
             auto_mesh,
             diff_min_area, ins_min_area, edge_min_area,
             shadow_min_area, light_min_area,
             grad_min_area, grad_max_delta);
    Ok = !ferror(fp);
  }
  for(i = 0; i < dcnt && Ok; i++)
    Ok = get_diff(i)->write(filename,fp);
  for(i = 0; i < icnt && Ok; i++)
    Ok = get_ins(i)->write(filename,fp);

  if(!Ok) {
    rt_Output->errorVar(get_class(), ":Cannot write ", filename, NULL);
    return FALSE;
  }
  if(aln) fclose(fp);
  return TRUE;
}

void RT_RS_Areas::print(FILE *f, char *n, int width, int decimalPlaces)
{
  if(n) fprintf(f, "%s\n", n);

  fprintf(f, "statistics:\n");
  fprintf(f, "  diffuse areas: %ld\n", dcnt);
  fprintf(f, "  non-diffuse areas: %ld\n", icnt);

  fprintf(f, "mode:\n");
  fprintf(f, "  automatic generation of meshing parameters: %d\n", auto_mesh);

  fprintf(f, "parameters:\n");
  sprintf(fmt,"  %s: %s%d.%df\n", "%s", "%", width, decimalPlaces);
  fprintf(f, fmt, "patch-size for constant divided diffuse areas", diff_min_area);
  fprintf(f, fmt, "patch-size for constant divided non-diffuse areas", ins_min_area);
  fprintf(f, fmt, "patch-size for primary light sources", light_min_area);
  fprintf(f, fmt, "patch-size at edges", edge_min_area);
  fprintf(f, fmt, "patch-size at shadow boundaries", shadow_min_area);
  fprintf(f, fmt, "intensity-gradient refinement - delta value", grad_max_delta);
  fprintf(f, fmt, "intensity-gradient refinement - area value", grad_min_area);
#ifdef RS_PRINT_ALL
  long i;
  char stitle[30];

  for(i = 0; i < dcnt; i++) {
    sprintf(stitle, "diffuse area %d:", i);
    get_diff(i)->print(f, stitle, width, decimalPlaces);
  }
  for(i = 0; i < icnt; i++) {
    sprintf(stitle,"non-diffuse area %d:", i);
    get_ins(i)->print(f, stitle, width, decimalPlaces);
  }
#endif
}
