////////////////////////////////////////////////////////////////////////////////
//  Polyhedron implementation.                                                //  
//  LAST EDIT: Wed Mar  8 15:40:31 1995 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1993 - 1995 YART team                                        //
////////////////////////////////////////////////////////////////////////////////

#include "polyhdrn.h"
#include "intrsect.h"

#ifndef DBL_MAX
#define DBL_MAX		1.7976931348623157e+308
#endif

const char *RTN_POLYHEDRON = "Polyhedron";

int RT_Polyhedron::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res = _classCMD( cd, ip, argc, argv );
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " { String Integer Integer Facet_List Vector_List Surface_List Vector_List} {  Creates a new polyhedron. {ARG 1 Name} of the new object, number of {ARG 2 Facets}, number of {ARG 3 Points}, list of {ARG 4 Facets}, list of {ARG 5 Points}, list of {ARG 6 Surfaces}, list of {ARG 7 Normals}.}}", 0 );
	return TCL_OK;
    }
    if (res == TCL_OK) { 
	if (argc != 8) { Tcl_AppendResult( ip, "Bad number of arguments. Try \"", RTN_POLYHEDRON, " ?\".", 0 ); return TCL_ERROR; }
	int nfacets;
	if ( !RT_string2int( argv[2], nfacets)) { Tcl_AppendResult( ip, "Arg 2 should be an integer.", 0 ); return TCL_ERROR; }
	if ( nfacets <= 0 ) { Tcl_AppendResult( ip, "Number of points should be positive!.", 0 ); return TCL_ERROR; }
	int npoints; 
	if ( !RT_string2int( argv[3], npoints)) { Tcl_AppendResult( ip, "Arg 3 should be an integer.", 0 ); return TCL_ERROR; }
	if ( npoints <= 0 ) { Tcl_AppendResult( ip, "Number of points should be positive!.", 0 ); return TCL_ERROR; }
	if ( npoints < 3)  { Tcl_AppendResult( ip, "Two points are not enough to draw a facet!.", 0 ); return TCL_ERROR; }

	RT_Facet   *facs = new RT_Facet[nfacets];
	RT_Vector  *vecs = new RT_Vector[npoints];
	RT_Surface *surfs = 0;
	RT_Vector *norms = 0;

	// initialize points:
	for (int i = 0; i < npoints; i++) vecs[i] = RT_NULL_NORMAL; 

	int xargc; char **xargv; int dum, cnt;
	int ret = TCL_ERROR;
        int facet_error = 0;
	if (Tcl_SplitList( ip, argv[4], &xargc, &xargv) == TCL_OK) {
	    if (xargc < nfacets) rt_Output->warningVar( argv[0], ": not enough facets specified.", 0);
	    cnt = xargc < nfacets ? xargc : nfacets;
	    for ( i = 0; i < cnt; i++) {
		char stmp[10];
		if (!RT_getFacet( &xargv[i], facs[i], dum)) {
		    sprintf(stmp,"%d",i);
		    rt_Output->errorVar( argv[0], ": found a bad facet. Sorry, can't create my own!", 0);
		    facet_error = 1;
		}
		if (facs[i].not_enough_points()) {
		    sprintf(stmp,"%d",i);
		    rt_Output->errorVar( argv[0], ": found a bad facet (# ",stmp,"). Need at least three points.", 0);
		    facet_error = 1;
		}
		if (facs[i].has_bad_pointer(npoints)) {
		    sprintf(stmp,"%d",i);
		    rt_Output->errorVar( argv[0], ": found a bad facet (# ",stmp,"). At least one index in list points to the universe!", 0);
		    facet_error = 1;
		}
	    }
	    free((char*)xargv);
	}
	if (facet_error) {
	    rt_Output->errorVar( argv[0], ": Could not create a polyhedron! Could not fix errors in facet list", 0);
	    delete facs;
	    return ret;
	}
	if (Tcl_SplitList( ip, argv[5], &xargc, &xargv) == TCL_OK) {
	    if (xargc < npoints) rt_Output->warningVar( argv[0], ": less vertices in list than specified! There could be some problems later ...", 0);
	    cnt = xargc < npoints ? xargc : npoints;
	    for ( i = 0; i < cnt; i++) {
		if (!RT_getVector( &xargv[i], vecs[i], dum)) 
		    rt_Output->warningVar( argv[0], ": found a bad point value. Using default value!", 0);
	    }
	    free((char*)xargv);
	    if (Tcl_SplitList( ip, argv[6], &xargc, &xargv) == TCL_OK) {
		surfs = new RT_Surface[npoints];	
		// initialize surfaces:
		for ( i = 0; i < npoints; i++) surfs[i] = RT_NULL_SURFACE;
		if (xargc < npoints && xargc) rt_Output->warningVar( argv[0], ": not enough surfaces specified. Using defaults!", 0);
		cnt = xargc < npoints ? xargc : npoints;
		for ( i = 0; i < cnt; i++) 
		    if (strlen( xargv[i]) && !RT_getSurface( &xargv[i], surfs[i], dum)) 
			rt_Output->warningVar( argv[0], ": found a bad surface value. Using default value!", 0);
		free((char*)xargv);
	    }

	    if (Tcl_SplitList( ip, argv[7], &xargc, &xargv) == TCL_OK) {
		norms = new RT_Vector[npoints];
		// initialize normals:
		for ( i = 0; i < npoints; i++) norms[i] = RT_NULL_NORMAL;
		if (xargc < npoints && xargc) rt_Output->warningVar( argv[0], ": not enough normals specified. Using defaults!", 0);
		cnt = xargc < npoints ? xargc : npoints;
		for ( i = 0; i < cnt; i++) if (strlen( xargv[i]) && !RT_getVector( &xargv[i], norms[i], dum)) 
		    rt_Output->warningVar( argv[0], ": found a bad normal value. Using default value!", 0);
		free((char*)xargv);
	    }
	    new RT_Polyhedron( argv[1], npoints, nfacets, facs, vecs, surfs, norms );
	    RTM_classReturn;
	}
	else rt_Output->errorVar( argv[0], ": Could not create a polyhedron! No vertices found.", 0);
	delete vecs; if (surfs) delete surfs; if (norms) delete norms;
	return ret;
    }
    return res;
}

RT_Polyhedron::RT_Polyhedron( char *a, int _npoints, int _nfacets, RT_Facet *_facets, 
RT_Vector *c, RT_Surface *d, RT_Vector *e):  RT_Primitive( a) {
    xfillstyle = 0;
    vertices = 0; facets = 0;
    // _npoints and _nfacets should be non-negative!!!
    npoints = _npoints; nfacets = _nfacets;
    if ( nfacets ) facets = new RT_Facet[nfacets];
    if (!facets) return;
    for (int k=0; k<nfacets; k++) facets[k] = _facets[k];
    if ( npoints ) vertices = new RT_Vertex[ npoints ];
    else vertices = 0;
    if (!vertices) return;
    RT_Vertex *tmp = vertices;
    int i;
    for (i = 0; i < npoints; i++) {
	if (c) { tmp->point( *c ); c++; }
	if (d) { if (*d == RT_NULL_SURFACE); else tmp->surface( *d ); d++; }
	if (e) {
	    if (*e == RT_NULL_NORMAL) ;
	    else { tmp->normal( *e ); e++; }
	}
	tmp++;
    }
    fnormals=0; fdist=0; ifunc=0; fvalid = 0; intersections=0;
}

RT_Polyhedron::~RT_Polyhedron() {
    if (facets) delete [] facets;
    if (vertices) delete [] vertices;

    // for ray-tracing:
    if (fnormals) delete [] fnormals;
    if (fdist) delete [] fdist;
    if (ifunc) delete [] ifunc;
    if (fvalid) delete [] fvalid;
    if (intersections) delete [] intersections;
}

void RT_Polyhedron::create() {
    int i,j;
    // create normal and distance for each facet here
    if (fnormals) delete [] fnormals;
    if (fdist) delete [] fdist;
    if (ifunc) delete [] ifunc;
    if (fvalid) delete [] fvalid;
    if (intersections) delete [] intersections; 
    fnormals = new RT_Vector[nfacets];
    fdist = new double[nfacets];
    ifunc = new RT_ifunc_ptr [nfacets];
    fvalid = new int [nfacets];
    intersections = new RT_Vector[nfacets];
    // computeNormals();
    // search maximal number of vertices
    int maxvert = 0;
    for(i=0; i<nfacets; i++) {
	int t = facets[i].get_number(); if (maxvert < t) maxvert = t;
    }
    RT_Vector *tmpvert = new RT_Vector [maxvert];
    for(i=0; i<nfacets; i++) {
	int t = facets[i].get_number();
	fvalid[i] = 0;
	fnormals[i] = RT_NULL_NORMAL;
	fdist[i] = 0;
	ifunc[i] = 0;
	if (t > 2) {
	    for(j=0;j<t;j++)
		tmpvert[j] = vertices[facets[i].get_pointer_to_vertex(j)].getPoint();
	    RT_calcNorm(t,tmpvert,fnormals[i]);
	    double xnabs = fnormals[i].ABS();
	    if (xnabs == 0.0) {
		rt_Output->warningVar(get_name(),": a normal vector for polyhedron is zero.",0);
		continue;
	    }
	    fdist[i] = fnormals[i].DOT(*tmpvert) / xnabs;
	    fnormals[i] = fnormals[i] * (1 / xnabs);
	    ifunc[i] = &RT_intersect_yz;
	    if (fabs(fnormals[i].y) > fabs(fnormals[i].x))
		ifunc[i] = &RT_intersect_xz;
	    if (fabs(fnormals[i].z) > fabs(fnormals[i].x) && fabs(fnormals[i].z) > fabs(fnormals[i].y))
		ifunc[i] = &RT_intersect_xy;
	    fvalid[i] = 1;
	}
    }
    delete [] tmpvert;
}

int RT_Polyhedron::intersect(const RT_Ray &ray, RT_InterSectionList &il) {
    int i,ret = 0;
    RT_Ray mray;
    // transform the ray into model coordinats
    RTM_wcRay2mcRay(ray,mray);
    for(i=0; i<nfacets; i++) {
	intersections[i] = RT_Vector(DBL_MAX,DBL_MAX,DBL_MAX);
	if (!fvalid[i]) continue;
	double t = fnormals[i].DOT(mray.dir);
	if (fabs(t) < rt_RayEps) continue;
	t = (fdist[i] - fnormals[i].DOT(mray.pt)) / t;
	if (t < 0.0) continue;
	RT_Vector p = mray.pt + mray.dir * t;
	int s = 0,is = 0;
	while (s < facets[i].get_number()) {
	    const RT_Vector *p1 = &vertices[facets[i].get_pointer_to_vertex(s)].getPoint();
	    const RT_Vector *p2 = (s < facets[i].get_number() - 1) ? &vertices[facets[i].get_pointer_to_vertex(s+1)].getPoint() : &vertices[facets[i].get_pointer_to_vertex(0)].getPoint();
	    if (ifunc[i](p, *p1, *p2 - *p1)) is++;
	    s++;
	}
	if (is & 1) {
	    double wt;
	    RTM_mcT2wcT(ray,mray,wt,t); // macro compute p again !
	    if (ray.valid(wt)) {
		RT_Surface surf;
		RT_Color zero_color(0,0,0);
		surf.ambi = zero_color; surf.diff = zero_color;
		surf.spec = zero_color; surf.emmi = zero_color;
		surf.tran = surf.refr = surf.shin = 0;
		double sum = 0;
		for (s = 0; s < facets[i].get_number(); s++) {
		    const RT_Surface *_surf = vertices[facets[i].get_pointer_to_vertex(s)].getSurface();
		    if (!_surf) _surf = &get_surface();
		    double a = (vertices[facets[i].get_pointer_to_vertex(s)].getPoint() - p).ABS();
		    if (fabs(a) < rt_RayEps) {
			surf = *_surf;
			sum = 1;
			break;
		    }
		    a = 1.0 / (a * a);
		    sum += a;
		    surf.ambi = surf.ambi + _surf->ambi * a;
		    surf.diff = surf.diff + _surf->diff * a;
		    surf.spec = surf.spec + _surf->spec * a;
		    surf.emmi = surf.emmi + _surf->emmi * a;
		    surf.tran += _surf->tran * a;
		    surf.refr += _surf->refr * a;
		    surf.shin += _surf->shin * a;		
		}
		sum = 1.0 / sum;
		surf.ambi = surf.ambi * sum;
		surf.diff = surf.diff * sum;
		surf.spec = surf.spec * sum;
		surf.emmi = surf.emmi * sum;
		surf.tran *= sum;
		surf.refr *= sum;
		surf.shin *= sum;		
		int enter = fnormals[i].DOT(mray.dir) < 0;
		il.add(new RT_InterSection(this,wt,enter,&surf));
		intersections[i] = ray.pt + ray.dir * wt;
		ret++;
	    }
	}
    }
    return ret;
}

void RT_Polyhedron::normal(const RT_Vector &R, RT_Vector &N) {
    int i;
    for(i = 0; i < nfacets; i++)
	if (intersections[i] == R) break;
    int oldi = i;
    if (i == nfacets) {
	RT_Vector p = wc2mc(R);
	for(i = 0; i < nfacets; i++)
	    if (fvalid[i] && fabs(fnormals[i].DOT(p) - fdist[i]) <= rt_RayEps) {
		int s = 0,is = 0;
		while (s < facets[i].get_number()) {
		    const RT_Vector *p1 = &vertices[facets[i].get_pointer_to_vertex(s)].getPoint();
		    const RT_Vector *p2 = (s < facets[i].get_number() - 1) ?
			&vertices[facets[i].get_pointer_to_vertex(s+1)].getPoint() :
		        &vertices[facets[i].get_pointer_to_vertex(0)].getPoint();
		    if (ifunc[i](p, *p1, *p2 - *p1)) is++;
		    s++;
		}
		if (is & 1) break;
	    }
    }
    if (i == nfacets) {
	rt_Output->warningVar(get_name(),": ask for normal at a non intersection point.",0);
	N = RT_NULL_NORMAL;
	return;
    }
    if (!fvalid[i]) {
	RTM_mcN2wcN(R,fnormals[i],N);
	return;
    }
    RT_Vector r = wc2mc(R);
    RT_Vector n;
    int s = 0;
    while (s < facets[i].get_number()) {
	const RT_Vector *_n = vertices[facets[i].get_pointer_to_vertex(s)].getNormal();
	if (!_n) _n = fnormals + i;
	double a = (vertices[facets[i].get_pointer_to_vertex(s)].getPoint() - r).ABS();
	if (fabs(a) < rt_RayEps) {
	    n = *_n;
	    break;
	} else
	    n = n + *_n * (1.0 / (a * a));
	s++;	
    }
    RTM_mcN2wcN(R,n,N);
}

void RT_Polyhedron::appendPoints( int number, const RT_Vector *c, const RT_Surface *d , const RT_Vector *e ) {
    if (number) {
	RT_Vertex *vtmp = new RT_Vertex[npoints];
	for (int i=0; i<npoints; i++) {
	    RT_Vector tmppnt = vertices[i].getPoint();
	    RT_Surface *tmpsurf = (RT_Surface*)vertices[i].getSurface();
	    const RT_Vector *tmpnorm = vertices[i].getNormal();
	    vtmp[i].point(tmppnt);
	    if (tmpsurf) vtmp[i].surface(*tmpsurf);
	    if (tmpnorm) vtmp[i].normal(*tmpnorm);
	}
	if (vertices) delete [] vertices;
	vertices = new RT_Vertex[npoints + number];
	for (i=0; i<npoints; i++) {
	    RT_Vector ttmppnt = vtmp[i].getPoint();
	    RT_Surface *ttmpsurf = (RT_Surface*)vtmp[i].getSurface();
	    const RT_Vector *ttmpnorm = vtmp[i].getNormal();
	    vertices[i].point(ttmppnt);
	    if (ttmpsurf) vertices[i].surface(*ttmpsurf);
	    if (ttmpnorm) vertices[i].normal(*ttmpnorm);
	}
	RT_Vertex *tmp = &vertices[npoints];
	for (i = 0; i < number; i++) {
	    if (c) { tmp->point( *c ); c++; }
	    if (d) { if (*d == RT_NULL_SURFACE) ; else tmp->surface( *d ); d++; }
	    if (e) { if (*e == RT_NULL_NORMAL) ; else tmp->normal( *e ); e++; }
	    tmp++;
	}
	npoints += number;
	delete [] vtmp;
    }
}

void RT_Polyhedron::newGeometry(int _npoints, int _nfacets) {
    npoints = _npoints; nfacets = _nfacets;
    if (vertices) delete [] vertices;
    if ( npoints ) vertices = new RT_Vertex[ npoints ];
    else vertices = 0;
    if (facets) delete [] facets;
    if ( nfacets ) facets = new RT_Facet[ nfacets ];
    else facets = 0;
}

RT_Vertex *RT_Polyhedron::get( int _nf, int _np) const {
    if ((_np<0) || (_np>=npoints) || (_nf<0) || (_nf>=nfacets)) {
	rt_Output->errorVar( get_name(), ": indices out of area!", 0 );
	return 0;
    }
    int index = facets[_nf].get_pointer_to_vertex(_np);
    return &vertices[index];
}

RT_Vertex *RT_Polyhedron::get( int np) const {
    if ( np <0 || np >= get_npoints() ) {
	rt_Output->errorVar( get_name(), ": indices out of area!", 0 );
	return 0;
    }
    return &vertices[np];
}

int RT_Polyhedron::getXF, RT_Polyhedron::getYF, RT_Polyhedron::cnF;

RT_ParseEntry RT_Polyhedron::table[] = {
    { "-get_npoints", RTP_NONE, 0, &getXF, "Get the number of vertices.", RTPS_NONE },
    { "-get_nfacets", RTP_NONE, 0, &getYF, "Get the number of facets.", RTPS_NONE },

    { "-appendPoints", RTP_SPECIAL, 0, 0, "Append {ARG 1 Number} of {ARG 2 Points} to vertex list. A list of {ARG 3 Surfaces} and {ARG 4 Normals} can be specified.", "Integer Vector_List Surface_List Vector_List" },
    { "-appendFacets", RTP_SPECIAL, 0, 0, "Append {ARG 1 Number} of {ARG 2 Facets} to the polyhedron.", "Integer Facet_List" },

    { "-vtPoint", RTP_SPECIAL, 0, 0, "Set new {ARG 3 Geometry} to vertex: {ARG 1 Facet}, {ARG 2 Number} of vertex in facet.", "Integer Integer Vector" },
    { "-get_vtPoint", RTP_SPECIAL, 0, 0, "Get point of vertex: {ARG 1 Facet}, {ARG 2 Number} in facet.", "Integer Integer" },

    { "-vtSurface", RTP_SPECIAL, 0, 0, "Set {ARG 3 Surface} for specified vertex: {ARG 1 Facet}, {ARG 2 Number} of vertex in facet.", "Integer Integer Surface" },
    { "-get_vtSurface", RTP_SPECIAL, 0, 0, "Get Surface of specified vertex: {ARG 1 Facet}, {ARG 2 Number} of vertex in facet. If there isn't a surface at the vertex return the object default surface.", "Integer Integer" },
    { "-vtDeleteSurface", RTP_SPECIAL, 0, 0, "Clear surface in {ARG 1 Facet}, {ARG 2 Number} of vertex in facet.", "Integer Integer" },

    { "-vtNormal", RTP_SPECIAL, 0, 0, "Set {ARG 3 Normal} for specified vertex: {ARG 1 Facet}, {ARG 2 Number} of vertex in facet.", "Integer Integer Vector" },
    { "-get_vtNormal", RTP_SPECIAL, 0, 0, "Get normal of specified vertex: {ARG 1 Facet}, {ARG 2 Number} in facet. If there isn't a normal at the vertex return the object default normal.", "Integer Integer" },
    { "-vt_deleteNormal", RTP_SPECIAL, 0, 0, "Clear normal at vertex: {ARG 1 Facet}, {ARG 2 Number} of vertex in facet.", "Integer Integer" },

    { "-computeNormals", RTP_NONE, 0, &cnF, "For each vertex a normal vector will be computed. Existing normals will be overwritten.", RTPS_NONE },
    
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_Polyhedron::objectCMD(char *argv[])  { 
    int ret =  RT_Primitive::objectCMD( argv );
    RT_parseTable( argv, table );
    if (getXF) {
	static char tmp[10]; ret++;
	RT_int2string( get_npoints(), tmp );
	RT_Object::result( tmp );
    }

    if (getYF) {
	static char tmp[10]; ret++;
	RT_int2string( get_nfacets(), tmp );
	RT_Object::result( tmp );
    }

    { 	int start, diff1 = 0, diff2 = 0, diff3 = 0;
	int posx, posy; RT_Surface s;
	if ((start = RT_findString( argv, "-vtSurface" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) 
		    if ( RT_getSurface( &argv[ start + 3 ], s, diff3) )	{ 
			vtSurface( posx, posy, s ); ret++; 
		    }
	RT_clearArgv( argv, start, diff1 + diff2 + diff3 );
    }

    { 	int start, diff1 = 0, diff2 = 0, diff3 = 0;
	int pnr; RT_Surface *s; RT_Vector *p, *n;
	if ((start = RT_findString( argv, "-appendPoints" )) >= 0) {
	    if ( RT_getInt( &argv[ start + 1 ], pnr, diff1) ) {
		s = new RT_Surface[pnr];
		p = new RT_Vector[pnr];
		n = new RT_Vector[pnr];
		for (int i=0; i<pnr; i++) {
		 int xargc; char **xargv; int dum, cnt;
		 if (Tcl_SplitList( rt_Ip, argv[start + 2], &xargc, &xargv) == TCL_OK) {
	    	   if (xargc < pnr) rt_Output->warningVar( argv[0], ": less vertices in append-list than specified! There could be some problems later ...", 0);
	    	   cnt = xargc < pnr ? xargc : pnr;
	    	   for ( int i = 0; i < cnt; i++) {
			if (!RT_getVector( &xargv[i], p[i], dum)) 
		    		rt_Output->warningVar( argv[0], ": found a bad point value. Using default value!", 0);
		   }
	    	   free((char*)xargv);
	    	   if (Tcl_SplitList( rt_Ip, argv[start + 3], &xargc, &xargv) == TCL_OK) {
			for ( i = 0; i < pnr; i++) s[i] = RT_NULL_SURFACE;
			if (xargc < pnr && xargc) rt_Output->warningVar( argv[0], ": not enough surfaces in append-list specified. Using defaults!", 0);
			cnt = xargc < pnr ? xargc : pnr;
			for ( i = 0; i < cnt; i++) 
		    	if (strlen( xargv[i]) && !RT_getSurface( &xargv[i], s[i], dum)) 
			   rt_Output->warningVar( argv[0], ": found a bad surface value. Using default value!", 0);
			free((char*)xargv);
		   }
	    	   if (Tcl_SplitList( rt_Ip, argv[start + 4], &xargc, &xargv) == TCL_OK) {
			for ( i = 0; i < pnr; i++) n[i] = RT_NULL_NORMAL;
			if (xargc < pnr && xargc) rt_Output->warningVar( argv[0], ": not enough normals in append-list specified. Using defaults!", 0);
			cnt = xargc < pnr ? xargc : pnr;
			for ( i = 0; i < cnt; i++) if (strlen( xargv[i]) && !RT_getVector( &xargv[i], n[i], dum)) 
		    	   rt_Output->warningVar( argv[0], ": found a bad normal value. Using default value!", 0);
			free((char*)xargv);
		   }
	    	}
	       }
	    }
	    appendPoints(pnr, p, s, n); ret++;
	    if (p) delete p; if(s) delete s; if (n) delete n;
	}
	RT_clearArgv( argv, start, diff1 + diff2 + diff3 );
    }

    { 	int start, diff1 = 0, diff2 = 0;
	int fnr; RT_Facet *f;
	int facet_error = 0;
	if ((start = RT_findString( argv, "-appendFacets" )) >= 0) {
	    if ( RT_getInt( &argv[ start + 1 ], fnr, diff1) ) {
		f = new RT_Facet[fnr];
		int xargc; char **xargv; int dum, cnt;
		if (Tcl_SplitList( rt_Ip, argv[start + 2], &xargc, &xargv) == TCL_OK) {
	    	    if (xargc < fnr) rt_Output->warningVar( argv[0], ": not enough facets in append-list specified.", 0);
	    	    cnt = xargc < fnr ? xargc : fnr;
		    diff2 = 1;
	    	    for ( int i = 0; i < cnt; i++) {
			char stmp[10];
			if (!RT_getFacet( &xargv[i], f[i], dum)) {
			    rt_Output->errorVar( argv[0], ": found a bad facet. Sorry, can't create my own!", 0);
			    facet_error = 1;
			}
			if (f[i].not_enough_points()) {
			    sprintf(stmp,"%d",i);
			    rt_Output->errorVar( argv[0], ": found a bad facet (# ",stmp,"). Need at least three points.", 0);
			    facet_error = 1;
			}
			if (f[i].has_bad_pointer(npoints)) {
			    sprintf(stmp,"%d",i);
			    rt_Output->errorVar( argv[0], ": found a bad facet (# ",stmp,"). At least one index in list points to the universe!", 0);
			    facet_error = 1;
			}
	    	    }
	    	    if (!facet_error) { appendFacets(fnr, f); ret++; }
	    	    if (f) delete f;
	    	    free((char*)xargv);
		}
	    }
	}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	int start, diff1 = 0, diff2 = 0;
	int posx, posy;
	if ((start = RT_findString( argv, "-get_vtSurface" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) {
		    static char tmp[200]; ret++;
		    RT_surface2string( get_vtSurface(posx, posy), tmp );
		    RT_Object::result( tmp );
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	int start, diff1 = 0, diff2 = 0;
	int posx, posy;
	if ((start = RT_findString( argv, "-vtDeleteSurface" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) {
		    vtDeleteSurface(posx, posy); ret++;
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	int start, diff1 = 0, diff2 = 0, diff3 = 0;
	int posx, posy; RT_Vector n;
	if ((start = RT_findString( argv, "-vtNormal" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) 
		    if ( RT_getVector( &argv[ start + 3 ], n, diff3) ) {
			vtNormal( posx, posy, n ); ret++;
		    }
	RT_clearArgv( argv, start, diff1 + diff2 + diff3 );
    }

    { 	int start, diff1 = 0, diff2 = 0;
	int posx, posy;
	if ((start = RT_findString( argv, "-get_vtNormal" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) {
		    char tmp[60]; ret++;
		    RT_vector2string( get_vtNormal(posx, posy), tmp );
		    RT_Object::result( tmp );
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	int start, diff1 = 0, diff2 = 0;
	int posx, posy;
	if ((start = RT_findString( argv, "-vtDeleteNormal" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) {
		    vtDeleteNormal(posx, posy); ret++;
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    { 	int start, diff1 = 0, diff2 = 0, diff3 = 0;
	int posx, posy; RT_Vector p;
	if ((start = RT_findString( argv, "-vtPoint" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) 
		    if ( RT_getVector( &argv[ start + 3 ], p, diff3) ) {
			vtPoint( posx, posy, p ); ret++;
		    }
	RT_clearArgv( argv, start, diff1 + diff2 + diff3 );
    }

    { 	int start, diff1 = 0, diff2 = 0;
	int posx, posy;
	if ((start = RT_findString( argv, "-get_vtPoint" )) >= 0) 
	    if ( RT_getInt( &argv[ start + 1 ], posx, diff1) ) 
		if ( RT_getInt( &argv[ start + 2 ], posy, diff2) ) {
		    char tmp[60]; ret++;
		    RT_vector2string( get_vtPoint(posx, posy), tmp );
		    RT_Object::result( tmp );
		}
	RT_clearArgv( argv, start, diff1 + diff2 );
    }

    if (cnF) { computeNormals(); ret++; }

    return ret;
}

void RT_Polyhedron::vtSurface(int _nf, int _np, const RT_Surface &s) {
    RT_Vertex *vr;
    if ( !(vr = get( _nf, _np ))) return;
    vr->surface( s );
    geomChanged();
}

const RT_Surface &RT_Polyhedron::get_vtSurface(int _nf, int _np ) const {
    RT_Vertex *vr;
    if ( !(vr = get( _nf, _np ))) return get_surface();
    return vr->getSurface() ? *vr->getSurface() : get_surface();
}

void RT_Polyhedron::vtDeleteSurface(int _np, int _nf) {
    RT_Vertex *vr;
    if ( !(vr = get( _nf, _np ))) return;
    vr->deleteSurface();
    geomChanged();
}

void RT_Polyhedron::vtNormal(int _nf, int _np, const RT_Vector &p) {
    RT_Vertex *vr;
    if ( !(vr = get( _nf, _np ))) return;
    vr->normal( p );
    geomChanged();
}

const RT_Vector &RT_Polyhedron::get_vtNormal(int _nf, int _np) const {
    RT_Vertex *vr;
    if ( !(vr = get( _nf, _np ))) return RT_NULL_NORMAL;
    return vr->getNormal() ? *vr->getNormal() : RT_NULL_NORMAL;
}

void RT_Polyhedron::vtDeleteNormal( int _nf, int _np ) {
    RT_Vertex *vr;
    if ( !(vr = get( _nf, _np ))) return;
    vr->deleteNormal();
    geomChanged();
}

void RT_Polyhedron::vtPoint(int _nf, int _np, const RT_Vector &p) {
    RT_Vertex *vr;
    if ( !(vr = get( _nf, _np ))) return;
    vr->point( p );
    geomChanged();
}

const RT_Vector &RT_Polyhedron::get_vtPoint(int _nf, int _np) const {
    RT_Vertex *vr;
    if ( !(vr = get( _nf, _np ))) return RT_BAD_VECTOR;
    return vr->getPoint();
}

void RT_Polyhedron::appendFacets( int number, const RT_Facet *newfacs) {
  if (number > 0) {
    RT_Facet *factmp = new RT_Facet[nfacets];
    int maxind = 0;
    int itmp;
    for (int i=0; i<nfacets; i++) {
	itmp = facets[i].get_number();
	maxind = (maxind > itmp) ? maxind : itmp;
    }
    int *iarray = new int[maxind];
    for (i=0; i<nfacets; i++) {
	itmp = facets[i].get_number();
	for (int j=0; j<itmp; j++) iarray[j] = facets[i].get_pointer_to_vertex(j);
	factmp[i].set(itmp, iarray);
    }
    if (facets) delete [] facets;
    facets = new RT_Facet[nfacets + number];
    for (i=0; i<nfacets; i++) {
	itmp = factmp[i].get_number();
	for (int j=0; j<itmp; j++) iarray[j] = factmp[i].get_pointer_to_vertex(j);
	facets[i].set( itmp, iarray );
    }
    if (iarray) delete iarray;
    maxind = 0;
    for (i=0; i<number; i++) {
	itmp = newfacs[i].get_number();
	maxind = (maxind > itmp) ? maxind : itmp;
    }
    iarray = new int[maxind];
    for (i=0; i<number; i++) {
	itmp = newfacs[i].get_number();
	for (int j=0; j<itmp; j++) iarray[j] = newfacs[i].get_pointer_to_vertex(j);
	facets[nfacets + i].set( itmp, iarray );
    }
    nfacets += number;
    if (iarray) delete iarray;
    delete [] factmp;
  }
}

RT_Bounds RT_Polyhedron::get_bounds() {
    RT_Bounds b;
    RT_Vertex *tmp = vertices;
    int fl = 0;
    for (int i = 0; i < npoints; i++) { 
	const RT_Vector &v = tmp->getPoint();
	if (fl) {
	    if (v.x < b.min.x) b.min.x = v.x;
	    if (v.x > b.max.x) b.max.x = v.x;
	    if (v.y < b.min.y) b.min.y = v.y;
	    if (v.y > b.max.y) b.max.y = v.y;
	    if (v.z < b.min.z) b.min.z = v.z;
	    if (v.z > b.max.z) b.max.z = v.z;
	}
	else {
	    b.max.x = b.min.x = v.x;
	    b.max.y = b.min.y = v.y;
	    b.max.z = b.min.z = v.z;
	    fl++;
	}
	tmp++;
    }
    return b;
} 

void RT_Polyhedron::printCon(FILE *f) const {
    int i;
    RT_Object::printCon( f );
    fprintf( f, "%i %i { ",get_nfacets(), get_npoints() );

    // save facets:
    for (i=0; i<nfacets; i++)  facets[i].print( f );
    fprintf( f, "} {" );

    // save points:
    RT_Vertex *tmp = vertices;
    for ( i = 0; i < npoints; i++) { tmp->getPoint().print( f ); tmp++; }
    fprintf( f, "} {" );
    tmp = vertices;

    // save the surfaces:
    for ( i = 0; i < npoints; i++) {
	const RT_Surface *su = tmp->getSurface();
	if (su) su->print( f );
	else fprintf( f, "{} " );
	tmp++;
    }
    fprintf( f, "} {" );
    tmp = vertices;

    // save the normals:
    for ( i = 0; i < npoints; i++) {
	const RT_Vector *no = tmp->getNormal();
	if (no) no->print( f );
	else fprintf( f, "{} " );
	tmp++;
    }
    fprintf( f, "}" );
}

void RT_Polyhedron::computeNormals() {
    RT_Vector tmp[3];
    RT_Vector norm;
    int i,j;
    typedef struct {
	int nn;
	double v[3];
    } NVTmp;
    NVTmp *ntmp = new NVTmp[npoints];
    for (i=0; i<npoints; i++) {
	ntmp[i].nn = 0;
	for (j=0; j<3; j++) ntmp[i].v[j] = 0.0;
	for (j=0; j<nfacets; j++) {
	   if (facets[j].vertex_in_facet(i)) {
		if (facets[j].first_vertex(i)) {
		   tmp[0] = get_vtPoint( j, facets[j].get_number() - 1 );
		   tmp[1] = get_vtPoint( j, 0 );
		   tmp[2] = get_vtPoint( j, 1 );
		}
		else if (facets[j].last_vertex(i)) {
		   int k = facets[j].get_number() - 1;
		   tmp[0] = get_vtPoint( j, k-1 );
		   tmp[1] = get_vtPoint( j, k );
		   tmp[2] = get_vtPoint( j, 0 );
		}
		else {
		    int k = facets[j].get_vertex_index( i );
		    tmp[0] = get_vtPoint( j, k-1 );
		    tmp[1] = get_vtPoint( j, k );
		    tmp[2] = get_vtPoint( j, k+1 );
		}
	    	RT_calcNorm( 3, tmp, norm );
		ntmp[i].v[0] += norm.x;
		ntmp[i].v[1] += norm.y;
		ntmp[i].v[2] += norm.z;
		ntmp[i].nn++;
	    }
       }
	if (ntmp[i].nn) {
	    norm.x = ntmp[i].v[0] / ntmp[i].nn;
	    norm.y = ntmp[i].v[1] / ntmp[i].nn;
	    norm.z = ntmp[i].v[2] / ntmp[i].nn;

	    get( i )->normal( norm );
	}
    }
    if (ntmp) delete ntmp;
    geomChanged();
}

const RT_Facet &RT_Polyhedron::get_facet(int nf) { 
    if ((nf<0) || (nf>=nfacets)) {
	rt_Output->errorVar( get_name(), ": index out of area!", 0 );
	return facets[0];
    }
    return facets[nf];
}

int RT_Polyhedron::copy(RT_Primitive *p) const {
    if ( p->isA( RTN_POLYHEDRON )) {
	RT_Polyhedron *ph = (RT_Polyhedron*)p;
	ph->newGeometry( get_npoints(), get_nfacets() );
	// copy facets:
	for (int i = 0; i < get_nfacets(); i++) 
	    ph->facets[i] = facets[i];

	// copy vertices:
	for (i = 0; i < get_npoints(); i++) 
	    ph->vertices[i] = vertices[i];
	
	return( 1 );
    }
    return( RT_Primitive::copy( p ) );
}
