////////////////////////////////////////////////////////////////////////////////
//  Quadric implementation.                                                   //  
//  LAST EDIT: Fri Aug  5 08:55:02 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRGHT which should be distributed with this  //
//  file. If COPYRGHT is not available or for more info please contact:       //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include "quadric.h"
#include "intrsect.h"

const char *RTN_QUADRIC = "Quadric";

int RT_Quadric::intersect(const RT_Ray &wray, RT_InterSectionList &list) {
    RT_Ray ray;
    RTM_wcRay2mcRay( wray, ray );
    
    // utility variables:
    double nb = 2 * quMixedTerms.x;
    double nc = 2 * quMixedTerms.y;
    double nd = 2 * quTerms.x;
    double nf = 2 * quMixedTerms.z;
    double ng = 2 * quTerms.y;
    double ni = 2 * quTerms.z;
    
    double aq = ray.dir.x * ( ray.dir.x * qu2Terms.x + ray.dir.y * nb + ray.dir.z * nc )
	+ ray.dir.y * ( ray.dir.y * qu2Terms.y + ray.dir.z * nf ) 
	    + ray.dir.z * ray.dir.z * qu2Terms.z;

    double nbq = ray.dir.x * ( ray.pt.x * qu2Terms.x + ray.pt.y * quMixedTerms.x + ray.pt.z * quMixedTerms.y + quTerms.x )
	+ ray.dir.y * ( ray.pt.x * quMixedTerms.x + ray.pt.y * qu2Terms.y + ray.pt.z * quMixedTerms.z + quTerms.y )
	    + ray.dir.z * ( ray.pt.x * quMixedTerms.y + ray.pt.y * quMixedTerms.z + ray.pt.z * qu2Terms.z + quTerms.z );

    double cq = ray.pt.x * ( ray.pt.x * qu2Terms.x + ray.pt.y * nb + ray.pt.z * nc + nd)
	+ ray.pt.y * ( ray.pt.y * qu2Terms.y + ray.pt.z * nf + ng) 
	    + ray.pt.z * ( qu2Terms.z * ray.pt.z + ni) + quConstant;

    int ret = 0;
    if (aq == 0.0) {
	double t = - 2 * cq/nbq;
	double wt;
	RTM_mcT2wcT( wray, ray, wt, t ); 
	if (wray.valid( wt)) {
	    list.add( new RT_InterSection( this, wt, 0, &get_surface() ));
	    ret++;
	}
    } 
    else {
	double ka = -nbq/aq;
	double kb = cq/aq;
	double df = ka * ka - cq/aq; 
	if (df < 0) return 0;
	double sq = sqrt( df );
	double t1 = ka - sq;
	double t2 = ka + sq;

	double wt1, wt2;
	RTM_mcT2wcT( wray, ray, wt1, t1 );
	RTM_mcT2wcT( wray, ray, wt2, t2 );
 
	if ( wray.valid( wt1) ) {
	    list.add( new RT_InterSection( this, wt1, 1,  &get_surface() ));
	    ret ++;
	}
	if ( wray.valid( wt2 )) {
	    list.add( new RT_InterSection( this, wt2, 0, &get_surface() ));
	    ret++;
	}
    }
    return ret;
}

void RT_Quadric::normal(const RT_Vector &wc, RT_Vector &n) {
    RT_Vector mc = wc2mc( wc);
    RT_Vector mcn;

    mcn.x = 2 * ( qmat.get( 0, 0 ) * mc.x + qmat.get( 0, 1 ) * mc.y 
		 + qmat.get( 0, 2 ) * mc.z + qmat.get( 0, 3 ));

    mcn.y = 2 * ( qmat.get( 1, 0 ) * mc.x + qmat.get( 1, 1 ) * mc.y 
		 + qmat.get( 1, 2 ) * mc.z + qmat.get( 1, 3 ));

    mcn.z = 2 * ( qmat.get( 2, 0 ) * mc.x + qmat.get( 2, 1 ) * mc.y 
		 + qmat.get( 2, 2 ) * mc.z + qmat.get( 2, 3 ));

    if (mcn.ABS() < rt_RayEps ) {
	// the normal is not defined at this point of the surface  
	// set it to any arbitrary direction
	mcn.x = 1.0;
	mcn.y = 1.0;
	mcn.z = 1.0;
    }

    // convert and normalize the normal vector:
    RTM_mcN2wcN( wc, mcn, n );
}

void RT_Quadric::quadric2Matrix() {
    qmat.set(0, 0, qu2Terms.x);
    qmat.set(1, 1, qu2Terms.y);
    qmat.set(2, 2, qu2Terms.z);
    qmat.set(3, 3, quConstant);
    qmat.set(0, 1,  quMixedTerms.x); qmat.set(1, 0, quMixedTerms.x);
    qmat.set(0, 2, quMixedTerms.y); qmat.set(2, 0, quMixedTerms.y);
    qmat.set(1, 2,  quMixedTerms.z); qmat.set(2, 1, quMixedTerms.z);
    qmat.set(0, 3, quTerms.x); qmat.set(3, 0, quTerms.x);
    qmat.set(1, 3, quTerms.y); qmat.set(3, 1, quTerms.y);
    qmat.set(2, 3, quTerms.z); qmat.set(3, 2, quTerms.z);
}

int RT_Quadric::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Vector Vector Vector Double} {Creates a new quadric called {ARG 1 Name}. Arguments: {ARG 2 Square} terms: A E H, {ARG 3 Mixed} terms: B C F, {ARG 4 Linear} terms: D G I, {ARG 5 Constant} : J. The quadric primitive will be used parameterized in analytical primitives. You can use it explicite, too - but there will no shading support for an arbitrary quadric.}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if (argc != 6) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <square[]> <mixed[]> <linear[]> <const>. ", NULL );
	    return TCL_ERROR;
	}
	int nr;
	RT_Vector sq, mx, ln; 
	double cn;
	if (!RT_getVector( &argv[2], sq, nr)) {
	    Tcl_AppendResult( ip, "Bad parameters. <square[]> should be a float vector. ", NULL );
	    return TCL_ERROR;
	}
	if (!RT_getVector( &argv[3], mx, nr)) {
	    Tcl_AppendResult( ip, "Bad parameters. <mixed[]> should be a float vector. ", NULL );
	    return TCL_ERROR;
	}
	if (!RT_getVector( &argv[4], ln, nr)) {
	    Tcl_AppendResult( ip, "Bad parameters. <linear[]> should be a float vector. ", NULL );
	    return TCL_ERROR;
	}
	if (!RT_getDouble( &argv[5], cn, nr)) {
	    Tcl_AppendResult( ip, "Bad parameters. <constant> should be a float. ", NULL );
	    return TCL_ERROR;
	}
	new RT_Quadric( argv[1], sq, mx, ln, cn ); 
	RTM_classReturn;
    }
    return res; 
}
