////////////////////////////////////////////////////////////////////////////////
//  implementation of parametric curves                                       //  
//  LAST EDIT: Wed Mar  8 11:20:34 1995 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRGHT which should be distributed with this  //
//  file. If COPYRGHT is not available or for more info please contact:       //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1993 - 1995 YART team                                        //
////////////////////////////////////////////////////////////////////////////////

#include "curves.h"

const char *RTN_CURVE = "Curve";

int RT_Curve::polygF;

RT_ParseEntry RT_Curve::table[] = {
    { "-polygon", RTP_NONE, 0, &polygF, "Draw the polygon of guiding points.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_Curve::objectCMD( char *argv[] ) {
    int ret = RT_Polyvertex::objectCMD( argv );
    RT_parseTable( argv, table );
    if( polygF ) {
	if( drawPolygon ) 
	    drawPolygon = 0;
	else
	    drawPolygon = 1;
	geomChanged(); 
	ret++;
    } 
    return ret;
}

int RT_Curve::copy(RT_Primitive *p) const {
    if (p->isA( RTN_CURVE)) return( RT_Polyvertex::copy( p ) );
    if (p->isA( RTN_POLYLINE)) return( xline->copy( p ) );
    return( RT_Primitive::copy(p) );
}

void RT_Curve::checkAttributes() {
    RT_Primitive::checkAttributes();
    if (xsolution->isChanged()) geomChanged();
    if (get_attribute( RTN_SURFACE_ATTR ).isChanged()) geomChanged();
}

void RT_Curve::createReferences( const RT_AttributeList &list ) {
    RT_Primitive::createReferences( list );
    // create a resolution reference:
    RT_ResolutionAttribute *tmp = xsolution;
    xsolution = (RT_ResolutionAttribute*)list.retraverse( RTN_RESOLUTION_ATTR, this );
    
    // create a surface reference:
    RT_SurfaceAttribute *tmp1 = (RT_SurfaceAttribute*)attributes->get( RTN_SURFACE_ATTR );
   RT_SurfaceAttribute *tmp2 = (RT_SurfaceAttribute*)list.retraverse( RTN_SURFACE_ATTR, this );

    // test changes:
    if ( (tmp != xsolution || *tmp != *xsolution) ||
	(tmp1 != tmp2 || *tmp1 != *tmp2) ) geomChanged();
}

void RT_Curve::resolution( double s ) {
    xsolution = (RT_ResolutionAttribute*)attributes->get( RTN_RESOLUTION_ATTR );
    if (!xsolution) attributes->insert( xsolution = new RT_ResolutionAttribute);
    xsolution->resolution( s );
    geomChanged();
}

double RT_Curve::get_resolution() const {
   return xsolution->get_resolution();
}

void RT_Curve::newGeometry() { 
    if( xline ) delete xline;
    xline = new RT_Polyline( 0, 0, 0, 0, 0 );
    xline->father(this);
}

void RT_Curve::drawLeadingPolygon() {
    if( xpolygon ) delete xpolygon;
    xpolygon = new RT_Polyline( 0, 0, 0, 0, 0 );
    xpolygon->father(this);
    
    for( int i = 0; i < get_number(); i++ )
	xpolygon->vtInsert( i, get_vtPoint(i), 0, 0 );
}

const char *RTN_BSPLINE = "BSpline";

int RT_BSpline::ordF, RT_BSpline::ordV, RT_BSpline::getOrdF, RT_BSpline::cfF, RT_BSpline::cfV, RT_BSpline::getModeF;

RT_ParseEntry RT_BSpline::table[] = {
    { "-order", RTP_INTEGER, (char*)&ordV, &ordF, "Change the {ARG 1 Order} of the b-spline.", RTPS_INTEGER },
    { "-get_order", RTP_NONE, 0, &getOrdF, "Get current order of the b-spline", RTPS_NONE },
    { "-mode",  RTP_INTEGER, (char*)&cfV, &cfF, "Set the  draw mode {ARG 1 Order} 0 - an opened or 1 - closed curve.", RTPS_INTEGER },
    { "-get_mode", RTP_NONE, 0, &getModeF, "Get current mode of the b-spline",  RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};
 
int RT_BSpline::classCMD( ClientData cd, Tcl_Interp *ip, int argc, char *argv[] ) { 
    int res = _classCMD( cd, ip, argc, argv );    
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " { String Integer Vector_List Surface_List Vector_List} {  Creates a new b-spline. {ARG 1 Name} of the new object, {ARG 2 Number} of guiging points, list of {ARG 3 Points}, list of {ARG 4 Surfaces}, list of {ARG 5 Normals}. The surface and normals list can be empty.}}", 0 );
	return TCL_OK;
    }
    if (res == TCL_OK) { 
	int ret = TCL_ERROR; int nr;
	RT_Vector  *vecs; RT_Surface *surfs; RT_Vector *norms;
	if ( createPoly( ip, argc, argv, nr, vecs, surfs, norms) == TCL_OK ) {
	    new RT_BSpline( argv[1], nr, vecs, surfs, norms );
	    RTM_classReturn;
	}
	else rt_Output->errorVar( argv[0], ": Could not create a new curve!", 0);
	if (vecs) delete [] vecs; if (surfs) delete [] surfs; if (norms) delete [] norms;
	return ret;
    }
    return res;
}

int RT_BSpline::objectCMD(char *argv[]) {
    int ret = RT_Curve::objectCMD( argv );
    RT_parseTable( argv, table );
    if (ordF) {
	xspline->setOrd( ordV );
	geomChanged();
	ret++;
    }
    if (getOrdF) {
      static char tmp[5]; 
      RT_int2string( xspline->getOrd(), tmp );
      RT_Object::result( tmp );
      ret++;
    }
    if (cfF) {
      xspline->setMod( (_SplineMode)cfV );
      geomChanged();
      ret++;
    }
    if (getModeF) { 
      static char tmp[5];
      RT_int2string( xspline->getMod(), tmp );
      RT_Object::result( tmp );
      ret++;
    }
    return ret;
}

RT_BSpline::RT_BSpline(  char *a, int b, RT_Vector *c, RT_Surface *d, RT_Vector *e ) : 
RT_Curve( a, b, c, d, e ) {
    int nr = get_number();
    xspline = new _bspline
	(nr, nr >= 3 ? 3 : nr, (int)( (100 - nr) * 0.2 + nr), _OPEN );
}

void RT_BSpline::createSpline() {
    // compute the limit for evaluation loop (min nr, max 100):
    int i, e, nr = get_number();
    int dim = (int)( ( 100-nr ) * get_resolution() + nr );
    
    if ( ( !(xspline->getNum() == nr) || !(xspline->getDim() == dim ) ) ) {
	xspline->setNum(nr);
	xspline->setDim(dim);
    }
    
    xspline->update();
  
    // set xline and xpolygon:
    newGeometry();
    
    _Vector parVec = xspline->getParVector();
    _Vector baseVec;
    
    RT_Vector *v = new RT_Vector[nr];
    RT_Vector *n = new RT_Vector[nr];
    RT_Surface *s = new RT_Surface[nr];
    
    for ( i = 0; i < nr; i++ ) v[i] = get_vtPoint(i);
    for( i = 0; i < nr; i++ ) n[i] = get_vtNormal(i);
    for( i = 0; i < nr; i++ ) s[i] = get_vtSurface(i);

    RT_Surface sf;
    RT_Vector vc, vn;

   // get the Matrix of basic functions:
    _Matrix N = xspline->getBase();

   for ( e = 0; e < N.Rows(); e++ ) {
       // reset values;   
       vc.x = vc.y = vc.z = 0;
       
       vn.x = vn.y = vn.z = 0;
       
       sf.tran = 0; sf.refr = 0; sf.shin = 0;
       sf.ambi = RT_Color(0, 0, 0);
       sf.diff = RT_Color(0, 0, 0);
       sf.emmi = RT_Color(0, 0, 0);
       sf.spec = RT_Color(0, 0, 0);
       
       int modDiv = xspline->getNum();
       int countTo = N.Cols();
       
       for ( i = 0; i < countTo; i++ ) {	   
	   vc.x = vc.x + N(e,i) * v[i%modDiv].x;
	   vc.y = vc.y + N(e,i) * v[i%modDiv].y;
	   vc.z = vc.z + N(e,i) * v[i%modDiv].z;
	   
	   // compute normal vector and surface:
	   vn.x = vn.x + N(e,i) * n[i%modDiv].x;
	   vn.y = vn.y + N(e,i) * n[i%modDiv].y;
	   vn.z = vn.z + N(e,i) * n[i%modDiv].z;
	   
	   sf.ambi.r = sf.ambi.r + N(e,i) * s[i%modDiv].ambi.r;
	   sf.ambi.g = sf.ambi.g + N(e,i) * s[i%modDiv].ambi.g;
	   sf.ambi.b = sf.ambi.b + N(e,i) * s[i%modDiv].ambi.b;
	       
	   sf.diff.r = sf.diff.r + N(e,i) * s[i%modDiv].diff.r,
	   sf.diff.g = sf.diff.g + N(e,i) * s[i%modDiv].diff.g;
	   sf.diff.b = sf.diff.b + N(e,i) * s[i%modDiv].diff.b;
	   
	   sf.emmi.r = sf.emmi.r + N(e,i) * s[i%modDiv].emmi.r;
	   sf.emmi.g = sf.emmi.g + N(e,i) * s[i%modDiv].emmi.g;
	   sf.emmi.b = sf.emmi.b + N(e,i) * s[i%modDiv].emmi.b;
	   
	   sf.spec.r = sf.spec.r + N(e,i) * s[i%modDiv].spec.r;
	   sf.spec.g = sf.spec.g + N(e,i) * s[i%modDiv].spec.g;
	   sf.spec.b = sf.spec.b + N(e,i) * s[i%modDiv].spec.b;
	   
	   sf.tran = sf.tran + N(e,i) * s[i%modDiv].tran;
	   sf.refr = sf.refr + N(e,i) * s[i%modDiv].refr;
	   sf.shin = sf.shin + N(e,i) * s[i%modDiv].shin;
       }   
   }
    // insert the new vertex:
    xline->vtInsert(e,vc,&sf,&vn);
    
    if ( drawPolygon ) drawLeadingPolygon();
    
    // cleanin' up:
    delete [] s;
    delete [] v;
    delete [] n;
}

const char *RTN_BEZIER = "Bezier";

int RT_Bezier::classCMD( ClientData cd, Tcl_Interp *ip, int argc, char *argv[] ) { 
    int res = _classCMD( cd, ip, argc, argv );
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " { String Integer Vector_List Surface_List Vector_List} {  Creates a new bezier curve. {ARG 1 Name} of the new object, {ARG 2 Number} of guiding points, list of {ARG 3 Points}, list of {ARG 4 Surfaces}, list of {ARG 5 Normals}. The surface and normals list can be empty.}}", 0 );
	return TCL_OK;
    }
    if (res == TCL_OK) { 
	int ret = TCL_ERROR; int nr;
	RT_Vector  *vecs; RT_Surface *surfs; RT_Vector *norms;
	if ( createPoly( ip, argc, argv, nr, vecs, surfs, norms) == TCL_OK ) {
	    new RT_Bezier( argv[1], nr, vecs, surfs, norms );
	    RTM_classReturn;
	}
	else rt_Output->errorVar( argv[0], ": Could not create a new curve!", 0);
	if (vecs) delete [] vecs; if (surfs) delete [] surfs; if (norms) delete [] norms;
	return ret;
    }
    return res;
}

RT_Bezier::RT_Bezier( char *a, int b , RT_Vector *c, RT_Surface *d, RT_Vector *e) :
RT_BSpline( a, b, c, d, e ) {}

void RT_Bezier::create() {
    if( xspline->getOrd() != get_number() ) {
	rt_Output->warningVar( get_name(), ": to create a bezier curve set order = number of vertices.", 0 );
	xspline->setOrd( get_number() );
    }
    createSpline(); 
}
