/*-
 * Copyright (c) 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 * Copyright (c) 1994, 1995
 *	Keith Bostic.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)svi_screen.h	9.21 (Berkeley) 2/16/95
 */

/*
 * Structure for mapping lines to the screen.  An SMAP is an array, with one
 * structure element per screen line, which holds information describing the
 * physical line which is displayed in the screen line.  The first two fields
 * (lno and off) are all that are necessary to describe a line.  The rest of
 * the information is useful to keep information from being re-calculated.
 *
 * Lno is the line number.  Off is the screen offset into the line.  For
 * example, the pair 2:1 would be the first screen of line 2, and 2:2 would
 * be the second.  If doing left-right scrolling, all of the offsets will be
 * the same, i.e. for the second screen, 1:2, 2:2, 3:2, etc.  If doing the
 * standard vi scrolling, it will be staggered, i.e. 1:1, 1:2, 1:3, 2:1, 3:1,
 * etc.
 *
 * The SMAP is always as large as the physical screen, plus a slot for the
 * info line, so that there is room to add any screen into another one at
 * screen exit.
 */
typedef struct _smap {
	recno_t  lno;		/* 1-N: Physical file line number. */
	size_t	 off;		/* 1-N: Screen offset in the line. */

				/* svi_line() cache information. */
	size_t	 c_sboff;	/* 0-N: offset of first character byte. */
	size_t	 c_eboff;	/* 0-N: offset of  last character byte. */
	u_int8_t c_scoff;	/* 0-N: offset into the first character. */
	u_int8_t c_eclen;	/* 1-N: columns from the last character. */
	u_int8_t c_ecsize;	/* 1-N: size of the  last character. */
} SMAP;

				/* Macros to flush/test cached information. */
#define	SMAP_CACHE(smp)		((smp)->c_ecsize != 0)
#define	SMAP_FLUSH(smp)		((smp)->c_ecsize = 0)

typedef struct _svi_private {
/* INITIALIZED AT SCREEN CREATE. */
	SMAP	*h_smap;	/* First slot of the line map. */
	SMAP	*t_smap;	/*  Last slot of the line map. */

	size_t	 exlinecount;	/* Ex overwrite count. */
	size_t	 extotalcount;	/* Ex overwrite count. */
	size_t	 exlcontinue;	/* Ex line continue value. */

				/* svi_opt_screens() cache information. */
#define	SVI_SCR_CFLUSH(svp)	svp->ss_lno = OOBLNO
	recno_t	 ss_lno;	/* 1-N: Line number. */
	size_t	 ss_screens;	/* Return value. */

	recno_t	 olno;		/* 1-N: old cursor file line. */
	size_t	 ocno;		/* 0-N: old file cursor column. */
	size_t	 sc_col;	/* 0-N: LOGICAL screen column. */
	SMAP	*sc_smap;	/* SMAP entry where sc_col occurs. */

/* PARTIALLY OR COMPLETELY COPIED FROM PREVIOUS SCREEN. */
	size_t	 srows;		/* 1-N: Rows in the terminal/window. */

				/* Add a byte string to the screen. */
	int	(*scr_addnstr) __P((SCR *, const char *, size_t));
				/* Add a string to the screen. */
	int	(*scr_addstr) __P((SCR *, const char *));
				/* Clear the screen. */
	int	(*scr_clear) __P((SCR *));
				/* Clear to the end of the line. */
	int	(*scr_clrtoeol) __P((SCR *));
				/* Return the cursor. */
	int	(*scr_cursor) __P((SCR *, size_t *, size_t *));
				/* Delete a line. */
	int	(*scr_deleteln) __P((SCR *));
				/* End the screen. */
	int	(*scr_end) __P((SCR *));
				/* Insert a line. */
	int	(*scr_insertln) __P((SCR *));
				/* Set an attribute. */
	int	(*scr_inverse) __P((SCR *, int));
				/* Set an attribute for a line. */
	int	(*scr_linverse) __P((SCR *, size_t));
				/* Move the cursor. */
	int	(*scr_move) __P((SCR *, size_t, size_t));
				/* Refresh the screen. */
	int	(*scr_refresh) __P((SCR *));
				/* Restore the screen. */
	int	(*scr_restore) __P((SCR *));
				/* Set the size of the screen. */
	int	(*scr_size) __P((SCR *, int));

/* Flags. */
#define	SVI_CUR_INVALID	0x001	/* Cursor position is unknown. */
#define	SVI_DIVIDER	0x002	/* Screen divider is displayed. */
#define	SVI_INFOLINE	0x004	/* The infoline is being used by v_ntext(). */
#define	SVI_SCR_DIRTY	0x008	/* Screen needs refreshing. */
#define	SVI_SCR_IVIDEO	0x010	/* Display in inverse video. */
#define	SVI_SCR_NUMBER	0x020	/* Screen numbering changed. */
	u_int8_t flags;
} SVI_PRIVATE;

#define	SVP(sp)		((SVI_PRIVATE *)((sp)->svi_private))

/*
 * Macros to get to the head/tail of the smap.  If the screen only has one
 * line, HMAP can be equal to TMAP, so the code has to understand the off-
 * by-one errors that can result.  If stepping through an SMAP and operating
 * on each entry, use sp->t_rows as the count of slots, don't use a loop
 * that compares <= TMAP.
 */
#define	_HMAP(sp)	(SVP(sp)->h_smap)
#define	HMAP		_HMAP(sp)
#define	_TMAP(sp)	(SVP(sp)->t_smap)
#define	TMAP		_TMAP(sp)

/*
 * One extra slot is always allocated for the map so that we can use
 * it to do vi :colon command input; see svi_get().
 */
#define	SIZE_HMAP(sp)	(SVP(sp)->srows + 1)

/*
 * We use the screen map pointer to indicate that the screen has been
 * initialized and it's ready to be manipulated (including split).
 */
#define	SVI_SCRINIT(sp)	(_HMAP(sp) != NULL)

#define	O_NUMBER_FMT	"%7lu "			/* O_NUMBER format, length. */
#define	O_NUMBER_LENGTH	8
#define	SCREEN_COLS(sp)				/* Screen columns. */	\
	((O_ISSET(sp, O_NUMBER) ? (sp)->cols - O_NUMBER_LENGTH : (sp)->cols))

#define	INFOLINE(sp)				/* Info line offset. */	\
	((sp)->rows == 1 ? 0 : (sp)->t_maxrows)

/*
 * Small screen (see svi/svi_refresh.c, section 7a) and one-line
 * screen test.  Note, both cannot be true for the same screen.
 */
#define	IS_SMALL(sp)	((sp)->t_minrows != (sp)->t_maxrows)
#define	IS_ONELINE(sp)	((sp)->rows == 1)

#define	HALFTEXT(sp)				/* Half text. */	\
	((sp)->t_rows == 1 ? 1 : (sp)->t_rows / 2)
#define	HALFSCREEN(sp)				/* Half text screen. */	\
	((sp)->t_maxrows == 1 ? 1 : (sp)->t_maxrows / 2)

/*
 * Next tab offset.
 *
 * !!!
 * There are problems with how the historical vi handled tabs.  For example,
 * by doing "set ts=3" and building lines that fold, you can get it to step
 * through tabs as if they were spaces and move inserted characters to new
 * positions when <esc> is entered.  I believe that nvi does tabs correctly,
 * but there may be some historical incompatibilities.
 */
#define	TAB_OFF(c)	COL_OFF((c), O_VAL(sp, O_TABSTOP))

/* The line relative to a specific window. */
#define	RLNO(sp, lno)	(sp)->woff + (lno)				\

/* Add a character into the text. */
#define	ADDCH(sp, svp, ch)						\
    (svp)->scr_addnstr(sp, KEY_NAME(sp, ch), KEY_LEN(sp, ch));

/* If messages waiting to be displayed. */
#define	MSGS_WAITING(sp)						\
	((sp)->msgq.lh_first != NULL &&					\
	    !F_ISSET((sp)->msgq.lh_first, M_EMPTY) ||			\
	 (sp)->gp->msgq.lh_first != NULL &&				\
	    !F_ISSET((sp)->gp->msgq.lh_first, M_EMPTY))

/* If more than one screen being shown. */
#define	IS_SPLIT(sp)							\
	((sp)->q.cqe_next != (void *)&(sp)->gp->dq ||			\
	(sp)->q.cqe_prev != (void *)&(sp)->gp->dq)

/* Screen adjustment operations. */
typedef enum { A_DECREASE, A_INCREASE, A_SET } adj_t;

/* Screen position operations. */
typedef enum { P_BOTTOM, P_FILL, P_MIDDLE, P_TOP } pos_t;

/* Scrolling operations. */
typedef enum {
	CNTRL_B, CNTRL_D, CNTRL_E, CNTRL_F,
	CNTRL_U, CNTRL_Y, Z_CARAT, Z_PLUS
} scroll_t;

/* Generic interfaces to the vi screen. */
int	svi_optchange __P((SCR *, int));
int	svi_screen_copy __P((SCR *, SCR *));
int	svi_screen_edit __P((SCR *));
int	svi_screen_end __P((SCR *));
int	svi_screen_init __P((SCR *));

/* Svi common messages. */
enum svimtype { SVIM_NOINIT };
void	svi_message __P((SCR *, char *, enum svimtype));

/* Svi function prototypes. */
int	svi_bg __P((SCR *));
int	svi_busy __P((SCR *, char const *));
int	svi_change __P((SCR *, recno_t, lnop_t));
int	svi_clrtoeos __P((SCR *));
size_t	svi_colpos __P((SCR *, recno_t, size_t));
int	svi_column __P((SCR *, size_t *));
conf_t	svi_confirm __P((SCR *, MARK *, MARK *));
int	svi_crel __P((SCR *, long));
int	svi_ex_cmd __P((SCR *, struct _excmdarg *, MARK *));
int	svi_ex_run __P((SCR *, MARK *, char *, size_t));
int	svi_ex_write __P((void *, const char *, int));
int	svi_fg __P((SCR *, CHAR_T *));
input_t	svi_get __P((SCR *, TEXTH *, ARG_CHAR_T, u_int));
int	svi_join __P((SCR *, SCR *, SCR *, SCR **));
int	svi_line __P((SCR *, SMAP *, size_t *, size_t *));
int	svi_msgflush __P((SCR *, int));
int	svi_number __P((SCR *));
size_t	svi_opt_screens __P((SCR *, recno_t, size_t *));
size_t	svi_rcm __P((SCR *, recno_t, int));
int	svi_refresh __P((SCR *));
int	svi_resize __P((SCR *, long, adj_t));
int	svi_sm_1down __P((SCR *));
int	svi_sm_1up __P((SCR *));
int	svi_sm_cursor __P((SCR *, SMAP **));
int	svi_sm_fill __P((SCR *, recno_t, pos_t));
int	svi_sm_next __P((SCR *, SMAP *, SMAP *));
recno_t	svi_sm_nlines __P((SCR *, SMAP *, recno_t, size_t));
int	svi_sm_position __P((SCR *, MARK *, u_long, pos_t));
int	svi_sm_prev __P((SCR *, SMAP *, SMAP *));
int	svi_sm_scroll __P((SCR *, MARK *, recno_t, scroll_t));
int	svi_split __P((SCR *, SCR **, SCR **));
int	svi_swap __P((SCR *, SCR **, char *));
