# Bindings for Tcl code
# Uses the regions module...regions are Tcl procs here, and filler is area
# between procs (including global variables/commands).

# This module uses the following heuristic for figuring out the boundaries for
# Tcl procs:
# A proc begins with the word 'proc ' and is *not* indented. The body of the
# proc begins with an open brace at the end of any line containing or after
# the 'proc'. (which is safe as long as the arglist doesn't begin that way, too)
# The proc ends after the close brace matching said open brace.

load_library_module regions.tcl
load_library_module balancebind.tcl

set tcl_region_beginner "proc "

# index points to a proc...check to see if it really is a procedure heading.
proc check_proc {t index} {
	scan [$t index $index] "%d.%d" row column
	if {($column == 0)} {return 1} else {return 0}
}

proc region_end {t index} {
	set ob "\{"	; set cb "\}"
	while {[$t get "$index lineend -1c"] != $ob} {
		set index [$t index "$index +1 line"]}
	return "[find_right_pair $t "$index lineend" $ob $cb] lineend"
}

proc region_prev {t index} {
	global tcl_region_beginner
	set trace "$index lineend"
	while {[set trace [text_string_last $t $tcl_region_beginner $trace]] != ""} {
		if {[check_proc $t $trace]} {return $trace}}
	return ""
}

proc region_next {t index} {
	global tcl_region_beginner
	set trace $index
	while {[set trace [text_string_first $t $tcl_region_beginner "$trace +1c"]] != ""} {
		if {[check_proc $t $trace]} {return $trace}}
	return ""
}

# For when the collapsing module is available
proc region_collapse_indicate {t start end} {
	if {[$t compare "$end linestart" == "$start linestart"]} {return ""}
	set ob "\{"  ;	set cb "\}"
	set index $start
	while {[$t get "$index lineend -1c"] != $ob} {
		set index [$t index "$index +1 line"]}
	return "[$t get $start "$index lineend"]...$cb"
}

# Since a 'filler' in Tcl code consists of some prev global code and, likely,
# a comment about the forthcoming function, return a description of the
# pre-comment stuff (... if nonempty), then a description of the comment
# (# start of comment ... end of comment
# Do not collapse if only one line break before comment and comment is
# only one line long. (Return text is two lines...this makes a space between
# functions and makes collapsed code look neater.)
proc filler_collapse_indicate {t start end} {
	if {[$t compare $start >= "$end -1 lines"] && ([$t get $start] == "\n")} {return ""}
	scan [$t index $end] "%d.%d" commline dummy
	while {[$t get "$commline.0"] != "#"} {incr commline -1}
	set commend $commline
	while {[$t get "$commline.0"] == "#"} {incr commline -1}

	if {[$t compare "$commline.0 -1c" == $start]} {
		set prefix ""} else {set prefix "..."}
	set commstart "$commline.0 +1 l"

	set result [selection_indicate $t $commstart "$commend.0 lineend"]
	if {($result == "") && ($prefix == "")} {return ""}
	if {($prefix != "") && ($result == "")} {
		if {[set result [$t get $commstart "$commend.0 lineend"]] == ""} {
			append prefix "\[[string range [gensym] 3 end]\]..."
	}}
	return "\n$prefix$result"
}


if $edit_flag {

set adjust_line_width 0

# Splits line if it is longer than length. Returns number of extra lines
# produced (0 is if line was not broken). Index is on line to break, length
# is desired length, string is contents of line.
proc split_line {t begin {prefix "#"} {length ""} {string ""}} {
	set begin [$t index $begin]
	if {($string == "")} {
		set string [$t get $begin "$begin lineend"]
	}
	if {($length == "")} {
		global adjust_line_width 
		if $adjust_line_width {
			set length $adjust_line_width
		} else {set length [lindex [$t configure -width] 4]}}
	set len [expr $length - [string length $prefix] - 1 ]

	if {([string length $string] < $len)} {
		$t insert $begin "$prefix "
		return 0}
	set offset [string last " " [string range $string 0 $len]]
	if {($offset < 0)} {return 0}
	set break [$t index "$begin +$offset chars"]
	$t delete $break
	$t insert $break "\n"
	set breaks [split_line $t "$break +1c" $prefix $length]
	$t insert $begin "$prefix "
	global modified ; set modified 1
	return [incr breaks]
}

proc do_adjust_region {t begin end prefix length} {
	set l [string length $prefix]
	global modified ; set modified 1
	for {set i $end} {$i >= $begin} {incr i -1} {
		set line [$t get "$i.0" "$i.0 lineend"]
		set pref [string first $prefix $line]
		set space [string first " " [string range $line $pref end]]
		$t delete "$i.0" "$i.0 +$pref c +$space c"
	}
	for {set i $end} {$i > $begin} {incr i -1} {$t delete "$i.0 -1c"}
	$t delete "$begin.0"
	split_line $t "$begin.0" $prefix $length
}

# Adjusts selected region to fit in length columns, so that no lines wrap
# If unspecified, length defaults to window width.
proc adjust_region {t begin end prefix {length ""}} {
	set chars [$t get "$begin.0" "$end.0 lineend"]
	set m1 [gensym] ; set m2 [gensym]
	$t mark set $m2 "$end.0 lineend"
	do_adjust_region $t $begin $end $prefix $length
	$t mark set $m1 "$begin.0"
	register_undoable_cmd $t [list undo_filter $t $m1 $m2 $chars] "Adjust $chars" "$m1 $m2"
}

proc adjust_function {t} {
	scan [$t index insert] "%d %d" begin dummy
	set end $begin
	set prefix [lindex [split [$t get "$begin.0" "$begin.0 lineend"] " "] 0]
	if {[string index $prefix 0] != "#"} {beep ; return}
	while {[string index $prefix 0] == [$t get "$begin.0 -1l"]} {
		incr begin -1
		if {$begin == 0} {incr begin ; break}}
	scan [$t index end] "%d.%d" endline dummy
	while {[string index $prefix 0] == [$t get "$end.0 +1l"]} {
		incr end
		if {$end > $endline} {incr end -1 ; break}}
	adjust_region $t $begin $end $prefix
}


# Tcl bindings.
proc tclbind {f m} {
	parse_bindings Text \
M-j			{adjust_function %W}

	if {[winfo exists $m]} {parse_menuentries $m.edit.m {
					{"Reformat Comment" 0 M-j}}
}}

tclbind $frame $menu
}

region_bind $frame $menu "Procedure" 2
set regions_defined 1
