/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1994, Martin Sjlin
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME:         tcp.c
 *
 * SCCSINFO:		@(#)tcp.c	1.4 6/6/94
 *
 * ORIGINAL AUTHOR(S):  Martin Sjlin 1994-04-29
 *
 * DESCRIPTION:
 * This module contains function to open a socket, make a server
 * socket, setting up a daemon.
 * 
 *
 * MODIFICATIONS:
 *	<list mods here with name and date>
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"			/* includes system  includes */
#include "machine.h"			/* t_u16bits etc */

#include <sys/socket.h>
#ifdef  HAVE_USLEEP			/* included in config.h ifndef */
#include <sys/time.h>
#endif /* n HAVE_USLEEP */
#include <rpc/rpc.h>
#ifndef IPPROTO_TCP			/* broken 4.3BSD headers */
#include <netinet/in.h>
#endif /* n IPPROTO_TCP */
#ifndef TCP_NODELAY			/* more broken header */
#include <netinet/tcp.h>
#endif /* n TCP_NODELAY */
#include <arpa/inet.h>
#include <netdb.h>

#include "liblincks.h"			/* to get SUCCESS/FAIL */

#ifdef  __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif /* n __STDC__ */

/* libc */
#ifdef sun
extern struct hostent *gethostbyname();
extern u_long inet_addr( /* char *cp */ );
#endif

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_tcp.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern char *sys_errlist[];	/* errno.h */
extern int sys_nerr;		/* errno.h */
extern int errno;		/* errno.h */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_rpc.h"

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define MAXTRIES 3
#define COMPILE_WARN \
"You must compile tcp.c with -DHAVE_SETPGRP2, -DHAVE_SETPGRP0\n\
or -DHAVE_SETSID or it will not work.\n"

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
/* none yet */

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* none */

/*  */
/*********************************************************************
 * Function: int tcp_lookup(char *host, t_u32bits *addr, struct hostent **php)
 *
 * Try to find IP addres for host (if not null), return in addr,
 * if no host, set to INADDR_ANY. If *php != NULL, set it to found host entry
 *
 * Modifications:
 *	<list mods here with name and date>
 *
 */
int tcp_lookup(host, addr, php)
  char *host;
  t_u32bits *addr;
  struct hostent **php;
{
  *addr = htonl(INADDR_ANY); /* default ... */
  if (host != NULL)  {
    struct hostent *hp = NULL;
    if (php != NULL) *php = NULL;

    /* first, check if numeric addr, then lookup */    
    if ((*addr = inet_addr(host)) == (t_u32bits) -1) {    /* network order */
      if ((hp = gethostbyname(host)) != (struct hostent *)NULL)  {
	*addr = *((unsigned int *) hp->h_addr_list[0]);  /* network order */
	if (php != NULL) *php = hp;
	return(SUCCESS);
      }
    } else
      return(SUCCESS);
  }
  return(FAIL);
}

/*  */
/*********************************************************************
 * Function: int tcp_connect(void (*error)(), char *host, int port)
 *
 * Create a tcp socket and connects to specificed address.
 *
 * Modifications:
 *	<list mods here with name and date>
 * 
 */
int tcp_connect(error, host, port)
  void (*error)P_((char *, ...));
  char *host;
  int port;
{
         int         i;
         int         sockfd;
  struct sockaddr_in server;
          t_u32bits  addr;

  /* get a TCP socket for our use */
  if ((sockfd=socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
    (*error)("tcp_open: Failed to setup socket -  %s\n", 
	       sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    return(-1);
  }

  /* lookup remote host */
  if (tcp_lookup(host, &addr, NULL) != 0) {
    (*error) ("tcp_open: failed to find host addr for %s - %s\n", 
	      host,
	      sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    (void) close(sockfd);
    return(-1);
  }

  /* host & port number to connect to */
  server.sin_family      = AF_INET;
  server.sin_addr.s_addr = addr;
  server.sin_port        = htons(((t_u16bits)port));

  /* ok, let us try to connect three times */
  for(i=MAXTRIES; i>0; i--)
   if (connect(sockfd, (struct sockaddr *) &server, sizeof(server)) == 0)
     break;
  else if (i == 1)  {
     (*error)("tcp_open: connect to %s failed third time - %s\n", 
	      host,
	      sys_errlist[(errno > sys_nerr) ? 0 : errno]);
     (void) close(sockfd);
     return(-1);
   }

  /* and return socket number */
  return(sockfd);
}

/*  */
/*********************************************************************
 * Function: int tcp_server(void (*error)(), int port, int clients)
 *
 * Create a tcp socket, bind and listens to specificed port.
 *
 * Modifications:
 *	<list mods here with name and date>
 * 
 */
int tcp_server(error, port, clients)
  void (*error)P_((char *, ...));
  int port;
  int clients;
{
         int         i;
         int         sockfd;
  struct sockaddr_in server;
         t_u32bits   addr = htonl(INADDR_ANY); /* default ... */

  /* get a TCP socket for our use */
  if ((sockfd=socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
    (*error)("tcp_open: Failed to setup socket - %s\n", 
	     sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    return(-1);
  }

  /* bind address */
  server.sin_family      = AF_INET;
  server.sin_addr.s_addr = addr;
  server.sin_port        = htons(((t_u16bits)port));

  /* ok, let us try to bind three times */
  for(i=MAXTRIES; i>0; i--)
   if (bind(sockfd, (struct sockaddr *) &server, sizeof(server)) == 0)
     break;
   else if (errno == EADDRINUSE && i > 1)
     sleep(30);
   else {
     (*error)("tcp_open: bind to port %d failed three times - %s\n",
	      port, 
	      sys_errlist[(errno > sys_nerr) ? 0 : errno]);
     (void) close(sockfd);
     return(-1);
   } 

  /* and then listen ... */
  if (listen(sockfd, clients) < 0) {
    (*error) ("tcp_server: Failed to listen on socket - %s\n", 
	      sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    (void) close(sockfd);
    return(-1);
  }

  /* return socket nummer */
  return(sockfd);
}

/*  */
/*********************************************************************
 * Function: int setup_daemon(void)
 *
 * setups the daemon stuff
 *
 * Modifications:
 *	<list mods here with name and date>
 * 
 */
int setup_daemon()
{
  /* fork off in case we were started from command line - it 
   * should not be the case, but one never knows (this to get
   * our own process group) */
  switch( fork() )  {
  case -1 : /* failed to fork */
    perror("Failed to fork first child");
    exit(1);
  case 0 : /* child ... */
    break;
  default : /* parent */
    _exit(0);
  }

#if !defined(HAVE_SETSID) && !defined(HAVE_SETPGRP0) && !defined(HAVE_SETPGRP2)
/*
 * You need to have one of HAVE_SETSID, HAVE_SETPGRP0 or HAVE_SETPGRP2 defined
 */
   (void)fprintf(stderr, COMPILE_WARN);
#endif

  /* Setup up process leader */
#ifdef HAVE_SETSID
  if ( setsid() < 0) 
#endif
#ifdef HAVE_SETPGRP0 /* System V */
  if ( setpgrp() == -1)  
#endif
#ifdef HAVE_SETPGRP2 /* BSD */
  if ( setpgrp(0,getpid()) == -1)  
#endif
    return(FAIL);
  else
    return(SUCCESS);
}

/*  */
/**********************************************************************
 * Function: void tcp_error(char *format, ....)
 *
 * error routine used for tcp_server ...
 *
 * Modifications:
 *      <list mods with name and date>
 */
#ifdef __STDC__
void tcp_error(char *format, ...)
#else
void tcp_error(va_alist)
  va_dcl
#endif /* n __STDC__ */
{
  va_list ap;
#ifndef __STDC__
  char *format;
  va_start(ap);
  format = va_arg(ap, char *);
#else
  va_start(ap,format);
#endif
  vfprintf(stderr, format, ap);
  va_end(ap);
}

/*  */
/**********************************************************************
 * Function: int tickle_rpc(char *host, u_long prg, u_long vers)
 *
 * tickle a rpc server, return FAIL if fails to find program else SUCCESS
 *
 * Modifications:
 *      <list mods with name and date>
 */
int tickle_rpc(host, prg, vers)
  char *host;
  u_long prg;
  u_long vers;
{
  enum clnt_stat st;
  if ((st = callrpc(host, prg, vers, (u_long)0,
		      xdr_void,(char *)NULL,
		      xdr_void,(char *)NULL))) {
    (void)fprintf(stderr,
		  "\nLooked for %s %ld %ld %ld:",
		  host, prg, vers, (u_long)0);
    clnt_perrno(st);
    return (FAIL);
  }
  return(SUCCESS);
}
