/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	usfuns.c
 *
 * SCCSINFO:		@(#)usfuns.c	1.12 5/17/94
 *
 * ORIGINAL AUTHOR(S): Ralph Rnnquist,  18 Sep 1987
 *
 * MODIFICATIONS:
 *      1994-01-06 Martin Sjlin. Added RCB_REGISTERCALLBACK
 *                 register callback function for 
 *                 edit/store/release/release exchanges with DBS!
 *      1994-01-06 Martin Sjlin. Joined register_pen_function.
 *      1994-01-12 Martin Sjlin. Restructured call back interface
 *                 for PEN - two we have two level, one lowlevel
 *                 corresponding to the original PEN interface and
 *                 as second corresponding to the high level events
 *                 we receive from the DBS. This will remove most of
 *                 duplicated code in pen_history.c as well as the 
 *                 low level constant copied from dbcodes.h!
 *      1994-05-17 Martin Sjlin. Added SA_SendAlive() which does
 *                 call LL_SyncTest to tell dbs that we are still
 *                 alive ... (WWW server was logged out ...)
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *
 * This file is part of the liblincks library.
 * It contains the functions that application programs call
 * in order to access a LINCKS database.
 *
 * ADDITIONAL DOCUMENTATION:
 *
 * See the document, "Application Interface to the LINCKS
 * Database System through the LIBLINCKS library.  Programmers Manual,"
 * found at ~lincks/Text/Misc/Liblincks.manual/liblincks_man.tex
 *
 * IF YOU CHANGE THIS FILE:
 *	-- YOU MIGHT ALSO NEED TO UPDATE THE MANUAL.
 *	-- the externally-callable functions of this file are
 *		declared in liblincks.h, so you may need to change it.
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "sunlincks.h"
#include "dbcodes.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_usfuns.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_comhist.h"
#include "f_dbcomm.h"
#include "f_structsupport.h"
#include "f_sysfuns.h"
#include "f_wssupport.h"
#include "f_logws.h"
#include "f_lowcom.h"
#include "f_linckshandler.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern char LL_loginuser[];	/* wssupport.c */
extern char LL_errorstring[];	/* Used for returning DB errors */
extern molecule *LL_moltbl;
extern int LL_uid;

extern char *sys_errlist[];
extern int sys_nerr;
extern int errno;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define ATTR(ix)	LL_moltbl[ix].attributes
#define LINKS(ix)	LL_moltbl[ix].links
#define DB_GET_UPDATE(lbl) \
  CH_update("DB_GET command", DB_OPERATION, (char *)NULL, (char *)NULL,\
	    (char *)NULL, (char *)NULL, 0, lbl, (label *)NULL, (label *)NULL)

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int BT_Condition P_(( label *lbl, label *btv, label *gpd, label
                              *root ));
static errcode CH_ULA_UNLINKAT P_(( label *lbl, char *group, char
                                     *field, int pos ));
static errcode DO_STOREOBJ P_(( label *lbl, int flag ));
static errcode Find_Binding_Table_Version P_(( label *bt, label *gpd,
                                                label *root, label *trans ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
static int calledfrom = 1;	/* from where is POLL_oob called */
static errcode usf_errcode;	/* Used to transfer error code through */

/*LINTLIBRARY*/

/*  */
/**********************************************************************
 * Function: errcode LOGIN(char *dbid)
 *
 * Login user on specified database (dbid), prompting for user
 * id and password on stdout/stdin. See also NOPROMPT_LOGIN()
 *
 * char *dbid      string with database name
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LOGIN(dbid)
  char *dbid;
{
  return LL_startup((char *)NULL, (char *)NULL, dbid);
}

/*  */
/**********************************************************************
 * Function: errcode NOPROMPT_LOGIN(char *user, char *passwd, char *dbid)
 *
 * Login user on specified database (dbid) with given
 *         userid and password, with no prompting. See also LOGIN()
 *
 * char *user      string with user name
 * char *passwd    string with password
 * char *dbid      string with database name
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode NOPROMPT_LOGIN(user, passwd, dbid)
  char *user;
  char *passwd;
  char *dbid;
{
  return LL_startup(user, passwd, dbid);
}

/*  */
/**********************************************************************
 * Function: errcode LOGOUT()
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LOGOUT()
{
  POLL_oob((void *)&calledfrom);

  /* LOGOUT should re-initialise the workspace structures !!! */
  (void)CH_update("LOGOUT", DB_OPERATION, (char *)NULL, (char *)NULL,
		  (char *)NULL, (char *)NULL, 0, (label *)NULL,
		  (label *)NULL, (label *)NULL);

  /* since we might get newer pens after CH_update (?) */
  POLL_oob((void *)&calledfrom);

  return LL_DBLogout();	/* Returns ERR_RCBAD if there are pending PENs */
}

/*  */
/**********************************************************************
 * Function:errcode PEN_RETRIEVE_MESSAGES(int(*mapfn)(),void *extra)
 *
 * Retrieves the pen messages and passes them to the mapfn together
 * with the extra argument. We ends the traversal (but we retrieve
 * all messages from the dbs) when the function returns non zero or
 * we get to the end of the list.
 *
 * int mapfn(void *extra, int index, int type, char *msg, int count, int len) 
 *
 * where 
 *
 *   void *extra   is the application program parameter
 *   int  index    is the index to the node object 
 *   int  type     is the type of pen message
 *   char *msg     is an extra paramter string allocated by
 *                 liblincks and should be free by you.
 *   int  count    is the current item, starting from 1. 
 *   const int len      is the number of messages in total
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode PEN_RETRIEVE_MESSAGE(mapfn, extra)
  int (*mapfn) ();
  void *extra;
{
   return(LL_DBPenMessage(mapfn, extra));
}
/*  */
/**********************************************************************
 * Function: int (*RCB_REGISTERCALLBACK(int what, int (*callback)(), void *extra))();
 *
 * Register call back function for EDIT/STORE/RELEASE/RETRIEVE - CODE
 * exchanges with the database. Functions are called with extra parameter
 * and the actual label ....
 *
 * Modifications:
 *      <list mods with name and date>
 */
int (*RCB_REGISTERCALLBACK(what, callback, extra))()
  int what;
  int (*callback)(/* void *extra, int index, int moltype */);
  void *extra;
{
  if (what == RCB_PEN_NOPOLL)
    calledfrom = * (int *) extra;
  if (RCB_PEN <= what && what <= RCB_PEN_POLL)
    return(register_pen_function( what, callback, extra));
  else if (RCB_EDIT <= what && what <= RCB_RELEASE)
    /* offset start at 0 */
    return(LL_registercallback( what - RCB_EDIT, callback, extra));
  else
    return((int (*)()) NULL);
}

/*  */
/**********************************************************************
 * Function: errcode CAGN_CHANGEATTRGROUPNAME(label *lbl, char *from, *to)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CAGN_CHANGEATTRGROUPNAME(lbl, from, to)
  label *lbl;
  char *from, *to;
{
  label xlbl;
  attrgroup *p;
  int oldinst, inws;

  inws = INWORKSPACE(lbl);
  if ((int) strlen(to) > MAXTAGLENGTH)
    return ERR_TAG;
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  if ((usf_errcode = LL_sysgetattrgroup(xlbl.inst, from, &p)))
    return usf_errcode;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & p)))
    return usf_errcode;

  if (!p)
    usf_errcode = LL_sysgetattrgroup(xlbl.inst, from, &p);

  free(p->tag);
  p->tag = LL_copystring(to);

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return CH_update("CAGN_CHANGEATTRGROUPNAME", inws, from, to, (char *)NULL,
		   (char *)NULL, 0, lbl, (label *)NULL, (label *)NULL);
}

/*  */
/**********************************************************************
 * Function: errcode CAN_CHANGEATTRNAME(label *lbl, char *group, *from, *to)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CAN_CHANGEATTRNAME(lbl, group, from, to)
  label *lbl;
  char *group, *from, *to;
{
  label xlbl;
  attribute *p;
  int oldinst, inws;

  inws = INWORKSPACE(lbl);
  if ((int) strlen(to) > MAXTAGLENGTH)
    return ERR_TAG;

  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  if ((usf_errcode = LL_sysgetattr(xlbl.inst, group, from, &p)))
    return usf_errcode;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & p)))
    return usf_errcode;

  if (!p)
    usf_errcode = LL_sysgetattr(xlbl.inst, group, from, &p);

  free(p->tag);
  p->tag = LL_copystring(to);

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return CH_update("CAN_CHANGEATTRNAME", inws, from, to, group, (char *)NULL,
		   0, lbl, (label *)NULL, (label *)NULL);

}

/*  */
/**********************************************************************
 * Function:errcode GAGN_GETATTRGROUPNAMES(label *lbl,int(*mapfn)(),void *extra)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode GAGN_GETATTRGROUPNAMES(lbl, mapfn, extra)
  label *lbl;
  int (*mapfn) ();
  void *extra;
{
  label xlbl;
  attrgroup *p, *this;
  int count, len, inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  this = LL_copyattributes(ATTR(xlbl.inst));

  count = len = 0;

  mapgroups(p, this)++ len;	/* macro defined in sunlincks.h */

  if ((*mapfn) (extra, (char *)NULL, 0, len) != 0) {
    LL_freeattributes(this);
    return NO_ERROR;
  }
  mapgroups(p, this) {
    ++count;
    if ((*mapfn) (extra, p->tag, count, len) != 0) {
      LL_freeattributes(this);
      return NO_ERROR;
    };
  };
  LL_freeattributes(this);
  return (inws) ? NO_ERROR : DB_GET_UPDATE(lbl);
}


/*  */
/**********************************************************************
 * Function: errcode GAN_GETATTRNAMES(FOUR PARAMETERS)
 *
 * Parameters:
 *  label *lbl
 *  char *group
 *  int (*mapfn)()
 *  void *extra
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode GAN_GETATTRNAMES(lbl, group, mapfn, extra)
  label *lbl;
  char *group;
  int (*mapfn) ();
  void *extra;
{
  label xlbl;
  attrgroup *this;
  attribute *p;
  int count, len, inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  if ((usf_errcode = LL_sysgetattrgroup(xlbl.inst, group, &this)))
    return usf_errcode;

  this = LL_copyattrgroup(this, (attrgroup *) NULL);
  count = len = 0;

  mapfields(p, this)++ len;

  if ((*mapfn) (extra, (char *)NULL, 0, len) != 0) {
    LL_freeattrgroup(this);
    return NO_ERROR;
  }
  mapfields(p, this) {
    ++count;
    if ((*mapfn) (extra, p->tag, count, len) != 0) {
      LL_freeattrgroup(this);
      return NO_ERROR;
    }
  };
  LL_freeattrgroup(this);
  return (inws) ? NO_ERROR : DB_GET_UPDATE(lbl);
}


/*  */
/**********************************************************************
 * Function: errcode GA_GETATTR(label *lbl, char *group, *field, attrval *val)
 *
 * Modifications:
 *                                and a call to CH_update added.
 *      <list mods with name and date>
 */
errcode GA_GETATTR(lbl, group, field, val)
  label *lbl;
  char *group, *field;
  attrval *val;
{
  int inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = CH_GA_GETATTR(lbl, group, field, val)))
    return usf_errcode;
  return (inws) ? NO_ERROR : DB_GET_UPDATE(lbl);
}

/*  */
/**********************************************************************
 * Function: errcode CH_GA_GETATTR(label *lbl,char *grp,char *fld, attrval *val)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_GA_GETATTR(lbl, group, field, val)
  label *lbl;
  char *group;
  char *field;
  attrval *val;
{
  label xlbl;
  attribute *p;

  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  if ((usf_errcode = LL_sysgetattr(xlbl.inst, group, field, &p)))
    return usf_errcode;

  val->attvalue = LL_copyval(p->size, p->value);
  val->attsize = p->size;

  return NO_ERROR;
}


/*  */
/**********************************************************************
 * Function: errcode RAGT_REMOVEATTRGROUPTAG(label *lbl, char *group)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode RAGT_REMOVEATTRGROUPTAG(lbl, group)
  label *lbl;
  char *group;
{
  label xlbl;
  attrgroup *prev, *this;
  int oldinst, inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  if ((usf_errcode = LL_findgrptag((cell *) ATTR(xlbl.inst), group,
				   (cell **) & prev, (cell **) & this)))
    return usf_errcode;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & this)))
    return usf_errcode;

  if (!this)
    (void)LL_findgrptag((cell *) ATTR(xlbl.inst), group,
			(cell **) & prev, (cell **) & this);

  if (!prev)
    ATTR(xlbl.inst) = this->next;
  else
    prev->next = this->next;

  LL_freeattrgroup(this);

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return CH_update("RAGT_REMOVEATTRGROUPTAG", inws, (char *)NULL, (char *)NULL,
		   group, (char *)NULL, 0, lbl, (label *)NULL, (label *)NULL);
}

/*  */
/**********************************************************************
 * Function: errcode RAGV_REMOVEATTRGROUPVAL(label *lbl, char *group)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode RAGV_REMOVEATTRGROUPVAL(lbl, group)
  label *lbl;
  char *group;
{
  label xlbl;
  attrgroup *p;
  int oldinst, inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  if ((usf_errcode = LL_sysgetattrgroup(xlbl.inst, group, &p)))
    return usf_errcode;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & p)))
    return usf_errcode;

  if (!p)
    (void)LL_sysgetattrgroup(xlbl.inst, group, &p);

  LL_freeattrgroupval(p->value);
  p->value = (attribute *)NULL;

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return CH_update("RAGV_REMOVEATTRGROUPVAL", inws, (char *)NULL, (char *)NULL,
		   group, (char *)NULL, 0, (label *)NULL, (label *)NULL, (label *)NULL);
}

/*  */
/**********************************************************************
 * Function: errcode RAT_REMOVEATTRTAG(label *lbl, char *group, *field)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode RAT_REMOVEATTRTAG(lbl, group, field)
  label *lbl;
  char *group;
  char *field;
{
  int inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = CH_RAT_REMOVEATTRTAG(lbl, group, field)))
    return usf_errcode;
  return CH_update("RAT_REMOVEATTRTAG", inws, (char *)NULL, (char *)NULL,
		   group, field, 0, (label *)NULL, (label *)NULL, (label *)NULL);
}

/*  */
/**********************************************************************
 * Function: errcode CH_RAT_REMOVEATTRTAG(label *lbl, char *group, char *field)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_RAT_REMOVEATTRTAG(lbl, group, field)
  label *lbl;
  char *group;
  char *field;
{
  label xlbl;
  attrgroup *gp;
  attribute *prev, *this;
  int oldinst;

  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  gp = ATTR(xlbl.inst);
  if ((usf_errcode = LL_findfieldtag((cell **) & gp, group, field,
				     (cell **) & prev, (cell **) & this)))
    return usf_errcode;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & this)))
    return usf_errcode;

  if (!this) {
    gp = ATTR(xlbl.inst);
    (void)LL_findfieldtag((cell **) & gp, group, field,
			  (cell **) & prev, (cell **) & this);
  }
  if (!prev)
    gp->value = this->next;
  else
    prev->next = this->next;

  LL_freeattr(this);

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return NO_ERROR;
}

/*  */
/**********************************************************************
 * Function: errcode RAV_REMOVEATTRVAL(label *lbl, char *group, *field)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode RAV_REMOVEATTRVAL(lbl, group, field)
  label *lbl;
  char *group, *field;
{
  label xlbl;
  attribute *p;
  int oldinst, inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  if ((usf_errcode = LL_sysgetattr(xlbl.inst, group, field, &p)))
    return usf_errcode;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & p)))
    return usf_errcode;

  if (!p)
    (void)LL_sysgetattr(xlbl.inst, group, field, &p);

  if (p->value)
    free(p->value);
  p->size = 0;
  p->value = (char *)NULL;

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return CH_update("RAV_REMOVEATTRVAL", inws, (char *)NULL, (char *)NULL,
		   group, field, 0, lbl, (label *)NULL, (label *)NULL);
}

/*  */
/**********************************************************************
 * Function: errcode SA_SETATTR(label *lbl, char *group, *field, attrval *val)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode SA_SETATTR(lbl, group, field, val)
  label *lbl;
  char *group, *field;
  attrval *val;
{
  int inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = CH_SA_SETATTR(lbl, group, field, val)))
    return usf_errcode;
  return CH_update("SA_SETATTR", inws, (char *)NULL, (char *)NULL, group,
		   field, 0, lbl, (label *)NULL, (label *)NULL);
}

/*  */
/**********************************************************************
 * Function: errcode CH_SA_SETATTR(label *lbl,char *group, *field, attrval *val)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_SA_SETATTR(lbl, group, field, val)
  label *lbl;
  char *group, *field;
  attrval *val;
{
  label xlbl;
  label *p;			/* Should point to something */
  int oldinst;

  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & p)))
    return usf_errcode;

  (void)LL_sysputattr(xlbl.inst, group, field, val);

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return NO_ERROR;
}

/*  */
/**********************************************************************
 * Function: errcode CLGN_CHANGELINKGROUPNAME(label *lbl, char *from, *to)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CLGN_CHANGELINKGROUPNAME(lbl, from, to)
  label *lbl;
  char *from, *to;
{
  label xlbl;
  linkgroup *p;
  int oldinst, inws;

  inws = INWORKSPACE(lbl);
  if ((int) strlen(to) > MAXTAGLENGTH)
    return ERR_TAG;
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  if ((usf_errcode = LL_sysgetlinkgroup(xlbl.inst, from, &p)))
    return usf_errcode;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & p)))
    return usf_errcode;

  if (!p)
    (void)LL_sysgetlinkgroup(xlbl.inst, from, &p);

  free(p->tag);
  p->tag = LL_copystring(to);

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return CH_update("CLGN_CHANGELINKGROUPNAME", inws, from, to, (char *)NULL,
		   (char *)NULL, 0, lbl, (label *)NULL, (label *)NULL);
}

/*  */
/**********************************************************************
 * Function: errcode CLN_CHANGELINKNAME(label *lbl, char *group, *from, *to)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CLN_CHANGELINKNAME(lbl, group, from, to)
  label *lbl;
  char *group, *from, *to;
{
  label xlbl;
  linktag *p;
  int oldinst, inws;

  inws = INWORKSPACE(lbl);
  if ((int)strlen(to) > MAXTAGLENGTH)
    return ERR_TAG;
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  if ((usf_errcode = LL_sysgetlink(xlbl.inst, group, from, &p)))
    return usf_errcode;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & p)))
    return usf_errcode;

  if (!p)
    (void)LL_sysgetlink(xlbl.inst, group, from, &p);

  free(p->tag);
  p->tag = LL_copystring(to);

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return CH_update("CLN_CHANGELINKNAME", inws, from, to, group, (char *)NULL,
		   0, lbl, (label *)NULL, (label *)NULL);
}

/*  */
/**********************************************************************
 * Function: errcode GLGN_GETLINKGROUPNAMES(THREE PARAMETERS)
 *
 * Parameters:
 *	label *lbl
 *	int (*mapfn)()
 *	void *extra
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode GLGN_GETLINKGROUPNAMES(lbl, mapfn, extra)
  label *lbl;
  int (*mapfn) ();
  void *extra;
{
  label xlbl;
  linkgroup *this, *p;
  int count, len, inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  this = LL_copylinks(LINKS(xlbl.inst));

  count = len = 0;

  mapgroups(p, this) len++;	/* macro defined in sunlincks.h */

  if ((*mapfn) (extra, (char *)NULL, 0, len) != 0) {
    LL_freelinks(this);
    return NO_ERROR;
  }
  mapgroups(p, this) {
    ++count;
    if ((*mapfn) (extra, p->tag, count, len) != 0) {
      LL_freelinks(this);
      return NO_ERROR;
    };
  };
  LL_freelinks(this);
  return (inws) ? NO_ERROR : DB_GET_UPDATE(lbl);
}

/*  */
/**********************************************************************
 * Function: errcode GLI_GETLINKITEM(FIVE PARAMETERS)
 *
 * Parameters:
 *	label *lbl
 *	char *group,
 *	char *field
 *	int pos
 *	label *item
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode GLI_GETLINKITEM(lbl, group, field, pos, item)
  label *lbl;
  char *group, *field;
  int pos;
  label *item;
{
  int inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = CH_GLI_GETLINKITEM(lbl, group, field, pos, item, 1)))
    return usf_errcode;
  return (inws) ? NO_ERROR : DB_GET_UPDATE(lbl);
}

/*  */
/**********************************************************************
 * Function: errcode CH_GLI_GETLINKITEM(SIX PARAMETERS)
 *
 * Parameters:
 *	label *lbl
 *	char *group,
 *	char *field
 *	int pos
 *	label *item
 *      int type
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_GLI_GETLINKITEM(lbl, group, field, pos, item, type)	/****/
  label *lbl;
  char *group, *field;
  int pos;
  label *item;
  int type;
{
  label xlbl;
  linktag *tp;
  linkitem *p;

  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  if ((usf_errcode =
       LL_sysgetlink((type == 1) ? xlbl.inst : xlbl.vs, group, field, &tp)))
    return usf_errcode;

  p = tp->value;

  if (!p) {
    item->vs = -1;
    item->inst = -1;
    return ERR_POS;
  };

  while (pos > 1 && p->next_item) {
    p = p->next_item;
    --pos;
  };

  item->vs = p->link.vs;
  item->inst = p->link.inst;

  if (pos != 1)
    return ERR_POS;
  else
    return NO_ERROR;
}

/*  */
/**********************************************************************
 * Function: errcode GLN_GETLINKNAMES(FOUR PARAMETERS)
 *
 * Parameters:
 *	label *lbl
 *	char *group
 *	int (*mapfn)()
 *	void *extra
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode GLN_GETLINKNAMES(lbl, group, mapfn, extra)
  label *lbl;
  char *group;
  int (*mapfn) ();
  void *extra;
{
  int inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = CH_GLN_GETLINKNAMES(lbl, group, mapfn, extra, 1)))
    return usf_errcode;
  return (inws) ? NO_ERROR : DB_GET_UPDATE(lbl);
}

/*  */
/**********************************************************************
 * Function: errcode CH_GLN_GETLINKNAMES(FIVE PARAMETERS)
 *
 * Parameters:
 *	label *lbl
 *	char *group
 *	int (*mapfn)()
 *	void *extra
 *      int  type
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_GLN_GETLINKNAMES(lbl, group, mapfn, extra, type)
  label *lbl;
  char *group;
  int (*mapfn) ();
  void *extra;
  int type;
{
  label xlbl;
  linkgroup *this;
  linktag *p;
  int count, len;

  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  if ((usf_errcode =
       LL_sysgetlinkgroup((type == 1) ? xlbl.inst : xlbl.vs, group, &this)))
    return usf_errcode;

  this = LL_copylinkgroup(this, (linkgroup *) NULL);
  count = len = 0;

  mapfields(p, this)++ len;

  if ((*mapfn) (extra, (char *)NULL, 0, len) != 0) {
    LL_freelinkgroup(this);
    return NO_ERROR;
  }
  mapfields(p, this) {
    ++count;
    if ((*mapfn) (extra, p->tag, count, len) != 0) {
      LL_freelinkgroup(this);
      return NO_ERROR;
    };
  };
  LL_freelinkgroup(this);
  return NO_ERROR;
}

/*  */
/**********************************************************************
 * Function: errcode GLV_GETLINKVAL(FIVE PARAMETERS)
 *
 * Parameters:
 *	label *lbl
 *	char *group
 *	char *name
 *	int (*mapfn)()
 *	void *extra
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode GLV_GETLINKVAL(lbl, group, name, mapfn, extra)
  label *lbl;
  char *group, *name;
  int (*mapfn) ();
  void *extra;
{
  int inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = CH_GLV_GETLINKVAL(lbl, group, name, mapfn, extra)))
    return usf_errcode;
  return (inws) ? NO_ERROR : DB_GET_UPDATE(lbl);
}

/*  */
/**********************************************************************
 * Function: errcode CH_GLV_GETLINKVAL(FIVE PARAMETERS)
 *
 * Parameters:
 *	label *lbl
 *	char *group
 *	char *name
 *	int (*mapfn)()
 *	void *extra
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_GLV_GETLINKVAL(lbl, group, name, mapfn, extra)	/****/
  label *lbl;
  char *group, *name;
  int (*mapfn) ();
  void *extra;
{
  label xlbl;
  linktag *this;
  linkitem *p;
  int count, len;

  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  if ((usf_errcode = LL_sysgetlink(xlbl.inst, group, name, &this)))
    return usf_errcode;

  this = LL_copylinktag(this, (linktag *)NULL);
  count = len = 0;

  mapitems(p, this)++ len;

  if ((*mapfn) (extra, (char *)NULL, 0, len) != 0) {
    LL_freelinktag(this);
    return NO_ERROR;
  }
  mapitems(p, this) {
    ++count;
    if ((*mapfn) (extra, &(p->link.vs), count, len) != 0) {
      LL_freelinktag(this);
      return NO_ERROR;
    }
  }
  LL_freelinktag(this);
  return NO_ERROR;
}

/*  */
/**********************************************************************
 * Function: errcode LINK(FIVE PARAMETERS)
 *
 * Parameters:
 *	label *from
 *	char *group
 *	char *field
 *	label *to
 *	int pos
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LINK(from, group, field, to, pos)
  label *from;
  char *group;
  char *field;
  label *to;
  int pos;
{
  int inws;

  if (!group || !field)
    return ERR_PARAMS;

  inws = INWORKSPACE(from);
  if ((usf_errcode = CH_LINK(from, group, field, to, pos)))
    return usf_errcode;
  return CH_update("LINK", inws, (char *)NULL, (char *)NULL, group,
		   field, pos, (label *)NULL, from, to);
}

/*  */
/**********************************************************************
 * Function: errcode CH_LINK(FIVE PARAMETERS)
 *
 * Parameters:
 *	label *from
 *	char *group
 *	char *field
 *	label *to
 *	int pos
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_LINK(from, group, field, to, pos)
  label *from;
  char *group;
  char *field;
  label *to;
  int pos;
{
  label xlbl;
  linkitem *p;
  int oldinst;

  if (pos < 1)
    return ERR_POS;
  if ((usf_errcode = LL_bindlbl(from, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;

  if (pos > 1 &&
      (usf_errcode = LL_sysgetlinkitem(xlbl.inst, group, field, pos - 1, &p)))
    return usf_errcode;

  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & p)))
    return usf_errcode;

  (void)LL_sysputlinkitem(xlbl.inst, group, field, pos, to);

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  from->inst = xlbl.inst;
  return NO_ERROR;
}

/*  */
/**********************************************************************
 * Function: errcode RLGT_REMOVELINKGROUPTAG(label *lbl, char *group)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode RLGT_REMOVELINKGROUPTAG(lbl, group)
  label *lbl;
  char *group;
{
  label xlbl;
  linkgroup *prev, *this;
  int oldinst, inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  if ((usf_errcode = LL_findgrptag((cell *) LINKS(xlbl.inst), group,
				   (cell **) & prev, (cell **) & this)))
    return usf_errcode;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & this)))
    return usf_errcode;

  if (!this)
    (void)LL_findgrptag((cell *) LINKS(xlbl.inst), group,
			(cell **) & prev, (cell **) & this);

  if (!prev)
    LINKS(xlbl.inst) = this->next;
  else
    prev->next = this->next;

  LL_freelinkgroup(this);

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return CH_update("RLGT_REMOVELINKGROUPTAG", inws, (char *)NULL, (char *)NULL,
		   group, (char *)NULL, 0, lbl, (label *)NULL, (label *)NULL);
}

/*  */
/**********************************************************************
 * Function: errcode RLGV_REMOVELINKGROUPVAL(label *lbl, char *group)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode RLGV_REMOVELINKGROUPVAL(lbl, group)
  label *lbl;
  char *group;
{
  label xlbl;
  linkgroup *p;
  int oldinst, inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  if ((usf_errcode = LL_sysgetlinkgroup(xlbl.inst, group, &p)))
    return usf_errcode;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & p)))
    return usf_errcode;

  if (!p)
    (void)LL_sysgetlinkgroup(xlbl.inst, group, &p);

  LL_freelinkgroupval(p->value);
  p->value = (linktag *)NULL;

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return CH_update("RLGV_REMOVELINKGROUPVAL", inws, (char *)NULL, (char *)NULL,
		   group, (char *)NULL, 0, lbl, (label *)NULL, (label *)NULL);
}

/*  */
/**********************************************************************
 * Function: errcode RLT_REMOVELINKTAG(label *lbl, char *group, char *field)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode RLT_REMOVELINKTAG(lbl, group, field)
  label *lbl;
  char *group;
  char *field;
{
  int inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = CH_RLT_REMOVELINKTAG(lbl, group, field)))
    return usf_errcode;
  return CH_update("RLT_REMOVELINKTAG", inws, (char *)NULL, (char *)NULL, group,
		   field, 0, lbl, (label *)NULL, (label *)NULL);
}

/*  */
/**********************************************************************
 * Function: errcode CH_RLT_REMOVELINKTAG(label *lbl, char *group, char *field)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_RLT_REMOVELINKTAG(lbl, group, field)
  label *lbl;
  char *group, *field;
{
  label xlbl;
  linkgroup *gp;
  linktag *prev, *this;
  int oldinst;

  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  gp = LINKS(xlbl.inst);
  if ((usf_errcode = LL_findfieldtag((cell **) & gp, group, field,
				     (cell **) & prev, (cell **) & this)))
    return usf_errcode;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & this)))
    return usf_errcode;

  if (!this) {
    gp = LINKS(xlbl.inst);
    (void)LL_findfieldtag((cell **) & gp, group, field,
			  (cell **) & prev, (cell **) & this);
  };

  if (!prev)
    gp->value = this->next;
  else
    prev->next = this->next;

  LL_freelinktag(this);

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return NO_ERROR;
}

/*  */
/**********************************************************************
 * Function: errcode RLV_REMOVELINKVAL(label *lbl, char *group, char *field)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode RLV_REMOVELINKVAL(lbl, group, field)
  label *lbl;
  char *group;
  char *field;
{
  int inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = CH_RLV_REMOVELINKVAL(lbl, group, field)))
    return usf_errcode;
  return CH_update("RLV_REMOVELINKVAL", inws, (char *)NULL, (char *)NULL, group,
		   field, 0, lbl, (label *)NULL, (label *)NULL);
}

/*  */
/**********************************************************************
 * Function: errcode CH_RLV_REMOVELINKVAL(label *lbl, char *group, char *field)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_RLV_REMOVELINKVAL(lbl, group, field)	/****/
  label *lbl;
  char *group, *field;
{
  label xlbl;
  linktag *p;
  int oldinst;

  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  if ((usf_errcode = LL_sysgetlink(xlbl.inst, group, field, &p)))
    return usf_errcode;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & p)))
    return usf_errcode;

  if (!p)
    (void)LL_sysgetlink(xlbl.inst, group, field, &p);

  LL_freelinkval(p->value);
  p->value = (linkitem *)NULL;

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return NO_ERROR;
}

/*  */
/**********************************************************************
 * Function: errcode ULA_UNLINKAT(label *lbl,char *group,char *field,int pos)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode ULA_UNLINKAT(lbl, group, field, pos)
  label *lbl;
  char *group;
  char *field;
  int pos;
{
  int inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = CH_ULA_UNLINKAT(lbl, group, field, pos)))
    return usf_errcode;
  return CH_update("ULA_UNLINKAT", inws, (char *)NULL, (char *)NULL,
		   group, field, pos, lbl, (label *)NULL, (label *)NULL);
}


/*  */
/**********************************************************************
 * Function: static errcode CH_ULA_UNLINKAT(label *lbl,char *group,char *field,int pos)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode CH_ULA_UNLINKAT(lbl, group, field, pos)
  label *lbl;
  char *group;
  char *field;
  int pos;
{
  label xlbl;
  linktag *p;
  linkitem *prev, *this;
  int oldinst;

  if (pos < 1)
    return ERR_POS;
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  if ((usf_errcode = LL_sysgetlinkitem(xlbl.inst, group, field, pos, &this)))
    return usf_errcode;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & this)))
    return usf_errcode;

  if (!this)
    (void)LL_sysgetlinkitem(xlbl.inst, group, field, pos, &this);

  (void)LL_sysgetlink(xlbl.inst, group, field, &p);
  prev = p->value;

  if (prev == this)
    p->value = this->next_item;
  else {
    while (prev->next_item != this)
      prev = prev->next_item;
    prev->next_item = this->next_item;
  };

  free((FREEPTR *)this);

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: errcode UL_UNLINK(label *lbl, char *group, char *field, label *to)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode UL_UNLINK(lbl, group, field, to)
  label *lbl;
  char *group;
  char *field;
  label *to;
{
  label xlbl;
  linktag *p;
  int oldinst, inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  if ((usf_errcode = LL_sysgetlink(xlbl.inst, group, field, &p)))
    return usf_errcode;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & p)))
    return usf_errcode;

  if (!p)
    (void)LL_sysgetlink(xlbl.inst, group, field, &p);

  LL_sysdellinkitem(&p->value, to);

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return CH_update("UL_UNLINK", inws, (char *)NULL, (char *)NULL, group,
		   field, 0, lbl, (label *)NULL, to);
}

/*  */
/**********************************************************************
 * Function: errcode GI_GETIMAGE(label *lbl, attrval *val)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode GI_GETIMAGE(lbl, val)
  label *lbl;
  attrval *val;
{
  label xlbl;
  image_desc *xval;
  int inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  xval = LL_moltbl[xlbl.inst].image;
  if (!xval) {
    val->attsize = 0;
    val->attvalue = (char *)NULL;
  } else {
    val->attvalue = LL_copyval(xval->size, xval->value);
    val->attsize = xval->size;
  };
  return (inws) ? NO_ERROR : DB_GET_UPDATE(lbl);
}

/*  */
/**********************************************************************
 * Function: errcode SI_SETIMAGE(label *lbl, attrval *val)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode SI_SETIMAGE(lbl, val)
  label *lbl;
  attrval *val;
{
  label xlbl;
  label *p;
  image_desc *xval;
  int oldinst, inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  oldinst = xlbl.inst;
  if ((usf_errcode = LL_checkbuffer(&xlbl, (cell **) & p)))
    return usf_errcode;

  if (!(xval = (image_desc *) malloc(sizeof(image_desc)))) {
    LogMess(LL_uid,
      "Liblincks: Couldn't allocate memory for image_desc in SI_SETIMAGE, %s",
	    sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    perror("SI_SETIMAGE: Cannot allocate memory for 'image_desc'\n");
  }
  xval->value = LL_copyval(val->attsize, val->attvalue);
  xval->size = val->attsize;

  if(LL_moltbl[xlbl.inst].image) {
    LL_freeimage(LL_moltbl[xlbl.inst].image);
  }
  LL_moltbl[xlbl.inst].image = xval;

  if (oldinst != xlbl.inst)
    (void)LL_update_VS(&xlbl);
  lbl->inst = xlbl.inst;
  return CH_update("SI_SETIMAGE", inws, (char *)NULL, (char *)NULL, (char *)NULL,
		   (char *)NULL, 0, lbl, (label *)NULL, (label *)NULL);
}

/*  */
/**********************************************************************
 * Function: errcode CO_CREATEOBJ(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CO_CREATEOBJ(lbl)
  label *lbl;
{
  POLL_oob((void *)&calledfrom);

  if ((usf_errcode = CH_CO_CREATEOBJ(lbl)))
    return usf_errcode;
  return CH_update("CO_CREATEOBJ", WS_OPERATION, (char *)NULL, (char *)NULL,
		   (char *)NULL, (char *)NULL, 0, lbl, (label *)NULL, (label *)NULL);
}

/*  */
/**********************************************************************
 * Function: errcode CH_CO_CREATEOBJ(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_CO_CREATEOBJ(lbl)	/****/
  label *lbl;
{
  label xlbl;

  xlbl.vs = LL_newmol();
  if ((usf_errcode = LL_newmolinfo(xlbl.vs)))
    return usf_errcode;
  xlbl.inst = INST_UNBOUND;
  if ((usf_errcode = LL_set_current(&xlbl)))
    return usf_errcode;

  lbl->vs = xlbl.vs;
  lbl->inst = xlbl.inst;

  return NO_ERROR;
}

/*  */
/**********************************************************************
 * Function: errcode CPO_COPYOBJ(label *obj, label *src)
 *
 * Copy the version of `src' to be a new transient version of `obj'.
 * If `src' is transient, then it is automatically stored prior to
 * copying.
 * If `obj' is transient, then it also is stored prior to copying.
 *
 * Modifications:
 *      <list mods with name and date>
 *
 */
errcode CPO_COPYOBJ(obj, src)
  label *obj;
  label *src;
{
  label xlbl;
  label ylbl;
  int inws;

  inws = INWORKSPACE(src);

  if ((usf_errcode = LL_bindlbl(src, &xlbl)))
    return usf_errcode;
  if ((usf_errcode = LL_bindlbl(obj, &ylbl)))
    return usf_errcode;

  /*
   * When a version is used to create another version, the old version
   * should no longer be editable.
   * If the original was editable, then we store that version and its
   * history structure node to the database.
   */
  if (LL_moltbl[xlbl.inst].edited == TRUE)
    switch (CH_SO_STOREOBJ(&xlbl)) {
    case SUCCESS:
      break;
    case ERR_LABEL:
      return ERR_LABEL;
    case FAIL:
      return FAIL;
    default:
      (void)printf("(CH_)SO_STOREOBJ returned something weird\n");
      break;
    }

  /*
   * if the 'obj' object is an editable version, then also this
   * object is stored
   */
  if (LL_moltbl[ylbl.inst].edited == TRUE)
    switch (CH_SO_STOREOBJ(&ylbl)) {
    case SUCCESS:
      break;
    case ERR_LABEL:
      return ERR_LABEL;
    case FAIL:
      return FAIL;
    default:
      (void)printf("(CH_)SO_STOREOBJ returned something weird\n");
      break;
    }
  /*
   * LL_checkbuffer makes a copy of xlbl (the bound 'src').
   * xlbl.inst will then point to the copy.
   *
   * LL_checkbuffer sets the edit history link
   * (SYSTEM:Parent) to point to the 'src' version
   */

  {
    cell *dummy;
    if ((usf_errcode = LL_checkbuffer(&xlbl, &dummy)))
      return usf_errcode;
  }

  /*
   * The copy belongs to the 'obj' object.
   * NOTE that we use xlbl to point to the result of the copy;
   * we want it to have the vs of "obj" and the inst of the copy.
   */
  ylbl.inst = xlbl.inst;

  (void)LL_update_VS(&ylbl);

  obj->inst = ylbl.inst;

  return CH_update("CPO_COPYOBJ", inws, (char *)NULL, (char *)NULL,
		   (char *)NULL, (char *)NULL, 0, (label *)NULL, src, obj);
}

/*  */
/**********************************************************************
 * Function: errcode GSR_GETSYSTEMROOT(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode GSR_GETSYSTEMROOT(lbl)
  label *lbl;
{
  POLL_oob((void *)&calledfrom);

  lbl->vs = VS_SYS_ROOT;
  lbl->inst = INST_UNBOUND;
  return NO_ERROR;
}

/*  */
/**********************************************************************
 * Function: errcode GC_GETCURRENT(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode GC_GETCURRENT(lbl)
  label *lbl;
{
  label xlbl;

  POLL_oob((void *)&calledfrom);

  lbl->inst = INST_UNBOUND;
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;
  lbl->inst = xlbl.inst;
  return NO_ERROR;
}

/*  */
/**********************************************************************
 * Function: errcode UBL_UNBINDLABEL(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode UBL_UNBINDLABEL(lbl)
  label *lbl;
{
  POLL_oob((void *)&calledfrom);

  if ((usf_errcode = LL_checkvs(lbl)))
    return usf_errcode;
  lbl->inst = INST_UNBOUND;

  return NO_ERROR;
}


/*  */
/**********************************************************************
 * Function: errcode GVS_GETVERSIONS(label *lbl,int (*mapfn)(),void *extra)
 *
 * Applies (*mapfn)() to all items in the *lbl history.
 *
 * Here we define the GVS function for the new history structure where
 * each slot consists of labels for:
 *  1. the version concerned
 *  2. the command history version for the store of 1
 *  3. the temporal predecessors of 1
 * The map function is applied to the items of the slots in order and
 * with the arguments:
 *   (void *)extra,	-- as given to GVS
 *   (label *)item,	-- the label
 *   int count,		-- value = -1, 0, 1, 2, or 3
 * The count values are interpreted as follows:
 *  -1	= Asking for loading a new history fragment
 *   0	= An initialisation call
 *   1	= The version call (beginning of slot)
 *   2	= The store command label call
 *   3	= predecessor call
 * The map function returns values GVS_CONTINUE, GVS_STOP or GVS_NEXT to
 * guide subsequent mapping.
 * GVS_CONTINUE	signals normal continuation, which for the -1 call case
 *		means to NOT load more fragments.
 * GVS_STOP	signals immediate termination of the whole mapping.
 * GVS_NEXT	signals a skip to next history slot.
 *
 * Note: the store command label is omitted from the transient slots, but
 * an extra case 2 call is inserted with a (label *)NULL pointer;
 * If the store command label values coincide with the version
 * concerned, then the command label is in reality unknown and therefore
 * the 2 call case is done with a (label *)NULL pointer.
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode GVS_GETVERSIONS(lbl, mapfn, extra)
  label *lbl;
  int (*mapfn) ();
  void *extra;
{
  linkgroup *history;
  linktag *slot;
  linkitem *version;
  label item;
  int count;
  int vs;
  int rtn;

  POLL_oob((void *)&calledfrom);

  if ((usf_errcode = LL_checkvs(lbl)))
    return usf_errcode;

  count = 0;

  if ((*mapfn) (extra, (label *)NULL, 0) == GVS_STOP)
    return NO_ERROR;
  
  if (!LL_sysgetlinkgroup(lbl->vs, TH_TRANSIENT, &history)) {
    mapfields(slot, history) {
      if (!LL_moltbl[slot->value->link.inst].inWS)
	continue;
      count = 1;
      for (version = slot->value; version; version = version->next_item) {
	item.vs = version->link.vs;
	item.inst = version->link.inst;
	rtn = (*mapfn) (extra, &item, count);
	if (rtn == GVS_STOP)
	  return NO_ERROR;
	if (rtn == GVS_NEXT)
	  break;
	if (count == 1) {
	  rtn = (*mapfn) (extra, (label *)NULL, 2);
	  if (rtn == GVS_STOP)
	    return NO_ERROR;
	  if (rtn == GVS_NEXT)
	    break;
	  count = 3;
	}
      }
    }
  }

  for (vs = lbl->vs; vs != -1; vs = next_vs_fragment(vs)) {
    if (!LL_moltbl[vs].inWS) {
      if ((*mapfn)(extra, (label *)NULL, -1) != GVS_NEXT)
	return NO_ERROR;
    }
    if (LL_sysgetlinkgroup(vs, VSNET, &history))
      continue;
    mapfields(slot, history) {
      count = 1;
      for (version = slot->value; version; version = version->next_item) {
	if ((count == 2) && (version->link.vs == item.vs)) {
	  rtn =(*mapfn) (extra, (label *)NULL, count);
	} else {
	  item.vs = version->link.vs;
	  item.inst = version->link.inst;
	  rtn =  (*mapfn) (extra, &item, count);
	}
	if (rtn == GVS_STOP)
	  return NO_ERROR;
	if (rtn == GVS_NEXT)
	  break;
	if (count < 3)
          count++;
      }
    }
  }
  return NO_ERROR;
}


/*  */
/**********************************************************************
 * Function: errcode GL_GETLATEST(label *lbl, int (*mapfn)(), void *extra)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode GL_GETLATEST(lbl, mapfn, extra)
  label *lbl;
  int (*mapfn) ();
  void *extra;
{
  int inws;

  POLL_oob((void *)&calledfrom);

  if ((usf_errcode = CH_GL_GETLATEST(lbl, mapfn, extra, &inws)))
    return usf_errcode;
  return (inws) ? NO_ERROR : DB_GET_UPDATE(lbl);
}

/*  */
/**********************************************************************
 * Function: errcode CH_GL_GETLATEST(label *lbl, int (*mapfn)(), void *extra, int *inws)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_GL_GETLATEST(lbl, mapfn, extra, inws)
  label *lbl;
  int (*mapfn) ();
  void *extra;
  int *inws;
{
  linktag *this;
  linkitem *p;
  int count, len;

  *inws = 1;
  if ((usf_errcode = LL_checkvs(lbl)))
    return usf_errcode;

  (void)LL_sysgetlink(lbl->vs, SYSTEM_LATEST, &this);

  this = LL_copylinktag(this, (linktag *)NULL);
  count = len = 0;
  mapitems(p, this)++ len;

  if ((*mapfn) (extra, (char *)NULL, 0, len)) {
    LL_freelinktag(this);
    return NO_ERROR;
  }
  mapitems(p, this) {
    ++count;
    if (*inws == 1)
      /* p -> link.vs changed to p->link */
      *inws = INWORKSPACE(&(p->link));
    if ((*mapfn) (extra, &(p->link.vs), count, len)) {
      LL_freelinktag(this);
      return NO_ERROR;
    }
  }
  LL_freelinktag(this);
  return NO_ERROR;
}

/*  */
/**********************************************************************
 * Function: errcode RFL_REMOVEFROMLATEST(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode RFL_REMOVEFROMLATEST(lbl)
  label *lbl;
{
  int inws;

  inws = INWORKSPACE(lbl);
  if ((usf_errcode = CH_RFL_REMOVEFROMLATEST(lbl)))
    return usf_errcode;
  return CH_update("RFL_REMOVEFROMLATEST", inws, (char *)NULL, (char *)NULL,
		   (char *)NULL, (char *)NULL, 0, lbl, (label *)NULL, (label *)NULL);
}

/*  */
/**********************************************************************
 * Function: errcode CH_RFL_REMOVEFROMLATEST(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_RFL_REMOVEFROMLATEST(lbl)
  label *lbl;
{
  label xlbl;
  linktag *tp = (linktag *)NULL;
  linkitem *px = (linkitem *)NULL, *py = (linkitem *)NULL;

  if (lbl->inst == INST_UNBOUND)
    return ERR_LABEL;
  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return usf_errcode;

  (void)LL_sysgetlink(xlbl.vs, SYSTEM_LATEST, &tp);

  if (!tp)
    return ERR_FIELD;		/* should never happen */
  else if (!tp->value->next_item && tp->value->link.inst == xlbl.inst)
    return (ERR_LASTLATEST);

  px = tp->value;
  while (px && px->link.inst != xlbl.inst) {
    py = px;
    px = px->next_item;
  };

  if (!py)
    tp->value = (px) ? px->next_item : (linkitem *)NULL;
  else
    py->next_item = (px) ? px->next_item : (linkitem *)NULL;

  return NO_ERROR;
}

/*  */
/**********************************************************************
 * Function: errcode RO_RELEASEOBJ(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode RO_RELEASEOBJ(lbl)
  label *lbl;
{
  label xlbl;
  linkgroup *gp;
  linktag *tp;
  int nextvs = -1;
  int vs = -1;
  int mol = -1;

  POLL_oob((void *)&calledfrom);

  xlbl.vs = lbl->vs;
  xlbl.inst = lbl->inst;

  if (!LL_moltbl[xlbl.vs].inWS)
    return NO_ERROR;
  if ((usf_errcode = LL_checkvs(&xlbl)))
    return usf_errcode;

  for (vs = xlbl.vs; vs != -1; vs = nextvs) {
    if (!LL_moltbl[vs].inWS)
      break;
    nextvs = next_vs_fragment(vs);
    if (!LL_sysgetlinkgroup(vs, VSNET, &gp))
      mapfields(tp, gp) {
	mol = tp->value->link.inst;
	if (LL_moltbl[mol].inWS)
	  (void)LL_DBRetry(mol, OTHER, LL_DBRelease);
      }
    (void)LL_DBRetry(vs, VS, LL_DBRelease);
  }

  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: errcode SO_STOREOBJ(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode SO_STOREOBJ(lbl)
  label *lbl;
{

  POLL_oob((void *)&calledfrom);

  if ((usf_errcode = DO_STOREOBJ(lbl,1)))
    return usf_errcode;
  return CH_update("SO_STOREOBJ", DB_OPERATION, (char *)NULL, (char *)NULL,
		   (char *)NULL, (char *)NULL, 0, lbl, (label *)NULL, (label *)NULL);
}



/*  */
/**********************************************************************
 * Function: errcode CH_SO_STOREOBJ(label *lbl)
 *
 * Description:
 *	Update history for the object and store it.
 *
 * Return codes:
 *	SUCCESS
 *	????
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_SO_STOREOBJ(lbl)
  label *lbl;
{
  return DO_STOREOBJ(lbl,0);
}

/*  */
/**********************************************************************
 * Function: static errcode DO_STOREOBJ(label *lbl,int flag)
 *
 * The flag tells whether or not to link to present user process node
 * version.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode DO_STOREOBJ(lbl,flag)
  label *lbl;
  int flag;
{
  label xlbl;
  linkgroup *trgrp = (linkgroup *)NULL;
  linkgroup *vsgrp = (linkgroup *)NULL;
  linktag *tr = (linktag *)NULL;
  linkitem *ti = (linkitem *)NULL;

  if (lbl->inst == INST_UNBOUND)
    return ERR_LABEL;

  /*
   * Note that we do NOT use EDITED(), because it has a side effect
   * of calling LL_bindlbl, which calls LL_set_current
   */

  if (!(LL_moltbl[lbl->vs].edited || LL_moltbl[lbl->inst].edited))
    return SUCCESS;

  xlbl.vs = lbl->vs;
  xlbl.inst = lbl->inst;

  if (LL_sysgetlinkgroup(xlbl.vs, TH_TRANSIENT, &trgrp))
    return FAIL;

  if ((usf_errcode = LL_sysgetlinkgroup(xlbl.vs, VSNET, &vsgrp)))
    return usf_errcode;

  /* Move transient history slot into proper history (VSNET) */
  if (trgrp && (tr = trgrp->value)) {
    trgrp->value = tr->next;
    tr->next = vsgrp->value;
    vsgrp->value = tr;
  }

  /* Insert link to storing command */
  if (tr) {
    ti = (linkitem *)malloc(sizeof(linkitem));
    ti->link.vs = xlbl.vs;
    ti->link.inst = xlbl.inst;
    ti->next_item = tr->value->next_item;
    tr->value->next_item = ti;
    if (flag)
      (void) CH_get_user_process_node(&(ti->link));
  }

  if ((usf_errcode = LL_DBRetry(xlbl.inst, OTHER, LL_DBStore)))
    return (usf_errcode);
  return LL_DBRetry(xlbl.vs, VS, LL_DBStore);
}


/*  */
/**********************************************************************
 * Function: errcode RTL_REMOVETRANSIENTLINKS(label *lbl)
 *
 * Description:
 *   RTL_REMOVETRANSIENTLINKS takes a bound label as input.
 *  The function searches for a field in the link group TH_TRANSIENT of
 *  the history structure node of 'lbl'
 *  that has 'lbl' as physically the last linkitem.
 *  If it finds such a field, it removes it.
 *  (The nodes that the field pointed to remain, however).
 *
 * Return codes:
 *	SUCCESS
 *	ERR_LABEL
 *	ERR_TH_TRANSIENT  There were no links that met the above conditions.
 *	ERR_xxx
 *
 * Modifications:
 *      <list mods with name and date>
 */

errcode RTL_REMOVETRANSIENTLINKS(lbl)
  label *lbl;
{
  linkgroup *gp;
  linktag *fpx, **fp;
  int somehit = FALSE;

  POLL_oob((void *)&calledfrom);

  /* input has to be a bound label */
  if (lbl->inst == INST_UNBOUND)
    return (ERR_LABEL);

  if ((usf_errcode = LL_sysgetlinkgroup(lbl->vs, TH_TRANSIENT, &gp)))
    return (usf_errcode);

  for (fp = &(gp->value); *fp; ) {
    if (LL_lastinst(*fp) == lbl->inst) {
      fpx = *fp;
      (*fp) = fpx->next;
      LL_freelinktag(fpx);	/* RECLAIM */
      LL_moltbl[lbl->vs].edited = TRUE;
      somehit = TRUE;
    } else {
      fp = &((*fp)->next);
    }
  }
  return (somehit) ? SUCCESS : ERR_TH_LINK;
}



/*  */
/**********************************************************************
 * Function: int INWORKSPACE(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
int INWORKSPACE(lbl)
  label *lbl;
{
  POLL_oob((void *)&calledfrom);

  if (lbl->inst != INST_UNBOUND)
    return LL_moltbl[lbl->inst].inWS;
  return LL_moltbl[lbl->vs].inWS;
}

/*  */
/**********************************************************************
 * Function: int ISTRANSIENT(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
int ISTRANSIENT(lbl)
  label *lbl;
{
  label xlbl;

  POLL_oob((void *)&calledfrom);

  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return FALSE;
  
  return (LL_moltbl[xlbl.inst].edited);
}
/*  */
/**********************************************************************
 * Function: int EDITED(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
int EDITED(lbl)
  label *lbl;
{
  label xlbl;

  POLL_oob((void *)&calledfrom);

  if ((usf_errcode = LL_bindlbl(lbl, &xlbl)))
    return FALSE;	/* <-- This is an error return, and should not happen
			 *  unless the application program is erroneous.
			 */
  return (LL_moltbl[xlbl.vs].edited || LL_moltbl[xlbl.inst].edited);
}

/*  */
/**********************************************************************
 * Function: int BOUND(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
int BOUND(lbl)
  label *lbl;
{
  POLL_oob((void *)&calledfrom);

  return (lbl->inst != INST_UNBOUND);
}


/*  */
/* BINDING TABLE MANAGEMENT */

/*  */
/**********************************************************************
 * Function: GBT_GETBINDINGTABLE(FOUR PARAMETERS)
 * Parameters:
 *	label *gpd	(in)	Generic presentation description label
 *	label *root	(in)	Root object label
 *	label *bt	(out)	Binding table label
 *	int *created	(out)	whether a new binding table was created
 *
 * Fetches the binding table for the gpd-root combination.
 *
 * We use the link value BTBL_LINK in the root object, `root', history node
 * to hold an association list of gpd-bt pairs. The binding table for a
 * given gpd-root combination is thus obtained by scanning the link value
 * BTBL_LINK for root, searching (every odd position) for the occurance of
 * the gpd label, and picking the next label as the binding table.
 *
 * If the search fails, a new object is created for binding table `bt', and
 * a gpd-bt association is made in the `root' history node. In this case, the
 * history node is also stored after the linking is made. The association is
 * made using unbound `gpd' and `bt' labels. The input `gpd' (as well as
 * `root') must anyhow be bound.
 *
 * Returns ERR_LABEL if label `gpd' or label `root' are unbound.
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode GBT_GETBINDINGTABLE(gpd, root, bt, created)
  label *gpd;
  label *root;
  label *bt;
  int *created;
{
  label btobj, btgpd, trans;
  linktag *this = (linktag *) NULL;
  linkitem *item;
  int wasedited;

  POLL_oob((void *)&calledfrom);

  *created = 0;

  if ((gpd->inst == INST_UNBOUND) || (root->inst == INST_UNBOUND))
    return ERR_LABEL;

  switch ((usf_errcode = LL_sysgetlink(root->vs, BTBL_LINK, &this))) {
  case SUCCESS:
    /*
     * The binding table object is located by associating on the gpd
     * object in BTBL_LINK of the root.
     */
    mapitems(item,this) {
      if (gpd->vs == item->link.vs) {
	item = item->next_item;
	break;
      }
      if ((item = item->next_item) == (linkitem *) NULL)
	break;
    }
    if (item != (linkitem *) NULL) {
      bt->vs = item->link.vs;
      bt->inst = item->link.inst;

      if (Find_Binding_Table_Version(bt,gpd,root,&trans)) {
	if ((usf_errcode = GC_GETCURRENT(bt)))
	  return usf_errcode;
	return ERR_VERSION;
      }
      return SUCCESS;
    }
  case ERR_GROUP:
  case ERR_FIELD:
    if ((usf_errcode = CH_CO_CREATEOBJ(bt)))
      return usf_errcode;
    btobj.vs = bt->vs;
    btobj.inst = INST_UNBOUND;
    btgpd.vs = gpd->vs;
    btgpd.inst = INST_UNBOUND;
    if ((usf_errcode =
	 (LL_sysputlinkitem(root->vs, BTBL_LINK, 1, &btobj/*unbound*/) ||
	  LL_sysputlinkitem(root->vs, BTBL_LINK, 1, &btgpd/*unbound*/))))
      return (usf_errcode);
    wasedited = LL_moltbl[root->vs].edited;
    LL_moltbl[root->vs].edited = TRUE;
    usf_errcode =
      (LL_DBRetry(root->vs, VS, LL_DBStore) ||
       CH_LINK(bt, BTBL_ROOT, root, 1) ||
       CH_LINK(bt, BTBL_GPD, gpd, 1));
    LL_moltbl[root->vs].edited = wasedited;
    *created = 1;
    return (usf_errcode);

  default:
    return (usf_errcode);
  }
}


/*  */
/**********************************************************************
 * Function: static int BT_Condition(lbl,btv,gpd,root)
 *
 * Check conditions by index i:
 * 0. lbl->inst == INST_UNBOUND
 * 1. `gpd' and `root' occur as BTBL_GPD and BTBL_ROOT resp. in btv.
 * 2. `root' occur as BTBL_ROOT in btv.
 * 3. `gpd' occur as BTBL_GPD in btv.
 *
 * When condition is satisfied, lbl->inst is set from btv->inst, and
 * i is returned. Otherwise 0 is returned.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int BT_Condition(lbl,btv,gpd,root)
  label *lbl;
  label *btv;
  label *gpd;
  label *root;
{
  linkitem *btvroot, *btvgpd;

  if ((btv->inst == INST_UNBOUND) ||
      LL_sysgetlinkitem(btv->inst, BTBL_ROOT, 1, &btvroot) ||
      LL_sysgetlinkitem(btv->inst, BTBL_GPD, 1, &btvgpd) ||
      (btvroot->link.inst != root->inst) ||
      (btvgpd->link.inst != gpd->inst))
    return FAIL;

  lbl->vs = btv->vs;
  lbl->inst = btv->inst;
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: static errcode Find_Binding_Table_Version(bt,gpd,root,trans)
 *
 * Scan the binding table history for the most appropriate version for
 * the given `gpd' and `root' labels.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode Find_Binding_Table_Version(bt,gpd,root,trans)
  label *bt;
  label *gpd;
  label *root;
  label *trans;
{
  linkgroup *history;
  linktag *slot;
  linkitem *version;
  int vs = -1;

  trans->vs = bt->vs;
  trans->inst = INST_UNBOUND;
  if (!LL_sysgetlinkgroup(bt->vs,TH_TRANSIENT,&history)) {
    mapfields(slot,history) {
      version = slot->value;
      if (LL_moltbl[version->link.inst].inWS) {
	trans->inst = version->link.inst;
	break;
      }
    }
  }
  if (!BT_Condition(bt,trans,gpd,root))
    return SUCCESS;

  if (LL_sysgetlinkgroup(bt->vs,VSNET,&history))
    return LL_set_current(bt);

  for (vs = bt->vs; vs != -1; vs = next_vs_fragment(vs)) {
    if (LL_sysgetlinkgroup(vs,VSNET,&history))
      continue;
    mapfields(slot,history) {
      version = slot->value;
      if (!BT_Condition(bt,&(version->link),gpd,root))
	return SUCCESS;
    }
  }

  return FAIL;
}

/*  */
/**********************************************************************
 * Function: errcode GBTC_GETBTCOMP(label *bt, *lbl, int changeable, int part)
 *
 * Bind `lbl' using binding table `bt'.
 *
 * The `lbl' history node is sought in BTBL_ROOT, BTBL_GPD, or BTBL_PARTS.
 * If found, `lbl' is bound to the binding table entry.
 * Otherwise `lbl' is bound using Bind_Object, and added to the binding
 * table.
 *
 * If changeable is TRUE, the bound `lbl' is then sought in its history
 * node's SYSTEM_LATEST to establish if there has been any update to the
 * object after the `lbl' version. If so, the binding table is updated to
 * contain the new component version.
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode GBTC_GETBTCOMP(bt, lbl, changeable, part)
  label *bt;		/* bound */
  label *lbl;		/* unbound */
  int changeable;	/* if set, bt updated if needed to contain
  			   latest version of component lbl */
  int part;		/* 0 = any, 1 = root, 2 = gpd */
{
#define IX_LIMIT (6)
#define BTBL(ix) acc[ix],acc[ix+1]
  static char *acc[] =
  {BTBL_ROOT, BTBL_GPD, BTBL_PARTS};
  linktag *this = (linktag *) NULL;
  linkitem *item, *xitem;
  int ix = 0;
  int pos = 1;

  POLL_oob((void *)&calledfrom);

  if (bt->inst == INST_UNBOUND)
    return usf_errcode;

  for (ix = ((part == 2)? 2: 0); ix < IX_LIMIT; ix += 2) {
    if (!LL_sysgetlink(bt->inst, BTBL(ix), &this)) {
      pos = 0;
      mapitems(item,this) {
	pos += 1;
	if (lbl->vs == item->link.vs)
	  break;
      }
      if (item != (linkitem *) NULL) {
	lbl->inst = item->link.inst;
	break;
      }
    }
    if (part) {
      ix = IX_LIMIT;
      break;
    }
  }

  if (ix >= IX_LIMIT) {		/* didn't find object */
    if (!changeable)
      return ERR_VERSION;
    switch(part) {
      case 1:
	ix = 0;
	break;
      case 2:
	ix = 2;
	break;
      default:
	ix = 4;
	break;
    }
    usf_errcode =
      (GC_GETCURRENT(lbl) ||
       CH_LINK(bt, BTBL(ix), lbl, 1));
    return usf_errcode;
  }

  if (!changeable)
    return SUCCESS;

  if ((usf_errcode = LL_sysgetlink(lbl->vs, SYSTEM_LATEST, &this)))
    return (usf_errcode);
  mapitems(xitem,this)
    if ((lbl->inst == xitem->link.inst) && (lbl->vs == xitem->link.vs))
      return SUCCESS;

  /* have a version, but not in list of latest */
  usf_errcode =
    (GC_GETCURRENT(lbl) ||
     CH_LINK(bt, BTBL(ix), lbl, pos) ||
     CH_ULA_UNLINKAT(bt, BTBL(ix), (pos+1)));
  return usf_errcode;
#undef IX_LIMIT
#undef BTBL
}


/*  */
/**********************************************************************
 * Function: errcode OS_OBJECTSYNC(label *lbl)
 *
 * Confirm with the database that the workspace copy of the object lbl
 * has an up-to-date history.
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode OS_OBJECTSYNC(lbl)
  label *lbl;
{
  POLL_oob((void *)&calledfrom);

  if (!LL_moltbl[lbl->vs].inWS)
    return SUCCESS;
  return LL_DBResolve(lbl->vs,VS);
}

/*  */
/**********************************************************************
 * Function: errcode SA_SENDALIVE()
 *
 * Tells the dbs that we are still alive and kicking'
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode SA_SENDALIVE()
{
  errcode ret;
  byte    status;

  POLL_oob((void *)&calledfrom);
  if ((ret = LL_SendCommand(ALIVE_CODE)) < 0)
    return(ret);
  return(LL_ReceiveStatus(&status));
}

