/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	linkpack.c
 *
 * SCCSINFO:		@(#)linkpack.c	1.8 5/3/94
 *
 * ORIGINAL AUTHOR(S):  Peter ]berg 1987-09-18
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 * This file contains the functions that perform link packing/unpacking
 *
 *********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************
 * errcode LL_PackGroup(linkgroup *group)
 * errcode LL_UnpackGroup(linkgroup **group)
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <netinet/in.h>

#include "sunlincks.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* Buffer definitions */
char *LL_packbuff;
char *LL_packbuffp;
int LL_packbuffsize;
int LL_packbuffspace;
int LL_packbufflen;

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_logws.h"
#include "f_dbcomm.h"
#include "f_attrpack.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern int LL_uid;
extern char *sys_errlist[];
extern int sys_nerr;
extern int errno;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* Itemtype keys */
#define NAMESTRING 0
#define ITEMLABEL 1
#define TERMINATE 2
#define COUNT 3

#define NULLABEL -1

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static void PackChars P_(( char *buff, unsigned count ));
static int PackItem P_(( linkitem *item ));
static int PackLinks P_(( linktag *links ));
static int PackString P_(( char *string, int mode ));
static void UnpackChars P_(( char *buff, unsigned count, int mode ));
static int UnpackItem P_(( linkitem **item ));
static int UnpackLinks P_(( linktag **links ));
static int UnpackString P_(( char **string, int mode ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* none */

/*  */
/**********************************************************************
 * Function: errcode LL_PackGroup(linkgroup *group)
 *
 * Recursively packs all link groups
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_PackGroup(group)
  linkgroup *group;
{
  static short termcount = 0;

  if (group) {
    if (PackString(group->tag, NAMESTRING) < 0)
      return (FAIL);

    if (PackLinks(group->value) < 0)
      return (FAIL);

    return (LL_PackGroup(group->next));
  }
  /* Insert termination marker */
  if (PackString((char *)&termcount, TERMINATE) < 0)
    return (FAIL);

  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int PackLinks(linktag *links)
 *
 * Recursively packs all links

 * Modifications:
 *      <list mods with name and date>
 */
static int PackLinks(links)
  linktag *links;
{
  static short termcount = 0;

  if (links) {
    if (PackString(links->tag, NAMESTRING) < 0)
      return (FAIL);

    if (PackItem(links->value) < 0)
      return (FAIL);

    return (PackLinks(links->next));
  }
  /* Insert termination marker */
  if (PackString((char *)&termcount, TERMINATE) < 0)
    return (FAIL);

  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int PackItem(linkitem *item)
 *
 * Recursively packs all link items

 * Modifications:
 *      <list mods with name and date>
 */
static int PackItem(item)
  linkitem *item;
{
  static int ITindex;
  static int termlabel = NULLABEL;

  if (item) {
    if (LL_MoltoIT(item->link.vs, &ITindex) < 0)
      return (FAIL);
    if (PackString((char *)(&ITindex), ITEMLABEL) < 0)
      return (FAIL);
    if (LL_MoltoIT(item->link.inst, &ITindex) < 0)
      return (FAIL);
    if (PackString((char *)(&ITindex), ITEMLABEL) < 0)
      return (FAIL);

    return (PackItem(item->next_item));
  }
  /* Insert termination marker */
  if (PackString((char *)(&termlabel), ITEMLABEL) < 0)
    return (FAIL);

  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int PackString(char *string, int mode)
 *
 * Packs a character string into the global LL_packbuff
 *
 * Modifications:
 *      1993-08-18 Martin Sjolin. Added htonl/htons
 *      <list mods with name and date>
 */
static int PackString(string, mode)
  char *string;
  int  mode;
{
  unsigned short len;
  unsigned short netshort;

  /* Check which mode we're in */
  switch (mode) {
  case NAMESTRING:
    /* Get length of string */
    len = strlen(string);
    netshort = htons(len);

    /* Adjust len according to how many characters we ACTUALLY will write */
    /* Make sure there is space for string in buffer */
    if (EnsureBuffer((int)(len + sizeof(netshort))) < 0)
        return (FAIL);

    /* Append length of string to follow */
    PackChars((char *)&netshort, sizeof(netshort));

    /* Append string */
    if (len > 0)
      PackChars(string, len);
    break;

  case TERMINATE:
    len = sizeof(netshort);

    /* Make sure there is space for string in buffer */
    if (EnsureBuffer((int)len) < 0)
      return (FAIL);

    /* Append string */
    PackChars(string, len);
    break;

  case ITEMLABEL:  {
    t_u32bits netlbl = htonl((*(t_u32bits *) string));

    /* Get length of label */
    len = sizeof(netlbl);

    /* Make sure there is space for string in buffer */
    if (EnsureBuffer((int)len) < 0)
      return (FAIL);

    PackChars((char *)&netlbl, len);
    break;
  }

  default:
    return (FAIL);
  }

  /* Return status */
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static void PackChars(char *buff, unsigned count)
 *
 * Packs characters into the pack buffer

 * Modifications:
 *      <list mods with name and date>
 */
static void PackChars(buff, count)
  register char *buff;
  register unsigned count;
{
  /* Adjust space left in pack buffer */
  LL_packbuffspace -= count;

  /* Copy chars */
 for(; count > 0; count--)
    *LL_packbuffp++ = *buff++;
}

/*  */
/**********************************************************************
 * Function: errcode LL_UnpackGroup(linkgroup **group)
 *
 * Recursively unpacks link groups from pack buffer
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode LL_UnpackGroup(group)
  linkgroup **group;
{
  static char *buff;

  if (LL_packbufflen > 0) {
    if (UnpackString(&buff, NAMESTRING) < 0)
      return (FAIL);

    if (buff) {
      if (!(*group = (linkgroup *) malloc(sizeof(linkgroup)))) {
	LogMess(LL_uid,
		"LL_UnpackGroup: malloc failed - error (%s)",
		sys_errlist[(errno > sys_nerr) ? 0 : errno]);
	perror("LL_UnpackGroup: malloc failed");
	return (FAIL);
      }
      (*group)->tag = buff;

      if (UnpackLinks(&((*group)->value)) < 0)
	return (FAIL);

      return (LL_UnpackGroup(&((*group)->next)));
    }
  }
  *group = (linkgroup *)NULL;
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int UnpackLinks(linktag **links)
 *
 * Recursively unpacks links from pack buffer
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int UnpackLinks(links)
  linktag **links;
{
  static char *buff;

  if (LL_packbufflen > 0) {
    if (UnpackString(&buff, NAMESTRING) < 0)
      return (FAIL);

    if (buff) {
      if (!(*links = (linktag *) malloc(sizeof(linktag)))) {
	LogMess(LL_uid,
		"UnpackLinks: malloc failed - error (%s)",
		sys_errlist[(errno > sys_nerr) ? 0 : errno]);
	perror("UnpackLinks: malloc failed");
	return (FAIL);
      }
      (*links)->tag = buff;

      if (UnpackItem(&((*links)->value)) < 0)
	return (FAIL);

      return (UnpackLinks(&((*links)->next)));
    }
  }
  *links = (linktag *)NULL;
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int UnpackItem(linkitem **item)
 *
 * Recursively unpacks link items from pack buffer
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int UnpackItem(item)
  linkitem **item;
{
  static int lbl;
  static char *lblp;

  if (LL_packbufflen > 0) {
    lblp = (char *)(&lbl);
    if (UnpackString(&lblp, ITEMLABEL) < 0)
      return (FAIL);

    if (lbl != NULLABEL) {
      if (!(*item = (linkitem *) malloc(sizeof(linkitem)))) {
	LogMess(LL_uid,
		"UnpackItem: malloc failed - error (%s)",
		sys_errlist[(errno > sys_nerr) ? 0 : errno]);
	perror("UnpackItem: malloc failed");
	return (FAIL);
      }
      if (LL_ITtoMol(lbl, &(*item)->link.vs) < 0)
	return (FAIL);

      lblp = (char *)(&lbl);
      if (UnpackString(&lblp, ITEMLABEL) < 0)
	return (FAIL);
      if (LL_ITtoMol(lbl, &(*item)->link.inst) < 0)
	return (FAIL);

      /* Get next item */
      return (UnpackItem(&((*item)->next_item)));
    }
  }
  *item = (linkitem *)NULL;
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int UnpackString(char **string, int mode)
 *
 * Unpacks a string from the pack buffer. Mallocs memory depending on mode
 *
 * Modifications:
 *      1993-08-18 Martin Sjolin. Added ntohl/ntohs
 *      <list mods with name and date>
 */
static int UnpackString(string, mode)
  char **string;
  int     mode;
{
  unsigned short len;
  unsigned short netshort;

  /* Check if string is to be read */
  switch (mode) {
  case NAMESTRING:
    /* Get length of entry */
    UnpackChars((char *)&netshort, (unsigned) sizeof(netshort), COUNT);
    len = ntohs(netshort);

    if (len) {
      if (!(*string = malloc((ALLOC_T)len + 1))) {
	LogMess(LL_uid,
		"UnpackString: malloc failed - error (%s)",
		sys_errlist[(errno > sys_nerr) ? 0 : errno]);
	perror("UnpackString: malloc failed");
	return (FAIL);
      }

      /* Copy into string */
      UnpackChars(*string, len, mode);
    } else 
      *string = (char *)NULL;

    break;

  case ITEMLABEL: {
    t_u32bits  netlbl;
  
    UnpackChars((char *)&netlbl, (unsigned) sizeof(netlbl), mode);

    /* this works since it is align correctly when calling */
    *((unsigned int *)(string[0])) = ntohl(netlbl);

    break;
  }

  default:
    return (FAIL);
  }

  /* Return status */
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static void UnpackChars(char *buff,unsigned count,int mode)
 *
 * Unpacks characters from the pack buffer.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void UnpackChars(buff, count, mode)
  register char *buff;
  register unsigned count;
  int mode;
{
  /* Adjust length of pack buffer */
  LL_packbufflen -= count;

  /* Copy */
  for(; count > 0; count--)
      *buff++ = *LL_packbuffp++;

  if (mode == NAMESTRING)
    *buff = '\0';
}
