/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME:             comhist.c
 *
 * SCCSINFO:                @(#)comhist.c	1.11 5/3/94
 *
 * ORIGINAL AUTHOR(S):      Thomas Hall, 1992-06-09
 *
 * MODIFICATIONS:
 *      <list mods with name and date>
 *
 * DESCRIPTION:
 * This module includes  all functions that updates, maintain and
 * explores the command history.
 *
 */
/**********************************************************************
 * INCLUDES:
 **********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "sunlincks.h"
#include "dbcodes.h"

/**********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 **********************************************************************/
#include "f_comhist.h"

/**********************************************************************
 * EXTERNALLY-AVAILABLE DATA FOUND IN THIS MODULE:
 **********************************************************************/
int global_comhistoff = 0;
int pen_off = 0;

/**********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 **********************************************************************/
#include "f_usfuns.h"
#include "f_sysfuns.h"
#include "f_structsupport.h"

/* libshared */
extern char *strdup();

/**********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 **********************************************************************/
extern molecule *LL_moltbl;     /* dbcomm.c */

/**********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 **********************************************************************/
#ifndef COMHISTOFF
static errcode CH_bad_lbl P_(( label *lbl ));
static errcode CH_find_ch_node P_(( void ));
static errcode DoCH_update P_(( char *func, int type, char *from_attr,
                                 char *to_attr, char *group, char *field,
                                 int pos, label *lbl, label *from_lbl,
                                 label *to_lbl ));
static errcode exist_ch_node P_(( void ));
static errcode set_userp_attr P_(( char *group, char *field, char *val ));
static errcode set_userp_link P_(( char *group, char *field, label *lbl ));
#endif	/* COMHISTOFF */

/**********************************************************************
 * INTERNAL (STATIC) DATA:
 **********************************************************************/
static label user_process_node;	/* The user (process) historynode*/
static char *application = (char *)NULL;	/* The application name */
static int logged_in = FALSE;	/* A flag to indicate if a LOGIN
                                   has been executed */
#ifndef COMHISTOFF
static label ch_node;		/* The main command history node */
static errcode ERRCODE;		/* Used to transfer error code
                                   through if statements */
static int prev_type = -1;	/* Flags that indicate if the prev- */
#endif	/* COMHISTOFF */

#define Labelcmp(a,b) \
	(((a)->vs == (b)->vs && (a)->inst == (b)->inst))

/*  */
/**********************************************************************
 * Function: errcode CH_update(TEN PARAMETERS)
 * Parameters:
 *	char *func;
 *	int type;
 *	char *from_attr, *to_attr, *group, *field;
 *	int pos;
 *	label *lbl, *from_lbl, *to_lbl;
 *
 * Parameters:
 *      label *lbl, *from_lbl, *to_lbl
 *      char  *func, *from_attr, *to_attr, *group, *field
 *      int   type, pos
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_update(func, type, from_attr, to_attr, group, field, pos,
		  lbl, from_lbl, to_lbl)
  char *func;
  int type;
  char *from_attr, *to_attr, *group, *field;
  int pos;
  label *lbl, *from_lbl, *to_lbl;
{
#ifdef COMHISTOFF
  return SUCCESS;
#else
  int pen_save = pen_off;
  errcode value = 0;

  pen_off = 1;
  value = DoCH_update(func, type, from_attr, to_attr, group, field, pos,
	      lbl, from_lbl, to_lbl);
  pen_off = pen_save;
  return value;
#endif	/* COMHISTOFF */
}

#ifndef COMHISTOFF
/*  */
/**********************************************************************
 * Function: errcode DoCH_update(TEN PARAMETERS)
 * Parameters:
 *	char *func;
 *	int type;
 *	char *from_attr, *to_attr, *group, *field;
 *	int pos;
 *	label *lbl, *from_lbl, *to_lbl;
 *
 * Parameters:
 *      label *lbl, *from_lbl, *to_lbl
 *      char  *func, *from_attr, *to_attr, *group, *field
 *      int   type, pos
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode DoCH_update(func, type, from_attr, to_attr, group, field, pos,
		  lbl, from_lbl, to_lbl)
  char *func;
  int type;
  char *from_attr, *to_attr, *group, *field;
  int pos;
  label *lbl, *from_lbl, *to_lbl;
{
  static char *optype[] = { "DB operation", "WS operation", "An operation" };
  char str[26];

  if (global_comhistoff)
    return SUCCESS;

  /* Make certain that the command history is initiated. */
  if (!logged_in) {
    if ((ERRCODE = CH_find_ch_node()) ||
	(ERRCODE = CH_CO_CREATEOBJ(&user_process_node)))
      return ERRCODE;
    logged_in = TRUE;
  }
  /* Check that all labels are okey. */
  if (CH_bad_lbl(lbl) || CH_bad_lbl(from_lbl) || CH_bad_lbl(to_lbl))
    return ERR_LABEL;

  /* Update user_process_node */

  /* update parameters */
  if ((ERRCODE = set_userp_attr("CALL", "Function", func)))
    return ERRCODE;

  /* Only change the type and application if neccessary. */
  if (prev_type != type) {
    if ((ERRCODE = set_userp_attr("CALL", "Type", optype[type])))
      return ERRCODE;
  }

  if (application) {
    if ((ERRCODE = set_userp_attr("CALL", "Application", application)))
      return ERRCODE;
    free(application);
    application = (char *)NULL;
  }


  (void)sprintf(str,"%d",pos);
  if ((ERRCODE=set_userp_attr("PARAMETER","From",from_attr)) ||
      (ERRCODE=set_userp_attr("PARAMETER","To",to_attr)) ||
      (ERRCODE=set_userp_attr("PARAMETER","Group",group)) ||
      (ERRCODE=set_userp_attr("PARAMETER","Field",field)) ||
      (ERRCODE=set_userp_attr("PARAMETER","Position",(pos)?str:(char*)NULL)) ||
      (ERRCODE=set_userp_link("PARAMETER","Lbl",lbl)) ||
      (ERRCODE=set_userp_link("PARAMETER","From",from_lbl)) ||
      (ERRCODE=set_userp_link("PARAMETER","To",to_lbl)))
    return ERRCODE;

  /* If database command */
  switch (type) {
  case DB_OPERATION:

    /* Link CHNET:front to userp */
    switch ((ERRCODE = CH_RLV_REMOVELINKVAL(&ch_node, "CHNET", "Front"))) {
    case ERR_GROUP:
    case ERR_FIELD:
    case SUCCESS:
      break;
    default:
      return ERRCODE;
    }
    /* Link User_p:Prev_command to CHNET:front. */
    /* Store ch_node */
    if ((ERRCODE=CH_LINK(&ch_node,"CHNET","Front",&user_process_node,1)) ||
	(ERRCODE = set_userp_link("CH", "Prev_command", &ch_node)) ||
	(ERRCODE = CH_SO_STOREOBJ(&ch_node)))
      return ERRCODE;
    break;
  case WS_OPERATION:
  case AN_OPERATION:
    /*
     * If the previous command was a DB-com and this command is a WS-com.
     * Remove the user_process_node:Prev_command link. WS-command should
     * never have this link.
     */
    if (prev_type == WS_OPERATION)
      break;
    if ((ERRCODE = set_userp_link("CH", "Prev_command", (label *) NULL)))
      return ERRCODE;
    break;
  default:
    return FAIL;
  }
  prev_type = type;

  /* Store user process node */
  if ((ERRCODE = CH_SO_STOREOBJ(&user_process_node)))
    return ERRCODE;

  /* Create next version of the user process node. */
  return (strcmp("LOGOUT",func) == 0  ? 
	  SUCCESS : set_userp_attr("CALL", "Function", "Dummy-edit"));
}
#endif	/* COMHISTOFF */

/*  */
/**********************************************************************
 * Function: errcode CH_init(char *app)
 *
 * Creates the command history node (ch_node) and initializes the
 * different attributes and links. This function should only be
 * called once, by the startup function.
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_init(app)
  char *app;
{
#ifdef COMHISTOFF
  return SUCCESS;
#else
  label lbl;
  label sysroot;
  static attrval attr = { 29, "No command has been executed" };
  int pen_save = pen_off;
  errcode value = SUCCESS;

  /* If the ch_node already exist -> do nothing */
  if (exist_ch_node())
    return value;

  pen_off = 1;

  (void) CH_set_application(app == (char *) NULL ? "Unspecified" : app);

  /* Create the 'no command executed'-node */
  /* Create and initialize the ch_node */
  /* Connect the ch_node to the system roots vs molecule */
  if ((ERRCODE = GSR_GETSYSTEMROOT(&sysroot)) ||
      (ERRCODE = CH_CO_CREATEOBJ(&lbl)) ||
      (ERRCODE = CH_SA_SETATTR(&lbl, "CALL", "Function", &attr)) ||
      (ERRCODE = CH_SO_STOREOBJ(&lbl)) ||
      (ERRCODE = CH_CO_CREATEOBJ(&ch_node)) ||
      (ERRCODE = CH_LINK(&ch_node, "CHNET", "Front", &lbl, 1)) ||
      (ERRCODE = CH_SO_STOREOBJ(&ch_node)) ||
      (ERRCODE = UBL_UNBINDLABEL(&ch_node)) ||
      (ERRCODE = LL_sysputlinkitem(sysroot.vs, "CH", "History", 1, &ch_node))){
    pen_off = pen_save;
    return ERRCODE;
  }

  /* Save the system root's history node. */
  LL_moltbl[sysroot.vs].edited = 1;
  value = LL_DBRetry(sysroot.vs, VSS, LL_DBStore);

  pen_off = pen_save;
  return value;
#endif	/* COMHISTOFF */
}

/*  */
/**********************************************************************
 * Function: char * CH_set_application(char *app)
 *
 * Parameters:
 *      char *app
 *
 * Modifications:
 *      <list mods with name and date>
 */
char * CH_set_application(app)
  char *app;
{
  char *prev = application;
  if (application != (char *)NULL)
    (void) free(application);
  application = strdup(app);
  return(prev);
}

/*  */
/**********************************************************************
 * Function: void CH_get_user_process_node(label *lbl)
 *
 * Returns the user_process_node label (in lbl).
 *
 * Parameters:
 *      label *lbl
 *
 * Modifications:
 *      <list mods with name and date>
 */
errcode CH_get_user_process_node(lbl)
  label *lbl;
{
  if (!logged_in)
    return(ERR_NODB);
  lbl->vs = user_process_node.vs;
  lbl->inst = user_process_node.inst;
  return(SUCCESS);
}

#ifndef	COMHISTOFF
/*  */
/**********************************************************************
 * Functions test if the ch_node exist and a function that finds
 * the ch_node.
 **********************************************************************/

/**********************************************************************
 * Function: static errcode exist_ch_node()
 *
 * Tests if the ch_node has been created already
 * If created : TRUE else FALSE
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode exist_ch_node()
{
  label sysroot;
  linkitem item, *tmp;

  tmp = &item;
  return (!GSR_GETSYSTEMROOT(&sysroot) && !LL_sysgetlinkitem(sysroot.vs,
						   "CH", "History", 1, &tmp));
}

/*  */
/**********************************************************************
 * Function: static errcode CH_find_ch_node()
 *
 * Sets the global variabel ch_node. If the ch_node don't exist it
 * returns error.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode CH_find_ch_node()
{
  label sysroot;
  linkitem item, *tmp;

  if ((ERRCODE = GSR_GETSYSTEMROOT(&sysroot)))
    return ERRCODE;
  tmp = &item;

  if ((ERRCODE = LL_sysgetlinkitem(sysroot.vs, "CH", "History", 1, &tmp)))
    return ERRCODE;
  ch_node.vs = tmp->link.vs;
  ch_node.inst = tmp->link.inst;
  LL_moltbl[ch_node.vs].mark = 1; /* Block PEN messages for this node */
  return SUCCESS;
}

/*  */
/**********************************************************************
 *
 * Other functions
 *
 **********************************************************************/

/*  */
/**********************************************************************
 * Function: static errcode CH_bad_lbl(label *lbl)
 *
 * Checks that the label lbl isen't either the ch_node or the
 * user_process_node.
 *
 * Parameters:
 *      label *lbl
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode CH_bad_lbl(lbl)
  label *lbl;
{
  if (!lbl)
    return SUCCESS;
  if (Labelcmp(&ch_node, lbl) || Labelcmp(&user_process_node, lbl))
    return FAIL;
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: static errcode set_userp_attr(char *group, char *field,
 *                                         char *val)
 *
 * Sets/unsets the given attribute in the user_process_node.
 *
 * Parameters:
 *      char *group
 *      char *field
 *      char *val
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode set_userp_attr(group, field, val)
  char *group;
  char *field;
  char *val;
{
  attrval attr;

  if (val) {
    attr.attsize = strlen(val) + 1;
    attr.attvalue = val;
    return CH_SA_SETATTR(&user_process_node, group, field, &attr);
  }
  switch (ERRCODE = CH_RAT_REMOVEATTRTAG(&user_process_node, group, field)) {
  case ERR_FIELD:
  case ERR_GROUP:
    return SUCCESS;
  default:
    return ERRCODE;
  }
}

/*  */
/**********************************************************************
 * Function: static errcode set_userp_link(char *group, char *field,
 *                                          label *lbl)
 *
 * Sets the links in the user_process_node.
 *
 * Parameters:
 *      char  *group
 *      char  *field
 *      label *lbl
 *
 * Modifications:
 *      <list mods with name and date>
 */
static errcode set_userp_link(group, field, lbl)
  char *group, *field;
  label *lbl;
{
  if (lbl) {
    switch ((ERRCODE=CH_RLV_REMOVELINKVAL(&user_process_node,group,field))) {
    case ERR_FIELD:
    case ERR_GROUP:
    case SUCCESS:
      break;
    default:
      return ERRCODE;
    }
    return CH_LINK(&user_process_node, group, field, lbl, 1);
  }
  switch ((ERRCODE=CH_RLT_REMOVELINKTAG(&user_process_node,group,field))) {
  case ERR_FIELD:
  case ERR_GROUP:
  case SUCCESS:
    break;
  default:
    return ERRCODE;
  }
  return SUCCESS;
}
#endif	/* COMHISTOFF */
