/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	showhistory.c
 *
 * SCCSINFO:		@(#)showhistory.c	1.7 6/6/94
 *
 * ORIGINAL AUTHOR(S):  Michael Jansson, 22-Mar-1991
 *
 * MODIFICATIONS:
 *      <list mods with name and date>
 *
 * DESCRIPTION:
 *	This module is used by the sunserv program and the xlincks program.
 *	It assumes that the application using this module is connected to a
 *	Lincks database. It will call the GVS_GETVERSIONS() and build a
 *	representation (a GraphNode structure) that can be displayed by the
 *	ShowTTY() function (in this module too) or by the GraphWidget (in the
 *	graph module).
 */

/******************************************************************************
 * INCLUDES:
 ******************************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <X11/StringDefs.h>

#include "graph.h"
#include "graphutils.h"
#include "liblincks.h"

/**********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 **********************************************************************/
#include "f_showhistory.h"

/**********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 **********************************************************************/
/* None */

/**********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 **********************************************************************/
#include "f_graphutils.h"

/**********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 **********************************************************************/
/* None */

/**********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 **********************************************************************/
struct _node_head {
  struct _node_head *next;
  int flag;
  GNode *node;
};

/**********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 **********************************************************************/
static struct _node_head *findnode P_(( void *value ));
static int histmapfn P_(( char *extra, label *item, int count ));

/**********************************************************************
 * INTERNAL (STATIC) DATA: 
 **********************************************************************/
static struct _node_head *nodes = (struct _node_head *)NULL;
static int load_fragment = 0;

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static struct _node_head *findnode(void *value)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static struct _node_head *findnode(value)
  void *value;
{
  struct _node_head *p;

  for (p = nodes; p; p = p->next)
    if (p->node->node.value == value)
      return p;

  p = (struct _node_head *)malloc(sizeof(struct _node_head));
  p->node = (GNode *)CreateNode((GNode *)NULL, value);
  p->next = nodes;
  p->flag = 0;
  nodes = p;
  return p;
}

/* ^L */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int histmapfn(char *extra, label *item, int count)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int histmapfn(extra,item,count)
  char *extra;
  label *item;
  int count;
{
  static struct _node_head *version;
  struct _node_head *parent;
  
  switch (count) {
  case GVS_MORE:	/* Load more fragments? (1=yes, 0=no) */
    if (load_fragment-- > 0)
      return GVS_NEXT;
    return GVS_STOP;
  case GVS_INIT:	/* Initialisation call */
    break;
  case GVS_VERSION:	/* Version call (happens first) */
    version = findnode((void *)(long)item->inst);
    break;
  case GVS_COMMAND:	/* Command call */
    break;
  case GVS_PREDECESSOR:	/* Predecessor call */
    parent = findnode((void *)(long)item->inst);
    version->flag = 1;
    (void)AddChild(parent->node,version->node);
    break;
  default:
    return 1;
  }
  return GVS_CONTINUE; /* Continue mapping */
}

/*  */
/**********************************************************************
 * Function: int BuildHistory(GNode *root, label *lbl,int flag)
 * 
 * Builds a graph below *root for the history of object *lbl. 'flag'
 * tells how many new hidden history fragments to reveal.
 *
 * Modifications:
 *      <list mods with name and date>
 */
int BuildHistory(root, lbl, flag)
  GNode *root;
  label *lbl;
  int flag;
{
  struct _node_head *p;
  GNode *node = root;
  load_fragment = flag;

  if (GVS_GETVERSIONS(lbl, histmapfn, (void *)NULL)) {
    (void)fprintf(stderr, "BuildHistory: GVS_GETVERSIONS returned error\n");
  }
  if (load_fragment < 0) {
    node = (GNode *)CreateNode((GNode *)NULL, (void *)NULL);
    (void)AddChild(root,node);
  }
  for (p = nodes; p; p = nodes) {
    nodes = p->next;
    if (!p->flag)
      (void)AddChild(node,p->node);
    (void)free((FREEPTR *)p);
  }
  return TRUE;
}

/*  */
/**********************************************************************
 * Function: void ShowGraphTTY(FILE *file, GNode *node, int level)
 * 
 * Traverse a GNode structure
 * and print it on stdout.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void ShowGraphTTY(file, node, level)
  FILE *file;
  GNode *node;
  int level;
{
  int i, j;

  if (!(node->node.flags & GNF_TOUCHED)) {
    /* Don't print nodes more than once. */
    node->node.flags |= GNF_TOUCHED;
    if (!node->used) {
      /* No children left on this branch. */
      (void)fprintf(file,"%ld.\n", (long)node->node.value);

    } else if (node->used>1) {
      /* We have a several children (a new branch). */
      (void)fprintf(file,"(%ld)\n", (long)node->node.value);
      for (i=0; node->node.children[i]; (i)++) {
	for (j=0; j<level; j++) (void)fprintf(file,"   ");
	(void)fprintf(file,"%ld > ", (long)node->node.value);
	ShowGraphTTY(file, (GNode *)node->node.children[i], level+1);
      }

    } else {
      /* We have a linear history. */
      (void)fprintf(file,"%ld > ", (long)node->node.value);
      ShowGraphTTY(file, (GNode *)node->node.children[0], level);
    }
  } else {
    /* We have already been to this node! */
    (void)fprintf(file,"[%ld]\n", (long)node->node.value);
  }
}
